import React from "react";
import { useParams } from "react-router-dom";
import {
  CheckCircle,
  Phone,
  Mail,
  TrendingUp,
  FileText,
  Shield,
  BarChart3,
  Target,
  MapPin,
  Users,
  Zap,
  Award,
  Calendar,
  DollarSign,
  Building,
  Activity,
  Briefcase,
  Star,
} from "lucide-react";
import { Button } from "../components/ui/button";

interface AdvisoryContent {
  title: string;
  description: string;
  features: string[];
  deliverables: string[];
  timeline: string;
  pricing?: string;
  icon: React.ReactNode;
}

const advisoryContent: Record<string, AdvisoryContent> = {
  "custom-report": {
    title: "Custom Advisory Report",
    description:
      "Personalized consultation tool providing custom insights based on user preferences, budget, goals, and locality interest with comprehensive risk evaluation and market analysis.",
    features: [
      "Personalized Property Recommendations based on budget and purpose",
      "Buyer's Risk Score Evaluation across financial, legal, builder, and market risks",
      "Best Time to Buy Analysis with seasonal trends and price forecasts",
      "Negotiation Guidance with ideal margins and market comparisons",
      "Financial Planning with EMI calculators and tax benefits",
      "Custom Property Comparisons with detailed pros & cons analysis",
      "Visual Insights with advisor icons and decision flowcharts",
      "Investment window optimization and demand consistency analysis",
    ],
    deliverables: [
      "Personalized property recommendations matching your profile",
      "Comprehensive risk assessment with score meters for all risk categories",
      "Market timing analysis with seasonal trends and price forecasts",
      "Negotiation strategy guide with builder offers and discount insights",
      "Complete financial planning toolkit with loan comparisons",
      "Side-by-side property comparison with value-for-money rankings",
      "Visual decision support with flowcharts and advisor recommendations",
    ],
    timeline: "7-10 business days",
    pricing: "Starting from ₹12,999",
    icon: <FileText className="w-8 h-8 text-[#0056D2]" />,
  },
};

export function AdvisoryPageCustom() {
  const { advisoryId } = useParams<{ advisoryId: string }>();
  const advisory = advisoryContent[advisoryId || ""];

  if (!advisory) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">
            Advisory Service Not Found
          </h1>
          <p className="text-gray-600">
            The requested advisory service could not be found.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-[#0056D2] to-[#003d9a] text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <div className="flex justify-center mb-4">{advisory.icon}</div>
            <h1 className="text-3xl md:text-4xl font-bold mb-4">
              {advisory.title}
            </h1>
            <p className="text-lg md:text-xl text-blue-100 max-w-4xl mx-auto leading-relaxed">
              {advisory.description}
            </p>
            <div className="mt-6 flex flex-col sm:flex-row gap-3 justify-center">
              {advisory.pricing && (
                <span className="inline-flex items-center bg-white/20 text-white px-5 py-2 rounded-full text-base font-semibold">
                  <DollarSign className="w-4 h-4 mr-2" />
                  {advisory.pricing}
                </span>
              )}
              <span className="inline-flex items-center bg-white/20 text-white px-5 py-2 rounded-full text-base font-semibold">
                <Calendar className="w-4 h-4 mr-2" />
                Delivery: {advisory.timeline}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Custom Advisory Report Detailed Sections */}
      {advisoryId === "custom-report" && (
        <>
          {/* Personalized Advisory Dashboard */}
          <div className="bg-gradient-to-br from-indigo-900 via-purple-900 to-pink-900 text-white py-16 relative overflow-hidden">
            <div className="absolute inset-0">
              <div className="absolute inset-0 bg-gradient-to-r from-indigo-500/10 to-pink-500/10"></div>
              <div className="absolute top-0 left-0 w-full h-full opacity-20">
                <svg className="w-full h-full" viewBox="0 0 100 100" preserveAspectRatio="none">
                  <circle cx="20" cy="20" r="2" fill="currentColor" opacity="0.3" />
                  <circle cx="80" cy="30" r="1.5" fill="currentColor" opacity="0.4" />
                  <circle cx="60" cy="70" r="2.5" fill="currentColor" opacity="0.2" />
                  <circle cx="30" cy="80" r="1" fill="currentColor" opacity="0.5" />
                </svg>
              </div>
            </div>
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
              <div className="text-center">
                <div className="flex items-center justify-center mb-8">
                  <div className="bg-white/10 p-6 rounded-3xl mr-6 backdrop-blur border border-white/20">
                    <Target className="w-20 h-20 text-white" />
                  </div>
                  <div>
                    <h2 className="text-5xl font-bold mb-4">Custom Advisory Intelligence</h2>
                    <p className="text-2xl text-indigo-200">Personalized Property Consultation & Risk Assessment</p>
                  </div>
                </div>
                
                <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mt-12">
                  <div className="bg-white/10 p-6 rounded-2xl backdrop-blur border border-white/20">
                    <div className="text-3xl font-bold text-indigo-200 mb-2">360°</div>
                    <div className="text-sm text-indigo-300">Personal Analysis</div>
                  </div>
                  <div className="bg-white/10 p-6 rounded-2xl backdrop-blur border border-white/20">
                    <div className="text-3xl font-bold text-indigo-200 mb-2">AI</div>
                    <div className="text-sm text-indigo-300">Smart Matching</div>
                  </div>
                  <div className="bg-white/10 p-6 rounded-2xl backdrop-blur border border-white/20">
                    <div className="text-3xl font-bold text-indigo-200 mb-2">Risk</div>
                    <div className="text-sm text-indigo-300">Score Evaluation</div>
                  </div>
                  <div className="bg-white/10 p-6 rounded-2xl backdrop-blur border border-white/20">
                    <div className="text-3xl font-bold text-indigo-200 mb-2">Custom</div>
                    <div className="text-sm text-indigo-300">Recommendations</div>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Advisory Modules */}
          <div className="bg-gradient-to-br from-slate-50 to-indigo-50 py-20">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
              <div className="space-y-20">
                
                {/* 1. Personalized Property Recommendations */}
                <div className="bg-white rounded-3xl shadow-2xl overflow-hidden border border-indigo-200">
                  <div className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white p-10">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center">
                        <div className="bg-indigo-500 p-4 rounded-2xl mr-6">
                          <Building className="w-12 h-12" />
                        </div>
                        <div>
                          <span className="text-indigo-200 font-semibold text-sm uppercase tracking-wide">Smart Matching</span>
                          <h3 className="text-3xl font-bold mb-2">Personalized Property Recommendations</h3>
                          <p className="text-indigo-100">AI-powered property matching based on your profile, budget, and preferences</p>
                        </div>
                      </div>
                      <div className="bg-indigo-500 px-6 py-3 rounded-full">
                        <span className="font-bold text-lg">MODULE 1</span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="p-10">
                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-10">
                      <div className="lg:col-span-2">
                        <div className="mb-8">
                          <h4 className="text-2xl font-bold text-gray-900 mb-6">Your Personalized Profile</h4>
                          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div className="bg-indigo-50 p-6 rounded-xl">
                              <h5 className="font-bold text-gray-900 mb-4 flex items-center">
                                <DollarSign className="w-5 h-5 text-indigo-600 mr-2" />
                                Budget Analysis
                              </h5>
                              <div className="space-y-3">
                                <div className="flex justify-between">
                                  <span className="text-gray-700">Budget Range</span>
                                  <span className="font-bold text-indigo-600">₹80L - ₹1.2Cr</span>
                                </div>
                                <div className="flex justify-between">
                                  <span className="text-gray-700">Down Payment</span>
                                  <span className="font-bold text-indigo-600">₹25L (25%)</span>
                                </div>
                                <div className="flex justify-between">
                                  <span className="text-gray-700">EMI Capacity</span>
                                  <span className="font-bold text-indigo-600">₹65,000/month</span>
                                </div>
                              </div>
                            </div>
                            
                            <div className="bg-indigo-50 p-6 rounded-xl">
                              <h5 className="font-bold text-gray-900 mb-4 flex items-center">
                                <Target className="w-5 h-5 text-indigo-600 mr-2" />
                                Investment Purpose
                              </h5>
                              <div className="space-y-3">
                                <div className="flex items-center">
                                  <CheckCircle className="w-5 h-5 text-green-500 mr-3" />
                                  <span className="text-gray-700">Self-use (Primary)</span>
                                </div>
                                <div className="flex items-center">
                                  <Activity className="w-5 h-5 text-yellow-500 mr-3" />
                                  <span className="text-gray-700">Investment (Secondary)</span>
                                </div>
                                <div className="flex items-center">
                                  <MapPin className="w-5 h-5 text-blue-500 mr-3" />
                                  <span className="text-gray-700">Preferred: Whitefield, Electronic City</span>
                                </div>
                              </div>
                            </div>
                          </div>
                        </div>
                        
                        <div className="bg-gradient-to-r from-indigo-500 to-purple-600 text-white p-8 rounded-2xl">
                          <h4 className="text-xl font-bold mb-6">Top 3 Recommended Properties</h4>
                          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div className="bg-white/10 p-4 rounded-xl backdrop-blur">
                              <div className="text-lg font-bold mb-2">Prestige Lakeside</div>
                              <div className="text-indigo-200 text-sm mb-3">Whitefield • 3BHK • ₹95L</div>
                              <div className="text-2xl font-bold text-green-300">96% Match</div>
                            </div>
                            <div className="bg-white/10 p-4 rounded-xl backdrop-blur">
                              <div className="text-lg font-bold mb-2">Brigade Cornerstone</div>
                              <div className="text-indigo-200 text-sm mb-3">Electronic City • 2BHK • ₹85L</div>
                              <div className="text-2xl font-bold text-green-300">94% Match</div>
                            </div>
                            <div className="bg-white/10 p-4 rounded-xl backdrop-blur">
                              <div className="text-lg font-bold mb-2">Sobha Dream Gardens</div>
                              <div className="text-indigo-200 text-sm mb-3">Whitefield • 3BHK • ₹1.1Cr</div>
                              <div className="text-2xl font-bold text-yellow-300">89% Match</div>
                            </div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-gradient-to-br from-slate-100 to-indigo-100 p-8 rounded-2xl">
                        <img
                          src="https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400&h=300&fit=crop"
                          alt="Property Recommendations"
                          className="w-full h-48 object-cover rounded-xl mb-6 shadow-lg"
                        />
                        <div className="space-y-4">
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-gray-900 mb-2">Matching Algorithm</h5>
                            <div className="space-y-2">
                              <div className="flex justify-between text-sm">
                                <span>Budget Fit</span>
                                <span className="font-bold text-green-600">98%</span>
                              </div>
                              <div className="flex justify-between text-sm">
                                <span>Location Preference</span>
                                <span className="font-bold text-green-600">95%</span>
                              </div>
                              <div className="flex justify-between text-sm">
                                <span>Amenity Match</span>
                                <span className="font-bold text-blue-600">87%</span>
                              </div>
                            </div>
                          </div>
                          
                          <div className="bg-indigo-500 text-white p-4 rounded-lg text-center">
                            <div className="text-2xl font-bold">96%</div>
                            <div className="text-indigo-200 text-sm">Overall Match Score</div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>

                {/* 2. Buyer's Risk Score Evaluation */}
                <div className="bg-white rounded-3xl shadow-2xl p-12 border border-red-200">
                  <div className="text-center mb-12">
                    <div className="flex items-center justify-center mb-6">
                      <div className="bg-red-100 p-4 rounded-2xl mr-6">
                        <Shield className="w-12 h-12 text-red-600" />
                      </div>
                      <div>
                        <h3 className="text-3xl font-bold text-gray-900">Buyer's Risk Score Evaluation</h3>
                        <p className="text-gray-600 mt-2">Comprehensive risk assessment across financial, legal, builder, and market factors</p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
                    <div className="bg-gradient-to-br from-green-50 to-emerald-100 p-8 rounded-2xl text-center border border-green-200">
                      <div className="bg-green-500 w-20 h-20 rounded-full mx-auto mb-6 flex items-center justify-center">
                        <DollarSign className="w-10 h-10 text-white" />
                      </div>
                      <h4 className="text-xl font-bold text-gray-900 mb-4">Financial Risk</h4>
                      <div className="relative mb-4">
                        <div className="w-24 h-24 mx-auto">
                          <svg className="w-24 h-24 transform -rotate-90" viewBox="0 0 36 36">
                            <path className="text-gray-300" strokeDasharray="100, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                            <path className="text-green-500" strokeDasharray="85, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                          </svg>
                          <div className="absolute inset-0 flex items-center justify-center">
                            <span className="text-2xl font-bold text-green-600">8.5</span>
                          </div>
                        </div>
                      </div>
                      <div className="text-sm text-gray-600 space-y-1">
                        <div>• EMI-to-income ratio: 35%</div>
                        <div>• Credit score: 780</div>
                        <div>• Debt obligations: Low</div>
                      </div>
                      <div className="mt-4 bg-green-100 text-green-800 px-3 py-1 rounded-full text-sm font-semibold">Low Risk</div>
                    </div>
                    
                    <div className="bg-gradient-to-br from-blue-50 to-cyan-100 p-8 rounded-2xl text-center border border-blue-200">
                      <div className="bg-blue-500 w-20 h-20 rounded-full mx-auto mb-6 flex items-center justify-center">
                        <FileText className="w-10 h-10 text-white" />
                      </div>
                      <h4 className="text-xl font-bold text-gray-900 mb-4">Legal Risk</h4>
                      <div className="relative mb-4">
                        <div className="w-24 h-24 mx-auto">
                          <svg className="w-24 h-24 transform -rotate-90" viewBox="0 0 36 36">
                            <path className="text-gray-300" strokeDasharray="100, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                            <path className="text-blue-500" strokeDasharray="75, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                          </svg>
                          <div className="absolute inset-0 flex items-center justify-center">
                            <span className="text-2xl font-bold text-blue-600">7.5</span>
                          </div>
                        </div>
                      </div>
                      <div className="text-sm text-gray-600 space-y-1">
                        <div>• Title verification: Clear</div>
                        <div>• RERA compliance: Yes</div>
                        <div>• Pending approvals: 2</div>
                      </div>
                      <div className="mt-4 bg-yellow-100 text-yellow-800 px-3 py-1 rounded-full text-sm font-semibold">Medium Risk</div>
                    </div>
                    
                    <div className="bg-gradient-to-br from-purple-50 to-pink-100 p-8 rounded-2xl text-center border border-purple-200">
                      <div className="bg-purple-500 w-20 h-20 rounded-full mx-auto mb-6 flex items-center justify-center">
                        <Building className="w-10 h-10 text-white" />
                      </div>
                      <h4 className="text-xl font-bold text-gray-900 mb-4">Builder Risk</h4>
                      <div className="relative mb-4">
                        <div className="w-24 h-24 mx-auto">
                          <svg className="w-24 h-24 transform -rotate-90" viewBox="0 0 36 36">
                            <path className="text-gray-300" strokeDasharray="100, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                            <path className="text-purple-500" strokeDasharray="82, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                          </svg>
                          <div className="absolute inset-0 flex items-center justify-center">
                            <span className="text-2xl font-bold text-purple-600">8.2</span>
                          </div>
                        </div>
                      </div>
                      <div className="text-sm text-gray-600 space-y-1">
                        <div>• Delivery record: 78%</div>
                        <div>• Quality rating: A-</div>
                        <div>• Financial stability: Good</div>
                      </div>
                      <div className="mt-4 bg-green-100 text-green-800 px-3 py-1 rounded-full text-sm font-semibold">Low Risk</div>
                    </div>
                    
                    <div className="bg-gradient-to-br from-orange-50 to-red-100 p-8 rounded-2xl text-center border border-orange-200">
                      <div className="bg-orange-500 w-20 h-20 rounded-full mx-auto mb-6 flex items-center justify-center">
                        <TrendingUp className="w-10 h-10 text-white" />
                      </div>
                      <h4 className="text-xl font-bold text-gray-900 mb-4">Market Risk</h4>
                      <div className="relative mb-4">
                        <div className="w-24 h-24 mx-auto">
                          <svg className="w-24 h-24 transform -rotate-90" viewBox="0 0 36 36">
                            <path className="text-gray-300" strokeDasharray="100, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                            <path className="text-orange-500" strokeDasharray="70, 100" strokeDashoffset="0" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831" fill="none" stroke="currentColor" strokeWidth="3"/>
                          </svg>
                          <div className="absolute inset-0 flex items-center justify-center">
                            <span className="text-2xl font-bold text-orange-600">7.0</span>
                          </div>
                        </div>
                      </div>
                      <div className="text-sm text-gray-600 space-y-1">
                        <div>• Price volatility: ±6%</div>
                        <div>• Demand consistency: Good</div>
                        <div>• Liquidity: Medium</div>
                      </div>
                      <div className="mt-4 bg-yellow-100 text-yellow-800 px-3 py-1 rounded-full text-sm font-semibold">Medium Risk</div>
                    </div>
                  </div>
                  
                  <div className="mt-12 bg-gradient-to-r from-indigo-600 to-purple-600 text-white p-8 rounded-2xl text-center">
                    <h4 className="text-2xl font-bold mb-4">Overall Risk Assessment</h4>
                    <div className="text-5xl font-bold mb-4">7.8/10</div>
                    <div className="text-xl font-semibold text-indigo-200 mb-2">MODERATE RISK PROFILE</div>
                    <div className="text-indigo-100">Suitable for investment with standard due diligence and precautions</div>
                  </div>
                </div>

                {/* 3. Best Time to Buy Analysis */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                  <div className="bg-white rounded-3xl shadow-xl p-10 border border-emerald-200">
                    <div className="flex items-center mb-8">
                      <div className="bg-emerald-100 p-4 rounded-2xl mr-6">
                        <Calendar className="w-10 h-10 text-emerald-600" />
                      </div>
                      <div>
                        <h3 className="text-2xl font-bold text-gray-900">Best Time to Buy Analysis</h3>
                        <p className="text-gray-600">Seasonal trends and optimal investment timing</p>
                      </div>
                    </div>
                    
                    <div className="space-y-6">
                      <div className="bg-emerald-50 p-6 rounded-xl">
                        <h4 className="font-bold text-gray-900 mb-4">Seasonal Market Trends</h4>
                        <div className="space-y-3">
                          <div className="flex justify-between items-center">
                            <span className="text-gray-700">Q1 (Jan-Mar)</span>
                            <div className="flex items-center">
                              <div className="w-20 bg-gray-200 rounded-full h-2 mr-3">
                                <div className="bg-green-500 h-2 rounded-full" style={{width: '85%'}}></div>
                              </div>
                              <span className="text-green-600 font-bold">Best</span>
                            </div>
                          </div>
                          <div className="flex justify-between items-center">
                            <span className="text-gray-700">Q2 (Apr-Jun)</span>
                            <div className="flex items-center">
                              <div className="w-20 bg-gray-200 rounded-full h-2 mr-3">
                                <div className="bg-yellow-500 h-2 rounded-full" style={{width: '65%'}}></div>
                              </div>
                              <span className="text-yellow-600 font-bold">Good</span>
                            </div>
                          </div>
                          <div className="flex justify-between items-center">
                            <span className="text-gray-700">Q3 (Jul-Sep)</span>
                            <div className="flex items-center">
                              <div className="w-20 bg-gray-200 rounded-full h-2 mr-3">
                                <div className="bg-red-500 h-2 rounded-full" style={{width: '45%'}}></div>
                              </div>
                              <span className="text-red-600 font-bold">Avoid</span>
                            </div>
                          </div>
                          <div className="flex justify-between items-center">
                            <span className="text-gray-700">Q4 (Oct-Dec)</span>
                            <div className="flex items-center">
                              <div className="w-20 bg-gray-200 rounded-full h-2 mr-3">
                                <div className="bg-blue-500 h-2 rounded-full" style={{width: '75%'}}></div>
                              </div>
                              <span className="text-blue-600 font-bold">Good</span>
                            </div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-emerald-50 p-6 rounded-xl">
                        <h4 className="font-bold text-gray-900 mb-4">Price Forecast</h4>
                        <div className="grid grid-cols-2 gap-4">
                          <div className="text-center p-3 bg-white rounded-lg">
                            <div className="text-lg font-bold text-emerald-600">+8-12%</div>
                            <div className="text-sm text-gray-600">Next 6 months</div>
                          </div>
                          <div className="text-center p-3 bg-white rounded-lg">
                            <div className="text-lg font-bold text-emerald-600">+25-30%</div>
                            <div className="text-sm text-gray-600">Next 2 years</div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-gradient-to-r from-emerald-500 to-teal-600 text-white p-6 rounded-xl text-center">
                        <h4 className="font-bold mb-2">Optimal Investment Window</h4>
                        <div className="text-2xl font-bold">January - March 2024</div>
                        <div className="text-emerald-100 text-sm">Maximum negotiation leverage</div>
                      </div>
                    </div>
                  </div>
                  
                  <div className="bg-white rounded-3xl shadow-xl p-10 border border-blue-200">
                    <div className="flex items-center mb-8">
                      <div className="bg-blue-100 p-4 rounded-2xl mr-6">
                        <Target className="w-10 h-10 text-blue-600" />
                      </div>
                      <div>
                        <h3 className="text-2xl font-bold text-gray-900">Negotiation Guidance</h3>
                        <p className="text-gray-600">Strategic pricing and deal optimization</p>
                      </div>
                    </div>
                    
                    <div className="space-y-6">
                      <img
                        src="https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=500&h=200&fit=crop"
                        alt="Negotiation Strategy"
                        className="w-full h-40 object-cover rounded-xl"
                      />
                      
                      <div className="bg-blue-50 p-6 rounded-xl">
                        <h4 className="font-bold text-gray-900 mb-4">Negotiation Margins</h4>
                        <div className="space-y-3">
                          <div className="flex justify-between">
                            <span className="text-gray-700">Listed Price</span>
                            <span className="font-bold text-gray-900">₹95,00,000</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-700">Market Rate</span>
                            <span className="font-bold text-blue-600">₹92,50,000</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-700">Ideal Target</span>
                            <span className="font-bold text-green-600">₹89,00,000</span>
                          </div>
                          <div className="flex justify-between border-t pt-2">
                            <span className="font-bold text-gray-900">Potential Savings</span>
                            <span className="font-bold text-green-600">₹6,00,000</span>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-blue-50 p-6 rounded-xl">
                        <h4 className="font-bold text-gray-900 mb-4">Builder Offers & Discounts</h4>
                        <ul className="space-y-2 text-gray-700">
                          <li className="flex items-center"><CheckCircle className="w-4 h-4 text-green-500 mr-2" />Free car parking worth ₹2L</li>
                          <li className="flex items-center"><CheckCircle className="w-4 h-4 text-green-500 mr-2" />Modular kitchen upgrade</li>
                          <li className="flex items-center"><CheckCircle className="w-4 h-4 text-green-500 mr-2" />2-year maintenance free</li>
                          <li className="flex items-center"><Activity className="w-4 h-4 text-yellow-500 mr-2" />Flexible payment plan available</li>
                        </ul>
                      </div>
                    </div>
                  </div>
                </div>

                {/* 4. Financial Planning */}
                <div className="bg-white rounded-3xl shadow-2xl p-12 border border-amber-200">
                  <div className="text-center mb-12">
                    <div className="flex items-center justify-center mb-6">
                      <div className="bg-amber-100 p-4 rounded-2xl mr-6">
                        <DollarSign className="w-12 h-12 text-amber-600" />
                      </div>
                      <div>
                        <h3 className="text-3xl font-bold text-gray-900">Comprehensive Financial Planning</h3>
                        <p className="text-gray-600 mt-2">Complete financial toolkit with EMI calculators and tax optimization</p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                    <div>
                      <h4 className="text-2xl font-bold text-gray-900 mb-6">Down Payment Planning</h4>
                      <div className="space-y-6">
                        <div className="bg-amber-50 p-6 rounded-xl">
                          <h5 className="font-bold text-gray-900 mb-4">Payment Structure</h5>
                          <div className="space-y-3">
                            <div className="flex justify-between">
                              <span className="text-gray-700">Property Value</span>
                              <span className="font-bold text-gray-900">₹89,00,000</span>
                            </div>
                            <div className="flex justify-between">
                              <span className="text-gray-700">Down Payment (25%)</span>
                              <span className="font-bold text-amber-600">₹22,25,000</span>
                            </div>
                            <div className="flex justify-between">
                              <span className="text-gray-700">Loan Amount</span>
                              <span className="font-bold text-blue-600">₹66,75,000</span>
                            </div>
                            <div className="flex justify-between">
                              <span className="text-gray-700">Registration & Charges</span>
                              <span className="font-bold text-red-600">₹5,34,000</span>
                            </div>
                            <div className="flex justify-between border-t pt-2">
                              <span className="font-bold text-gray-900">Total Initial Cost</span>
                              <span className="font-bold text-amber-600">₹27,59,000</span>
                            </div>
                          </div>
                        </div>
                        
                        <div className="bg-amber-50 p-6 rounded-xl">
                          <h5 className="font-bold text-gray-900 mb-4">EMI Affordability Calculator</h5>
                          <div className="grid grid-cols-2 gap-4">
                            <div className="text-center p-3 bg-white rounded-lg">
                              <div className="text-lg font-bold text-amber-600">₹58,450</div>
                              <div className="text-sm text-gray-600">Monthly EMI</div>
                            </div>
                            <div className="text-center p-3 bg-white rounded-lg">
                              <div className="text-lg font-bold text-amber-600">20 Years</div>
                              <div className="text-sm text-gray-600">Loan Tenure</div>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div>
                      <h4 className="text-2xl font-bold text-gray-900 mb-6">Loan Interest Comparison</h4>
                      <div className="space-y-6">
                        <div className="bg-amber-50 p-6 rounded-xl">
                          <h5 className="font-bold text-gray-900 mb-4">Bank Comparison</h5>
                          <div className="space-y-4">
                            <div className="flex justify-between items-center p-3 bg-white rounded-lg">
                              <div>
                                <div className="font-semibold text-gray-900">SBI Home Loan</div>
                                <div className="text-sm text-gray-600">Processing: ₹25,000</div>
                              </div>
                              <div className="text-right">
                                <div className="text-lg font-bold text-green-600">8.50%</div>
                                <div className="text-sm text-gray-600">Interest Rate</div>
                              </div>
                            </div>
                            
                            <div className="flex justify-between items-center p-3 bg-white rounded-lg">
                              <div>
                                <div className="font-semibold text-gray-900">HDFC Home Loan</div>
                                <div className="text-sm text-gray-600">Processing: ₹30,000</div>
                              </div>
                              <div className="text-right">
                                <div className="text-lg font-bold text-blue-600">8.75%</div>
                                <div className="text-sm text-gray-600">Interest Rate</div>
                              </div>
                            </div>
                            
                            <div className="flex justify-between items-center p-3 bg-white rounded-lg">
                              <div>
                                <div className="font-semibold text-gray-900">ICICI Home Loan</div>
                                <div className="text-sm text-gray-600">Processing: ₹35,000</div>
                              </div>
                              <div className="text-right">
                                <div className="text-lg font-bold text-yellow-600">8.90%</div>
                                <div className="text-sm text-gray-600">Interest Rate</div>
                              </div>
                            </div>
                          </div>
                        </div>
                        
                        <div className="bg-gradient-to-r from-amber-500 to-orange-600 text-white p-6 rounded-xl">
                          <h5 className="font-bold mb-4">Tax Benefits (Section 80C, 24b)</h5>
                          <div className="grid grid-cols-2 gap-4">
                            <div className="text-center">
                              <div className="text-2xl font-bold">₹1.5L</div>
                              <div className="text-amber-100 text-sm">Principal (80C)</div>
                            </div>
                            <div className="text-center">
                              <div className="text-2xl font-bold">₹2L</div>
                              <div className="text-amber-100 text-sm">Interest (24b)</div>
                            </div>
                          </div>
                          <div className="text-center mt-4">
                            <div className="text-lg font-bold">Annual Tax Savings: ₹1,05,000</div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>

                {/* 5. Custom Property Comparisons */}
                <div className="bg-white rounded-3xl shadow-2xl overflow-hidden border border-teal-200">
                  <div className="bg-gradient-to-r from-teal-600 to-cyan-600 text-white p-10">
                    <h3 className="text-3xl font-bold mb-4">Custom Property Comparisons</h3>
                    <p className="text-teal-100 text-lg">Side-by-side analysis of your shortlisted properties with detailed pros & cons</p>
                  </div>
                  
                  <div className="p-10">
                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                      <div className="bg-teal-50 p-6 rounded-2xl border border-teal-200">
                        <img
                          src="https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400&h=200&fit=crop"
                          alt="Property 1"
                          className="w-full h-32 object-cover rounded-xl mb-4"
                        />
                        <h4 className="text-xl font-bold text-gray-900 mb-4">Prestige Lakeside Habitat</h4>
                        
                        <div className="space-y-4">
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-green-700 mb-2">✓ PROS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Best price per sq ft</li>
                              <li>• Excellent connectivity</li>
                              <li>• Reputed builder</li>
                              <li>• Ready to move</li>
                            </ul>
                          </div>
                          
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-red-700 mb-2">✗ CONS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Limited amenities</li>
                              <li>• Older construction</li>
                              <li>• No club house</li>
                            </ul>
                          </div>
                          
                          <div className="bg-teal-500 text-white p-4 rounded-lg text-center">
                            <div className="text-2xl font-bold">8.5/10</div>
                            <div className="text-teal-100">Value Score</div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-blue-50 p-6 rounded-2xl border border-blue-200">
                        <img
                          src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?w=400&h=200&fit=crop"
                          alt="Property 2"
                          className="w-full h-32 object-cover rounded-xl mb-4"
                        />
                        <h4 className="text-xl font-bold text-gray-900 mb-4">Brigade Cornerstone Utopia</h4>
                        
                        <div className="space-y-4">
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-green-700 mb-2">✓ PROS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Premium amenities</li>
                              <li>• Modern architecture</li>
                              <li>• Good resale value</li>
                              <li>• Brand reputation</li>
                            </ul>
                          </div>
                          
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-red-700 mb-2">✗ CONS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Higher price point</li>
                              <li>• Under construction</li>
                              <li>• Traffic congestion</li>
                            </ul>
                          </div>
                          
                          <div className="bg-blue-500 text-white p-4 rounded-lg text-center">
                            <div className="text-2xl font-bold">7.8/10</div>
                            <div className="text-blue-100">Value Score</div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-purple-50 p-6 rounded-2xl border border-purple-200">
                        <img
                          src="https://images.unsplash.com/photo-1449824913935-59a10b8d2000?w=400&h=200&fit=crop"
                          alt="Property 3"
                          className="w-full h-32 object-cover rounded-xl mb-4"
                        />
                        <h4 className="text-xl font-bold text-gray-900 mb-4">Sobha Dream Gardens</h4>
                        
                        <div className="space-y-4">
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-green-700 mb-2">✓ PROS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Luxury specifications</li>
                              <li>• Large apartment size</li>
                              <li>• Premium location</li>
                              <li>• High-end amenities</li>
                            </ul>
                          </div>
                          
                          <div className="bg-white p-4 rounded-lg">
                            <h5 className="font-bold text-red-700 mb-2">✗ CONS</h5>
                            <ul className="text-sm text-gray-700 space-y-1">
                              <li>• Exceeds budget</li>
                              <li>• High maintenance</li>
                              <li>• Limited parking</li>
                            </ul>
                          </div>
                          
                          <div className="bg-purple-500 text-white p-4 rounded-lg text-center">
                            <div className="text-2xl font-bold">8.2/10</div>
                            <div className="text-purple-100">Value Score</div>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div className="mt-12 bg-gradient-to-r from-teal-500 to-cyan-600 text-white p-8 rounded-2xl">
                      <h4 className="text-2xl font-bold mb-6 text-center">Value-for-Money Ranking</h4>
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur text-center">
                          <div className="text-4xl font-bold mb-2">🥇</div>
                          <div className="text-xl font-bold">Prestige Lakeside</div>
                          <div className="text-teal-200">Best Overall Value</div>
                        </div>
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur text-center">
                          <div className="text-4xl font-bold mb-2">🥈</div>
                          <div className="text-xl font-bold">Sobha Dream Gardens</div>
                          <div className="text-teal-200">Premium Choice</div>
                        </div>
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur text-center">
                          <div className="text-4xl font-bold mb-2">🥉</div>
                          <div className="text-xl font-bold">Brigade Cornerstone</div>
                          <div className="text-teal-200">Future Potential</div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>

                {/* 6. Visual Insights & Decision Support */}
                <div className="bg-gradient-to-br from-slate-800 to-indigo-900 text-white rounded-3xl shadow-2xl p-12">
                  <div className="text-center mb-12">
                    <div className="flex items-center justify-center mb-6">
                      <div className="bg-white/20 p-6 rounded-2xl mr-6">
                        <BarChart3 className="w-16 h-16" />
                      </div>
                      <div>
                        <h3 className="text-4xl font-bold mb-3">Visual Insights & Decision Flowcharts</h3>
                        <p className="text-indigo-200 text-xl">Smart visual guidance for informed property decisions</p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                    <div>
                      <h4 className="text-2xl font-bold mb-6">Decision Flowchart</h4>
                      <div className="space-y-4">
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur">
                          <div className="flex items-center mb-4">
                            <div className="bg-green-500 w-8 h-8 rounded-full flex items-center justify-center text-white font-bold mr-4">1</div>
                            <h5 className="font-bold">Budget Alignment Check</h5>
                          </div>
                          <div className="ml-12">
                            <div className="text-green-300">✓ Within budget range</div>
                            <div className="text-green-300">✓ EMI affordable</div>
                            <div className="text-green-300">✓ Down payment ready</div>
                          </div>
                        </div>
                        
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur">
                          <div className="flex items-center mb-4">
                            <div className="bg-blue-500 w-8 h-8 rounded-full flex items-center justify-center text-white font-bold mr-4">2</div>
                            <h5 className="font-bold">Risk Assessment</h5>
                          </div>
                          <div className="ml-12">
                            <div className="text-blue-300">✓ Low financial risk</div>
                            <div className="text-yellow-300">⚠ Medium legal risk</div>
                            <div className="text-blue-300">✓ Builder credible</div>
                          </div>
                        </div>
                        
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur">
                          <div className="flex items-center mb-4">
                            <div className="bg-purple-500 w-8 h-8 rounded-full flex items-center justify-center text-white font-bold mr-4">3</div>
                            <h5 className="font-bold">Market Timing</h5>
                          </div>
                          <div className="ml-12">
                            <div className="text-green-300">✓ Optimal buying season</div>
                            <div className="text-green-300">✓ Price appreciation expected</div>
                            <div className="text-green-300">✓ Good negotiation window</div>
                          </div>
                        </div>
                        
                        <div className="bg-gradient-to-r from-green-500 to-emerald-600 p-6 rounded-xl">
                          <div className="text-center">
                            <div className="text-2xl font-bold mb-2">RECOMMENDATION: PROCEED</div>
                            <div className="text-green-100">All conditions favorable for investment</div>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div>
                      <h4 className="text-2xl font-bold mb-6">Advisor Insights</h4>
                      <div className="space-y-6">
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur">
                          <div className="flex items-center mb-4">
                            <Users className="w-8 h-8 text-indigo-300 mr-3" />
                            <h5 className="font-bold">Expert Advisor</h5>
                          </div>
                          <div className="bg-white/10 p-4 rounded-lg">
                            <p className="text-indigo-200 italic">"Based on your profile and market analysis, Prestige Lakeside offers the best value proposition. The timing is optimal for negotiation, and the risk profile aligns with your investment goals."</p>
                          </div>
                        </div>
                        
                        <div className="bg-white/10 p-6 rounded-xl backdrop-blur">
                          <h5 className="font-bold mb-4">Key Success Factors</h5>
                          <div className="space-y-3">
                            <div className="flex items-center">
                              <CheckCircle className="w-5 h-5 text-green-400 mr-3" />
                              <span className="text-indigo-200">Strong financial position</span>
                            </div>
                            <div className="flex items-center">
                              <CheckCircle className="w-5 h-5 text-green-400 mr-3" />
                              <span className="text-indigo-200">Optimal market timing</span>
                            </div>
                            <div className="flex items-center">
                              <CheckCircle className="w-5 h-5 text-green-400 mr-3" />
                              <span className="text-indigo-200">Well-researched choice</span>
                            </div>
                            <div className="flex items-center">
                              <Activity className="w-5 h-5 text-yellow-400 mr-3" />
                              <span className="text-indigo-200">Monitor legal clearances</span>
                            </div>
                          </div>
                        </div>
                        
                        <div className="bg-gradient-to-r from-indigo-500 to-purple-600 p-6 rounded-xl text-center">
                          <h5 className="font-bold mb-3">Confidence Score</h5>
                          <div className="text-4xl font-bold mb-2">92%</div>
                          <div className="text-indigo-200">High confidence in recommendation</div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </>
      )}

      {/* Features Section */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              What's Included
            </h2>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Comprehensive analysis and insights tailored to your specific needs
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {advisory.features.map((feature, index) => (
              <div key={index} className="flex items-start">
                <CheckCircle className="w-6 h-6 text-green-500 mt-1 mr-3 flex-shrink-0" />
                <span className="text-gray-700">{feature}</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Deliverables Section */}
      <div className="bg-gray-50 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              What You'll Receive
            </h2>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Detailed reports and actionable insights delivered to your inbox
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {advisory.deliverables.map((deliverable, index) => (
              <div key={index} className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
                <div className="flex items-start">
                  <FileText className="w-6 h-6 text-[#0056D2] mt-1 mr-3 flex-shrink-0" />
                  <span className="text-gray-700">{deliverable}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-[#0056D2] py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl font-bold text-white mb-4">
            Ready to Get Started?
          </h2>
          <p className="text-xl text-blue-100 mb-8 max-w-2xl mx-auto">
            Get your personalized advisory report and make informed property decisions
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-[#0056D2] hover:bg-gray-100">
              Order Report Now
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-[#0056D2]">
              <Phone className="w-4 h-4 mr-2" />
              Schedule Consultation
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}