import React, { useState } from 'react';
import { ArrowLeft, Camera, Navigation } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import MapLibreMap from '../components/MapLibreMap';

const AerialView: React.FC = () => {
  const navigate = useNavigate();
  const [coordinates, setCoordinates] = useState({ lat: '', lng: '' });
  const [mapCenter, setMapCenter] = useState<{lat: number, lng: number} | null>(null);
  const [clickedCoords, setClickedCoords] = useState<{lat: number, lng: number} | null>(null);
  const [error, setError] = useState('');

  const handleCoordinateSubmit = () => {
    // Clean the input values
    const latStr = coordinates.lat.replace(/[^0-9.-]/g, '');
    const lngStr = coordinates.lng.replace(/[^0-9.-]/g, '');
    
    const lat = parseFloat(latStr);
    const lng = parseFloat(lngStr);

    if (isNaN(lat) || isNaN(lng)) {
      setError('Please enter valid numeric coordinates');
      return;
    }

    if (lat < -90 || lat > 90) {
      setError('Latitude must be between -90 and 90');
      return;
    }

    if (lng < -180 || lng > 180) {
      setError('Longitude must be between -180 and 180');
      return;
    }

    setError('');
    setMapCenter({ lat, lng });
  };

  const handleMapClick = (lat: number, lng: number) => {
    setClickedCoords({ lat, lng });
  };

  const getCurrentLocation = () => {
    if (!navigator.geolocation) {
      setError('Geolocation is not supported by this browser');
      return;
    }

    setError('Getting your location...');
    
    navigator.geolocation.getCurrentPosition(
      (position) => {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        console.log('Got location:', lat, lng);
        setCoordinates({ lat: lat.toString(), lng: lng.toString() });
        setMapCenter({ lat, lng });
        setError('');
      },
      (error) => {
        console.error('Geolocation error:', error);
        setError(`Unable to get location: ${error.message}`);
      },
      {
        enableHighAccuracy: true,
        timeout: 10000,
        maximumAge: 0
      }
    );
  };

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <button
                onClick={() => navigate('/')}
                className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <ArrowLeft className="w-5 h-5" />
              </button>
              <div className="flex items-center space-x-2">
                <Camera className="w-6 h-6 text-purple-600" />
                <h1 className="text-xl font-semibold">Aerial Property View</h1>
              </div>
            </div>
            <div className="text-sm text-gray-600">
              {clickedCoords ? (
                <span>📍 {clickedCoords.lat.toFixed(6)}, {clickedCoords.lng.toFixed(6)}</span>
              ) : (
                <span>Click on map to get coordinates</span>
              )}
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto p-4">
        {/* Coordinate Input Section */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-4">
          <h2 className="text-lg font-semibold mb-4">Enter Property Coordinates</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Latitude
              </label>
              <input
                type="number"
                step="any"
                value={coordinates.lat}
                onChange={(e) => setCoordinates({...coordinates, lat: e.target.value})}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                placeholder="12.9716"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Longitude
              </label>
              <input
                type="number"
                step="any"
                value={coordinates.lng}
                onChange={(e) => setCoordinates({...coordinates, lng: e.target.value})}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                placeholder="77.5946"
              />
            </div>
          </div>
          
          <div className="flex flex-wrap gap-2 mb-4">
            <button
              onClick={handleCoordinateSubmit}
              className="px-4 py-2 bg-purple-600 text-white rounded-md hover:bg-purple-700 transition-colors"
            >
              Show Aerial View
            </button>
            <button
              onClick={getCurrentLocation}
              className="px-4 py-2 bg-green-600 text-black rounded-md hover:bg-green-700 transition-colors flex items-center font-semibold"
            >
              <Navigation className="w-4 h-4 mr-2 text-black" />
              <span className="text-black">Use Current Location</span>
            </button>
          </div>

          {error && (
            <div className="text-red-600 text-sm">
              {error}
            </div>
          )}
        </div>

        {/* Map Section */}
        {mapCenter && (
          <div className="bg-white rounded-lg shadow-sm overflow-hidden mb-4 border-2 border-blue-300">
            <div className="p-2 bg-blue-50 text-sm text-blue-700">
              📍 Showing aerial view for: {mapCenter.lat.toFixed(6)}, {mapCenter.lng.toFixed(6)}
            </div>
            <MapLibreMap
              latitude={mapCenter.lat}
              longitude={mapCenter.lng}
              zoom={18}
              showMarker={true}
              interactive={true}
              onCoordinateChange={handleMapClick}
              className="w-full h-[600px] border-t border-blue-200"
              style={{ minHeight: '600px' }}
            />
          </div>
        )}
        
        <div className="bg-white rounded-lg shadow-sm p-6">
          <h2 className="text-lg font-semibold mb-4">Aerial View Features</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center p-4 bg-purple-50 rounded-lg">
              <Camera className="w-8 h-8 mx-auto mb-2 text-purple-600" />
              <h3 className="font-semibold">4K Satellite Imagery</h3>
              <p className="text-sm text-gray-600">High-resolution aerial views</p>
            </div>
            <div className="text-center p-4 bg-blue-50 rounded-lg">
              <div className="w-8 h-8 mx-auto mb-2 bg-blue-600 rounded-full flex items-center justify-center">
                <span className="text-black text-sm font-bold">360°</span>
              </div>
              <h3 className="font-semibold">Interactive Navigation</h3>
              <p className="text-sm text-gray-600">Explore from every angle</p>
            </div>
            <div className="text-center p-4 bg-green-50 rounded-lg">
              <div className="w-8 h-8 mx-auto mb-2 bg-green-600 rounded-full flex items-center justify-center">
                <span className="text-white text-xs">📍</span>
              </div>
              <h3 className="font-semibold">Precise Location</h3>
              <p className="text-sm text-gray-600">Accurate property mapping</p>
            </div>
          </div>
          
          {clickedCoords && (
            <div className="mt-6 p-4 bg-gray-50 rounded-lg">
              <h3 className="font-semibold mb-2">Selected Location</h3>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-gray-600">Latitude:</span>
                  <span className="ml-2 font-mono">{clickedCoords.lat.toFixed(6)}</span>
                </div>
                <div>
                  <span className="text-gray-600">Longitude:</span>
                  <span className="ml-2 font-mono">{clickedCoords.lng.toFixed(6)}</span>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default AerialView;