import React, { useState } from 'react';

const UNITS = [
  { key: 'sqft', label: 'Square Feet', toSqft: 1 },
  { key: 'sqm', label: 'Square Meter', toSqft: 10.7639 },
  { key: 'sqyd', label: 'Square Yard', toSqft: 9 },
  { key: 'acre', label: 'Acre', toSqft: 43560 },
  { key: 'hectare', label: 'Hectare', toSqft: 107639 },
];

export default function AreaConverter() {
  const [formData, setFormData] = useState({
    value: '1000',
    fromUnit: 'sqft',
    toUnit: 'sqm'
  });

  const [result, setResult] = useState<{
    convertedValue: number;
    fromLabel: string;
    toLabel: string;
  } | null>(null);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const convertArea = () => {
    const value = parseFloat(formData.value);
    if (!value || value <= 0) {
      alert('Please enter a valid area value');
      return;
    }
    if (formData.fromUnit === formData.toUnit) {
      alert('Please select different units for conversion');
      return;
    }

    const fromUnit = UNITS.find(u => u.key === formData.fromUnit)!;
    const toUnit = UNITS.find(u => u.key === formData.toUnit)!;
    const sqft = value * fromUnit.toSqft;
    const converted = sqft / toUnit.toSqft;

    setResult({
      convertedValue: converted,
      fromLabel: fromUnit.label,
      toLabel: toUnit.label
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-4xl mx-auto px-4">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Area Converter</h1>
          <p className="text-gray-600 max-w-3xl mx-auto">
            Convert property measurements between different units for accurate comparison and documentation. 
            Essential for buyers, builders, and agents dealing with properties listed in various measurement units.
          </p>
          <div className="mt-4 p-3 bg-orange-50 rounded-lg inline-block">
            <p className="text-sm text-orange-700 font-medium">📏 Compare properties accurately across different unit systems</p>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-lg p-8">
          <div className="grid md:grid-cols-2 gap-8">
            <div>
              <h2 className="text-xl font-semibold mb-6">Enter Conversion Details</h2>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Value: {parseInt(formData.value).toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="value"
                    min="100"
                    max="10000"
                    step="50"
                    value={formData.value}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>100</span>
                    <span>10,000</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    From Unit
                  </label>
                  <select
                    name="fromUnit"
                    value={formData.fromUnit}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    {UNITS.map(unit => (
                      <option key={unit.key} value={unit.key}>{unit.label}</option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    To Unit
                  </label>
                  <select
                    name="toUnit"
                    value={formData.toUnit}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    {UNITS.map(unit => (
                      <option key={unit.key} value={unit.key}>{unit.label}</option>
                    ))}
                  </select>
                </div>

                <button
                  onClick={convertArea}
                  className="w-full bg-orange-600 text-white py-3 px-4 rounded-md hover:bg-orange-700 transition-colors font-medium flex items-center justify-center gap-2"
                >
                  <span>📏</span>
                  Convert Area
                </button>
              </div>
            </div>

            <div>
              <h2 className="text-xl font-semibold mb-6">Conversion Result</h2>
              
              {result ? (
                <div className="space-y-4">
                  <div className="bg-green-50 border border-green-200 rounded-lg p-6 text-center">
                    <h3 className="text-lg font-semibold text-green-800 mb-4">Converted Value</h3>
                    <p className="text-3xl font-bold text-green-600 mb-2">
                      {result.convertedValue.toLocaleString('en-IN', { maximumFractionDigits: 4 })}
                    </p>
                    <p className="text-sm text-green-700">
                      {formData.value} {result.fromLabel} = {result.convertedValue.toLocaleString('en-IN', { maximumFractionDigits: 4 })} {result.toLabel}
                    </p>
                  </div>

                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-blue-800 mb-2">Quick Reference</h3>
                    <div className="grid grid-cols-2 gap-2 text-sm text-blue-700">
                      <div>1 Sqft = 0.0929 Sqm</div>
                      <div>1 Sqm = 10.764 Sqft</div>
                      <div>1 Acre = 43,560 Sqft</div>
                      <div>1 Hectare = 2.471 Acres</div>
                    </div>
                  </div>

                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-gray-800 mb-2">Common Uses</h3>
                    <ul className="text-sm text-gray-600 space-y-1">
                      <li>• Property size comparison</li>
                      <li>• Construction planning</li>
                      <li>• Real estate documentation</li>
                      <li>• Land measurement verification</li>
                    </ul>
                  </div>
                </div>
              ) : (
                <div className="bg-gray-50 border border-gray-200 rounded-lg p-8 text-center">
                  <p className="text-gray-500">Enter values to convert area units</p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}