import { useMemo } from "react";
import { useParams, useNavigate } from "react-router-dom";
import { Card, CardContent } from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import { Calendar, Clock, ArrowLeft, TrendingUp } from "lucide-react";

const BLOGS: Record<string, any> = {
  "1": {
    title: "Complete Guide to Home Loans in 2026",
    category: "Finance",
    readTime: "8 min read",
    publishedAt: "2 days ago",
    trending: true,
    hero: "https://images.unsplash.com/photo-1664892798972-079f15663b16?auto=format&fit=crop&w=1600&q=80",
    sections: [
      {
        heading: "Overview",
        body:
          "Home loan products have evolved in 2026 with flexible rates, linked benchmarks, and prepayment options. This guide walks you through interest types, eligibility, documents, and ways to lower total interest.",
      },
      {
        heading: "Interest Types",
        body:
          "Floating rates follow repo or MCLR; fixed rates lock for a period. Hybrid products start fixed, then convert to floating. Choose based on risk appetite and market cycle.",
      },
      {
        heading: "How to reduce EMI",
        body:
          "Increase down payment, extend tenure judiciously, improve credit score, and consider a balance transfer when rate gap exceeds 75–100 bps.",
      },
    ],
  },
  "2": {
    title: "Real Estate Investment Strategies for Beginners",
    category: "Investment",
    readTime: "12 min read",
    publishedAt: "1 week ago",
    trending: false,
    hero: "https://images.unsplash.com/photo-1635108199395-8cd24af60af1?auto=format&fit=crop&w=1600&q=80",
    sections: [
      {
        heading: "Start with goals",
        body:
          "Define whether you want cash flow, appreciation, or diversification. Goals drive asset type, city, and holding period.",
      },
      {
        heading: "Due diligence",
        body:
          "Evaluate micro-markets, supply pipeline, developer track record, and rental demand. Run conservative scenarios before committing.",
      },
    ],
  },
  "3": {
    title: "Top 10 Localities in Mumbai for First-time Buyers",
    category: "Location Guide",
    readTime: "6 min read",
    publishedAt: "3 days ago",
    trending: true,
    hero: "https://images.unsplash.com/photo-1754797007288-cfa09b51b056?auto=format&fit=crop&w=1600&q=80",
    sections: [
      {
        heading: "Selection criteria",
        body:
          "Short commute, upcoming infra, social fabric, and affordability. Areas listed balance livability with value.",
      },
    ],
  },
  "4": {
    title: "Legal Checklist for Property Purchase",
    category: "Legal",
    readTime: "10 min read",
    publishedAt: "5 days ago",
    trending: false,
    hero: "https://images.unsplash.com/photo-1668911494481-1643ee3e1235?auto=format&fit=crop&w=1600&q=80",
    sections: [
      {
        heading: "Documents to verify",
        body:
          "Title deed chain, encumbrance certificate, RERA registration, approved plans, possession/OC, and society NOC where applicable.",
      },
      {
        heading: "Pro tips",
        body:
          "Use a lawyer for diligence, prefer escrow payments, and record all deviations in an addendum before agreement signing.",
      },
    ],
  },
};

export default function BlogDetail() {
  const params = useParams();
  const navigate = useNavigate();
  const id = params.id ?? "";
  const blog = useMemo(() => BLOGS[id], [id]);

  if (!blog) {
    return (
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="w-full px-4 sm:px-6 lg:px-8 max-w-5xl mx-auto text-center">
          <h1 className="text-2xl font-semibold text-gray-900 mb-4">Blog not found</h1>
          <Button onClick={() => navigate(-1)} className="bg-[#0056D2] hover:bg-[#0056D2]/90 text-white">Go Back</Button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white">
      <div className="w-full">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          {/* Header without images */}
          <div className="mb-4">
            <div className="flex items-center gap-2 mb-2">
              <Badge className="bg-[#0056D2] text-white text-xs">{blog.category}</Badge>
              {blog.trending && (
                <Badge className="bg-[#00BFA6] text-white text-xs"><TrendingUp className="w-3 h-3 mr-1"/>Trending</Badge>
              )}
            </div>
            <h1 className="text-2xl md:text-3xl font-bold text-gray-900">{blog.title}</h1>
            <div className="mt-1 text-gray-600 text-xs md:text-sm flex items-center gap-4">
              <span className="inline-flex items-center"><Clock className="w-3 h-3 mr-1" />{blog.readTime}</span>
              <span className="inline-flex items-center"><Calendar className="w-3 h-3 mr-1" />{blog.publishedAt}</span>
            </div>
          </div>

          <div className="flex items-center justify-between mb-6">
            <Button variant="outline" onClick={() => navigate(-1)}>
              <ArrowLeft className="w-4 h-4 mr-2" /> Back
            </Button>
          </div>

          <div className="grid grid-cols-1 gap-8">
            {/* Main content spans full width */}
            <div>
              <Card>
                <CardContent className="p-6 md:p-8">
                  {blog.sections.map((s: any, idx: number) => (
                    <div key={idx} className="mb-8">
                      <h2 className="text-2xl font-semibold text-gray-900 mb-2">{s.heading}</h2>
                      <p className="leading-relaxed text-gray-700">{s.body}</p>
                    </div>
                  ))}

                  {/* Add a themed highlight box */}
                  <div className="p-4 rounded-lg bg-[#F5FAFF] border border-[#0056D2]/20 mb-6">
                    <h3 className="font-semibold text-[#0056D2] mb-2">Key Takeaways</h3>
                    <ul className="list-disc list-inside text-gray-700 space-y-1">
                      <li>Follow a clear checklist before committing to a property.</li>
                      <li>Benchmark rates and fees across lenders/developers.</li>
                      <li>Use NAL tools to estimate budget, EMI, and eligibility.</li>
                    </ul>
                  </div>

                  {/* Long-form guidance paragraph for new apartment buyers */}
                  <p className="text-gray-800 leading-8 text-[15px] mb-6">
                    Buying a new apartment should start with clarity on budget and location. Begin by calculating your affordable range using your take-home income, existing obligations, and a realistic down payment. Aim for a total EMI that does not exceed 35–40% of monthly net income. Shortlist localities that balance commute time, social infrastructure (schools, hospitals, markets), and future appreciation drivers such as upcoming metro corridors or expressways. For each shortlisted project, verify developer reputation, RERA registration, stage of construction, and the sanctioned layout/amenities. During site visits, evaluate daylight, cross-ventilation, usable carpet area, tower distance, and maintenance costs. Compare at least three projects side-by-side on price per sq.ft, effective all-in cost (including parking, floor rise, GST, registration), delivery timelines, and payment plans. Once you narrow down a unit, run an EMI scenario at slightly higher rates to stress test affordability, and keep a 6–9 month emergency buffer. Before booking, get a preliminary legal check done, ensure the allotment/homebuyer agreement reflects promised specifications, and use digital payment/escrow options whenever possible. Following these steps keeps the process transparent, reduces risk, and helps you negotiate confidently.
                  </p>

                  <div className="space-y-6">
                    <div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Financing & Eligibility</h3>
                      <p className="text-gray-700 leading-7">Use our Budget Calculator to estimate a safe price band, then validate with the EMI Calculator and Eligibility Check. Target a down payment of 15–25% to lower total interest. If you have existing EMIs, ensure debt-to-income stays under 45% after the home loan.</p>
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Site Visit Checklist</h3>
                      <ul className="list-disc list-inside text-gray-700 leading-7 space-y-1">
                        <li>Measure usable carpet vs. super built-up; confirm balcony/loading.</li>
                        <li>Check sound levels, ventilation, water pressure, and lift count.</li>
                        <li>Ask for sample agreement, maintenance estimate, and handover timeline.</li>
                      </ul>
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Legal Due Diligence</h3>
                      <p className="text-gray-700 leading-7">Verify title chain, EC, RERA number, approvals, and OC/CC as applicable. Match unit number and area in drawings, allotment, and demand letters.</p>
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Negotiation Tips</h3>
                      <p className="text-gray-700 leading-7">Negotiate on total outflow, not just base rate. Seek waiver on floor rise or parking, or additional fittings. Confirm final figure in the agreement before booking.</p>
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Post‑Purchase Prep</h3>
                      <p className="text-gray-700 leading-7">Plan interiors, snag checks, and registration timeline. Maintain a small buffer for moving, furnishings, and initial society deposits.</p>
                    </div>
                  </div>
                  {/* Article meta & actions inline bottom */}
                  <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-4">
                    <Card>
                      <CardContent className="p-4">
                        <h4 className="font-semibold text-gray-900 mb-2">Article Info</h4>
                        <div className="text-sm text-gray-700 space-y-1">
                          <div className="flex items-center"><Clock className="w-3 h-3 mr-2" />{blog.readTime}</div>
                          <div className="flex items-center"><Calendar className="w-3 h-3 mr-2" />{blog.publishedAt}</div>
                          <div>Category: <span className="font-medium">{blog.category}</span></div>
                        </div>
                      </CardContent>
                    </Card>
                    <Card className="md:col-span-2">
                      <CardContent className="p-4">
                        <h4 className="font-semibold text-gray-900 mb-2">Recommended Actions</h4>
                        <ul className="list-disc list-inside text-sm text-gray-700 space-y-1">
                          <li>Use Budget Calculator to set a realistic range.</li>
                          <li>Check Loan Eligibility before site visits.</li>
                          <li>Compare shortlisted properties side-by-side.</li>
                        </ul>
                      </CardContent>
                    </Card>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}


