import React, { useState } from 'react';

export default function BudgetCalculator() {
  const [formData, setFormData] = useState({
    monthlyIncome: '100000',
    existingEmi: '0',
    downPayment: '1000000',
    interestRate: '8.5',
    tenureYears: '20'
  });

  const [result, setResult] = useState<{
    maxEmi: number;
    maxLoan: number;
    propertyBudget: number;
  } | null>(null);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const calculateBudget = () => {
    const income = parseFloat(formData.monthlyIncome);
    const existingEmi = parseFloat(formData.existingEmi);
    const downPayment = parseFloat(formData.downPayment);
    const rate = parseFloat(formData.interestRate);
    const tenure = parseInt(formData.tenureYears);

    if (!income || income <= 0) {
      alert('Please enter a valid monthly income');
      return;
    }
    if (!downPayment || downPayment <= 0) {
      alert('Please enter a valid down payment amount');
      return;
    }
    if (!rate || rate <= 0) {
      alert('Please enter a valid interest rate');
      return;
    }

    const maxEmi = Math.max(0, (income - existingEmi) * 0.4);
    const r = rate / 12 / 100;
    const n = tenure * 12;
    const maxLoan = r > 0 ? (maxEmi * (Math.pow(1 + r, n) - 1)) / (r * Math.pow(1 + r, n)) : 0;
    const ltv = 0.8;
    const propertyBudget = (maxLoan + downPayment) / ltv;

    setResult({
      maxEmi: Math.round(maxEmi),
      maxLoan: Math.round(maxLoan),
      propertyBudget: Math.round(propertyBudget)
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-4xl mx-auto px-4">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Budget Calculator</h1>
          <p className="text-gray-600 max-w-3xl mx-auto">
            Determine your realistic property affordability range based on income, expenses, and down payment capacity. 
            Prevents financial overcommitment by calculating maximum EMI, eligible loan amount, and ideal property budget.
          </p>
          <div className="mt-4 p-3 bg-purple-50 rounded-lg inline-block">
            <p className="text-sm text-purple-700 font-medium">💡 Get your affordable price range before property hunting</p>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-lg p-8">
          <div className="grid md:grid-cols-2 gap-8">
            <div>
              <h2 className="text-xl font-semibold mb-6">Enter Your Details</h2>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Monthly Income: ₹{parseInt(formData.monthlyIncome).toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="monthlyIncome"
                    min="25000"
                    max="500000"
                    step="5000"
                    value={formData.monthlyIncome}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹25K</span>
                    <span>₹5L</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Existing EMI: ₹{parseInt(formData.existingEmi).toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="existingEmi"
                    min="0"
                    max="100000"
                    step="1000"
                    value={formData.existingEmi}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹0</span>
                    <span>₹1L</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Down Payment: ₹{parseInt(formData.downPayment).toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="downPayment"
                    min="500000"
                    max="10000000"
                    step="100000"
                    value={formData.downPayment}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹5L</span>
                    <span>₹1Cr</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Interest Rate (%)
                  </label>
                  <input
                    type="number"
                    step="0.1"
                    name="interestRate"
                    value={formData.interestRate}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                    placeholder="Enter interest rate"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Loan Tenure (Years)
                  </label>
                  <select
                    name="tenureYears"
                    value={formData.tenureYears}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="10">10 Years</option>
                    <option value="15">15 Years</option>
                    <option value="20">20 Years</option>
                    <option value="25">25 Years</option>
                    <option value="30">30 Years</option>
                  </select>
                </div>

                <button
                  onClick={calculateBudget}
                  className="w-full bg-purple-600 text-white py-3 px-4 rounded-md hover:bg-purple-700 transition-colors font-medium flex items-center justify-center gap-2"
                >
                  <span>🧮</span>
                  Calculate Budget
                </button>
              </div>
            </div>

            <div>
              <h2 className="text-xl font-semibold mb-6">Budget Results</h2>
              
              {result ? (
                <div className="space-y-4">
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-blue-800 mb-2">Max EMI</h3>
                    <p className="text-2xl font-bold text-blue-600">
                      ₹{result.maxEmi.toLocaleString('en-IN')}
                    </p>
                    <p className="text-sm text-blue-500">(₹{(result.maxEmi/100000).toFixed(2)} Lakhs)</p>
                  </div>

                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-green-800 mb-2">Max Loan</h3>
                    <p className="text-2xl font-bold text-green-600">
                      ₹{result.maxLoan.toLocaleString('en-IN')}
                    </p>
                    <p className="text-sm text-green-500">(₹{result.maxLoan >= 10000000 ? (result.maxLoan/10000000).toFixed(2) + ' Crores' : (result.maxLoan/100000).toFixed(2) + ' Lakhs'})</p>
                  </div>

                  <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-purple-800 mb-2">Property Budget</h3>
                    <p className="text-2xl font-bold text-purple-600">
                      ₹{result.propertyBudget.toLocaleString('en-IN')}
                    </p>
                    <p className="text-sm text-purple-500">(₹{result.propertyBudget >= 10000000 ? (result.propertyBudget/10000000).toFixed(2) + ' Crores' : (result.propertyBudget/100000).toFixed(2) + ' Lakhs'})</p>
                  </div>

                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-gray-800 mb-2">Tips</h3>
                    <ul className="text-sm text-gray-600 space-y-1">
                      <li>• Based on 40% of disposable income for EMI</li>
                      <li>• Assumes 80% loan-to-value ratio</li>
                      <li>• Higher down payment increases budget</li>
                      <li>• Consider additional costs like registration</li>
                    </ul>
                  </div>
                </div>
              ) : (
                <div className="bg-gray-50 border border-gray-200 rounded-lg p-8 text-center">
                  <p className="text-gray-500">Enter your details to calculate budget</p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}