import React, { useEffect, useState } from 'react';
import { ArrowLeft, Plus, Target, TrendingUp, Users, BarChart3, Play, Pause, Edit, Trash2 } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import { Button } from '../components/ui/button';
import { getCampaigns } from '../api/campaigns';
import { isAdmin } from '../utils/auth';

interface Campaign {
  id?: string | number;
  name?: string;
  campaignName?: string;
  status?: string;
  budget?: number;
  budgetAmount?: number;
  spent?: number;
  impressions?: number;
  clicks?: number;
  conversions?: number;
  startDate?: string;
  endDate?: string;
  propertyLocation?: string;
  targetCity?: string;
  platforms?: string[];
}

export default function CampaignManagement() {
  const navigate = useNavigate();
  const staticCampaigns: Campaign[] = [
    {
      id: 1,
      name: "Luxury Apartments - Bangalore",
      status: "active",
      budget: 50000,
      spent: 32000,
      impressions: 125000,
      clicks: 2500,
      conversions: 45,
      startDate: "2024-01-15",
      endDate: "2024-02-15"
    },
    {
      id: 2,
      name: "Budget Homes - Mumbai",
      status: "paused",
      budget: 30000,
      spent: 18000,
      impressions: 85000,
      clicks: 1800,
      conversions: 28,
      startDate: "2024-01-10",
      endDate: "2024-02-10"
    }
  ];
  const [campaigns, setCampaigns] = useState<Campaign[]>([]);
  const [loading, setLoading] = useState(false);

  const getCurrentRole = (): 'buyer' | 'seller' | 'agent' | 'admin' | 'user' => {
    try {
      const accessToken = localStorage.getItem('access_token');
      if (accessToken) {
        const payload = JSON.parse(atob(accessToken.split('.')[1]));
        const roles: string[] = payload.roles || [];
        if (roles.includes('Admin')) return 'admin';
        if (roles.includes('Agent')) return 'agent';
        if (roles.includes('Seller')) return 'seller';
        if (roles.includes('Buyer')) return 'buyer';
      }
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const role = (user.user_role || user.role || 'user').toString().toLowerCase();
      if (role === 'admin') return 'admin';
      if (role === 'agent') return 'agent';
      if (role === 'seller') return 'seller';
      if (role === 'buyer') return 'buyer';
      return 'user';
    } catch {
      return 'user';
    }
  };

  const role = getCurrentRole();
  const hasSellerToken = Boolean(localStorage.getItem('sellerToken'));
  const hasAgentToken = Boolean(localStorage.getItem('agentToken'));
  const isSellerAgentAdmin =
    isAdmin() ||
    role === 'seller' ||
    role === 'agent' ||
    role === 'admin' ||
    hasSellerToken ||
    hasAgentToken;
  const isBuyerView = !isSellerAgentAdmin;

  useEffect(() => {
    if (!isBuyerView) return;
    const fetchAllCampaigns = async () => {
      try {
        setLoading(true);
        const response = await getCampaigns({ admin: true });
        setCampaigns(response.campaigns || []);
      } catch (error) {
        console.error('Error fetching campaigns:', error);
        setCampaigns([]);
      } finally {
        setLoading(false);
      }
    };
    fetchAllCampaigns();
  }, [isBuyerView]);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'paused': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-gray-100 text-gray-800';
      case 'approved': return 'bg-blue-100 text-blue-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const displayCampaigns = isBuyerView ? campaigns : staticCampaigns;

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="mb-6">
          <Button
            variant="outline"
            onClick={() => navigate('/')}
            className="flex items-center gap-2"
          >
            <ArrowLeft className="w-4 h-4" />
            Back to Home
          </Button>
        </div>

        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="p-6 border-b bg-white">
            <div className="flex flex-col gap-4 md:flex-row md:justify-between md:items-center">
              <div>
                <h1 className="text-4xl font-bold text-gray-900 mb-2">Campaign Management</h1>
                <p className="text-lg text-gray-700">
                  {isBuyerView
                    ? 'Browse all active campaigns posted by sellers, agents, and admins'
                    : 'Create and manage your property marketing campaigns'}
                </p>
              </div>
              {isBuyerView && (
                <div className="bg-blue-50 text-blue-700 px-4 py-2 rounded-full text-sm font-semibold w-fit">
                  Verified Listings Only
                </div>
              )}
              {!isBuyerView && (
                <Button 
                  className="bg-blue-600 text-black hover:bg-blue-700 px-6 py-3 font-semibold shadow-lg"
                  onClick={() => {
                    const agentToken = localStorage.getItem('agentToken');
                    const sellerToken = localStorage.getItem('sellerToken');
                    
                    if (agentToken || sellerToken) {
                      navigate('/campaign-form');
                    } else {
                      alert('Only sellers and agents can create campaigns');
                    }
                  }}
                >
                  <Plus className="w-4 h-4 mr-2" />
                  New Campaign
                </Button>
              )}
            </div>
            {!isBuyerView && (
              <></>
            )}
          </div>

          {/* Stats Overview */}
          <div className="p-6 bg-gray-50 border-b">
            <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
              {isBuyerView ? (
                <>
                  <div className="bg-white p-6 rounded-2xl shadow-sm border border-blue-100">
                    <div className="flex items-center gap-4">
                      <div className="w-12 h-12 rounded-2xl bg-blue-100 flex items-center justify-center">
                        <Target className="w-6 h-6 text-blue-600" />
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Total Campaigns</p>
                        <p className="text-2xl font-bold text-gray-900">{displayCampaigns.length}</p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-2xl shadow-sm border border-green-100">
                    <div className="flex items-center gap-4">
                      <div className="w-12 h-12 rounded-2xl bg-green-100 flex items-center justify-center">
                        <TrendingUp className="w-6 h-6 text-green-600" />
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Active/Approved</p>
                        <p className="text-2xl font-bold text-gray-900">
                          {displayCampaigns.filter(c => ['active', 'approved'].includes((c.status || '').toLowerCase())).length}
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-2xl shadow-sm border border-purple-100">
                    <div className="flex items-center gap-4">
                      <div className="w-12 h-12 rounded-2xl bg-purple-100 flex items-center justify-center">
                        <Users className="w-6 h-6 text-purple-600" />
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Cities</p>
                        <p className="text-2xl font-bold text-gray-900">
                          {new Set(displayCampaigns.map(c => (c.propertyLocation || c.targetCity || '').toLowerCase()).filter(Boolean)).size}
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-2xl shadow-sm border border-orange-100">
                    <div className="flex items-center gap-4">
                      <div className="w-12 h-12 rounded-2xl bg-orange-100 flex items-center justify-center">
                        <BarChart3 className="w-6 h-6 text-orange-600" />
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Total Budget</p>
                        <p className="text-2xl font-bold text-gray-900">
                          ₹{displayCampaigns.reduce((sum, c) => sum + (c.budgetAmount || c.budget || 0), 0).toLocaleString()}
                        </p>
                      </div>
                    </div>
                  </div>
                </>
              ) : (
                <>
                  <div className="bg-white p-6 rounded-xl shadow-sm">
                    <div className="flex items-center">
                      <Target className="w-8 h-8 text-blue-600 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600">Active Campaigns</p>
                        <p className="text-2xl font-bold text-gray-900">12</p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-xl shadow-sm">
                    <div className="flex items-center">
                      <TrendingUp className="w-8 h-8 text-green-600 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600">Total Impressions</p>
                        <p className="text-2xl font-bold text-gray-900">2.1M</p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-xl shadow-sm">
                    <div className="flex items-center">
                      <Users className="w-8 h-8 text-purple-600 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600">Total Clicks</p>
                        <p className="text-2xl font-bold text-gray-900">45.2K</p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white p-6 rounded-xl shadow-sm">
                    <div className="flex items-center">
                      <BarChart3 className="w-8 h-8 text-orange-600 mr-3" />
                      <div>
                        <p className="text-sm text-gray-600">Conversions</p>
                        <p className="text-2xl font-bold text-gray-900">892</p>
                      </div>
                    </div>
                  </div>
                </>
              )}
            </div>
          </div>

          {/* Campaign List */}
          <div className="p-6">
            <div className="flex flex-col gap-2 md:flex-row md:items-center md:justify-between mb-6">
              <div>
                <h2 className="text-xl font-bold text-gray-900">{isBuyerView ? 'All Campaigns' : 'Your Campaigns'}</h2>
                {isBuyerView && (
                  <p className="text-sm text-gray-500">Discover promoted listings and compare active campaigns.</p>
                )}
              </div>
            </div>
            <div className="space-y-4">
              {loading && (
                <div className="text-sm text-gray-500">Loading campaigns...</div>
              )}
              {!loading && displayCampaigns.length === 0 && (
                <div className="text-sm text-gray-500">No campaigns available.</div>
              )}
              {!loading && displayCampaigns.map((campaign, index) => (
                <div key={campaign.id ?? index} className="bg-white border border-gray-200 rounded-2xl p-6 hover:shadow-lg transition-shadow">
                  <div className="flex flex-col gap-3 md:flex-row md:justify-between md:items-start mb-4">
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900">
                        {campaign.campaignName || campaign.name || `Campaign ${index + 1}`}
                      </h3>
                      <div className="flex flex-wrap items-center gap-2 mt-2">
                        <span className={`inline-block px-3 py-1 rounded-full text-sm font-medium ${getStatusColor(campaign.status || 'active')}`}>
                          {(campaign.status || 'active').toString().charAt(0).toUpperCase() + (campaign.status || 'active').toString().slice(1)}
                        </span>
                        {(campaign.propertyLocation || campaign.targetCity) && (
                          <span className="inline-block px-3 py-1 rounded-full text-sm font-medium bg-gray-100 text-gray-700">
                            {campaign.propertyLocation || campaign.targetCity}
                          </span>
                        )}
                        {campaign.platforms?.length ? (
                          <span className="inline-block px-3 py-1 rounded-full text-sm font-medium bg-blue-50 text-blue-700">
                            {campaign.platforms.length} Platforms
                          </span>
                        ) : null}
                      </div>
                    </div>
                    {!isBuyerView && (
                      <div className="flex space-x-2">
                        <Button variant="outline" size="sm">
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button variant="outline" size="sm">
                          {(campaign.status || 'active') === 'active' ? <Pause className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                        </Button>
                        <Button variant="outline" size="sm" className="text-red-600 hover:text-red-700">
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    )}
                  </div>

                  <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                    <div>
                      <p className="text-sm text-gray-600">Budget</p>
                      <p className="font-semibold">₹{(campaign.budgetAmount || campaign.budget || 0).toLocaleString()}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Spent</p>
                      <p className="font-semibold">₹{(campaign.spent || 0).toLocaleString()}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Impressions</p>
                      <p className="font-semibold">{(campaign.impressions || 0).toLocaleString()}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Clicks</p>
                      <p className="font-semibold">{(campaign.clicks || 0).toLocaleString()}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Conversions</p>
                      <p className="font-semibold">{campaign.conversions || 0}</p>
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">CTR</p>
                      <p className="font-semibold">
                        {campaign.impressions && campaign.clicks
                          ? `${((campaign.clicks / campaign.impressions) * 100).toFixed(2)}%`
                          : '0.00%'}
                      </p>
                    </div>
                  </div>

                  <div className="mt-4">
                    <div className="flex justify-between text-sm text-gray-600 mb-1">
                      <span>Budget Progress</span>
                      <span>
                        {campaign.budgetAmount || campaign.budget
                          ? Math.round(((campaign.spent || 0) / (campaign.budgetAmount || campaign.budget || 1)) * 100)
                          : 0}%
                      </span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-blue-600 h-2 rounded-full" 
                        style={{ width: `${campaign.budgetAmount || campaign.budget ? ((campaign.spent || 0) / (campaign.budgetAmount || campaign.budget || 1)) * 100 : 0}%` }}
                      ></div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Campaign Creation Guide */}
          {!isBuyerView && (
            <div className="p-6 bg-gray-50 border-t">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Campaign Creation Guide</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="bg-white p-4 rounded-lg">
                  <div className="text-blue-600 font-semibold mb-2">1. Define Target Audience</div>
                  <p className="text-sm text-gray-600">Set demographics, location, and buyer preferences for precise targeting</p>
                </div>
                <div className="bg-white p-4 rounded-lg">
                  <div className="text-green-600 font-semibold mb-2">2. Set Budget & Duration</div>
                  <p className="text-sm text-gray-600">Allocate budget and timeline for optimal campaign performance</p>
                </div>
                <div className="bg-white p-4 rounded-lg">
                  <div className="text-purple-600 font-semibold mb-2">3. Monitor & Optimize</div>
                  <p className="text-sm text-gray-600">Track performance metrics and adjust campaigns for better results</p>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}
