import React, { useEffect, useState, useRef } from 'react';
import { useNavigate } from 'react-router-dom';
import { useCompare } from '../context/CompareContext';
import jsPDF from 'jspdf';
import html2canvas from 'html2canvas';

interface Property {
  id: number;
  title: string;
  price: string;
  sale_price?: string;
  property_type: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  carpet_area?: number;
  city: string;
  locality: string;
  furnishing: string;
  property_age: string;
  facing: string;
  floor_number?: string;
  total_floors?: string;
  balconies?: number;
  price_per_sqft?: number;
  loan_availability: boolean;
  price_negotiable: boolean;
  possession_status: string;
  rera_verified?: boolean;
  kyc_verified: boolean;
  amenities: string[];
  photo: string;
}

export const ComparePage: React.FC = () => {
  const { compareList, removeFromCompare, clearCompare } = useCompare();
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [imageErrors, setImageErrors] = useState<Record<number, boolean>>({});
  const [downloading, setDownloading] = useState(false);
  const contentRef = useRef<HTMLDivElement>(null);
  const navigate = useNavigate();

  useEffect(() => {
    if (compareList.length === 0) {
      navigate('/');
      return;
    }

    const fetchProperties = async () => {
      try {
        setLoading(true);
        const promises = compareList.map(id =>
          fetch(`http://localhost:8090/api/properties/${id}`).then(res => res.json())
        );
        const results = await Promise.all(promises);
        setProperties(results);
      } catch (error) {
        console.error('Error fetching properties:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchProperties();
  }, [compareList, navigate]);

  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === 'string' ? parseFloat(price) : price;
    if (isNaN(numPrice)) return 'N/A';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  const getImageUrl = (photo: string, propertyId: number) => {
    if (imageErrors[propertyId]) {
      return 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDQwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI0MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjZTVlN2ViIi8+Cjx0ZXh0IHg9IjIwMCIgeT0iMTUwIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBmaWxsPSIjNmI3MjgwIiBmb250LXNpemU9IjE2Ij5ObyBJbWFnZTwvdGV4dD4KPC9zdmc+';
    }
    if (!photo) return 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDQwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI0MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjZTVlN2ViIi8+Cjx0ZXh0IHg9IjIwMCIgeT0iMTUwIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBmaWxsPSIjNmI3MjgwIiBmb250LXNpemU9IjE2Ij5ObyBJbWFnZTwvdGV4dD4KPC9zdmc+';
    if (photo.startsWith('http')) return photo;
    return `http://localhost:8000${photo.startsWith('/') ? photo : '/' + photo}`;
  };

  const handleImageError = (propertyId: number) => {
    setImageErrors(prev => ({ ...prev, [propertyId]: true }));
  };

  const downloadPDF = () => {
    setDownloading(true);
    
    // Create print-specific styles
    const printStyles = `
      <style>
        @media print {
          body * { visibility: hidden; }
          .print-content, .print-content * { visibility: visible; }
          .print-content { position: absolute; left: 0; top: 0; width: 100%; }
          button { display: none !important; }
          .absolute { display: none !important; }
        }
      </style>
    `;
    
    // Add print styles to head
    const styleElement = document.createElement('style');
    styleElement.innerHTML = printStyles.replace(/<\/?style>/g, '');
    document.head.appendChild(styleElement);
    
    // Add print class to content
    if (contentRef.current) {
      contentRef.current.classList.add('print-content');
    }
    
    // Trigger print
    setTimeout(() => {
      window.print();
      
      // Cleanup
      setTimeout(() => {
        document.head.removeChild(styleElement);
        if (contentRef.current) {
          contentRef.current.classList.remove('print-content');
        }
        setDownloading(false);
      }, 1000);
    }, 100);
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4">
        {/* Header */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Compare Properties</h1>
              <p className="text-gray-600">Comparing {properties.length} properties across 20+ parameters</p>
            </div>
            <div className="flex gap-3">
              <button
                onClick={() => navigate(-1)}
                className="px-4 py-2 text-gray-600 hover:text-gray-800 font-medium"
              >
                Back
              </button>
              <button
                onClick={downloadPDF}
                disabled={downloading}
                className="px-4 py-2 bg-blue-600 text-black rounded-lg hover:bg-blue-700 font-medium disabled:opacity-50 flex items-center gap-2"
              >
                {downloading ? (
                  <>
                    <svg className="animate-spin h-4 w-4" fill="none" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                      <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                    Generating...
                  </>
                ) : (
                  <>
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    Download PDF
                  </>
                )}
              </button>
              <button
                onClick={clearCompare}
                className="px-4 py-2 bg-red-100 text-red-600 rounded-lg hover:bg-red-200 font-medium"
              >
                Clear All
              </button>
            </div>
          </div>
        </div>

        {/* Comparison Table */}
        <div ref={contentRef} className="bg-white rounded-lg shadow-sm overflow-x-auto">
          <table className="w-full">
            <thead>
              <tr className="border-b">
                <th className="sticky left-0 bg-gray-50 p-4 text-left font-semibold text-gray-900 w-48">Parameter</th>
                {properties.map((property) => (
                  <th key={property.id} className="p-4 min-w-[250px]">
                    <div className="relative">
                      <button
                        onClick={() => removeFromCompare(property.id)}
                        className="absolute top-0 right-0 p-1 text-red-500 hover:text-red-700"
                      >
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                        </svg>
                      </button>
                      <img
                        src={getImageUrl(property.photo, property.id)}
                        alt={property.title}
                        className="w-full h-32 object-cover rounded-lg mb-3"
                        onError={() => handleImageError(property.id)}
                      />
                      <h3 className="font-semibold text-gray-900 text-sm mb-1">{property.title}</h3>
                      <p className="text-xs text-gray-600">{property.locality}, {property.city}</p>
                    </div>
                  </th>
                ))}
              </tr>
            </thead>
            <tbody>
              {/* Basic Details */}
              <tr className="bg-blue-50">
                <td colSpan={properties.length + 1} className="sticky left-0 p-3 font-bold text-blue-900">
                  Basic Details
                </td>
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Price</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">
                    <span className="text-lg font-bold text-blue-600">{formatPrice(p.sale_price || p.price)}</span>
                  </td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Property Type</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center capitalize">{p.property_type}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Built-up Area</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.built_up_area || p.carpet_area || 'N/A'} sq.ft</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Bedrooms</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.bedrooms || 'N/A'} BHK</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Bathrooms</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.bathrooms || 'N/A'}</td>
                ))}
              </tr>

              {/* Property Features */}
              <tr className="bg-green-50">
                <td colSpan={properties.length + 1} className="sticky left-0 p-3 font-bold text-green-900">
                  Property Features
                </td>
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Furnishing</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center capitalize">{p.furnishing?.replace('-', ' ')}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Property Age</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center capitalize">{p.property_age?.replace('-', ' to ')}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Facing</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center capitalize">{p.facing?.replace('-', ' ')}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Floor</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.floor_number || 'N/A'}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Total Floors</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.total_floors || 'N/A'}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Balconies</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.balconies || 'N/A'}</td>
                ))}
              </tr>

              {/* Financial */}
              <tr className="bg-yellow-50">
                <td colSpan={properties.length + 1} className="sticky left-0 p-3 font-bold text-yellow-900">
                  Financial Details
                </td>
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Price per sq.ft</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">{p.price_per_sqft ? `₹${p.price_per_sqft}` : 'N/A'}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Loan Available</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">
                    <span className={`px-2 py-1 rounded-full text-xs ${p.loan_availability ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                      {p.loan_availability ? 'Yes' : 'No'}
                    </span>
                  </td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Price Negotiable</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">
                    <span className={`px-2 py-1 rounded-full text-xs ${p.price_negotiable ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                      {p.price_negotiable ? 'Yes' : 'No'}
                    </span>
                  </td>
                ))}
              </tr>

              {/* Legal & Verification */}
              <tr className="bg-purple-50">
                <td colSpan={properties.length + 1} className="sticky left-0 p-3 font-bold text-purple-900">
                  Legal & Verification
                </td>
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">RERA Verified</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">
                    <span className={`px-2 py-1 rounded-full text-xs ${p.rera_verified ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'}`}>
                      {p.rera_verified ? 'Verified' : 'Not Verified'}
                    </span>
                  </td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">KYC Verified</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center">
                    <span className={`px-2 py-1 rounded-full text-xs ${p.kyc_verified ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'}`}>
                      {p.kyc_verified ? 'Verified' : 'Not Verified'}
                    </span>
                  </td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Possession Status</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center capitalize">{p.possession_status}</td>
                ))}
              </tr>

              {/* Amenities */}
              <tr className="bg-indigo-50">
                <td colSpan={properties.length + 1} className="sticky left-0 p-3 font-bold text-indigo-900">
                  Amenities
                </td>
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Total Amenities</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4 text-center font-semibold">{p.amenities?.length || 0}</td>
                ))}
              </tr>
              <tr className="border-b hover:bg-gray-50">
                <td className="sticky left-0 bg-white p-4 font-medium text-gray-700">Available Amenities</td>
                {properties.map((p) => (
                  <td key={p.id} className="p-4">
                    <div className="flex flex-wrap gap-1 justify-center">
                      {p.amenities?.slice(0, 5).map((amenity, idx) => (
                        <span key={idx} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
                          {amenity}
                        </span>
                      ))}
                      {p.amenities?.length > 5 && (
                        <span className="px-2 py-1 bg-gray-100 text-gray-600 text-xs rounded-full">
                          +{p.amenities.length - 5} more
                        </span>
                      )}
                    </div>
                  </td>
                ))}
              </tr>
            </tbody>
          </table>
        </div>

        {/* Action Buttons */}
        <div className="mt-6 flex justify-center gap-4">
          {properties.map((property) => (
            <button
              key={property.id}
              onClick={() => navigate(`/property/${property.id}`)}
              className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-semibold"
            >
              View Property #{property.id}
            </button>
          ))}
        </div>
      </div>
    </div>
  );
};
