import React, { useState, useRef } from 'react';
import { Upload, Lock, Unlock, FileText, Image, File, Eye, Download, ArrowLeft } from 'lucide-react';
import { Button } from '../components/ui/button';
import { useNavigate } from 'react-router-dom';

interface UploadedFile {
  id: string;
  name: string;
  size: number;
  type: string;
  url: string;
}

export function DigitalVaultDemo() {
  const navigate = useNavigate();
  const [step, setStep] = useState<'folder' | 'upload' | 'password' | 'unlock' | 'vault'>('folder');
  const [propertyName, setPropertyName] = useState('');
  const [files, setFiles] = useState<UploadedFile[]>([]);
  const [password, setPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [storedPassword, setStoredPassword] = useState('');
  const [unlockPassword, setUnlockPassword] = useState('');
  const [isLocked, setIsLocked] = useState(true);
  const [passwordError, setPasswordError] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFiles = Array.from(event.target.files || []);
    
    selectedFiles.forEach(file => {
      const reader = new FileReader();
      reader.onload = (e) => {
        const newFile: UploadedFile = {
          id: Date.now().toString() + Math.random().toString(36).substr(2, 9),
          name: file.name,
          size: file.size,
          type: file.type,
          url: e.target?.result as string
        };
        setFiles(prev => [...prev, newFile]);
      };
      reader.readAsDataURL(file);
    });
  };

  const handleCreatePassword = () => {
    setPasswordError('');
    
    if (password.length < 6) {
      setPasswordError('Password must be at least 6 characters long');
      return;
    }
    
    if (password !== confirmPassword) {
      setPasswordError('Passwords do not match');
      return;
    }
    
    setStoredPassword(password);
    setPassword('');
    setConfirmPassword('');
    setStep('unlock');
  };

  const handleUnlock = () => {
    setPasswordError('');
    
    if (unlockPassword === storedPassword) {
      setIsLocked(false);
      setUnlockPassword('');
      setStep('vault');
    } else {
      setPasswordError('Incorrect password. Please try again.');
    }
  };

  const getFileIcon = (type: string) => {
    if (type.startsWith('image/')) return <Image className="w-6 h-6 text-blue-500" />;
    if (type.includes('pdf')) return <FileText className="w-6 h-6 text-red-500" />;
    return <File className="w-6 h-6 text-gray-500" />;
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="mb-8">
          <Button 
            variant="ghost" 
            onClick={() => navigate(-1)}
            className="mb-4"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Digital Vault
          </Button>
          <h1 className="text-3xl font-bold text-gray-900">Digital Vault</h1>
          <p className="text-gray-600 mt-2">Experience secure document storage with password protection</p>
        </div>

        {/* Step 1: Create Property Folder */}
        {step === 'folder' && (
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <div className="text-center mb-8">
              <div className="w-16 h-16 bg-[#0056D2] text-white rounded-full flex items-center justify-center text-2xl font-bold mx-auto mb-4">
                1
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Create Property Folder</h2>
              <p className="text-gray-600">Start by creating a dedicated folder for your property documents</p>
            </div>
            
            <div className="max-w-md mx-auto">
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Property Name
              </label>
              <input
                type="text"
                value={propertyName}
                onChange={(e) => setPropertyName(e.target.value)}
                placeholder="e.g., Luxury Villa - Marine Drive"
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
              />
            </div>

            <div className="mt-6 flex justify-center">
              <Button 
                onClick={() => setStep('upload')}
                disabled={!propertyName.trim()}
                className="bg-[#0056D2] hover:bg-[#0041A3] text-white"
              >
                Create Folder & Continue
              </Button>
            </div>
          </div>
        )}

        {/* Step 2: Upload Documents */}
        {step === 'upload' && (
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <div className="text-center mb-8">
              <div className="w-16 h-16 bg-[#0056D2] text-white rounded-full flex items-center justify-center text-2xl font-bold mx-auto mb-4">
                2
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Upload Documents Securely</h2>
              <p className="text-gray-600">Upload your property documents to the secure folder: {propertyName}</p>
            </div>
            
            <div 
              className="border-2 border-dashed border-gray-300 rounded-lg p-12 text-center hover:border-[#0056D2] transition-colors cursor-pointer"
              onClick={() => fileInputRef.current?.click()}
            >
              <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Drop files here or click to upload</h3>
              <p className="text-gray-500">Support for PDF, Images, Word, Excel and other documents</p>
              <input
                ref={fileInputRef}
                type="file"
                multiple
                accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.gif,.xls,.xlsx"
                onChange={handleFileUpload}
                className="hidden"
              />
            </div>

            {files.length > 0 && (
              <div className="mt-8">
                <h3 className="text-lg font-semibold text-gray-900 mb-4">Uploaded Files ({files.length})</h3>
                <div className="space-y-3">
                  {files.map(file => (
                    <div key={file.id} className="flex items-center gap-3 p-3 bg-gray-50 rounded-lg">
                      {getFileIcon(file.type)}
                      <div className="flex-1">
                        <p className="font-medium text-gray-900">{file.name}</p>
                        <p className="text-sm text-gray-500">{formatFileSize(file.size)}</p>
                      </div>
                    </div>
                  ))}
                </div>
                
                <div className="mt-6 flex justify-between">
                  <Button 
                    variant="outline"
                    onClick={() => setStep('folder')}
                  >
                    Back
                  </Button>
                  <Button 
                    onClick={() => setStep('password')}
                    className="bg-[#0056D2] hover:bg-[#0041A3] text-white"
                  >
                    Set Password
                  </Button>
                </div>
              </div>
            )}
          </div>
        )}

        {/* Step 3: Set Password */}
        {step === 'password' && (
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <div className="text-center mb-8">
              <div className="w-16 h-16 bg-[#0056D2] text-white rounded-full flex items-center justify-center text-2xl font-bold mx-auto mb-4">
                3
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Set Vault Password</h2>
              <p className="text-gray-600">Create a secure password to protect your documents</p>
            </div>
            
            <div className="max-w-md mx-auto">
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Password
                  </label>
                  <input
                    type="password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    placeholder="Enter password (minimum 6 characters)"
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Confirm Password
                  </label>
                  <input
                    type="password"
                    value={confirmPassword}
                    onChange={(e) => setConfirmPassword(e.target.value)}
                    placeholder="Confirm your password"
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  />
                </div>
                
                {passwordError && (
                  <div className="p-3 bg-red-50 border border-red-200 rounded-md">
                    <p className="text-sm text-red-600">{passwordError}</p>
                  </div>
                )}
                
                <div className="p-3 bg-blue-50 border border-blue-200 rounded-md">
                  <div className="flex items-center gap-2 mb-2">
                    <Lock className="w-4 h-4 text-blue-600" />
                    <span className="text-sm font-medium text-blue-800">Security Features:</span>
                  </div>
                  <ul className="text-sm text-blue-700 space-y-1">
                    <li>• End-to-end encryption</li>
                    <li>• Secure password protection</li>
                    <li>• No password recovery - keep it safe!</li>
                  </ul>
                </div>
              </div>
            </div>

            <div className="mt-8 flex justify-between">
              <Button 
                variant="outline"
                onClick={() => setStep('upload')}
              >
                Back
              </Button>
              <Button 
                onClick={handleCreatePassword}
                disabled={!password || !confirmPassword}
                className="bg-[#0056D2] hover:bg-[#0041A3] text-white"
              >
                Create Password
              </Button>
            </div>
          </div>
        )}

        {/* Step 4: Unlock Vault */}
        {step === 'unlock' && (
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <div className="text-center mb-8">
              <div className="w-16 h-16 bg-[#0056D2] text-white rounded-full flex items-center justify-center text-2xl font-bold mx-auto mb-4">
                4
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Unlock Your Vault</h2>
              <p className="text-gray-600">Enter your password to access the secure vault</p>
            </div>
            
            <div className="max-w-md mx-auto">
              <div className="space-y-4">
                <div className="text-center">
                  <div className="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Lock className="w-10 h-10 text-gray-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">Vault: {propertyName}</h3>
                  <p className="text-sm text-gray-600">{files.length} documents secured</p>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Enter Password
                  </label>
                  <input
                    type="password"
                    value={unlockPassword}
                    onChange={(e) => setUnlockPassword(e.target.value)}
                    placeholder="Enter your vault password"
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    onKeyPress={(e) => e.key === 'Enter' && handleUnlock()}
                  />
                </div>
                
                {passwordError && (
                  <div className="p-3 bg-red-50 border border-red-200 rounded-md">
                    <p className="text-sm text-red-600">{passwordError}</p>
                  </div>
                )}
              </div>
            </div>

            <div className="mt-8 flex justify-center">
              <Button 
                onClick={handleUnlock}
                disabled={!unlockPassword}
                className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-8"
              >
                <Unlock className="w-4 h-4 mr-2" />
                Unlock Vault
              </Button>
            </div>
          </div>
        )}

        {/* Step 5: Vault Access */}
        {step === 'vault' && (
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <div className="text-center mb-8">
              <div className="w-16 h-16 bg-green-500 text-white rounded-full flex items-center justify-center mx-auto mb-4">
                <Unlock className="w-8 h-8" />
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Vault Unlocked Successfully!</h2>
              <p className="text-gray-600">Your documents are now accessible. Vault will auto-lock when you leave.</p>
            </div>
            
            <div className="mb-6">
              <div className="bg-gradient-to-r from-green-50 to-emerald-50 rounded-lg p-6 mb-6 border border-green-200">
                <div className="flex items-center gap-3 mb-3">
                  <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                    <Lock className="w-5 h-5 text-green-600" />
                  </div>
                  <div>
                    <h3 className="font-semibold text-gray-900">🔓 Vault: {propertyName}</h3>
                    <p className="text-sm text-green-600">Status: Unlocked & Secure</p>
                  </div>
                </div>
                <p className="text-sm text-gray-600">{files.length} documents encrypted and protected</p>
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {files.map(file => (
                <div key={file.id} className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow">
                  <div className="flex items-center gap-3 mb-3">
                    {getFileIcon(file.type)}
                    <div className="flex-1 min-w-0">
                      <p className="font-medium text-gray-900 truncate">{file.name}</p>
                      <p className="text-sm text-gray-500">{formatFileSize(file.size)}</p>
                    </div>
                  </div>
                  
                  <div className="flex gap-2">
                    <Button 
                      size="sm" 
                      variant="outline" 
                      className="flex-1"
                      onClick={() => {
                        const newWindow = window.open();
                        if (newWindow) {
                          newWindow.document.write(`
                            <html>
                              <head><title>${file.name}</title></head>
                              <body style="margin:0;display:flex;justify-content:center;align-items:center;min-height:100vh;background:#f5f5f5;">
                                ${file.type.startsWith('image/') 
                                  ? `<img src="${file.url}" style="max-width:100%;max-height:100vh;object-fit:contain;" alt="${file.name}" />` 
                                  : `<iframe src="${file.url}" style="width:100vw;height:100vh;border:none;" title="${file.name}"></iframe>`
                                }
                              </body>
                            </html>
                          `);
                          newWindow.document.close();
                        }
                      }}
                    >
                      <Eye className="w-4 h-4 mr-1" />
                      View
                    </Button>
                    <Button 
                      size="sm" 
                      variant="outline" 
                      className="flex-1"
                      onClick={() => {
                        const link = document.createElement('a');
                        link.href = file.url;
                        link.download = file.name;
                        document.body.appendChild(link);
                        link.click();
                        document.body.removeChild(link);
                      }}
                    >
                      <Download className="w-4 h-4 mr-1" />
                      Download
                    </Button>
                  </div>
                </div>
              ))}
            </div>

            <div className="mt-8 p-6 bg-green-50 border border-green-200 rounded-lg text-center">
              <h3 className="font-semibold text-green-800 mb-2">🎉 Digital Vault Active!</h3>
              <p className="text-green-700 text-sm mb-4">
                Your documents are securely encrypted and protected with your password.
              </p>
              <div className="flex justify-center gap-4">
                <Button 
                  variant="outline"
                  onClick={() => {
                    setStep('folder');
                    setPropertyName('');
                    setFiles([]);
                    setPassword('');
                    setConfirmPassword('');
                    setStoredPassword('');
                    setUnlockPassword('');
                    setIsLocked(true);
                    setPasswordError('');
                  }}
                >
                  Create Another Vault
                </Button>
                <Button 
                  onClick={() => {
                    setIsLocked(true);
                    setStep('unlock');
                  }}
                  className="bg-red-600 hover:bg-red-700 text-white"
                >
                  <Lock className="w-4 h-4 mr-2" />
                  Lock Vault
                </Button>
                <Button 
                  onClick={() => navigate('/services/digital-vault')}
                  className="bg-[#0056D2] hover:bg-[#0041A3] text-white"
                >
                  Learn More
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}