import { useState, useRef } from 'react';
import { useNavigate } from 'react-router-dom';
import { Upload, FileText, AlertTriangle, CheckCircle, Loader2, ArrowRight } from 'lucide-react';

interface SegregationResult {
  success: boolean;
  session_id: string;
  processing_time: number;
  total_pages: number;
  document_segments: number;
  property_identities: number;
  ownership_chains: number;
  overall_confidence: number;
  risk_level: string;
  recommendations: string[];
  error?: string;
}

export function DocumentSegregation() {
  const [file, setFile] = useState<File | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [progress, setProgress] = useState(0);
  const [currentStep, setCurrentStep] = useState(0);
  const [result, setResult] = useState<SegregationResult | null>(null);
  const [error, setError] = useState<string | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const navigate = useNavigate();

  const processingSteps = [
    { message: "Validating PDF structure and security...", duration: 2000 },
    { message: "Streaming pages without loading full document...", duration: 3000 },
    { message: "Detecting document boundaries using AI...", duration: 4000 },
    { message: "Classifying document types with confidence scoring...", duration: 3500 },
    { message: "Processing OCR with multiple engines...", duration: 5000 },
    { message: "Extracting property identities and legal entities...", duration: 4000 },
    { message: "Building ownership chains with temporal validation...", duration: 3500 },
    { message: "Performing risk assessment and verification...", duration: 2500 },
    { message: "Generating legally defensible results...", duration: 1500 }
  ];

  const validateAuth = (): string => {
    try {
      const userStr = localStorage.getItem('user');
      if (!userStr) {
        return '550e8400-e29b-41d4-a716-446655440000';
      }
      
      const user = JSON.parse(userStr);
      const userId = user?.user_id || user?.id || '550e8400-e29b-41d4-a716-446655440000';
      
      return userId;
    } catch {
      return '550e8400-e29b-41d4-a716-446655440000';
    }
  };

  const validateFile = (file: File): string | null => {
    if (file.type !== 'application/pdf') {
      return 'Only PDF files are supported';
    }
    
    const maxSize = 200 * 1024 * 1024; // 200MB
    if (file.size > maxSize) {
      return 'File size must be less than 200MB';
    }
    
    if (file.size < 1000) {
      return 'File is too small or empty';
    }
    
    return null;
  };

  const handleFileSelect = (selectedFile: File) => {
    const validationError = validateFile(selectedFile);
    if (validationError) {
      setError(validationError);
      return;
    }
    
    setFile(selectedFile);
    setError(null);
    setResult(null);
  };

  const handleUpload = async () => {
    if (!file) return;
    
    setIsProcessing(true);
    setProgress(0);
    setCurrentStep(0);
    setError(null);
    setResult(null);
    
    try {
      const userId = validateAuth();
      const sessionId = crypto.randomUUID();
      
      // Animate through processing steps
      const totalDuration = processingSteps.reduce((sum, step) => sum + step.duration, 0);
      let elapsedTime = 0;
      
      for (let i = 0; i < processingSteps.length; i++) {
        setCurrentStep(i);
        
        const stepDuration = processingSteps[i].duration;
        const startProgress = (elapsedTime / totalDuration) * 100;
        const endProgress = ((elapsedTime + stepDuration) / totalDuration) * 100;
        
        // Animate progress for this step
        await new Promise<void>((resolve) => {
          const startTime = Date.now();
          const animate = () => {
            const elapsed = Date.now() - startTime;
            const stepProgress = Math.min(elapsed / stepDuration, 1);
            const currentProgress = startProgress + (endProgress - startProgress) * stepProgress;
            setProgress(currentProgress);
            
            if (stepProgress < 1) {
              requestAnimationFrame(animate);
            } else {
              resolve();
            }
          };
          animate();
        });
        
        elapsedTime += stepDuration;
      }
      
      // Actual API call
      const formData = new FormData();
      formData.append('file', file);
      formData.append('user_id', userId);
      formData.append('session_id', sessionId);
      
      const response = await fetch('http://localhost:8090/api/documents/segregation/segregate-large-pdf', {
        method: 'POST',
        body: formData
      });
      
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({ detail: 'Unknown error' }));
        throw new Error(errorData.detail || `HTTP ${response.status}`);
      }
      
      const resultData = await response.json();
      setResult(resultData);
      setProgress(100);
      
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Processing failed';
      setError(errorMessage);
      console.error('PDF segregation error:', err);
    } finally {
      setIsProcessing(false);
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  };

  const getRiskLevelColor = (riskLevel: string) => {
    switch (riskLevel.toLowerCase()) {
      case 'low': return 'text-green-600 bg-green-50 border-green-200';
      case 'medium': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'high': return 'text-red-600 bg-red-50 border-red-200';
      case 'critical': return 'text-red-800 bg-red-100 border-red-300';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 font-display">
      <main className="flex-grow">
        <div className="container mx-auto max-w-4xl px-4 sm:px-6 lg:px-8 py-12">
          <div className="flex flex-col gap-8">
            {/* Header */}
            <div className="text-center">
              <h1 className="text-4xl font-black tracking-tighter text-black mb-4">
                Large PDF Document Segregation
              </h1>
              <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                Upload a single large PDF containing mixed property documents. Our AI will automatically 
                segregate, classify, and reconstruct ownership chains without any assumptions.
              </p>
            </div>

            {/* Upload Section */}
            {!result && (
              <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">
                <h2 className="text-xl font-bold text-gray-900 mb-6">Upload Large PDF</h2>
                
                {!file ? (
                  <div
                    onClick={() => fileInputRef.current?.click()}
                    className="flex flex-col items-center justify-center w-full py-12 px-6 border-2 border-dashed border-gray-300 rounded-xl hover:border-blue-500 hover:bg-blue-50 transition-all duration-200 group cursor-pointer"
                  >
                    <Upload className="w-16 h-16 mb-4 text-gray-400 group-hover:text-blue-500 transition-colors" />
                    <p className="text-lg font-semibold mb-2 text-gray-700 group-hover:text-blue-600">
                      Click to upload large PDF
                    </p>
                    <p className="text-sm text-gray-500 group-hover:text-blue-500 mb-4">
                      PDF files up to 200MB supported
                    </p>
                    <div className="text-xs text-gray-400 space-y-1">
                      <p>• Mixed documents in any order</p>
                      <p>• Multiple properties supported</p>
                      <p>• Automatic document classification</p>
                      <p>• Ownership chain reconstruction</p>
                    </div>
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-green-50 border border-green-200 rounded-lg">
                      <div className="flex items-center gap-3">
                        <FileText className="w-6 h-6 text-green-600" />
                        <div>
                          <p className="font-semibold text-gray-900">{file.name}</p>
                          <p className="text-sm text-gray-600">{formatFileSize(file.size)}</p>
                        </div>
                      </div>
                      <button
                        onClick={() => setFile(null)}
                        className="text-gray-500 hover:text-red-500 transition-colors"
                      >
                        ✕
                      </button>
                    </div>
                    
                    <button
                      onClick={handleUpload}
                      disabled={isProcessing}
                      className="w-full flex items-center justify-center px-6 py-4 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {isProcessing ? (
                        <>
                          <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                          Processing PDF...
                        </>
                      ) : (
                        <>
                          <span>Start AI Segregation</span>
                          <ArrowRight className="ml-2 w-5 h-5" />
                        </>
                      )}
                    </button>
                  </div>
                )}
                
                <input
                  ref={fileInputRef}
                  type="file"
                  accept=".pdf"
                  onChange={(e) => {
                    const selectedFile = e.target.files?.[0];
                    if (selectedFile) {
                      handleFileSelect(selectedFile);
                    }
                  }}
                  className="hidden"
                />
              </div>
            )}

            {/* Processing Overlay */}
            {isProcessing && (
              <div className="fixed inset-0 bg-gradient-to-br from-blue-900/90 to-indigo-900/90 backdrop-blur-sm flex items-center justify-center z-50">
                <div className="bg-white rounded-2xl p-8 max-w-lg mx-4 shadow-2xl">
                  <div className="text-center mb-6">
                    <div className="w-16 h-16 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-4">
                      <FileText className="w-8 h-8 text-white" />
                    </div>
                    <h3 className="text-xl font-bold text-gray-900 mb-2">AI Document Segregation</h3>
                    <p className="text-gray-600">Processing your large PDF with advanced AI</p>
                  </div>
                  
                  {/* Progress Bar */}
                  <div className="mb-6">
                    <div className="flex justify-between items-center mb-2">
                      <span className="text-sm font-medium text-gray-700">Progress</span>
                      <span className="text-sm font-bold text-blue-600">{Math.round(progress)}%</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                      <div 
                        className="h-full bg-gradient-to-r from-blue-500 to-indigo-600 rounded-full transition-all duration-300 ease-out"
                        style={{ width: `${progress}%` }}
                      ></div>
                    </div>
                  </div>
                  
                  {/* Current Step */}
                  <div className="bg-blue-50 rounded-lg p-4 border border-blue-200">
                    <div className="flex items-center gap-3">
                      <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center">
                        <div className="w-3 h-3 bg-white rounded-full animate-pulse"></div>
                      </div>
                      <div>
                        <p className="text-xs font-semibold text-blue-900 mb-1">
                          Step {currentStep + 1} of {processingSteps.length}
                        </p>
                        <p className="text-sm text-blue-700">
                          {processingSteps[currentStep]?.message}
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Error Display */}
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-6">
                <div className="flex items-center gap-3 mb-3">
                  <AlertTriangle className="w-6 h-6 text-red-600" />
                  <h3 className="text-lg font-semibold text-red-900">Processing Error</h3>
                </div>
                <p className="text-red-700 mb-4">{error}</p>
                <button
                  onClick={() => {
                    setError(null);
                    setFile(null);
                  }}
                  className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                >
                  Try Again
                </button>
              </div>
            )}

            {/* Results Display */}
            {result && (
              <div className="space-y-6">
                {/* Success Header */}
                <div className="bg-green-50 border border-green-200 rounded-lg p-6">
                  <div className="flex items-center gap-3 mb-3">
                    <CheckCircle className="w-8 h-8 text-green-600" />
                    <h3 className="text-xl font-bold text-green-900">Segregation Completed</h3>
                  </div>
                  <p className="text-green-700">
                    Successfully processed {result.total_pages} pages in {result.processing_time.toFixed(1)} seconds
                  </p>
                </div>

                {/* Results Summary */}
                <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-6">
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Processing Results</h3>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                    <div className="text-center p-4 bg-blue-50 rounded-lg border border-blue-200">
                      <div className="text-2xl font-bold text-blue-700">{result.document_segments}</div>
                      <div className="text-sm text-blue-600">Document Segments</div>
                    </div>
                    <div className="text-center p-4 bg-purple-50 rounded-lg border border-purple-200">
                      <div className="text-2xl font-bold text-purple-700">{result.property_identities}</div>
                      <div className="text-sm text-purple-600">Property Identities</div>
                    </div>
                    <div className="text-center p-4 bg-indigo-50 rounded-lg border border-indigo-200">
                      <div className="text-2xl font-bold text-indigo-700">{result.ownership_chains}</div>
                      <div className="text-sm text-indigo-600">Ownership Chains</div>
                    </div>
                    <div className="text-center p-4 bg-green-50 rounded-lg border border-green-200">
                      <div className="text-2xl font-bold text-green-700">
                        {Math.round(result.overall_confidence * 100)}%
                      </div>
                      <div className="text-sm text-green-600">Overall Confidence</div>
                    </div>
                  </div>

                  {/* Risk Assessment */}
                  <div className={`p-4 rounded-lg border ${getRiskLevelColor(result.risk_level)} mb-4`}>
                    <h4 className="font-semibold mb-2">Risk Assessment: {result.risk_level.toUpperCase()}</h4>
                    <p className="text-sm">
                      Based on document quality, chain completeness, and legal consistency analysis
                    </p>
                  </div>

                  {/* Recommendations */}
                  {result.recommendations && result.recommendations.length > 0 && (
                    <div className="bg-amber-50 border border-amber-200 rounded-lg p-4">
                      <h4 className="font-semibold text-amber-900 mb-2">Recommendations</h4>
                      <ul className="text-sm text-amber-800 space-y-1">
                        {result.recommendations.map((rec, index) => (
                          <li key={index} className="flex items-start gap-2">
                            <span className="text-amber-600">•</span>
                            <span>{rec}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}
                </div>

                {/* Action Buttons */}
                <div className="flex justify-center gap-4">
                  <button
                    onClick={() => navigate(`/document-segregation/results/${result.session_id}`)}
                    className="px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                  >
                    View Detailed Results
                  </button>
                  <button
                    onClick={() => navigate(`/ownership-chains/${result.session_id}`)}
                    className="px-6 py-3 bg-indigo-500 text-white rounded-lg hover:bg-indigo-600 transition-colors"
                  >
                    View Ownership Chains
                  </button>
                  <button
                    onClick={() => {
                      setFile(null);
                      setResult(null);
                      setError(null);
                    }}
                    className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    Process Another PDF
                  </button>
                </div>
              </div>
            )}
          </div>
        </div>
      </main>
    </div>
  );
}