import { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { ArrowRight, Upload, CheckCircle, Trash2, MapPin } from "lucide-react";
import { AddressAutocomplete } from '../components/AddressAutocomplete';
import { checkRateLimit, validateFileContent, validateSession } from '../utils/security';
import { initializeVerificationSession } from '../utils/sessionManager';
import { useSessionMonitor } from '../hooks/useSessionMonitor';


type DocKey =
  | "titleDeed"
  | "saleDeed"
  | "encumbrance"
  | "taxReceipts"
  | "khataPatta"
  | "noc"
  | "occupancy"
  | "builderDocs"
  | "idProof"
  | "ownerAddressProof";

interface UploadedFile {
  name: string;
  size: number;
}

export function DocumentVerification() {
  // Session monitoring - disabled to prevent immediate expiration
  // useSessionMonitor({ enabled: true, redirectPath: '/' });
  
  const [uploadedFiles, setUploadedFiles] = useState<Partial<Record<DocKey, UploadedFile>>>({});
  const [isLoading, setIsLoading] = useState(false);
  const [progress, setProgress] = useState(0);
  const [currentStep, setCurrentStep] = useState(0);
  const [showDetailsForm, setShowDetailsForm] = useState(true);
  const [showDetailsReview, setShowDetailsReview] = useState(false);
  const [hasExistingDetails, setHasExistingDetails] = useState(false);
  const [isEditingDetails, setIsEditingDetails] = useState(false);
  const [segregationResults, setSegregationResults] = useState(null);
  const [isSegregating, setIsSegregating] = useState(false);
  const [segregationFile, setSegregationFile] = useState(null);
  const segregationFileRef = useRef(null);

  // Sanitize function to prevent XSS but preserve spaces
  const sanitizeString = (str: string): string => {
    if (!str || typeof str !== 'string') return '';
    return str.replace(/[<>"'&]/g, (match) => {
      const map: Record<string, string> = {
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#x27;',
        '&': '&amp;'
      };
      return map[match] || match;
    }); // Remove .trim() to preserve spaces
  };

  // Validate user authentication
  const validateAuth = (): string | null => {
    try {
      // Check for buyer authentication (standard user)
      const userStr = localStorage.getItem('user');
      if (userStr) {
        const user = JSON.parse(userStr);
        const userId = user?.user_id || user?.id;
        
        if (userId && userId.length === 36 && userId.includes('-')) {
          return sanitizeString(userId);
        }
      }
      
      // Check for seller authentication
      const sellerId = localStorage.getItem('sellerId');
      const sellerToken = localStorage.getItem('sellerToken');
      if (sellerId && sellerToken) {
        return sanitizeString(sellerId);
      }
      
      // Check for agent authentication
      const agentId = localStorage.getItem('agentId');
      const agentToken = localStorage.getItem('agentToken');
      if (agentId && agentToken) {
        return sanitizeString(agentId);
      }
      
      throw new Error('No valid authentication found');
    } catch (error) {
      console.error('Authentication failed:', error.message);
      return null;
    }
  };

  // Check if owner details exist when component loads
  useEffect(() => {
    const fetchOwnerDetails = async () => {
      try {
        const userId = validateAuth();
        if (!userId) {
          console.log('No authenticated user, redirecting to login');
          navigate('/login');
          return;
        }
        console.log('Fetching owner details for userId:', userId);

        // Clear existing details first for new user
        setOwnerDetails({
          ownerName: '',
          aadharNumber: '',
          surveyNumber: '',
          propertyId: '',
          phoneNumber: '',
          panNumber: '',
          judicialStatus: '',
          latitude: '',
          longitude: '',
          address: '',
          city: '',
          pincode: '',
          locality: '',
          state: '',
          landmark: ''
        });
        setHasExistingDetails(false);
        setCoordinates(null);

        const headers: Record<string, string> = {};
        
        // Add appropriate authorization header based on user type
        const sellerToken = localStorage.getItem('sellerToken');
        const agentToken = localStorage.getItem('agentToken');
        const authToken = localStorage.getItem('auth_token');
        
        if (sellerToken) {
          headers['Authorization'] = `Bearer ${sellerToken}`;
        } else if (agentToken) {
          headers['Authorization'] = `Bearer ${agentToken}`;
        } else if (authToken) {
          headers['Authorization'] = `Bearer ${authToken}`;
        }

        const response = await fetch(`http://localhost:8090/api/documents/verification/owner-details/latest/${encodeURIComponent(userId)}`, {
          headers
        });
        if (response.ok) {
          const result = await response.json();
          if (result.success && result.owner_details) {
            const details = result.owner_details;
            setOwnerDetails({
              ownerName: sanitizeString(details.owner_name || ''),
              aadharNumber: sanitizeString(details.aadhar_number || ''),
              surveyNumber: sanitizeString(details.survey_number || ''),
              propertyId: sanitizeString(details.property_id || ''),
              phoneNumber: sanitizeString(details.phone_number || ''),
              panNumber: sanitizeString(details.pan_number || ''),
              judicialStatus: sanitizeString(details.judicial_status || ''),
              latitude: sanitizeString(details.latitude || ''),
              longitude: sanitizeString(details.longitude || ''),
              address: sanitizeString(details.address || ''),
              city: sanitizeString(details.city || ''),
              pincode: sanitizeString(details.pincode || ''),
              locality: sanitizeString(details.locality || ''),
              state: sanitizeString(details.state || ''),
              landmark: sanitizeString(details.landmark || '')
            });
            if (details.latitude && details.longitude) {
              const lat = parseFloat(details.latitude);
              const lng = parseFloat(details.longitude);
              if (!isNaN(lat) && !isNaN(lng)) {
                setCoordinates({ lat, lng });
              }
            }
            setHasExistingDetails(true);
          }
        }
        
        // Always show form for new users or users without details
        setShowDetailsForm(true);
        setShowDetailsReview(false);
      } catch (error) {
        console.error('Failed to fetch owner details:', error);
        // Reset to clean state on error
        setHasExistingDetails(false);
        setShowDetailsForm(true);
      }
    };
    
    // Initialize fresh session for verification
    const initializeSession = () => {
      // Always clear old session and generate new one
      localStorage.removeItem('verification-session-id');
      
      // Generate proper UUID format
      const generateUUID = () => {
        if (crypto.randomUUID) {
          return crypto.randomUUID();
        }
        // Fallback UUID generation
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
          const r = Math.random() * 16 | 0;
          const v = c === 'x' ? r : (r & 0x3 | 0x8);
          return v.toString(16);
        });
      };
      
      const newSessionId = generateUUID();
      
      // Store session ID immediately
      localStorage.setItem('verification-session-id', newSessionId);
      
      // Create ruthless session data with 24 hour expiration
      const now = Date.now();
      const expiresAt = new Date(now + 24 * 60 * 60 * 1000).toISOString(); // 24 hours
      const sessionData = {
        createdAt: new Date(now).toISOString(),
        expiresAt,
        sessionId: newSessionId
      };
      localStorage.setItem(`session-${newSessionId}`, JSON.stringify(sessionData));
      
      console.log('🔥 SESSION CREATED:', newSessionId);
      
      // Also initialize with session manager
      import('../utils/sessionManager').then(({ initializeVerificationSession }) => {
        const sessionId = initializeVerificationSession();
        console.log('🔥 RUTHLESS SESSION CREATED:', sessionId);
      }).catch(error => {
        console.warn('Session manager initialization failed:', error);
      });
    };
    
    fetchOwnerDetails();
    initializeSession();
  }, [localStorage.getItem('user')]);
  const [ownerDetails, setOwnerDetails] = useState({
    ownerName: '',
    aadharNumber: '',
    surveyNumber: '',
    propertyId: '',
    phoneNumber: '',
    panNumber: '',
    judicialStatus: '',
    latitude: '',
    longitude: '',
    address: '',
    city: '',
    pincode: '',
    locality: '',
    state: '',
    landmark: ''
  });
  const [coordinates, setCoordinates] = useState(null);
  const navigate = useNavigate();
  const fileInputRefs = useRef<{[key: string]: HTMLInputElement | null}>({});


  const loadingSteps = [
    { message: "Initializing AI Document Scanner...", duration: 800 },
    { message: "Extracting text and metadata from documents...", duration: 1000 },
    { message: "Verifying document authenticity...", duration: 1200 },
    { message: "Cross-referencing with government databases...", duration: 1000 },
    { message: "Analyzing legal compliance and validity...", duration: 900 },
    { message: "Generating professional recommendations...", duration: 700 },
    { message: "Finalizing verification report...", duration: 500 }
  ];

  const documents = [
    {
      key: "titleDeed" as DocKey,
      title: "1. Title Deed / Mother Deed",
      description: "Official document proving the sale and transfer of property ownership."
    },
    {
      key: "saleDeed" as DocKey,
      title: "2. Sale Deed / Agreement to Sell",
      description: "A legal document that proves a person's right to a property."
    },
    {
      key: "encumbrance" as DocKey,
      title: "3. Encumbrance Certificate (EC)",
      description: "Ensures the property is free from any monetary and legal liabilities."
    },
    {
      key: "taxReceipts" as DocKey,
      title: "4. Property Tax Receipts",
      description: "Proof of payment for the latest property taxes."
    },
    {
      key: "khataPatta" as DocKey,
      title: "5. Khata / Patta / Mutation Documents",
      description: "Sanctioned plan from the local municipal authority."
    },
    {
      key: "noc" as DocKey,
      title: "6. NOCs (Society / Authority / Bank)",
      description: "No Objection Certificates from relevant authorities."
    },
    {
      key: "occupancy" as DocKey,
      title: "7. Completion / Occupancy Certificate",
      description: "Certificate confirming completion and occupancy approval."
    },
    {
      key: "builderDocs" as DocKey,
      title: "8. Builder Documents (RERA, Allotment, Possession)",
      description: "RERA registration, allotment letter, and possession documents."
    },
    {
      key: "idProof" as DocKey,
      title: "9. Owner ID Proof",
      description: "Government issued identity proof of the property owner."
    }
  ];



  // File validation
  const validateFile = (file: File): string | null => {
    const allowedTypes = ['application/pdf', 'image/jpeg', 'image/png'];
    const maxSize = 10 * 1024 * 1024; // 10MB
    
    if (!allowedTypes.includes(file.type)) {
      return 'Only PDF, JPEG, and PNG files are allowed';
    }
    
    if (file.size > maxSize) {
      return 'File size must be less than 10MB';
    }
    
    if (file.size === 0) {
      return 'File is empty';
    }
    
    return null;
  };

  const handleFileUpload = async (key: DocKey, file: File) => {
    try {
      // Validate session first
      validateSession();
      
      // Validate authentication
      const userId = validateAuth();
      if (!userId) {
        alert('Please login to upload documents');
        return;
      }
      console.log('Upload using userId:', userId);
      
      // Rate limiting
      checkRateLimit(userId, 5, 30000); // 5 uploads per 30 seconds
      
      // Validate file
      const fileError = validateFile(file);
      if (fileError) {
        alert(fileError);
        return;
      }
      
      // Validate file content
      await validateFileContent(file);
      
      // IMMEDIATE UI UPDATE: Show document card right away
      setUploadedFiles(prev => ({
        ...prev,
        [key]: { name: file.name, size: file.size }
      }));
      
      // Get or create session ID
      let sessionId = localStorage.getItem('verification-session-id');
      if (!sessionId) {
        // Generate proper UUID format
        const generateUUID = () => {
          if (crypto.randomUUID) {
            return crypto.randomUUID();
          }
          // Fallback UUID generation
          return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
            const r = Math.random() * 16 | 0;
            const v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
          });
        };
        
        sessionId = generateUUID();
        localStorage.setItem('verification-session-id', sessionId);
        
        // Create ruthless session data
        const now = Date.now();
        const expiresAt = new Date(now + 24 * 60 * 60 * 1000).toISOString();
        const sessionData = {
          createdAt: new Date(now).toISOString(),
          expiresAt,
          sessionId
        };
        localStorage.setItem(`session-${sessionId}`, JSON.stringify(sessionData));
        
        console.log('Created new session ID:', sessionId);
      }
      
      // Create FormData for verification upload
      const formData = new FormData();
      formData.append('file', file);
      formData.append('session_id', sessionId);
      formData.append('user_id', userId);
      formData.append('doc_type', key);
      
      // Check if owner details exist, if not provide defaults
      if (!ownerDetails.ownerName || !ownerDetails.propertyId) {
        console.warn('Owner details missing, using defaults for upload');
        // Don't block upload, let backend handle with fallback
      }
      
      // Call verification upload API (owner details already saved separately)
      const headers: Record<string, string> = {};
      
      // Add appropriate authorization header based on user type
      const sellerToken = localStorage.getItem('sellerToken');
      const agentToken = localStorage.getItem('agentToken');
      const authToken = localStorage.getItem('auth_token');
      
      if (sellerToken) {
        headers['Authorization'] = `Bearer ${sellerToken}`;
      } else if (agentToken) {
        headers['Authorization'] = `Bearer ${agentToken}`;
      } else if (authToken) {
        headers['Authorization'] = `Bearer ${authToken}`;
      }
      
      const response = await fetch('http://localhost:8090/api/documents/verification/upload', {
        method: 'POST',
        headers,
        body: formData
      });
      
      if (response.ok) {
        const result = await response.json();
        console.log('Verification upload successful:', result);
        // UI already updated above - no need to update again
        
      } else {
        // Remove from UI if upload failed
        setUploadedFiles(prev => {
          const updated = { ...prev };
          delete updated[key];
          return updated;
        });
        
        let errorMessage = 'Upload failed';
        try {
          const errorData = await response.json();
          errorMessage = errorData.detail || errorMessage;
        } catch {
          const errorText = await response.text();
          errorMessage = errorText || errorMessage;
        }
        console.error('Upload failed:', response.status, response.statusText, errorMessage);
        alert(`Upload failed: ${errorMessage}`);
      }
    } catch (error) {
      // Remove from UI if upload failed
      setUploadedFiles(prev => {
        const updated = { ...prev };
        delete updated[key];
        return updated;
      });
      
      console.error('Upload error:', error);
      alert(`Upload failed: ${error.message || 'Please try again.'}`);
    }
  };

  const handleFileRemove = (key: DocKey) => {
    setUploadedFiles(prev => {
      const updated = { ...prev };
      delete updated[key];
      return updated;
    });
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  };

  const getDocumentSpecificMessage = (docKey: string, status: string, index: number) => {
    const documentMessages = {
      titleDeed: {
        verified: "Title deed verification confirms clear ownership chain with all historical transfers properly recorded. Property boundaries and survey numbers match revenue records perfectly.",
        pending: "Title deed shows valid ownership but requires updated survey settlement for boundary verification. Minor discrepancies in plot dimensions need municipal confirmation.",
        rejected: "Title deed contains significant inconsistencies in ownership history. Multiple gaps in transfer chain require immediate legal review before proceeding."
      },
      saleDeed: {
        verified: "Sale deed is authentic with proper stamp duty payment and registration. All parties' signatures verified against government database.",
        pending: "Sale deed is valid but missing witness signatures on page 2. Document requires notarization completion for full legal compliance.",
        rejected: "Sale deed shows signs of tampering with altered dates and amounts. Registration number doesn't match sub-registrar records."
      },
      encumbrance: {
        verified: "Encumbrance certificate confirms property is free from all legal and financial liabilities. Clean title with no pending disputes.",
        pending: "Encumbrance certificate is valid but shows one pending mortgage entry that needs bank clearance confirmation.",
        rejected: "Encumbrance certificate reveals multiple unresolved liens and legal notices. Property has significant financial encumbrances."
      },
      taxReceipts: {
        verified: "Property tax receipts are current and all dues cleared. Municipal records show consistent payment history.",
        pending: "Tax receipts are mostly current but missing last quarter payment. Requires updated receipt from municipal corporation.",
        rejected: "Tax receipts show significant arrears with penalty notices. Property has outstanding dues exceeding ₹50,000."
      },
      khataPatta: {
        verified: "Khata documents are updated with correct owner details and property classification. Revenue records fully synchronized.",
        pending: "Khata shows correct ownership but property classification needs update from agricultural to residential.",
        rejected: "Khata documents contain wrong owner name and outdated property details. Requires complete mutation process."
      },
      noc: {
        verified: "All NOCs from society, bank, and authorities are valid and current. No objections for property transfer.",
        pending: "Society NOC is valid but bank NOC requires loan closure certificate attachment for completion.",
        rejected: "Multiple NOCs are expired and society has raised objections regarding maintenance dues and violations."
      },
      occupancy: {
        verified: "Occupancy certificate is valid with all safety and construction compliance confirmed by municipal authority.",
        pending: "Occupancy certificate is issued but requires fire safety clearance attachment for complete compliance.",
        rejected: "Occupancy certificate is fake - building plan doesn't match approved construction. Major violations detected."
      },
      builderDocs: {
        verified: "Builder documents including RERA registration and allotment letter are authentic and valid. Project is legally compliant.",
        pending: "RERA registration is valid but possession letter requires builder's signature and official seal for completion.",
        rejected: "Builder documents show RERA violations and project approval is suspended. Multiple buyer complaints registered."
      },
      idProof: {
        verified: "Owner ID proof is authentic government document with all security features intact. Identity verification successful.",
        pending: "ID proof is valid but requires address update to match property location for complete verification.",
        rejected: "ID proof appears to be forged with mismatched photo and details. Document fails authenticity verification."
      },
      ownerAddressProof: {
        verified: "Address proof is current and matches property location. Utility bills confirm genuine residence at the address.",
        pending: "Address proof is valid but utility bill is 4 months old. Requires recent bill within 3 months for compliance.",
        rejected: "Address proof shows different location from property address. Utility connection appears to be fraudulent."
      }
    };

    return documentMessages[docKey as keyof typeof documentMessages]?.[status as keyof typeof documentMessages.titleDeed] || 
           `Document verification ${status} - requires review for ${docKey}.`;
  };

  const getShortMessage = (docKey: string, status: string, index: number) => {
    const shortMessages = {
      titleDeed: {
        verified: "Title deed verification completed successfully with clear ownership chain confirmed. All historical transfers are properly documented and registered. Property boundaries match survey records with no encumbrances detected. Ready for transaction approval.",
        pending: "Title deed shows valid ownership but requires clarification on recent mutation entries. Minor discrepancies in historical records need verification from sub-registrar office. Overall document appears genuine but needs administrative completion. Estimated resolution time: 5-7 business days.",
        rejected: "Critical issues identified in title documentation that prevent transaction approval. Multiple gaps in ownership chain and unresolved disputes detected. Document authenticity concerns require immediate legal consultation. Transaction cannot proceed in current status."
      },
      saleDeed: {
        verified: "Sale deed authentication successful with all transaction details verified. Stamp duty calculations are accurate and registration procedures properly followed. Parties' signatures match government records with no irregularities found. Document supports clear ownership transfer.",
        pending: "Sale deed appears genuine but missing witness signatures on final page. Registration details mostly accurate but require notarization completion. Minor procedural requirements need fulfillment for full legal compliance. Can be resolved through proper documentation.",
        rejected: "Sale deed examination reveals serious authenticity concerns and document tampering evidence. Forged signatures and incorrect stamp duty calculations identified. Registration irregularities suggest fraudulent documentation. Legal action may be necessary for resolution."
      },
      encumbrance: {
        verified: "Encumbrance certificate confirms property is completely free from all legal and financial liabilities. No mortgages, liens, or pending litigation recorded against the property. All previous financial obligations properly discharged with clearance certificates on record. Clean title for transaction.",
        pending: "Encumbrance certificate shows generally clean title with one pending mortgage entry requiring bank confirmation. Recent loan closure needs documentation verification from financial institution. Administrative matter that can be resolved with proper bank clearance certificate. No major legal concerns identified.",
        rejected: "Multiple serious encumbrances detected that pose major transaction risks. Outstanding mortgages, legal notices, and court cases recorded against property. Financial liens and pending litigation create significant legal complications. Professional legal assistance essential before proceeding."
      },
      taxReceipts: {
        verified: "Property tax compliance excellent with all municipal obligations current and properly documented. Payment history shows consistent compliance with no outstanding dues or penalties. Tax classification appropriate for property type with recent assessments satisfied. Meets all transaction requirements.",
        pending: "Tax compliance generally satisfactory but missing current quarter payment receipt. Municipal records show good payment history with minor administrative update needed. Outstanding amount is minimal and can be cleared immediately. No serious compliance issues identified.",
        rejected: "Serious tax compliance failures with substantial arrears and penalty assessments outstanding. Municipal legal notices issued for non-payment with potential lien threats. Tax obligations exceed ₹75,000 requiring immediate resolution. Could prevent transaction completion until cleared."
      },
      khataPatta: {
        verified: "Revenue records comprehensive and accurately reflect current ownership details. All mutations properly recorded with correct property classification confirmed. Ownership entries match supporting documentation with no administrative gaps. Revenue department records fully support claimed ownership.",
        pending: "Revenue records substantially accurate but property classification requires update from agricultural to residential. Recent ownership mutation pending processing by village revenue officer. Administrative procedure that typically takes 15-20 days for completion. No ownership disputes indicated.",
        rejected: "Critical discrepancies in revenue records create serious ownership validation concerns. Incorrect owner details and conflicting property information require immediate resolution. Disputed mutations and administrative errors pose significant transaction risks. Legal consultation advisable for record correction."
      },
      noc: {
        verified: "All required No Objection Certificates obtained and verified as current and valid. Society clearances confirm no outstanding dues with bank NOCs establishing proper loan closure. Authority approvals properly documented with no stakeholder objections. Clear for property transfer.",
        pending: "Most NOCs in order but society clearance requires resolution of minor maintenance dues (₹12,000). Bank NOC pending loan closure certificate attachment for completion. Routine administrative matters that can be resolved within 7-10 days. No major objections identified.",
        rejected: "Multiple NOC objections create serious impediments to property transfer. Society disputes over ₹85,000 in dues with bank obligations unresolved. Authority violations and compliance issues identified. These objections could prevent transaction completion requiring immediate attention."
      },
      occupancy: {
        verified: "Occupancy certificate verification confirms full building compliance with safety and construction standards. Municipal approvals current with structural requirements met and clearances properly documented. Building legally authorized for occupancy meeting all regulatory requirements.",
        pending: "Occupancy documentation substantially complete but fire safety clearance certificate requires attachment. Building meets most compliance standards with minor safety requirements pending. Municipal approval process typically takes 10-15 days for completion. No major violations identified.",
        rejected: "Critical building compliance failures invalidate occupancy authorization. Construction violations and safety non-compliance issues identified by municipal authorities. Unauthorized modifications detected that pose legal and safety risks. Professional consultation essential for compliance resolution."
      },
      builderDocs: {
        verified: "Builder documentation confirms full regulatory compliance with RERA registration current and valid. Project approvals properly obtained with construction and delivery obligations met. Builder demonstrates good standing with authorities fulfilling all legal development requirements.",
        pending: "Builder credentials generally satisfactory but RERA compliance confirmation requires updating. Project completion certificate needs builder's official seal and signature for finalization. Administrative documentation that can be completed within 5-7 business days. No serious compliance concerns.",
        rejected: "Serious builder compliance issues pose significant project risks. RERA violations and project approval irregularities identified with builder financial difficulties reported. These issues could affect project completion and legal title. Professional consultation strongly recommended before proceeding."
      },
      idProof: {
        verified: "Identity verification complete with authentic government documentation confirmed. All security features intact with personal details matching property ownership records. Photographs current and verified meeting all identification requirements. Owner identity established beyond reasonable doubt.",
        pending: "Identity documentation authentic but address correction needed to match current property location. Government ID valid but requires administrative update for complete compliance. Routine procedure that can be completed at nearest government office. No authenticity concerns identified.",
        rejected: "Identity verification failed due to serious authenticity concerns in submitted documentation. Document forgery suspected with mismatched personal details and security features. These issues pose significant security risks indicating possible identity fraud. Professional verification essential before proceeding."
      },
      ownerAddressProof: {
        verified: "Address verification confirms genuine residence through authentic utility documentation. Service connections legitimate with billing addresses matching property records perfectly. Residential status properly established satisfying all verification requirements. Supports legitimate ownership claims.",
        pending: "Address documentation generally acceptable but utility bill dated 4 months ago requires recent replacement. Service connections verified as legitimate but need current billing statement within 3 months. Administrative requirement easily fulfilled through utility provider. No authenticity concerns.",
        rejected: "Address verification failed due to serious discrepancies in utility documentation. Fraudulent service connections suspected with billing addresses not matching property records. These issues raise concerns about residence authenticity and document manipulation. Immediate resolution required for verification."
      }
    };

    return shortMessages[docKey as keyof typeof shortMessages]?.[status as keyof typeof shortMessages.titleDeed] || 
           `${status.charAt(0).toUpperCase() + status.slice(1)} - review required.`;
  };

  const getDetailedMessage = (docKey: string, status: string, index: number) => {
    const detailedMessages = {
      titleDeed: {
        verified: "Comprehensive title verification confirms an unbroken chain of ownership with all historical transfers properly documented and registered. Property boundaries align with survey records, and no encumbrances or disputes are recorded against the title. All stamp duty payments are current, and registration formalities have been completed in accordance with applicable laws. The title is marketable and suitable for immediate transaction without legal impediments.",
        pending: "Initial title examination reveals generally sound ownership documentation; however, certain historical entries require additional verification to ensure complete chain continuity. Minor discrepancies in mutation records need clarification from revenue authorities. While no major legal issues are apparent, obtaining updated certified copies from the sub-registrar will resolve these administrative gaps and provide complete title assurance.",
        rejected: "Critical deficiencies in title documentation pose significant legal risks that preclude transaction approval. Multiple gaps in ownership chain, unresolved disputes, or fraudulent alterations have been identified. The current title status cannot support legitimate ownership claims and may result in future litigation. Immediate legal consultation is essential to address these fundamental title defects before any transaction consideration."
      },
      saleDeed: {
        verified: "Sale deed authentication confirms all transaction details are accurate and legally compliant. Parties' signatures have been verified against government records, stamp duty calculations are correct, and registration procedures were properly followed. The document establishes clear transfer of ownership rights and contains all necessary legal provisions for a valid property transaction. No irregularities or compliance issues have been identified.",
        pending: "Sale deed appears genuine with most transaction details verified; however, certain procedural requirements need completion for full legal compliance. Missing witness signatures, incomplete notarization, or minor registration discrepancies require attention. These are typically administrative matters that can be resolved through proper documentation and do not indicate fundamental transaction invalidity.",
        rejected: "Sale deed examination reveals serious authenticity concerns that invalidate the transaction. Evidence of document tampering, forged signatures, incorrect stamp duty, or registration irregularities has been identified. The document cannot support legitimate ownership transfer and may constitute fraudulent documentation. Legal action may be necessary to address these critical deficiencies."
      },
      encumbrance: {
        verified: "Encumbrance certificate analysis confirms the property is completely free from all legal and financial liabilities. No mortgages, liens, legal notices, or pending litigation are recorded against the property. All previous financial obligations have been properly discharged, and clearance certificates are on record. The property presents no encumbrance risks for prospective buyers or lenders.",
        pending: "Encumbrance examination shows generally clean title with minor outstanding items requiring clarification. Pending mortgage entries need bank confirmation of closure, or recent transactions require updated documentation. These matters are typically administrative and do not indicate serious legal concerns, but resolution is necessary for complete clearance certification.",
        rejected: "Significant encumbrances pose major transaction risks that must be resolved before proceeding. Multiple mortgages, legal notices, court cases, or financial liens are recorded against the property. These encumbrances could result in legal complications, financial liability, or transaction failure. Professional legal assistance is essential to evaluate and resolve these critical issues."
      },
      taxReceipts: {
        verified: "Property tax compliance is exemplary with all obligations current and properly documented. Municipal records show consistent payment history with no outstanding dues or penalty assessments. Tax classification is appropriate for the property type, and all recent assessments have been satisfied. The property meets all municipal tax requirements for transaction approval.",
        pending: "Tax compliance is generally satisfactory with minor administrative requirements pending completion. Recent quarter payments or updated assessment acknowledgments may be needed for complete compliance. These are routine municipal requirements that can be easily satisfied and do not indicate serious tax issues or compliance failures.",
        rejected: "Serious tax compliance failures pose significant transaction obstacles that require immediate resolution. Substantial arrears, penalty assessments, or legal notices for tax non-compliance are recorded. These issues could result in municipal liens, legal action, or transaction prohibition. Professional assistance is recommended to resolve these critical tax matters."
      },
      khataPatta: {
        verified: "Revenue records are comprehensive and accurately reflect current ownership and property details. All mutations have been properly recorded, property classification is correct, and ownership entries match supporting documentation. Revenue department records fully support the claimed ownership and property characteristics without discrepancies or administrative gaps.",
        pending: "Revenue records show substantial accuracy with minor administrative updates required for complete compliance. Property classification changes, address updates, or recent ownership mutations may need processing by revenue authorities. These are standard administrative procedures that do not indicate ownership disputes or serious compliance issues.",
        rejected: "Critical discrepancies in revenue records create serious ownership validation concerns. Incorrect owner details, disputed mutations, or conflicting property information require immediate resolution. These issues could indicate ownership disputes, fraudulent claims, or administrative errors that pose significant transaction risks. Legal consultation is advisable to address these fundamental record discrepancies."
      },
      noc: {
        verified: "All required No Objection Certificates have been obtained and verified as current and valid. Society clearances confirm no outstanding dues or violations, bank NOCs establish loan closure where applicable, and authority approvals are properly documented. No stakeholder objections exist that would impede property transfer or create future complications.",
        pending: "Most required NOCs are in order with minor administrative completions needed for full compliance. Outstanding society dues, pending bank clearances, or authority approvals may require additional documentation or processing time. These are typically routine administrative matters that can be resolved through proper follow-up with relevant authorities.",
        rejected: "Multiple NOC objections create serious impediments to property transfer that require immediate attention. Society disputes, unresolved bank obligations, or authority violations have been identified. These objections could prevent transaction completion and may indicate underlying compliance issues or financial obligations that pose significant risks to prospective buyers."
      },
      occupancy: {
        verified: "Occupancy certificate verification confirms full building compliance with all safety and construction standards. Municipal approvals are current, structural requirements have been met, and all necessary clearances are properly documented. The building is legally authorized for occupancy and meets all regulatory requirements for safe habitation and commercial use.",
        pending: "Occupancy documentation is substantially complete with minor safety clearances requiring finalization. Fire safety certificates, structural compliance confirmations, or updated municipal approvals may be needed for complete regulatory compliance. These are standard safety requirements that can typically be satisfied through proper administrative procedures.",
        rejected: "Critical building compliance failures invalidate occupancy authorization and pose serious safety and legal risks. Construction violations, safety non-compliance, or unauthorized modifications have been identified. These issues could result in municipal action, safety hazards, or legal liability. Professional consultation is essential to address these serious compliance deficiencies."
      },
      builderDocs: {
        verified: "Builder documentation verification confirms full regulatory compliance and project legitimacy. RERA registration is current and valid, project approvals are properly obtained, and all construction and delivery obligations have been met. The builder demonstrates good standing with regulatory authorities and has fulfilled all legal requirements for project development and completion.",
        pending: "Builder credentials are generally satisfactory with minor documentation updates required for complete verification. RERA compliance confirmations, project completion certificates, or financial standing verifications may need updating. These are typically administrative requirements that can be satisfied through proper documentation from relevant authorities.",
        rejected: "Serious builder compliance issues pose significant project and investment risks that require immediate investigation. RERA violations, project approval irregularities, or builder financial difficulties have been identified. These issues could affect project completion, legal title, or investment security. Professional legal and financial consultation is strongly recommended before proceeding."
      },
      idProof: {
        verified: "Identity verification is complete and confirms authentic government documentation with all security features intact. Personal details match property ownership records, photographs are current and verified, and all identification requirements are satisfied. The owner's identity is established beyond reasonable doubt through proper government-issued documentation.",
        pending: "Identity documentation is authentic with minor administrative updates required for complete compliance. Address corrections, photograph updates, or additional supporting documentation may be needed to fully satisfy verification requirements. These are routine administrative matters that can be easily resolved through proper documentation procedures.",
        rejected: "Identity verification has failed due to serious authenticity concerns that require immediate resolution. Document forgery, mismatched details, or fraudulent identification has been detected. These issues pose significant security and legal risks and could indicate identity fraud or ownership misrepresentation. Professional verification and legal consultation are essential before proceeding."
      },
      ownerAddressProof: {
        verified: "Address verification confirms genuine residence at the claimed location through authentic utility documentation. Service connections are legitimate, billing addresses match property records, and residential status is properly established. The address proof satisfies all verification requirements and supports legitimate ownership and residence claims.",
        pending: "Address documentation is generally acceptable with minor updates required for complete compliance. Recent utility bills, service connection confirmations, or address updates may be needed to satisfy current verification standards. These are routine administrative requirements that can be easily fulfilled through proper utility documentation.",
        rejected: "Address verification has failed due to serious discrepancies that raise concerns about residence claims and document authenticity. Fraudulent utility connections, address mismatches, or fabricated documentation has been identified. These issues could indicate fraudulent residence claims or document manipulation that pose significant verification risks requiring immediate resolution."
      }
    };

    return detailedMessages[docKey as keyof typeof detailedMessages]?.[status as keyof typeof detailedMessages.titleDeed] || 
           `Detailed ${status} analysis pending for ${docKey} documentation.`;
  };

  const handleSegregationFileUpload = (e) => {
    const file = e.target.files?.[0];
    if (file) {
      const fileError = validateFile(file);
      if (fileError && !file.name.toLowerCase().endsWith('.pdf')) {
        alert('Please upload a PDF file for segregation');
        return;
      }
      if (file.size > 200 * 1024 * 1024) { // 200MB limit for segregation
        alert('File size must be less than 200MB for segregation');
        return;
      }
      setSegregationFile(file);
      setSegregationResults(null); // Clear previous results
    }
  };

  const handleSegregation = async () => {
    if (!segregationFile) {
      alert('Please upload a PDF file first');
      return;
    }
    
    setIsSegregating(true);
    try {
      const userId = validateAuth();
      if (!userId) {
        alert('Please login to use document segregation');
        return;
      }
      
      const formData = new FormData();
      formData.append('file', segregationFile);
      formData.append('user_id', userId);
      
      const response = await fetch('http://localhost:8090/api/documents/segregation/segregate-large-pdf', {
        method: 'POST',
        body: formData
      });
      
      if (response.ok) {
        const result = await response.json();
        setSegregationResults(result);
        console.log('Segregation completed:', result);
        
        // Display results immediately if available
        if (result.success && result.document_segments > 0) {
          // Fetch detailed results
          try {
            const detailsResponse = await fetch(`http://localhost:8090/api/documents/segregation/segregation-results/${result.session_id}?user_id=${userId}`);
            if (detailsResponse.ok) {
              const detailsResult = await detailsResponse.json();
              if (detailsResult.success && detailsResult.results.document_segments) {
                const segments = JSON.parse(detailsResult.results.document_segments);
                setSegregationResults({
                  ...result,
                  segregated_documents: segments.map((segment, index) => ({
                    document_type: segment.classification || 'Unknown',
                    page_number: segment.page_range || `${index + 1}`,
                    confidence: segment.confidence || 0.5,
                    risk_level: segment.risk_level || 'medium'
                  }))
                });
              }
            }
          } catch (detailsError) {
            console.error('Error fetching detailed results:', detailsError);
          }
        }
      } else {
        const errorText = await response.text();
        console.error('Segregation failed:', errorText);
        alert('Document segregation failed. Please try again.');
      }
    } catch (error) {
      console.error('Segregation error:', error);
      alert('Failed to process document segregation');
    } finally {
      setIsSegregating(false);
    }
  };

  const handleSubmit = async () => {
    setIsLoading(true);
    setProgress(0);
    setCurrentStep(0);
    
    // Calculate dynamic loading duration based on document count
    const documentCount = Object.keys(uploadedFiles).length;
    const getTotalLoadingTime = (count: number) => {
      if (count >= 1 && count <= 5) return 12000; // 12 seconds
      if (count >= 6 && count <= 10) return 16000; // 16 seconds
      return 12000; // default fallback
    };
    
    const totalLoadingTime = getTotalLoadingTime(documentCount);
    const stepDuration = totalLoadingTime / loadingSteps.length;
    
    // Animate through each step
    for (let i = 0; i < loadingSteps.length; i++) {
      setCurrentStep(i);
      
      // Animate progress for this step
      const stepProgress = (i / loadingSteps.length) * 100;
      const nextStepProgress = ((i + 1) / loadingSteps.length) * 100;
      
      // Smooth progress animation
      const animateProgress = (start: number, end: number, duration: number) => {
        return new Promise<void>((resolve) => {
          const startTime = Date.now();
          const animate = () => {
            const elapsed = Date.now() - startTime;
            const progress = Math.min(elapsed / duration, 1);
            const currentProgress = start + (end - start) * progress;
            setProgress(currentProgress);
            
            if (progress < 1) {
              requestAnimationFrame(animate);
            } else {
              resolve();
            }
          };
          animate();
        });
      };
      
      await animateProgress(stepProgress, nextStepProgress, stepDuration);
    }
    
    // Get session ID for navigation
    const sessionId = localStorage.getItem('verification-session-id');
    
    setProgress(100);
    await new Promise(resolve => setTimeout(resolve, 500));
    
    setIsLoading(false);
    navigate(`/docverification/results?session=${sessionId}`);
  };



  const getFinalRecommendation = (verified: number, pending: number, rejected: number, total: number) => {
    const verificationRate = (verified / total) * 100;
    
    if (verificationRate >= 90) {
      return "Excellent documentation quality! Your property demonstrates exceptional legal compliance with minimal risk. All critical documents are verified, indicating a clean title and strong marketability. Recommendation: Proceed with confidence - this property is well-positioned for smooth transactions with institutional lenders and serious buyers.";
    } else if (verificationRate >= 70) {
      return "Strong documentation foundation with minor areas requiring attention. The majority of your documents are verified, indicating good legal standing. Recommendation: Address the pending items promptly to achieve complete compliance and optimize transaction readiness. Consider engaging a property consultant for the remaining items.";
    } else if (verificationRate >= 50) {
      return "Mixed verification results require strategic document management. While some documents are properly authenticated, the pending and rejected items need immediate attention to ensure complete legal compliance. Recommendation: Prioritize resolving rejected documents first, then address pending items. Professional legal consultation is advisable.";
    } else {
      return "Significant documentation gaps require immediate attention. The current verification status indicates potential legal and financial risks that must be addressed before proceeding with any property transactions. Recommendation: Engage a qualified property lawyer immediately for comprehensive document review and remedial action.";
    }
  };

  // Input validation
  const validateOwnerDetails = (): string | null => {
    const { ownerName, aadharNumber, surveyNumber, propertyId, phoneNumber, panNumber, judicialStatus, address, city, state } = ownerDetails;
    
    if (!ownerName.trim() || ownerName.length < 2) return 'Owner name must be at least 2 characters';
    if (!/^\d{12}$/.test(aadharNumber.replace(/\s/g, ''))) return 'Aadhar number must be 12 digits';
    if (!surveyNumber.trim()) return 'Survey number is required';
    if (!propertyId.trim()) return 'Property ID is required';
    if (!/^\d{10}$/.test(phoneNumber.replace(/\s/g, ''))) return 'Phone number must be 10 digits';
    if (!/^[A-Z]{5}\d{4}[A-Z]$/.test(panNumber.replace(/\s/g, ''))) return 'Invalid PAN format';
    if (!['clear', 'pending', 'disputed', 'mortgaged'].includes(judicialStatus)) return 'Invalid judicial status';
    if (!address.trim() || address.length < 10) return 'Address must be at least 10 characters';
    if (!city.trim()) return 'City is required';
    if (!state.trim()) return 'State is required';
    
    return null;
  };

  const handleDetailsSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      // Validate session
      validateSession();
      
      const userId = validateAuth();
      if (!userId) {
        alert('Please login to save details');
        return;
      }
      console.log('Details submit using userId:', userId);
      
      // Rate limiting
      checkRateLimit(userId, 3, 60000); // 3 submissions per minute
      
      // Validate inputs
      const validationError = validateOwnerDetails();
      if (validationError) {
        alert(validationError);
        return;
      }

      // Use coordinates from state or input fields
      let finalLat = '';
      let finalLng = '';
      
      if (coordinates) {
        finalLat = coordinates.lat.toString();
        finalLng = coordinates.lng.toString();
      } else if (ownerDetails.latitude && ownerDetails.longitude) {
        const lat = parseFloat(ownerDetails.latitude);
        const lng = parseFloat(ownerDetails.longitude);
        if (!isNaN(lat) && !isNaN(lng)) {
          finalLat = lat.toString();
          finalLng = lng.toString();
        }
      }

      // Create FormData for owner details
      const formData = new FormData();
      formData.append('user_id', userId);
      formData.append('owner_name', ownerDetails.ownerName.trim());
      formData.append('aadhar_number', ownerDetails.aadharNumber.trim());
      formData.append('survey_number', ownerDetails.surveyNumber.trim());
      formData.append('property_id', ownerDetails.propertyId.trim());
      formData.append('phone_number', ownerDetails.phoneNumber.trim());
      formData.append('pan_number', ownerDetails.panNumber.trim());
      formData.append('judicial_status', ownerDetails.judicialStatus.trim());
      formData.append('latitude', finalLat);
      formData.append('longitude', finalLng);
      
      console.log('Sending coordinates:', { latitude: finalLat, longitude: finalLng });
      formData.append('address', ownerDetails.address || '');
      formData.append('city', ownerDetails.city || '');
      formData.append('pincode', ownerDetails.pincode || '');
      formData.append('locality', ownerDetails.locality || '');
      formData.append('state', ownerDetails.state || 'Karnataka');
      formData.append('landmark', ownerDetails.landmark || '');

        const headers: Record<string, string> = {};
        
        // Add appropriate authorization header based on user type
        const sellerToken = localStorage.getItem('sellerToken');
        const agentToken = localStorage.getItem('agentToken');
        const authToken = localStorage.getItem('auth_token');
        
        if (sellerToken) {
          headers['Authorization'] = `Bearer ${sellerToken}`;
        } else if (agentToken) {
          headers['Authorization'] = `Bearer ${agentToken}`;
        } else if (authToken) {
          headers['Authorization'] = `Bearer ${authToken}`;
        }

        const response = await fetch('http://localhost:8090/api/documents/verification/owner-details', {
          method: 'POST',
          headers,
          body: formData
        });

      if (response.ok) {
        const result = await response.json();
        console.log('Owner details saved:', result);
        console.log('FormData sent:', {
          latitude: finalLat,
          longitude: finalLng,
          coordinates: coordinates
        });
        if (isEditingDetails) {
          setIsEditingDetails(false);
          setHasExistingDetails(true);
        } else {
          setShowDetailsForm(false);
        }
      } else {
        const errorText = await response.text();
        console.error('Failed to save owner details:', errorText);
        console.error('Response status:', response.status);
        alert('Failed to save owner details');
      }
    } catch (error) {
      console.error('Error saving owner details:', error);
      alert('Failed to save owner details');
    }
  };

  const handleInputChange = (field: string, value: string) => {
    // Sanitize input to prevent XSS
    const sanitizedValue = sanitizeString(value);
    setOwnerDetails(prev => ({ ...prev, [field]: sanitizedValue }));
  };







  // Show owner details form only if no details exist
  if (showDetailsForm) {
    return (
      <div className="min-h-screen bg-gray-50 font-display">
        <main className="flex-grow">
          <div className="container mx-auto max-w-4xl px-4 sm:px-6 lg:px-8 py-12">
            <div className="flex flex-col gap-8">
              {/* Header Section */}
              <div className="flex flex-col gap-3 text-center">
                <p className="text-base font-semibold text-blue-600">Step 1 of 5</p>
                <p className="text-4xl font-black tracking-tighter text-black">Property Owner Details</p>
                <p className="text-lg font-normal text-gray-600 max-w-2xl mx-auto">
                  {hasExistingDetails ? 'Review your property and personal information below' : 'Please provide your property and personal information to proceed with document verification'}
                </p>
              </div>

              {/* Progress Steps */}
              <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">
                <div className="flex items-center max-w-6xl mx-auto">
                  {/* Step 1 - Active */}
                  <div className="flex flex-col items-center text-center">
                    <div className="flex items-center justify-center w-12 h-12 bg-blue-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                      1
                    </div>
                    <p className="text-sm font-semibold text-gray-900 mb-1">Property Owner Details</p>
                    <p className="text-xs text-blue-500 font-medium">In Progress</p>
                  </div>
                  
                  {/* Connector 1 */}
                  <div className="flex-1 flex items-center mx-4">
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                    <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                  </div>
                  
                  {/* Step 2 - Pending */}
                  <div className="flex flex-col items-center text-center">
                    <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                      2
                    </div>
                    <p className="text-sm font-medium text-black mb-1">Document Upload</p>
                    <p className="text-xs text-gray-500">Pending</p>
                  </div>
                  
                  {/* Connector 2 */}
                  <div className="flex-1 flex items-center mx-4">
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                    <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                  </div>
                  
                  {/* Step 3 - Pending */}
                  <div className="flex flex-col items-center text-center">
                    <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                      3
                    </div>
                    <p className="text-sm font-medium text-black mb-1">Information Review</p>
                    <p className="text-xs text-gray-500">Pending</p>
                  </div>
                  
                  {/* Connector 3 */}
                  <div className="flex-1 flex items-center mx-4">
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                    <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                  </div>
                  
                  {/* Step 4 - Pending */}
                  <div className="flex flex-col items-center text-center">
                    <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                      4
                    </div>
                    <p className="text-sm font-medium text-black mb-1">RIBIL Score</p>
                    <p className="text-xs text-gray-500">Pending</p>
                  </div>
                  
                  {/* Connector 4 */}
                  <div className="flex-1 flex items-center mx-4">
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                    <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                    <div className="flex-1 h-0.5 bg-gray-300"></div>
                  </div>
                  
                  {/* Step 5 - Pending */}
                  <div className="flex flex-col items-center text-center">
                    <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                      5
                    </div>
                    <p className="text-sm font-medium text-black mb-1">Final Verification</p>
                    <p className="text-xs text-gray-500">Pending</p>
                  </div>
                </div>
              </div>

              {/* Existing Details Review Section */}
              {hasExistingDetails && !isEditingDetails && (
                <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8 mb-8">
                  <div className="mb-6">
                    <div className="flex items-center mb-4">
                      <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center mr-3">
                        <span className="text-white text-sm font-bold">✓</span>
                      </div>
                      <h2 className="text-xl font-bold text-gray-900">Your Property Owner Information</h2>
                    </div>
                    <p className="text-gray-600">Review the details below and proceed to document upload when ready.</p>
                  </div>

                  <div className="space-y-6">
                    {/* Personal Details */}
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-4">Personal Information</h3>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Owner Name</p>
                          <p className="text-base text-gray-900">{ownerDetails.ownerName}</p>
                        </div>
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Phone Number</p>
                          <p className="text-base text-gray-900">{ownerDetails.phoneNumber}</p>
                        </div>
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Aadhar Number</p>
                          <p className="text-base text-gray-900">{ownerDetails.aadharNumber}</p>
                        </div>
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">PAN Number</p>
                          <p className="text-base text-gray-900">{ownerDetails.panNumber}</p>
                        </div>
                      </div>
                    </div>

                    {/* Property Details */}
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Information</h3>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Survey Number</p>
                          <p className="text-base text-gray-900">{ownerDetails.surveyNumber}</p>
                        </div>
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Property ID</p>
                          <p className="text-base text-gray-900">{ownerDetails.propertyId}</p>
                        </div>
                        <div className="bg-gray-50 p-4 rounded-lg md:col-span-2">
                          <p className="text-sm font-medium text-gray-700">Judicial Status</p>
                          <p className="text-base text-gray-900">{ownerDetails.judicialStatus}</p>
                        </div>
                      </div>
                    </div>

                    {/* Location Details */}
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-4">Location Details</h3>
                      <div className="space-y-4">
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <p className="text-sm font-medium text-gray-700">Property Address</p>
                          <p className="text-base text-gray-900">{ownerDetails.address}</p>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                          <div className="bg-gray-50 p-4 rounded-lg">
                            <p className="text-sm font-medium text-gray-700">City</p>
                            <p className="text-base text-gray-900">{ownerDetails.city}</p>
                          </div>
                          <div className="bg-gray-50 p-4 rounded-lg">
                            <p className="text-sm font-medium text-gray-700">Pincode</p>
                            <p className="text-base text-gray-900">{ownerDetails.pincode}</p>
                          </div>
                          <div className="bg-gray-50 p-4 rounded-lg">
                            <p className="text-sm font-medium text-gray-700">State</p>
                            <p className="text-base text-gray-900">{ownerDetails.state}</p>
                          </div>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div className="bg-gray-50 p-4 rounded-lg">
                            <p className="text-sm font-medium text-gray-700">Locality</p>
                            <p className="text-base text-gray-900">{ownerDetails.locality}</p>
                          </div>
                          <div className="bg-gray-50 p-4 rounded-lg">
                            <p className="text-sm font-medium text-gray-700">Landmark</p>
                            <p className="text-base text-gray-900">{ownerDetails.landmark || 'Not specified'}</p>
                          </div>
                        </div>
                        {coordinates && (
                          <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                            <p className="text-sm font-medium text-gray-700 mb-2">Map Location</p>
                            <p className="text-base text-gray-900 flex items-center">
                              <MapPin className="w-4 h-4 mr-2 text-green-600" />
                              {coordinates.lat.toFixed(6)}, {coordinates.lng.toFixed(6)}
                            </p>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  {/* Action Buttons */}
                  <div className="flex justify-between pt-8 border-t border-gray-200 mt-8">
                    <button
                      onClick={() => {
                        setIsEditingDetails(true);
                      }}
                      className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                    >
                      Edit Details
                    </button>
                    <button
                      onClick={() => {
                        setShowDetailsForm(false);
                      }}
                      className="flex items-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                    >
                      <span>Proceed to Document Upload</span>
                      <ArrowRight className="ml-2 w-5 h-5" />
                    </button>
                  </div>
                </div>
              )}

              {/* Owner Details Form */}
              {(!hasExistingDetails || isEditingDetails) && (
              <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">

                <div className="mb-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-2">
                    {isEditingDetails ? 'Update Property Owner Information' : 'Complete Property Owner Information'}
                  </h2>
                  <p className="text-gray-600">
                    {isEditingDetails ? 'Update your details below and save changes.' : 'Please provide comprehensive details about yourself and your property, including location and media files.'}
                  </p>
                </div>
                <form onSubmit={handleDetailsSubmit} className="space-y-8">
                  {/* Personal Details */}
                  <div className="mb-8">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Personal Information</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Owner Name *</label>
                        <input
                          type="text"
                          value={ownerDetails.ownerName}
                          onChange={(e) => handleInputChange('ownerName', e.target.value)}
                          onKeyDown={(e) => {
                            // Allow space key explicitly
                            if (e.key === ' ') {
                              e.stopPropagation();
                            }
                          }}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter full name"
                          required
                        />
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Phone Number *</label>
                        <input
                          type="tel"
                          value={ownerDetails.phoneNumber}
                          onChange={(e) => handleInputChange('phoneNumber', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter 10-digit mobile number"
                          pattern="[0-9]{10}"
                          required
                        />
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Aadhar Card Number *</label>
                        <input
                          type="text"
                          value={ownerDetails.aadharNumber}
                          onChange={(e) => handleInputChange('aadharNumber', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter 12-digit Aadhar number"
                          pattern="[0-9]{12}"
                          required
                        />
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">PAN Number *</label>
                        <input
                          type="text"
                          value={ownerDetails.panNumber}
                          onChange={(e) => handleInputChange('panNumber', e.target.value.toUpperCase())}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter PAN number"
                          pattern="[A-Z]{5}[0-9]{4}[A-Z]{1}"
                          required
                        />
                      </div>
                    </div>
                  </div>

                  {/* Property Details */}
                  <div className="mb-8">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Information</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Survey Number *</label>
                        <input
                          type="text"
                          value={ownerDetails.surveyNumber}
                          onChange={(e) => handleInputChange('surveyNumber', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter survey number"
                          required
                        />
                      </div>
                      
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Property ID *</label>
                        <input
                          type="text"
                          value={ownerDetails.propertyId}
                          onChange={(e) => handleInputChange('propertyId', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="Enter property ID"
                          required
                        />
                      </div>
                    </div>
                  </div>




                  
                  {/* Location Details */}
                  <div className="mb-8">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Location Details</h3>
                    <div className="space-y-6">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Property Address *</label>
                        <AddressAutocomplete
                          value={ownerDetails.address}
                          onChange={(value) => handleInputChange('address', value)}
                          onLocationSelect={(location) => {
                            setCoordinates({ lat: location.lat, lng: location.lng });
                            handleInputChange('address', location.address);
                            handleInputChange('latitude', location.lat.toString());
                            handleInputChange('longitude', location.lng.toString());
                          }}
                          placeholder="Enter complete property address"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">City *</label>
                        <input
                          type="text"
                          value={ownerDetails.city}
                          onChange={(e) => handleInputChange('city', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="e.g., Mumbai"
                        />
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">Pincode *</label>
                          <input
                            type="text"
                            value={ownerDetails.pincode}
                            onChange={(e) => handleInputChange('pincode', e.target.value)}
                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            placeholder="e.g., 400050"
                          />
                        </div>

                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">Locality *</label>
                          <input
                            type="text"
                            value={ownerDetails.locality}
                            onChange={(e) => handleInputChange('locality', e.target.value)}
                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            placeholder="e.g., Bandra West"
                          />
                        </div>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">State *</label>
                        <input
                          type="text"
                          value={ownerDetails.state || ''}
                          onChange={(e) => handleInputChange('state', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="e.g., Maharashtra"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Nearby Landmark</label>
                        <input
                          type="text"
                          value={ownerDetails.landmark || ''}
                          onChange={(e) => handleInputChange('landmark', e.target.value)}
                          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="e.g., Near Bandra Station"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Map Location</label>
                        <p className="text-sm text-gray-500 mb-4">Auto-filled from address/pincode or select on map</p>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-4">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">Latitude</label>
                            <input
                              type="text"
                              value={ownerDetails.latitude}
                              onChange={(e) => handleInputChange('latitude', e.target.value)}
                              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                              placeholder="e.g., 19.0760"
                            />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">Longitude</label>
                            <input
                              type="text"
                              value={ownerDetails.longitude}
                              onChange={(e) => handleInputChange('longitude', e.target.value)}
                              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                              placeholder="e.g., 72.8777"
                            />
                          </div>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                          <button
                            type="button"
                            onClick={() => {
                              const mapsUrl = coordinates 
                                ? `https://www.google.com/maps/@${coordinates.lat},${coordinates.lng},15z`
                                : 'https://www.google.com/maps/@12.9716,77.5946,15z';
                              window.open(mapsUrl, '_blank', 'width=1000,height=700');
                            }}
                            className="flex items-center justify-center gap-2 px-4 py-3 rounded-lg font-medium text-sm" style={{backgroundColor: '#3B82F6', color: 'white'}}
                          >
                            <MapPin className="w-4 h-4" />
                            <span>Select from Map</span>
                          </button>
                          <button
                            type="button"
                            onClick={() => {
                              const lat = parseFloat(ownerDetails.latitude);
                              const lng = parseFloat(ownerDetails.longitude);
                              if (!isNaN(lat) && !isNaN(lng)) {
                                setCoordinates({ lat, lng });
                                alert(`Coordinates set: ${lat.toFixed(6)}, ${lng.toFixed(6)}`);
                              } else {
                                alert('Please enter valid latitude and longitude values first');
                              }
                            }}
                            className="flex items-center justify-center gap-2 px-4 py-3 border rounded-lg font-medium text-sm" style={{backgroundColor: '#F3F4F6', color: '#1F2937', borderColor: '#D1D5DB'}}
                          >
                            <MapPin className="w-4 h-4" />
                            <span>Get Coordinates</span>
                          </button>
                          <button
                            type="button"
                            onClick={() => {
                              if (!navigator.geolocation) {
                                alert('Geolocation is not supported by this browser.');
                                return;
                              }
                              
                              navigator.geolocation.getCurrentPosition(
                                (position) => {
                                  const { latitude, longitude } = position.coords;
                                  setCoordinates({ lat: latitude, lng: longitude });
                                  handleInputChange('latitude', latitude.toString());
                                  handleInputChange('longitude', longitude.toString());
                                  
                                  // Auto-fill address details based on coordinates
                                  if (latitude >= 12.8 && latitude <= 13.2 && longitude >= 77.4 && longitude <= 77.8) {
                                    handleInputChange('address', '25, Arkavathy Layout Main Road, Jakkur, Bengaluru, Karnataka 560064');
                                    handleInputChange('city', 'Bengaluru');
                                    handleInputChange('pincode', '560064');
                                    handleInputChange('locality', 'Jakkur');
                                    handleInputChange('state', 'Karnataka');
                                    handleInputChange('landmark', 'Near Arkavathy Layout Main Road');
                                  } else if (latitude >= 18.8 && latitude <= 19.3 && longitude >= 72.7 && longitude <= 73.1) {
                                    handleInputChange('address', '12, Linking Road, Bandra West, Mumbai, Maharashtra 400050');
                                    handleInputChange('city', 'Mumbai');
                                    handleInputChange('pincode', '400050');
                                    handleInputChange('locality', 'Bandra West');
                                    handleInputChange('state', 'Maharashtra');
                                    handleInputChange('landmark', 'Near Bandra Station');
                                  } else {
                                    handleInputChange('address', `Current Location (${latitude.toFixed(4)}, ${longitude.toFixed(4)})`);
                                    handleInputChange('city', 'Bengaluru');
                                    handleInputChange('state', 'Karnataka');
                                  }
                                  
                                  alert(`Location detected: ${latitude.toFixed(6)}, ${longitude.toFixed(6)}`);
                                },
                                (error) => {
                                  alert('Could not get your current location. Please try again or enter coordinates manually.');
                                },
                                {
                                  enableHighAccuracy: true,
                                  timeout: 10000,
                                  maximumAge: 60000
                                }
                              );
                            }}
                            className="flex items-center justify-center gap-2 px-4 py-3 rounded-lg font-medium text-sm" style={{backgroundColor: '#059669', color: 'white'}}
                          >
                            <MapPin className="w-4 h-4" />
                            <span>Use My Location</span>
                          </button>
                        </div>
                        
                        {coordinates && (
                          <div className="bg-green-50 border border-green-200 rounded-lg p-3 mt-4">
                            <p className="text-sm text-green-700 flex items-center">
                              <MapPin className="w-4 h-4 mr-2" />
                              Location selected: {coordinates.lat.toFixed(6)}, {coordinates.lng.toFixed(6)}
                            </p>
                            <button
                              type="button"
                              onClick={() => {
                                setCoordinates(null);
                                handleInputChange('latitude', '');
                                handleInputChange('longitude', '');
                              }}
                              className="mt-2 text-xs text-red-600 hover:text-red-800"
                            >
                              Clear location
                            </button>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  {/* Legal Status */}
                  <div className="mb-8">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Legal Status</h3>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Judicial Status *</label>
                      <select
                        value={ownerDetails.judicialStatus}
                        onChange={(e) => handleInputChange('judicialStatus', e.target.value)}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        required
                      >
                        <option value="">Select judicial status</option>
                        <option value="clear">Clear - No Legal Issues</option>
                        <option value="pending">Pending - Legal Proceedings Ongoing</option>
                        <option value="disputed">Disputed - Property Under Dispute</option>
                        <option value="mortgaged">Mortgaged - Property Under Loan</option>
                      </select>
                    </div>
                  </div>
                  
                  {/* Summary Section */}
                  {(ownerDetails.ownerName || coordinates) && (
                    <div className="bg-gray-50 border border-gray-200 rounded-lg p-6">
                      <h3 className="text-lg font-semibold text-gray-900 mb-4">📋 Summary</h3>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div>
                          <p className="font-medium text-gray-700">Personal Info</p>
                          <p className="text-gray-600">
                            {ownerDetails.ownerName ? `✓ Owner: ${ownerDetails.ownerName}` : '• Owner name pending'}
                          </p>
                          <p className="text-gray-600">
                            {ownerDetails.phoneNumber ? `✓ Phone: ${ownerDetails.phoneNumber}` : '• Phone pending'}
                          </p>
                        </div>
                        <div>
                          <p className="font-medium text-gray-700">Location Info</p>
                          <p className="text-gray-600">
                            {ownerDetails.city ? `✓ City: ${ownerDetails.city}` : '• City pending'}
                          </p>
                          <p className="text-gray-600">
                            {coordinates ? '✓ Map location selected' : '• Map location pending'}
                          </p>
                        </div>
                      </div>
                    </div>
                  )}
                  
                  <div className="flex justify-between pt-6">
                    {isEditingDetails && (
                      <button
                        type="button"
                        onClick={() => {
                          setIsEditingDetails(false);
                        }}
                        className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                      >
                        Cancel
                      </button>
                    )}
                    <button
                      type="submit"
                      className="flex min-w-[200px] cursor-pointer items-center justify-center overflow-hidden rounded-lg h-12 px-6 bg-blue-500 text-white text-base font-bold leading-normal tracking-wide shadow-lg shadow-blue-500/30 hover:bg-blue-600 focus:outline-none focus:ring-4 focus:ring-blue-500/50 transition-all ml-auto"
                    >
                      <span className="truncate">{isEditingDetails ? 'Update Details' : 'Proceed to Document Upload'}</span>
                      <ArrowRight className="ml-2 w-5 h-5" />
                    </button>
                  </div>
                </form>
              </div>
              )}
            </div>
          </div>
        </main>
        

      </div>
    );
  }
  return (
    <div className="min-h-screen bg-gray-50 font-display">


      {/* Main Content */}
      <main className="flex-grow">
        <div className="container mx-auto max-w-4xl px-4 sm:px-6 lg:px-8 py-12">
          <div className="flex flex-col gap-12">
            {/* Header Section */}
            <div className="flex flex-col gap-3 text-center">
              <p className="text-base font-semibold text-blue-600">Step 2 of 5</p>
              <p className="text-4xl font-black tracking-tighter text-black">Document Upload</p>
              <p className="text-lg font-normal text-gray-600 max-w-2xl mx-auto">Please upload the required documents for property verification.</p>
            </div>

            {/* Progress Steps */}
            <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">
              <div className="flex items-center max-w-6xl mx-auto">
                {/* Step 1 - Completed */}
                <div className="flex flex-col items-center text-center">
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Property Owner Details</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </div>
                
                {/* Connector 1 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>
                
                {/* Step 2 - Active */}
                <div className="flex flex-col items-center text-center">
                  <div className="flex items-center justify-center w-12 h-12 bg-blue-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    2
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Document Upload</p>
                  <p className="text-xs text-blue-500 font-medium">In Progress</p>
                </div>
                
                {/* Connector 2 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>
                
                {/* Step 3 - Pending */}
                <button 
                  onClick={() => navigate('/docverification/results')}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    3
                  </div>
                  <p className="text-sm font-medium text-black mb-1">Information Review</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>
                
                {/* Connector 3 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>
                
                {/* Step 4 - Pending */}
                <button 
                  onClick={() => navigate('/ribil-scorecard-page')}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    4
                  </div>
                  <p className="text-sm font-medium text-black mb-1">RIBIL Score</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>
                
                {/* Connector 4 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>
                
                {/* Step 5 - Pending */}
                <button 
                  onClick={() => navigate('/docverification/final')}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    5
                  </div>
                  <p className="text-sm font-medium text-black mb-1">Final Verification</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>
              </div>
            </div>

            {/* Document Segregation Upload Section */}
            <div className="bg-gradient-to-r from-purple-50 to-indigo-50 rounded-xl border border-purple-200 p-6">
              <div className="flex items-center justify-between mb-4">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-1">📄 Quick Document Segregation</h3>
                  <p className="text-sm text-gray-600">Upload a single PDF to automatically segregate and classify all pages</p>
                </div>
              </div>
              
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <div>
                  {segregationFile ? (
                    <div className="flex items-center justify-between w-full p-4 border border-green-200 rounded-xl bg-green-50">
                      <div className="flex items-center gap-4 overflow-hidden">
                        <CheckCircle className="text-green-600 flex-shrink-0 w-6 h-6" />
                        <div className="flex flex-col overflow-hidden">
                          <p className="text-sm font-semibold truncate text-gray-900">{segregationFile.name}</p>
                          <p className="text-xs text-gray-600 mt-1">{formatFileSize(segregationFile.size)}</p>
                        </div>
                      </div>
                      <button 
                        onClick={() => setSegregationFile(null)}
                        className="text-gray-500 hover:text-red-500 transition-colors flex-shrink-0 ml-4 p-1"
                      >
                        <Trash2 className="w-5 h-5" />
                      </button>
                    </div>
                  ) : (
                    <div className="relative">
                      <input
                        ref={(el) => segregationFileRef.current = el}
                        type="file"
                        accept=".pdf"
                        onChange={handleSegregationFileUpload}
                        className="hidden"
                      />
                      <div
                        onClick={() => segregationFileRef.current?.click()}
                        className="flex flex-col items-center justify-center w-full py-8 px-6 border-2 border-dashed border-purple-300 rounded-xl hover:border-purple-500 hover:bg-purple-50 transition-all duration-200 group cursor-pointer"
                      >
                        <Upload className="w-12 h-12 mb-4 text-purple-400 group-hover:text-purple-500 transition-colors" />
                        <p className="text-sm font-semibold mb-1 text-gray-700 group-hover:text-purple-600">Upload PDF for Segregation</p>
                        <p className="text-xs text-gray-500 group-hover:text-purple-500">PDF only (max. 200MB)</p>
                      </div>
                    </div>
                  )}
                </div>
                
                <div className="flex flex-col justify-center">
                  <button
                    onClick={handleSegregation}
                    disabled={!segregationFile || isSegregating}
                    className="flex items-center justify-center px-6 py-3 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed mb-3"
                  >
                    {isSegregating ? (
                      <>
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                        <span>Processing...</span>
                      </>
                    ) : (
                      <>
                        <span>Start Segregation</span>
                        <ArrowRight className="ml-2 w-4 h-4" />
                      </>
                    )}
                  </button>
                  <p className="text-xs text-gray-500 text-center">AI will analyze and classify each page</p>
                </div>
              </div>
              
              {segregationResults && (
                <div className="mt-6 p-4 bg-white rounded-lg border border-gray-200">
                  <h4 className="font-medium text-gray-900 mb-3">📊 Segregation Results</h4>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                    {segregationResults.segregated_documents?.map((doc, index) => (
                      <div key={index} className="bg-gray-50 rounded-lg p-3 border border-gray-200">
                        <div className="flex items-center justify-between mb-2">
                          <p className="font-medium text-sm text-gray-900">{doc.document_type}</p>
                          <span className="text-xs text-gray-500">Page {doc.page_number}</span>
                        </div>
                        <div className="flex items-center justify-between">
                          <p className="text-xs text-gray-600">Confidence: {(doc.confidence * 100).toFixed(1)}%</p>
                          {doc.risk_level && (
                            <span className={`inline-block px-2 py-1 rounded text-xs font-medium ${
                              doc.risk_level === 'low' ? 'bg-green-100 text-green-800' :
                              doc.risk_level === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                              'bg-red-100 text-red-800'
                            }`}>
                              {doc.risk_level}
                            </span>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>

            {/* Document Upload Sections */}
            <div className="bg-white rounded-xl border border-gray-200 shadow-sm">
              <div className="flex flex-col">
                {documents.map((doc, index) => (
                  <div key={doc.key}>
                    <div className="grid grid-cols-1 lg:grid-cols-5 gap-8 p-6 items-center">
                      <div className="flex flex-col lg:col-span-2">
                        <p className="text-base font-semibold text-gray-900 mb-2">{doc.title}</p>
                        <p className="text-sm text-gray-600 leading-relaxed">{doc.description}</p>
                      </div>
                      <div className="lg:col-span-3">
                        {uploadedFiles[doc.key] ? (
                          <div className="flex items-center justify-between w-full p-4 border border-green-200 rounded-xl bg-green-50">
                            <div className="flex items-center gap-4 overflow-hidden">
                              <CheckCircle className="text-green-600 flex-shrink-0 w-6 h-6" />
                              <div className="flex flex-col overflow-hidden">
                                <p className="text-sm font-semibold truncate text-gray-900">{uploadedFiles[doc.key]!.name}</p>
                                <p className="text-xs text-gray-600 mt-1">{formatFileSize(uploadedFiles[doc.key]!.size)}</p>
                              </div>
                            </div>
                            <button 
                              onClick={() => handleFileRemove(doc.key)}
                              className="text-gray-500 hover:text-red-500 transition-colors flex-shrink-0 ml-4 p-1"
                            >
                              <Trash2 className="w-5 h-5" />
                            </button>
                          </div>
                        ) : (
                          <div className="relative">
                            <input
                              ref={(el) => fileInputRefs.current[doc.key] = el}
                              type="file"
                              accept=".pdf,.jpg,.jpeg,.png"
                              onChange={async (e) => {
                                const file = e.target.files?.[0];
                                if (file) {
                                  await handleFileUpload(doc.key, file);
                                  e.target.value = ''; // Reset input
                                }
                              }}
                              className="hidden"
                            />
                            <div
                              onClick={() => {
                                const input = fileInputRefs.current[doc.key];
                                if (input) input.click();
                              }}
                              className="flex flex-col items-center justify-center w-full py-8 px-6 border-2 border-dashed border-gray-300 rounded-xl hover:border-blue-500 hover:bg-blue-50 transition-all duration-200 group cursor-pointer"
                            >
                              <Upload className="w-12 h-12 mb-4 text-gray-400 group-hover:text-blue-500 transition-colors" />
                              <p className="text-sm font-semibold mb-1 text-gray-700 group-hover:text-blue-600">Click to upload or drag & drop</p>
                              <p className="text-xs text-gray-500 group-hover:text-blue-500">PDF, JPG, PNG (max. 10MB)</p>
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                    {index < documents.length - 1 && <div className="border-t border-gray-100"></div>}
                  </div>
                ))}
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex justify-between items-center pt-4">
              <div className="flex gap-3 items-center">
                <div className="text-xs text-gray-500">
                  Session: {localStorage.getItem('verification-session-id')?.substring(0, 8) || 'None'}...
                </div>
                {Object.keys(uploadedFiles).length > 0 && (
                  <div className="text-sm text-gray-600 flex items-center">
                    {Object.keys(uploadedFiles).length} document(s) in current session
                  </div>
                )}
              </div>
              <button 
                onClick={handleSubmit}
                disabled={isLoading || Object.keys(uploadedFiles).length === 0}
                className="flex min-w-[200px] cursor-pointer items-center justify-center overflow-hidden rounded-lg h-12 px-6 bg-blue-500 text-white text-base font-bold leading-normal tracking-wide shadow-lg shadow-blue-500/30 hover:bg-blue-600 focus:outline-none focus:ring-4 focus:ring-blue-500/50 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                    <span className="truncate">AI Scanning Documents...</span>
                  </>
                ) : (
                  <>
                    <span className="truncate">Proceed to Verification</span>
                    <ArrowRight className="ml-2 w-5 h-5" />
                  </>
                )}
              </button>
            </div>
            
            {/* Professional Loading Overlay */}
            {isLoading && (
              <div className="fixed inset-0 bg-gradient-to-br from-blue-900/90 to-indigo-900/90 backdrop-blur-sm flex items-center justify-center z-50">
                <div className="bg-white rounded-2xl p-6 max-w-md mx-4 shadow-2xl border border-gray-100">
                  {/* Header */}
                  <div className="text-center mb-5">
                   
                    <h3 className="text-lg font-bold text-gray-900 mb-1">AI Document Analysis</h3>
                    <p className="text-sm text-gray-600">Advanced AI processing your documents</p>
                  </div>
                  
                  {/* Progress Bar */}
                  <div className="mb-4">
                    <div className="flex justify-between items-center mb-2">
                      <span className="text-sm font-medium text-gray-700">Progress</span>
                      <span className="text-sm font-bold text-blue-600">{Math.round(progress)}%</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-2.5 overflow-hidden">
                      <div 
                        className="h-full bg-gradient-to-r from-blue-500 to-indigo-600 rounded-full transition-all duration-300 ease-out shadow-sm"
                        style={{ width: `${progress}%` }}
                      ></div>
                    </div>
                  </div>
                  
                  {/* Current Step */}
                  <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-3 border border-blue-100 mb-4">
                    <div className="flex items-center gap-3">
                      <div className="w-7 h-7 bg-blue-500 rounded-full flex items-center justify-center flex-shrink-0">
                        <div className="w-2 h-2 bg-white rounded-full animate-pulse"></div>
                      </div>
                      <div>
                        <p className="text-xs font-semibold text-blue-900 mb-0.5">
                          Step {currentStep + 1} of {loadingSteps.length}
                        </p>
                        <p className="text-sm text-blue-700">
                          {loadingSteps[currentStep]?.message}
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  {/* Compact Steps List */}
                  <div className="space-y-1.5">
                    {loadingSteps.map((step, index) => (
                      <div key={index} className={`flex items-center gap-2 p-2 rounded transition-all duration-300 ${
                        index < currentStep ? 'bg-green-50' :
                        index === currentStep ? 'bg-blue-50' :
                        'bg-gray-50'
                      }`}>
                        <div className={`w-5 h-5 rounded-full flex items-center justify-center text-xs font-bold ${
                          index < currentStep ? 'bg-green-500 text-white' :
                          index === currentStep ? 'bg-blue-500 text-white' :
                          'bg-gray-300 text-gray-600'
                        }`}>
                          {index < currentStep ? '✓' : index + 1}
                        </div>
                        <p className={`text-xs ${
                          index < currentStep ? 'text-green-700 font-medium' :
                          index === currentStep ? 'text-blue-700 font-medium' :
                          'text-gray-500'
                        }`}>
                          {step.message}
                        </p>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </main>
    </div>
  );
}