import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { CheckCircle2, AlertTriangle, XCircle, ArrowLeft, FileText, Shield, Download } from "lucide-react";
import { authenticatedFetch } from '../utils/auth';

type VerificationResult = {
  documentName: string;
  fileName: string;
  fileSize: number;
  status: 'verified' | 'pending' | 'rejected';
  message: string;
  uploadedAt: string;
};

// Helper function to get authentication data for all user types
const getAuthData = (): { userId: string; authToken: string } | null => {
  // Check for buyer authentication (access_token)
  const userStr = localStorage.getItem('user');
  const accessToken = localStorage.getItem('access_token');
  if (userStr && accessToken) {
    const user = JSON.parse(userStr);
    const userId = user?.user_id || user?.id;
    if (userId) {
      return { userId, authToken: accessToken };
    }
  }

  // Fallback to auth_token for backward compatibility
  const authToken = localStorage.getItem('auth_token');
  if (userStr && authToken) {
    const user = JSON.parse(userStr);
    const userId = user?.user_id || user?.id;
    if (userId) {
      return { userId, authToken };
    }
  }

  // Check for seller authentication
  const sellerToken = localStorage.getItem('sellerToken');
  const sellerId = localStorage.getItem('sellerId');
  if (sellerToken && sellerId) {
    return { userId: sellerId, authToken: sellerToken };
  }

  // Check for agent authentication
  const agentToken = localStorage.getItem('agentToken');
  const agentId = localStorage.getItem('agentId');
  if (agentToken && agentId) {
    return { userId: agentId, authToken: agentToken };
  }

  return null;
};

export default function DocumentVerificationFinal() {
  const [results, setResults] = useState<VerificationResult[]>([]);
  const [finalRecommendation, setFinalRecommendation] = useState<string>("");
  const navigate = useNavigate();

  const [ownerDetails, setOwnerDetails] = useState<{judicial_status: string; latitude: number | null; longitude: number | null; owner_name?: string; survey_number?: string; property_id?: string} | null>(null);
  const [ribilScore, setRibilScore] = useState<{overall_score: number; verification_score: number; completeness_score: number; quality_score: number; grade?: string; trust_level?: string} | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  console.log('Component state - loading:', loading, 'ownerDetails:', ownerDetails, 'error:', error);

  useEffect(() => {
    const loadData = async () => {
      
      try {
        setLoading(true);
        const authData = getAuthData();
        const userId = authData?.userId || '550e8400-e29b-41d4-a716-446655440000';
        const authToken = authData?.authToken;
        
        if (!userId) {
          // Use default user ID for testing
          const defaultUserId = '550e8400-e29b-41d4-a716-446655440000';
          console.warn('No user ID found, using default:', defaultUserId);
          // Don't return, continue with default user
        }

        const sessionId = new URLSearchParams(window.location.search).get('session') || localStorage.getItem('verification-session-id');
        
        // Load verification documents from API
        if (sessionId) {
          const documentsResponse = await fetch(`http://localhost:8090/api/documents/verification/session/${sessionId}?user_id=${userId}`, {
            headers: {
              ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
              'Content-Type': 'application/json'
            }
          });
          if (documentsResponse.ok) {
            const documentsData = await documentsResponse.json();
            if (documentsData.success && documentsData.documents) {
              const transformedResults = documentsData.documents.map((doc: any) => ({
                documentName: getDocumentDisplayName(doc.doc_type),
                fileName: doc.filename,
                fileSize: doc.file_size,
                status: doc.verification_status === 'completed' ? 'verified' : 
                       doc.verification_status === 'pending' ? 'pending' : 'rejected',
                message: getVerificationMessage(doc),
                uploadedAt: doc.created_at
              }));
              setResults(transformedResults);
            }
          }
        }

        // Load owner details
        const ownerResponse = await fetch(`http://localhost:8090/api/documents/verification/owner-details/latest/${userId}`, {
          headers: {
            ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
            'Content-Type': 'application/json'
          }
        });
        console.log('Owner details response status:', ownerResponse.status);
        if (ownerResponse.ok) {
          const ownerData = await ownerResponse.json();
          console.log('Owner details data:', ownerData);
          if (ownerData.success && ownerData.owner_details) {
            const details = ownerData.owner_details;
            console.log('Processing owner details:', details);
            setOwnerDetails({
              judicial_status: details.judicial_status || 'Unknown',
              latitude: details.latitude !== null ? parseFloat(details.latitude) : null,
              longitude: details.longitude !== null ? parseFloat(details.longitude) : null,
              owner_name: details.owner_name,
              survey_number: details.survey_number,
              property_id: details.property_id
            });
          } else {
            console.log('No owner details found or API returned unsuccessful response');
          }
        } else {
          console.log('Owner details API call failed with status:', ownerResponse.status);
        }
      } catch (err) {
        setError('Failed to load verification data');
        console.error('Error loading data:', err);
        console.error('Full error details:', err);
      } finally {
        setLoading(false);
      }
    };
    
    loadData();
  }, []);
  
  // Load RIBIL score after owner details are loaded
  useEffect(() => {
    if (!ownerDetails?.property_id) return;
    
    const loadRibilScore = async () => {
      const sessionId = new URLSearchParams(window.location.search).get('session') || localStorage.getItem('verification-session-id');
      const authData = getAuthData();
      const userId = authData?.userId;
      const authToken = authData?.authToken;
      
      if (sessionId && userId) {
        try {
          const ribilResponse = await authenticatedFetch(`http://localhost:8090/api/ribil/score/${ownerDetails.property_id}?session_id=${sessionId}&user_id=${userId}`, {
            headers: {
              ...(authToken && { 'Authorization': `Bearer ${authToken}` })
            }
          });
          const ribilData = await ribilResponse.json();
          console.log('RIBIL API response:', ribilData);
          if (ribilData.success && ribilData.data) {
            setRibilScore({
              overall_score: ribilData.data.overall_score || 0,
              verification_score: ribilData.data.verification_score || 0,
              completeness_score: ribilData.data.completeness_score || 0,
              quality_score: ribilData.data.quality_score || 0,
              grade: ribilData.data.grade,
              trust_level: ribilData.data.trust_level
            });
          } else {
            console.log('RIBIL API returned no data or unsuccessful response');
          }
        } catch (error) {
          console.log('RIBIL score fetch failed:', error);
          console.error('RIBIL fetch error details:', error);
        }
      }
    };
    
    loadRibilScore();
  }, [ownerDetails]);


  const getDocumentDisplayName = (docType: string) => {
    const typeMap: Record<string, string> = {
      'titleDeed': 'Title Deed / Mother Deed',
      'saleDeed': 'Sale Deed / Agreement to Sell',
      'encumbrance': 'Encumbrance Certificate (EC)',
      'taxReceipts': 'Property Tax Receipts',
      'khataPatta': 'Khata / Patta / Mutation Documents',
      'noc': 'NOCs (Society / Authority / Bank)',
      'occupancy': 'Completion / Occupancy Certificate',
      'builderDocs': 'Builder Documents (RERA, Allotment, Possession)',
      'idProof': 'Owner ID Proof'
    };
    return typeMap[docType] || docType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
  };

  const getVerificationMessage = (doc: any) => {
    const result = doc.processed_data?.verification_result;
    if (result?.message) return result.message;
    
    switch (doc.verification_status) {
      case 'completed':
        return `Comprehensive ${getDocumentDisplayName(doc.doc_type).toLowerCase()} verification confirms an unbroken chain of ownership with all historical transfers properly documented and registered. Property boundaries align with survey records, and no encumbrances or disputes are recorded against the title. All stamp duty payments are current, and registration formalities have been completed in accordance with applicable laws. The title is marketable and suitable for immediate transaction without legal impediments.`;
      case 'pending':
        return `${getDocumentDisplayName(doc.doc_type)} appears genuine with most transaction details verified; however, certain procedural requirements need completion for full legal compliance. Missing witness signatures, incomplete notarization, or minor registration discrepancies require attention. These are typically administrative matters that can be resolved through proper documentation and do not indicate fundamental transaction invalidity.`;
      case 'failed':
        return `Significant encumbrances pose major transaction risks that must be resolved before proceeding. Multiple mortgages, legal notices, court cases, or financial liens are recorded against the property. These encumbrances could result in legal complications, financial liability, or transaction failure. Professional legal assistance is essential to evaluate and resolve these critical issues.`;
      default:
        return 'Document processing status unknown.';
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const generatePDF = async () => {
    // Convert favicon to base64
    const getBase64Image = async (url: string): Promise<string> => {
      try {
        const response = await fetch(url);
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result as string);
          reader.readAsDataURL(blob);
        });
      } catch {
        return '';
      }
    };

    const logoBase64 = await getBase64Image('/favicon.png');

    const reportData = {
      generatedAt: new Date().toISOString(),
      totalDocuments: results.length,
      verifiedCount: results.filter(d => d.status === 'verified').length,
      pendingCount: results.filter(d => d.status === 'pending').length,
      rejectedCount: results.filter(d => d.status === 'rejected').length,
      documents: results.map(doc => ({
        documentName: doc.documentName,
        fileName: doc.fileName,
        fileSize: doc.fileSize,
        status: doc.status,
        uploadedAt: doc.uploadedAt,
        message: doc.message
      })),
      finalRecommendation: getConsultantMessage(
        results.filter(d => d.status === 'verified').length,
        results.filter(d => d.status === 'pending').length,
        results.filter(d => d.status === 'rejected').length,
        results.length
      )
    };

    // Create HTML content for PDF
    const htmlContent = `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>NAL Verification Report</title>
  <style>
    @media print {
      body { margin: 0; }
      .no-print { display: none; }
    }
    body { font-family: Arial, sans-serif; margin: 20px; color: #333; line-height: 1.4; }
    .header { position: relative; text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2B256D; padding-bottom: 20px; }
    .header-logo { position: absolute; top: 0; right: 0; width: 60px; height: 60px; }
    .summary { display: flex; justify-content: space-around; margin: 20px 0; gap: 10px; }
    .summary-item { text-align: center; padding: 15px; border-radius: 8px; flex: 1; }
    .verified { background-color: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
    .pending { background-color: #fef3c7; color: #92400e; border: 1px solid #fde68a; }
    .rejected { background-color: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
    .document { margin: 15px 0; padding: 15px; border: 1px solid #e5e7eb; border-radius: 8px; page-break-inside: avoid; }
    .status-verified { border-left: 4px solid #10b981; background-color: #f0fdf4; }
    .status-pending { border-left: 4px solid #f59e0b; background-color: #fffbeb; }
    .status-rejected { border-left: 4px solid #ef4444; background-color: #fef2f2; }
    .recommendation { background-color: #eff6ff; padding: 20px; border-radius: 8px; margin-top: 20px; border: 1px solid #bfdbfe; }
    h1 { color: #2B256D; margin: 0 0 10px 0; }
    h2 { color: #374151; margin: 20px 0 10px 0; }
    h3 { color: #1f2937; margin: 0 0 10px 0; }
    .status-badge { padding: 4px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; text-transform: uppercase; }
    .status-verified-badge { background-color: #dcfce7; color: #166534; }
    .status-pending-badge { background-color: #fef3c7; color: #92400e; }
    .status-rejected-badge { background-color: #fee2e2; color: #991b1b; }
    p { margin: 8px 0; }
    .print-btn { background: #2B256D; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; margin: 20px; }
  </style>
</head>
<body>
  <button class="print-btn no-print" onclick="window.print()">Print/Save as PDF</button>
  
  <div class="header">
    ${logoBase64 ? `<img src="${logoBase64}" alt="NAL Logo" class="header-logo" />` : ''}
    <h1>NAL Verification Report</h1>
    <p>Generated on: ${new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</p>
  </div>
  
  <div class="summary">
    <div class="summary-item verified">
      <h3>${reportData.verifiedCount}</h3>
      <p>Verified Documents</p>
    </div>
    <div class="summary-item pending">
      <h3>${reportData.pendingCount}</h3>
      <p>Pending Review</p>
    </div>
    <div class="summary-item rejected">
      <h3>${reportData.rejectedCount}</h3>
      <p>Rejected Documents</p>
    </div>
  </div>
  
  <h2>Document Details</h2>
  ${reportData.documents.map(doc => `
    <div class="document status-${doc.status}">
      <h3>${doc.documentName}</h3>
      <p><strong>File:</strong> ${doc.fileName}</p>
      <p><strong>Size:</strong> ${formatFileSize(doc.fileSize)}</p>
      <p><strong>Status:</strong> <span class="status-badge status-${doc.status}-badge">${doc.status.toUpperCase()}</span></p>
      <p><strong>Uploaded:</strong> ${formatDate(doc.uploadedAt)}</p>
      <p><strong>Analysis:</strong> ${doc.message}</p>
    </div>
  `).join('')}
  
  <div class="recommendation">
    <h2>Professional Consultant Analysis</h2>
    <p>${reportData.finalRecommendation}</p>
  </div>
</body>
</html>`;

    // Create blob and download
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    // Open in new window for printing
    const printWindow = window.open(url, '_blank');
    if (printWindow) {
      printWindow.onload = () => {
        setTimeout(() => {
          printWindow.focus();
          printWindow.print();
        }, 500);
      };
    } else {
      // Fallback: direct download
      const a = document.createElement('a');
      a.href = url;
      a.download = `document-verification-report-${new Date().toISOString().split('T')[0]}.html`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    }
  };

  const getConsultantMessage = (verified: number, pending: number, rejected: number, total: number) => {
    const verificationRate = (verified / total) * 100;
    const timestamp = Date.now();
    
    const excellentMessages = [
      `Based on our comprehensive analysis of ${total} documents, your property demonstrates exceptional legal compliance with ${verified} documents fully verified. The documentation quality exceeds industry standards, indicating meticulous record-keeping and strong legal foundation. This level of verification significantly reduces transaction risks and enhances property marketability for institutional buyers and lenders.`,
      `Professional assessment reveals outstanding documentation standards across ${total} submitted documents. With ${verified} documents achieving full verification, your property portfolio reflects exemplary legal diligence and compliance management. This comprehensive verification status positions your property advantageously for expedited transactions and premium valuations.`,
      `Our detailed verification process confirms exceptional document quality with ${verified} of ${total} documents meeting all regulatory requirements. The thorough authentication demonstrates professional property management and legal compliance that will facilitate smooth transactions with minimal due diligence delays.`
    ];
    
    const goodMessages = [
      `Analysis of ${total} documents shows strong legal foundation with ${verified} documents verified and ${pending} requiring minor clarifications. The majority verification rate indicates good compliance standards. Addressing the ${pending} pending items will optimize your documentation for complete transaction readiness and enhanced buyer confidence.`,
      `Professional review confirms solid documentation quality across ${total} submitted files. With ${verified} documents authenticated, your property shows good legal standing. The ${pending} pending verifications represent standard administrative requirements that can be resolved promptly to achieve complete compliance.`,
      `Comprehensive assessment reveals commendable document management with ${verified} verified documents out of ${total} total submissions. The pending ${pending} items are typical administrative matters that, once resolved, will establish optimal documentation standards for enhanced market positioning.`
    ];
    
    const mixedMessages = [
      `Our analysis of ${total} documents reveals mixed verification results requiring strategic attention. While ${verified} documents are properly authenticated, the ${pending} pending and ${rejected} rejected items need immediate resolution to ensure complete legal compliance and minimize transaction risks.`,
      `Professional evaluation shows partial verification success with ${verified} documents confirmed out of ${total} submitted. The ${pending + rejected} items requiring attention represent moderate compliance concerns that should be addressed to strengthen your property's legal standing and market viability.`,
      `Document review indicates moderate quality standards with ${verified} verified documents and ${pending + rejected} items needing resolution. This mixed verification status suggests both documentation strengths and areas requiring improvement to achieve optimal transaction readiness.`
    ];
    
    const concernMessages = [
      `Critical analysis of ${total} documents reveals significant compliance gaps with only ${verified} documents verified and ${rejected} rejected items indicating serious legal concerns. The current verification status poses substantial risks for property transactions and requires immediate professional legal intervention.`,
      `Professional assessment identifies major documentation deficiencies across ${total} submitted documents. With ${rejected} rejected items and only ${verified} verified documents, the current status suggests substantial legal and financial risks that must be addressed before proceeding with any property transactions.`,
      `Comprehensive review reveals concerning verification results with ${rejected} document rejections out of ${total} total submissions. The low verification rate of ${Math.round(verificationRate)}% indicates critical compliance issues requiring immediate remedial action and professional legal consultation.`
    ];
    
    // Use timestamp to ensure different messages for different sessions
    const messageIndex = Math.floor((timestamp / 1000) % 3);
    
    if (verificationRate >= 85) {
      return excellentMessages[messageIndex];
    } else if (verificationRate >= 65) {
      return goodMessages[messageIndex];
    } else if (verificationRate >= 40) {
      return mixedMessages[messageIndex];
    } else {
      return concernMessages[messageIndex];
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 font-display">
      <main className="flex-grow">
        <div className="container mx-auto max-w-6xl px-4 sm:px-6 lg:px-8 py-12">
          <div className="flex flex-col gap-8">
            {/* Header Section */}
            <div className="flex flex-col gap-3 text-center">
              <p className="text-base font-semibold text-blue-600">Step 5 of 5</p>
              <p className="text-4xl font-black tracking-tighter text-black">Final Verification Report</p>
              <p className="text-lg font-normal text-gray-600 max-w-2xl mx-auto">Comprehensive analysis and recommendations for your property documents.</p>
            </div>

            {/* Final Professional Recommendation */}
            <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
              <div className="flex items-start gap-4">
                <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center flex-shrink-0">
                  <Shield className="w-6 h-6 text-white" />
                </div>
                <div className="flex-1">
                  <h3 className="text-xl font-semibold text-gray-900 mb-4">Professional Recommendation</h3>
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                    <p className="text-sm text-blue-800 leading-relaxed">
                      {(() => {
                        const verifiedCount = results.filter(r => r.status === 'verified').length;
                        const pendingCount = results.filter(r => r.status === 'pending').length;
                        const rejectedCount = results.filter(r => r.status === 'rejected').length;
                        const totalCount = results.length;
                        
                        if (totalCount === 0) {
                          return "No documents have been uploaded for verification. Please upload your property documents to receive a comprehensive analysis.";
                        } else if (rejectedCount > 0) {
                          return `Critical issues identified in ${rejectedCount} document(s). Immediate attention required before proceeding with property transaction. Legal consultation recommended to resolve compliance matters.`;
                        } else if (pendingCount > 0) {
                          return `${verifiedCount} of ${totalCount} documents verified successfully. ${pendingCount} document(s) require additional review. Please address pending items to complete verification process.`;
                        } else if (verifiedCount === totalCount) {
                          return `Excellent! All ${totalCount} documents have been successfully verified. Your property documentation meets all legal requirements and the transaction can proceed with full confidence.`;
                        } else {
                          return "Based on the comprehensive analysis of your property documents, our professional assessment indicates the overall verification status and recommended next steps for your property transaction.";
                        }
                      })()}
                    </p>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                    <div className="text-center p-4 bg-green-50 rounded-lg border border-green-200">
                      <div className="text-2xl font-bold text-green-600 mb-1">
                        {results.filter(r => r.status === 'verified').length}
                      </div>
                      <div className="text-sm text-green-800 font-medium">Verified Documents</div>
                    </div>
                    <div className="text-center p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                      <div className="text-2xl font-bold text-yellow-600 mb-1">
                        {results.filter(r => r.status === 'pending').length}
                      </div>
                      <div className="text-sm text-yellow-800 font-medium">Pending Review</div>
                    </div>
                    <div className="text-center p-4 bg-red-50 rounded-lg border border-red-200">
                      <div className="text-2xl font-bold text-red-600 mb-1">
                        {results.filter(r => r.status === 'rejected').length}
                      </div>
                      <div className="text-sm text-red-800 font-medium">Rejected Documents</div>
                    </div>
                  </div>

                  <div className="flex justify-center gap-4">
                    <button 
                      onClick={() => {
                        const sessionId = new URLSearchParams(window.location.search).get('session') || localStorage.getItem('verification-session-id');
                        navigate(`/ribil-scorecard-page?session=${sessionId}`);
                      }}
                      className="flex items-center justify-center px-6 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors"
                    >
                      <ArrowLeft className="w-5 h-5 mr-2" />
                      Back to RIBIL Score
                    </button>
                    
                    <button 
                      onClick={generatePDF}
                      className="flex items-center justify-center px-6 py-3 border border-blue-500 text-blue-500 rounded-lg hover:bg-blue-50 transition-colors"
                    >
                      <Download className="w-5 h-5 mr-2" />
                      Download Report
                    </button>
                    
                    <button 
                      onClick={() => navigate('/user/dashboard')}
                      className="flex items-center justify-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                    >
                      Go to Dashboard
                    </button>
                  </div>
                </div>
              </div>
            </div>

            {/* Document Results Section */}
            {results.length > 0 && (
              <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <div className="flex justify-between items-center mb-6">
                  <h3 className="text-xl font-semibold text-gray-900">Document Verification Details</h3>
                  <button 
                    onClick={generatePDF}
                    className="flex items-center px-4 py-2 text-sm bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Download Report
                  </button>
                </div>
                <div className="space-y-4">
                  {results.map((result, index) => (
                    <div key={index} className="border border-gray-200 rounded-lg p-4">
                      <div className="flex items-start justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <div className={`w-8 h-8 rounded-full flex items-center justify-center ${
                            result.status === 'verified' ? 'bg-green-100' :
                            result.status === 'pending' ? 'bg-yellow-100' : 'bg-red-100'
                          }`}>
                            {result.status === 'verified' ? (
                              <CheckCircle2 className="w-5 h-5 text-green-600" />
                            ) : result.status === 'pending' ? (
                              <AlertTriangle className="w-5 h-5 text-yellow-600" />
                            ) : (
                              <XCircle className="w-5 h-5 text-red-600" />
                            )}
                          </div>
                          <div>
                            <h4 className="font-semibold text-gray-900">{result.documentName}</h4>
                            <div className="flex items-center gap-4 text-sm text-gray-500">
                              <span>{result.fileName}</span>
                              <span>{formatFileSize(result.fileSize)}</span>
                              <span>{formatDate(result.uploadedAt)}</span>
                            </div>
                          </div>
                        </div>
                        <span className={`px-3 py-1 rounded-full text-xs font-medium ${
                          result.status === 'verified' ? 'bg-green-100 text-green-800' :
                          result.status === 'pending' ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800'
                        }`}>
                          {result.status === 'verified' ? 'Verified' :
                           result.status === 'pending' ? 'Pending' : 'Rejected'}
                        </span>
                      </div>
                      <p className="text-sm text-gray-600 leading-relaxed">{result.message}</p>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Loading State */}
            {loading && (
              <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-8 text-center">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                <p className="text-gray-600">Loading verification results...</p>
              </div>
            )}

            {/* Error State */}
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-xl p-6">
                <div className="flex items-center gap-3">
                  <XCircle className="w-6 h-6 text-red-600" />
                  <div>
                    <h3 className="font-semibold text-red-900">Error Loading Data</h3>
                    <p className="text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}

            {/* No Documents State */}
            {!loading && !error && results.length === 0 && (
              <div className="bg-gray-50 border border-gray-200 rounded-xl p-8 text-center">
                <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No Documents Found</h3>
                <p className="text-gray-600">No verification documents were found for this session.</p>
              </div>
            )}
          </div>
        </div>
      </main>
    </div>
  );
}