import { useEffect, useState } from "react";
import { useNavigate, useSearchParams } from "react-router-dom";
import { CheckCircle2, AlertTriangle, XCircle, ArrowLeft, FileText, Download, Loader2, Shield } from "lucide-react";
import { ErrorBoundary, useErrorHandler } from '../components/ErrorBoundary';
import { VERIFICATION_STATUS, JUDICIAL_STATUS, type VerificationStatus } from '../utils/constants';
import '../styles/document-verification.css';
import { formatDate } from '../utils/regionalFormatters';
import { VerificationCharts } from '../components/VerificationCharts';

type VerificationDocument = {
  id: number;
  doc_type: string;
  filename: string;
  file_size: number;
  verification_status: VerificationStatus;
  created_at: string;
  processed_data?: {
    verification_result?: {
      message?: string;
      status?: string;
      score?: number;
      data_validation?: { valid: boolean; issues: string[] };
      forgery_detection?: { status: string; fraud_risk: string };
      duplicate_detection?: { status: string };
      authenticity_score?: { overall: number };
      dispute_risk?: { risk_level: string; risk_score: string };
    };
  };
  features?: {
    feature_1_enhanced_validation?: {
      status: string;
      issues: string[];
      anomaly_score: number;
      anomaly_detected: boolean;
    };
    feature_2_forgery_detection?: {
      fraud_detected: boolean;
      fraud_score: number;
      fraud_level: string;
      ela_score: number;
    };
    feature_3_duplicate_detection?: {
      is_duplicate: boolean;
      hash: string;
    };
    feature_4_authenticity_scoring?: {
      authenticity_score: number;
      overall_status: string;
    };
    feature_5_dispute_prediction?: {
      dispute_probability: number;
      dispute_risk_level: string;
      recommendation: string;
    };
  };
};

type OwnerDetails = {
  owner_name: string;
  judicial_status: string;
  latitude: number | null;
  longitude: number | null;
};

// Sanitize function to prevent XSS
const sanitizeString = (str: string): string => {
  if (!str || typeof str !== 'string') return '';
  return str.replace(/[<>"'&]/g, (match) => {
    const map: Record<string, string> = {
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#x27;',
      '&': '&amp;'
    };
    return map[match] || match;
  }).trim();
};

// Validate user authentication with support for all user types
const validateAuth = (): { userId: string; authToken: string } | null => {
  try {
    // Check for buyer authentication (access_token)
    const userStr = localStorage.getItem('user');
    const accessToken = localStorage.getItem('access_token');
    if (userStr && accessToken) {
      const user = JSON.parse(userStr);
      const userId = user?.user_id || user?.id;
      if (userId && userId.length === 36 && userId.includes('-')) {
        return { userId: sanitizeString(userId), authToken: accessToken };
      }
    }

    // Fallback to auth_token for backward compatibility
    const authToken = localStorage.getItem('auth_token');
    if (userStr && authToken) {
      const user = JSON.parse(userStr);
      const userId = user?.user_id || user?.id;
      if (userId && userId.length === 36 && userId.includes('-')) {
        return { userId: sanitizeString(userId), authToken };
      }
    }

    // Check for seller authentication
    const sellerToken = localStorage.getItem('sellerToken');
    const sellerId = localStorage.getItem('sellerId');
    if (sellerToken && sellerId) {
      return { userId: sanitizeString(sellerId), authToken: sellerToken };
    }

    // Check for agent authentication
    const agentToken = localStorage.getItem('agentToken');
    const agentId = localStorage.getItem('agentId');
    if (agentToken && agentId) {
      return { userId: sanitizeString(agentId), authToken: agentToken };
    }

    throw new Error('User not authenticated');
  } catch (error) {
    console.error('Authentication failed:', error.message);
    return null;
  }
};

function DocumentVerificationResultsContent() {
  const [documents, setDocuments] = useState<VerificationDocument[]>([]);
  const [ownerDetails, setOwnerDetails] = useState<OwnerDetails | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const { handleError } = useErrorHandler();

  useEffect(() => {
    const loadRealData = async () => {
      try {
        setLoading(true);
        setError(null);

        const authData = validateAuth();
        if (!authData) {
          setError('Authentication required. Please login to view results.');
          setLoading(false);
          return;
        }
        console.log('Using userId for results:', authData.userId);

        let sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
        if (!sessionId) {
          // Generate new session if missing
          sessionId = crypto.randomUUID ? crypto.randomUUID() : 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
            const r = Math.random() * 16 | 0;
            const v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
          });
          localStorage.setItem('verification-session-id', sessionId);
          console.log('Generated new session for results:', sessionId);
        } else {
          // Ensure localStorage is synced with URL session
          localStorage.setItem('verification-session-id', sessionId);
          console.log('Using session for results:', sessionId);
        }

        // Fetch documents for current session only
        console.log('Fetching documents for session:', sessionId);
        const documentsResponse = await fetch(`http://localhost:8090/api/documents/verification/results/${encodeURIComponent(sessionId)}?user_id=${encodeURIComponent(authData.userId)}`, {
          headers: {
            'Authorization': `Bearer ${authData.authToken}`,
            'Content-Type': 'application/json'
          }
        });

        if (!documentsResponse || !documentsResponse.ok) {
          throw new Error('Failed to fetch verification documents');
        }

        const documentsData = await documentsResponse.json();
        console.log('Documents API response:', documentsData);
        console.log('Documents array:', documentsData.documents);
        console.log('Documents count:', documentsData.documents?.length);
        if (!documentsData.success) {
          throw new Error(documentsData.message || 'Failed to load documents');
        }

        console.log('Setting documents:', documentsData.documents);
        setDocuments(documentsData.documents || []);

        // Fetch owner details (still needs user_id for owner-specific data)
        const ownerResponse = await fetch(`http://localhost:8090/api/documents/verification/owner-details/latest/${encodeURIComponent(authData.userId)}`, {
          headers: {
            'Authorization': `Bearer ${authData.authToken}`,
            'Content-Type': 'application/json'
          }
        });
        if (ownerResponse.ok) {
          const ownerData = await ownerResponse.json();
          console.log('Owner details response:', ownerData);
          if (ownerData.success && ownerData.owner_details) {
            const details = ownerData.owner_details;
            console.log('Owner details data:', details);
            const lat = details.latitude !== null ? parseFloat(details.latitude) : null;
            const lng = details.longitude !== null ? parseFloat(details.longitude) : null;
            console.log('Parsed coordinates:', { lat, lng });
            setOwnerDetails({
              owner_name: sanitizeString(details.owner_name || details.ownerName || ''),
              judicial_status: sanitizeString(details.judicial_status || details.judicialStatus || ''),
              latitude: lat,
              longitude: lng
            });
          }
        } else {
          console.error('Failed to fetch owner details:', ownerResponse.status);
        }

      } catch (err) {
        console.error('Error loading verification data:', err);
        const errorMessage = err instanceof Error ? err.message : 'Failed to load verification data';
        setError(errorMessage);
        handleError(err instanceof Error ? err : new Error(errorMessage));
      } finally {
        setLoading(false);
      }
    };

    loadRealData();
  }, [searchParams]);

  const getDocumentDisplayName = (docType: string) => {
    const typeMap: Record<string, string> = {
      'title_deed': 'Title Deed / Mother Deed',
      'sale_deed': 'Sale Deed / Agreement to Sell',
      'encumbrance_certificate': 'Encumbrance Certificate (EC)',
      'property_card': 'Property Card / Pahani',
      'survey_settlement': 'Survey Settlement Record',
      'tax_receipt': 'Property Tax Receipt',
      'possession_certificate': 'Possession Certificate'
    };
    return typeMap[docType] || docType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
  };

  const getVerificationMessage = (doc: VerificationDocument) => {
    // Check if duplicate (ENHANCED with time-based warning)
    if (doc.verification_status === 'failed' && doc.features?.feature_3_duplicate_detection?.is_duplicate) {
      const f3 = doc.features.feature_3_duplicate_detection;
      const originalFile = f3.original_filename || 'unknown';
      const dupType = f3.duplicate_type || 'verified_document';
      const matchPct = Math.round((f3.similarity_score || 1.0) * 100);
      const minutesAgo = f3.minutes_ago || 0;
      const waitMinutes = Math.max(0, 240 - minutesAgo);
      const waitHours = Math.floor(waitMinutes / 60);
      const waitMins = waitMinutes % 60;
      
      if (dupType === 'verified_document') {
        return `⚠️ DUPLICATE DOCUMENT DETECTED\n\nREASON: This document was already verified within the past 4 hours.\n\nORIGINAL FILE: ${originalFile}\nVERIFIED: ${minutesAgo} minutes ago\nMATCH: ${matchPct}% similarity\nSURVEY NUMBER: ${doc.survey_number}\nPROPERTY ID: ${doc.property_number}\n\n⏰ WAIT TIME: ${waitHours}h ${waitMins}m remaining (4-hour window)\n\nThe system detected an existing verified document for this property. Please wait for the verification window to expire before uploading again.`;
      } else if (dupType === 'exact_content') {
        return `🚫 DUPLICATE DOCUMENT REJECTED\n\nREASON: This exact document has already been uploaded to the system.\n\nORIGINAL FILE: ${originalFile}\nUPLOADED: ${minutesAgo} minutes ago\nMATCH: ${matchPct}% (Identical content)\nSURVEY NUMBER: ${doc.survey_number}\nPROPERTY ID: ${doc.property_number}\n\n⏰ WAIT TIME: ${waitHours}h ${waitMins}m remaining\n\nThe system detected that this file is byte-for-byte identical to a previously uploaded document. Please verify you are uploading the correct document.`;
      } else if (dupType === 'content_match') {
        // Enhanced content match with field details
        let message = `🚫 DUPLICATE DOCUMENT REJECTED\n\nREASON: Document content ${matchPct}% identical to existing document.\n\nORIGINAL FILE: ${originalFile}\nSURVEY NUMBER: ${doc.survey_number}\nPROPERTY ID: ${doc.property_number}\n`;
        
        if (matchingFields.length > 0) {
          message += `\nMATCHING FIELDS (${matchDetails.total_matching || matchingFields.length}): ${matchingFields.slice(0, 5).join(', ')}${matchingFields.length > 5 ? '...' : ''}`;
        }
        
        if (differingFields.length > 0) {
          message += `\nDIFFERING FIELDS (${matchDetails.total_differing || differingFields.length}): ${differingFields.slice(0, 3).map((d: any) => d.field).join(', ')}${differingFields.length > 3 ? '...' : ''}`;
        }
        
        message += `\n\nThe system found an existing document with ${matchPct}% content similarity. This indicates the same document or a very similar version already exists.`;
        
        return message;
      } else {
        return `🚫 DUPLICATE DOCUMENT REJECTED\n\nREASON: A document for this property already exists in the system.\n\nORIGINAL FILE: ${originalFile}\nMATCH: ${matchPct}% (Same property)\nSURVEY NUMBER: ${doc.survey_number}\nPROPERTY ID: ${doc.property_number}\n\nThe system found an existing document with matching property identifiers. If you need to upload a different document type for the same property, please contact support.`;
      }
    }
    
    const result = doc.processed_data?.verification_result;
    if (result?.message) {
      return result.message;
    }

    // Generate rejection with REAL validation reasons from backend
    if (doc.verification_status === 'failed') {
      const f1 = doc.features?.feature_1_enhanced_validation;
      const f2 = doc.features?.feature_2_forgery_detection;
      const f4 = doc.features?.feature_4_authenticity_scoring;
      const f5 = doc.features?.feature_5_dispute_prediction;
      
      // Priority 1: Use F1 validation issues (real field-level reasons)
      if (f1 && f1.issues && f1.issues.length > 0) {
        const issuesList = f1.issues.join('\n');
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED:\n\n${issuesList}`;
      }
      
      // Priority 2: Check missing fields from F1
      if (f1 && f1.missing_fields && f1.missing_fields.length > 0) {
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED: Missing fields - ${f1.missing_fields.join(', ')}. OCR could not extract these from document.`;
      }
      
      // Priority 3: Check mismatched fields from F1
      if (f1 && f1.mismatched_fields && f1.mismatched_fields.length > 0) {
        const mismatches = f1.mismatched_fields.map((m: any) => 
          `${m.field}: extracted '${m.extracted}' vs provided '${m.user_provided}'`
        ).join('\n');
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED: Field mismatches detected:\n\n${mismatches}`;
      }
      
      // Priority 4: OCR/Authenticity failure
      if (f4 && f4.authenticity_score < 30) {
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED: OCR extraction failed (score: ${f4.authenticity_score}/100). Document quality too poor - upload clearer scan.`;
      }
      
      // Priority 5: Forgery detected
      if (f2 && f2.fraud_score > 0.5) {
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED: Tampering detected (fraud: ${Math.round(f2.fraud_score * 100)}%). Upload original unmodified document.`;
      }
      
      // Priority 6: High dispute risk
      if (f5 && f5.dispute_probability > 0.7) {
        return `${getDocumentDisplayName(doc.doc_type)} REJECTED: High dispute risk (${Math.round(f5.dispute_probability * 100)}%). Data inconsistencies detected.`;
      }
      
      // Default rejection
      return `${getDocumentDisplayName(doc.doc_type)} REJECTED: Verification failed. Upload higher quality document or contact support.`;
    }

    switch (doc.verification_status) {
      case 'completed':
        return `${getDocumentDisplayName(doc.doc_type)} verification completed successfully. Document meets all validation requirements.`;
      case 'pending':
        return `${getDocumentDisplayName(doc.doc_type)} requires additional review. Some validation criteria need manual verification.`;
      default:
        return 'Document processing status unknown.';
    }
  };

  const getJudicialStatusDisplay = (status: string) => {
    const statusMap: Record<string, string> = {
      'clear': 'Clear - No Legal Issues',
      'pending': 'Pending - Legal Proceedings Ongoing',
      'disputed': 'Disputed - Property Under Dispute',
      'mortgaged': 'Mortgaged - Property Under Loan'
    };
    return statusMap[status] || sanitizeString(status);
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  };

  const formatDateTime = (dateString: string) => {
    const base = formatDate(dateString);
    const d = new Date(dateString);
    const time = isNaN(d.getTime())
      ? ''
      : ` ${d.toLocaleTimeString('en-IN', { hour: '2-digit', minute: '2-digit' })}`;
    return `${base}${time}`.trim();
  };

  const verifiedCount = documents.filter(d => d.verification_status === VERIFICATION_STATUS.COMPLETED).length;
  const pendingCount = documents.filter(d => d.verification_status === VERIFICATION_STATUS.PENDING).length;
  const rejectedCount = documents.filter(d => d.verification_status === VERIFICATION_STATUS.FAILED).length;
  
  // Filter out rejected/duplicate documents for charts
  const validDocuments = documents.filter(d => d.verification_status !== VERIFICATION_STATUS.FAILED);

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 font-display flex items-center justify-center">
        <div className="text-center">
          <Loader2 className="w-8 h-8 animate-spin mx-auto mb-4 text-blue-600" />
          <p className="text-gray-600">Loading verification results...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 font-display flex items-center justify-center">
        <div className="text-center max-w-md">
          <XCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Error Loading Results</h2>
          <p className="text-gray-600 mb-4">{error}</p>
          <button
            onClick={() => navigate('/docverification')}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Back to Upload
          </button>
        </div>
      </div>
    );
  }

  const generatePDF = async () => {
    // Convert favicon to base64
    const getBase64Image = async (url: string): Promise<string> => {
      try {
        const response = await fetch(url);
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result as string);
          reader.readAsDataURL(blob);
        });
      } catch {
        return '';
      }
    };

    const logoBase64 = await getBase64Image('/favicon.png');

    const reportData = {
      generatedAt: new Date().toISOString(),
      totalDocuments: documents.length,
      verifiedCount: documents.filter(d => d.verification_status === 'completed').length,
      pendingCount: documents.filter(d => d.verification_status === 'pending').length,
      rejectedCount: documents.filter(d => d.verification_status === 'failed').length,
      documents: documents.map(doc => ({
        documentName: getDocumentDisplayName(doc.doc_type),
        fileName: doc.filename,
        fileSize: doc.file_size,
        status: doc.verification_status,
        uploadedAt: doc.created_at,
        message: getVerificationMessage(doc)
      })),
      finalRecommendation: getConsultantMessage(
        documents.filter(d => d.verification_status === 'completed').length,
        documents.filter(d => d.verification_status === 'pending').length,
        documents.filter(d => d.verification_status === 'failed').length,
        documents.length
      )
    };

    // Create HTML content for PDF
    const htmlContent = `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>NAL Verification Report</title>
  <style>
    @media print {
      body { margin: 0; }
      .no-print { display: none; }
    }
    body { font-family: Arial, sans-serif; margin: 20px; color: #333; line-height: 1.4; }
    .header { position: relative; text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2B256D; padding-bottom: 20px; }
    .header-logo { position: absolute; top: 0; right: 0; width: 60px; height: 60px; }
    .summary { display: flex; justify-content: space-around; margin: 20px 0; gap: 10px; }
    .summary-item { text-align: center; padding: 15px; border-radius: 8px; flex: 1; }
    .verified { background-color: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
    .pending { background-color: #fef3c7; color: #92400e; border: 1px solid #fde68a; }
    .rejected { background-color: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
    .document { margin: 15px 0; padding: 15px; border: 1px solid #e5e7eb; border-radius: 8px; page-break-inside: avoid; }
    .status-verified { border-left: 4px solid #10b981; background-color: #f0fdf4; }
    .status-pending { border-left: 4px solid #f59e0b; background-color: #fffbeb; }
    .status-rejected { border-left: 4px solid #ef4444; background-color: #fef2f2; }
    .recommendation { background-color: #eff6ff; padding: 20px; border-radius: 8px; margin-top: 20px; border: 1px solid #bfdbfe; }
    h1 { color: #2B256D; margin: 0 0 10px 0; }
    h2 { color: #374151; margin: 20px 0 10px 0; }
    h3 { color: #1f2937; margin: 0 0 10px 0; }
    .status-badge { padding: 4px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; text-transform: uppercase; }
    .status-verified-badge { background-color: #dcfce7; color: #166534; }
    .status-pending-badge { background-color: #fef3c7; color: #92400e; }
    .status-rejected-badge { background-color: #fee2e2; color: #991b1b; }
    p { margin: 8px 0; }
    .print-btn { background: #2B256D; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; margin: 20px; }
  </style>
</head>
<body>
  <button class="print-btn no-print" onclick="window.print()">Print/Save as PDF</button>
  
  <div class="header">
    ${logoBase64 ? `<img src="${logoBase64}" alt="NAL Logo" class="header-logo" />` : ''}
    <h1>NAL Verification Report</h1>
    <p>Generated on: ${formatDateTime(new Date().toISOString())}</p>
  </div>
  
  <div class="summary">
    <div class="summary-item verified">
      <h3>${reportData.verifiedCount}</h3>
      <p>Verified Documents</p>
    </div>
    <div class="summary-item pending">
      <h3>${reportData.pendingCount}</h3>
      <p>Pending Review</p>
    </div>
    <div class="summary-item rejected">
      <h3>${reportData.rejectedCount}</h3>
      <p>Rejected Documents</p>
    </div>
  </div>
  
  <h2>Document Details</h2>
  ${reportData.documents.map(doc => `
    <div class="document status-${doc.status}">
      <h3>${doc.documentName}</h3>
      <p><strong>File:</strong> ${doc.fileName}</p>
      <p><strong>Size:</strong> ${formatFileSize(doc.fileSize)}</p>
      <p><strong>Status:</strong> <span class="status-badge status-${doc.status}-badge">${doc.status.toUpperCase()}</span></p>
      <p><strong>Uploaded:</strong> ${formatDateTime(doc.uploadedAt)}</p>
      <p><strong>Analysis:</strong> ${doc.message}</p>
    </div>
  `).join('')}
  
  <div class="recommendation">
    <h2>Professional Consultant Analysis</h2>
    <p>${reportData.finalRecommendation}</p>
  </div>
</body>
</html>`;

    // Create blob and download
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);

    // Open in new window for printing
    const printWindow = window.open(url, '_blank');
    if (printWindow) {
      printWindow.onload = () => {
        setTimeout(() => {
          printWindow.focus();
          printWindow.print();
        }, 500);
      };
    } else {
      // Fallback: direct download
      const a = document.createElement('a');
      a.href = url;
      a.download = `document-verification-report-${new Date().toISOString().split('T')[0]}.html`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    }
  };

  const getConsultantMessage = (verified: number, pending: number, rejected: number, total: number) => {
    const verificationRate = (verified / total) * 100;
    const timestamp = Date.now();

    const excellentMessages = [
      `Based on our comprehensive analysis of ${total} documents, your property demonstrates exceptional legal compliance with ${verified} documents fully verified. The documentation quality exceeds industry standards, indicating meticulous record-keeping and strong legal foundation. This level of verification significantly reduces transaction risks and enhances property marketability for institutional buyers and lenders.`,
      `Professional assessment reveals outstanding documentation standards across ${total} submitted documents. With ${verified} documents achieving full verification, your property portfolio reflects exemplary legal diligence and compliance management. This comprehensive verification status positions your property advantageously for expedited transactions and premium valuations.`,
      `Our detailed verification process confirms exceptional document quality with ${verified} of ${total} documents meeting all regulatory requirements. The thorough authentication demonstrates professional property management and legal compliance that will facilitate smooth transactions with minimal due diligence delays.`
    ];

    const goodMessages = [
      `Analysis of ${total} documents shows strong legal foundation with ${verified} documents verified and ${pending} requiring minor clarifications. The majority verification rate indicates good compliance standards. Addressing the ${pending} pending items will optimize your documentation for complete transaction readiness and enhanced buyer confidence.`,
      `Professional review confirms solid documentation quality across ${total} submitted files. With ${verified} documents authenticated, your property shows good legal standing. The ${pending} pending verifications represent standard administrative requirements that can be resolved promptly to achieve complete compliance.`,
      `Comprehensive assessment reveals commendable document management with ${verified} verified documents out of ${total} total submissions. The pending ${pending} items are typical administrative matters that, once resolved, will establish optimal documentation standards for enhanced market positioning.`
    ];

    const mixedMessages = [
      `Our analysis of ${total} documents reveals mixed verification results requiring strategic attention. While ${verified} documents are properly authenticated, the ${pending} pending and ${rejected} rejected items need immediate resolution to ensure complete legal compliance and minimize transaction risks.`,
      `Professional evaluation shows partial verification success with ${verified} documents confirmed out of ${total} submitted. The ${pending + rejected} items requiring attention represent moderate compliance concerns that should be addressed to strengthen your property's legal standing and market viability.`,
      `Document review indicates moderate quality standards with ${verified} verified documents and ${pending + rejected} items needing resolution. This mixed verification status suggests both documentation strengths and areas requiring improvement to achieve optimal transaction readiness.`
    ];

    const concernMessages = [
      `Critical analysis of ${total} documents reveals significant compliance gaps with only ${verified} documents verified and ${rejected} rejected items indicating serious legal concerns. The current verification status poses substantial risks for property transactions and requires immediate professional legal intervention.`,
      `Professional assessment identifies major documentation deficiencies across ${total} submitted documents. With ${rejected} rejected items and only ${verified} verified documents, the current status suggests substantial legal and financial risks that must be addressed before proceeding with any property transactions.`,
      `Comprehensive review reveals concerning verification results with ${rejected} document rejections out of ${total} total submissions. The low verification rate of ${Math.round(verificationRate)}% indicates critical compliance issues requiring immediate remedial action and professional legal consultation.`
    ];

    // Use timestamp to ensure different messages for different sessions
    const messageIndex = Math.floor((timestamp / 1000) % 3);

    if (verificationRate >= 85) {
      return excellentMessages[messageIndex];
    } else if (verificationRate >= 65) {
      return goodMessages[messageIndex];
    } else if (verificationRate >= 40) {
      return mixedMessages[messageIndex];
    } else {
      return concernMessages[messageIndex];
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 font-display">
      <main className="flex-grow">
        <div className="container mx-auto max-w-6xl px-4 sm:px-6 lg:px-8 py-12">
          <div className="flex flex-col gap-8">
            {/* Header Section */}
            <div className="flex flex-col gap-3 text-center">
              <p className="text-base font-semibold text-blue-600">Step 3 of 4</p>
              <p className="text-4xl font-black tracking-tighter text-black">Information Review</p>
              <p className="text-lg font-normal text-gray-600 max-w-2xl mx-auto">Review the verification status of your uploaded documents.</p>
            </div>

            {/* Progress Steps */}
            <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">
              <div className="flex items-center max-w-6xl mx-auto">
                {/* Step 1 - Completed */}
                <button
                  onClick={() => {
                    const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                    navigate(`/docverification?session=${sessionId}`);
                  }}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Property Owner Details</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </button>

                {/* Connector 1 */}
                <div className="flex-1 flex items-center mx-2">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-1"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>

                {/* Step 2 - Completed */}
                <button
                  onClick={() => {
                    const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                    navigate(`/docverification?session=${sessionId}`);
                  }}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Document Upload</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </button>

                {/* Connector 2 */}
                <div className="flex-1 flex items-center mx-2">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-1"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>

                {/* Step 3 - Active */}
                <div className="flex flex-col items-center text-center">
                  <div className="flex items-center justify-center w-12 h-12 bg-blue-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    3
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Information Review</p>
                  <p className="text-xs text-blue-500 font-medium">In Progress</p>
                </div>

                {/* Connector 3 */}
                <div className="flex-1 flex items-center mx-2">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-1"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>

                {/* Step 4 - Pending */}
                <button
                  onClick={() => {
                    const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                    navigate(`/ribil-scorecard-page?session=${sessionId}`);
                  }}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    4
                  </div>
                  <p className="text-sm font-medium text-black mb-1">RIBIL Score</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>

                {/* Connector 4 */}
                <div className="flex-1 flex items-center mx-2">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-1"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>

                {/* Step 5 - Pending */}
                <button
                  onClick={() => {
                    const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                    navigate(`/docverification/final?session=${sessionId}`);
                  }}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    5
                  </div>
                  <p className="text-sm font-medium text-black mb-1">Final Verification</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>
              </div>
            </div>

            {/* Verification Results Table - MOVED TO TOP */}
            <div className="bg-white rounded-xl border border-gray-200 shadow-sm overflow-hidden">
              <div className="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-blue-50 to-indigo-50">
                <div>
                  <h3 className="text-xl font-bold text-gray-900">Document Verification Status</h3>
                  <p className="text-sm text-gray-600 mt-1">Review the verification results for each uploaded document</p>
                </div>
              </div>

              {documents.length === 0 ? (
                <div className="p-8 text-center">
                  <FileText className="w-12 h-12 text-gray-300 mx-auto mb-4" />
                  <p className="text-gray-500">No documents found for verification</p>
                  <button
                    onClick={() => navigate('/docverification')}
                    className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    Upload Documents
                  </button>
                </div>
              ) : (
                <div className="overflow-x-auto">
                  <table className="w-full border-collapse border border-gray-300">
                    <thead className="bg-gradient-to-r from-gray-100 to-gray-200">
                      <tr>
                        <th className="px-6 py-3 text-left text-xs font-bold text-gray-800 uppercase tracking-wider border border-gray-300">Document</th>
                        <th className="px-6 py-3 text-center text-xs font-bold text-gray-800 uppercase tracking-wider border border-gray-300 bg-green-100">Verified</th>
                        <th className="px-6 py-3 text-center text-xs font-bold text-gray-800 uppercase tracking-wider border border-gray-300 bg-amber-100">Pending</th>
                        <th className="px-6 py-3 text-center text-xs font-bold text-gray-800 uppercase tracking-wider border border-gray-300 bg-red-100">Rejected</th>
                        <th className="px-6 py-3 text-left text-xs font-bold text-gray-800 uppercase tracking-wider border border-gray-300">Verification Details</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white">
                      {documents.map((doc) => (
                        <tr key={`doc-${doc.id}-${doc.filename}`} className={`hover:bg-gray-50 transition-colors ${doc.verification_status === 'pending' ? 'bg-amber-25 border-l-4 border-l-amber-400' : ''
                          }`}>
                          <td className="px-6 py-4 border border-gray-300">
                            <div className="flex items-center">
                              <div className="ml-3">
                                <div className="text-sm font-medium text-gray-900">{getDocumentDisplayName(doc.doc_type)}</div>
                                <div className="text-xs text-gray-500">{doc.filename}</div>
                                <div className="text-xs text-gray-400">{formatFileSize(doc.file_size)}</div>
                              </div>
                            </div>
                          </td>
                          <td className={`px-6 py-4 text-center border border-gray-300 ${doc.verification_status === 'completed' ? 'bg-green-50' : 'bg-gray-50'
                            }`}>
                            {doc.verification_status === 'completed' ? (
                              <div className="flex items-center justify-center">
                                <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center">
                                  <CheckCircle2 className="w-5 h-5 text-white" />
                                </div>
                              </div>
                            ) : (
                              <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center mx-auto">
                                <div className="w-2 h-2 bg-gray-400 rounded-full"></div>
                              </div>
                            )}
                          </td>
                          <td className={`px-6 py-4 text-center border border-gray-300 ${doc.verification_status === 'pending' ? 'bg-amber-50 border-amber-300' : 'bg-gray-50'
                            }`}>
                            {doc.verification_status === 'pending' ? (
                              <div className="flex items-center justify-center">
                                <div className="relative w-10 h-10 bg-gradient-to-r from-amber-400 to-orange-500 rounded-full flex items-center justify-center shadow-lg">
                                  <AlertTriangle className="w-6 h-6 text-white animate-bounce" />
                                  <div className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full animate-ping"></div>
                                </div>
                              </div>
                            ) : (
                              <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center mx-auto">
                                <div className="w-2 h-2 bg-gray-400 rounded-full"></div>
                              </div>
                            )}
                          </td>
                          <td className={`px-6 py-4 text-center border border-gray-300 ${doc.verification_status === 'failed' ? 'bg-red-50' : 'bg-gray-50'
                            }`}>
                            {doc.verification_status === 'failed' ? (
                              <div className="flex items-center justify-center">
                                <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center">
                                  <XCircle className="w-5 h-5 text-white" />
                                </div>
                              </div>
                            ) : (
                              <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center mx-auto">
                                <div className="w-2 h-2 bg-gray-400 rounded-full"></div>
                              </div>
                            )}
                          </td>
                          <td className="px-6 py-4 border border-gray-300">
                            <div
                              className="text-sm p-4 rounded-lg shadow-sm"
                              style={{
                                backgroundColor: doc.verification_status === 'completed' ? '#dcfce7' :
                                  doc.verification_status === 'pending' ? '#fef3c7' :
                                    doc.verification_status === 'failed' ? '#fee2e2' : '#f3f4f6',
                                color: doc.verification_status === 'completed' ? '#166534' :
                                  doc.verification_status === 'pending' ? '#92400e' :
                                    doc.verification_status === 'failed' ? '#991b1b' : '#374151',
                                border: `2px solid ${doc.verification_status === 'completed' ? '#86efac' :
                                  doc.verification_status === 'pending' ? '#fcd34d' :
                                    doc.verification_status === 'failed' ? '#fca5a5' : '#d1d5db'
                                  }`
                              }}
                            >
                              <div className="font-semibold mb-2 whitespace-pre-line">{getVerificationMessage(doc)}</div>

                              {/* Hide feature scores for duplicates */}
                              {!(doc.verification_status === 'failed' && doc.features?.feature_3_duplicate_detection?.is_duplicate) && (doc.features || doc.processed_data?.verification_result) && (
                                <div className="mt-3 pt-3 border-t space-y-2 text-xs" style={{ borderColor: doc.verification_status === 'completed' ? '#86efac' : doc.verification_status === 'pending' ? '#fcd34d' : '#fca5a5' }}>
                                  {(doc.features?.feature_1_enhanced_validation || doc.processed_data?.verification_result?.data_validation) && (
                                    <div className="flex items-center justify-between bg-white bg-opacity-50 p-2 rounded">
                                      <span className="font-semibold">📋 F1 Data Validation:</span>
                                      <span className={
                                        (doc.features?.feature_1_enhanced_validation?.status === 'PASS' || doc.processed_data?.verification_result?.data_validation?.valid)
                                          ? 'text-green-700 font-bold' : 'text-orange-700 font-bold'
                                      }>
                                        {doc.features?.feature_1_enhanced_validation
                                          ? `${doc.features.feature_1_enhanced_validation.status} (${Math.round(doc.features.feature_1_enhanced_validation.anomaly_score * 100)}%)`
                                          : (doc.processed_data?.verification_result?.data_validation?.valid ? '✓ PASS' : '✗ FAIL')}
                                      </span>
                                    </div>
                                  )}

                                  {(doc.features?.feature_2_forgery_detection || doc.processed_data?.verification_result?.forgery_detection) && (
                                    <div className="flex items-center justify-between bg-white bg-opacity-50 p-2 rounded">
                                      <span className="font-semibold">🔍 F2 Forgery Detection:</span>
                                      <span className={
                                        (doc.features?.feature_2_forgery_detection?.fraud_level === 'LOW' || doc.processed_data?.verification_result?.forgery_detection?.status === 'CLEAN')
                                          ? 'text-green-700 font-bold' : 'text-orange-700 font-bold'
                                      }>
                                        {doc.features?.feature_2_forgery_detection
                                          ? `${doc.features.feature_2_forgery_detection.fraud_level} (${Math.round(doc.features.feature_2_forgery_detection.fraud_score * 100)}%)`
                                          : `${doc.processed_data?.verification_result?.forgery_detection?.status} (${doc.processed_data?.verification_result?.forgery_detection?.fraud_risk})`}
                                      </span>
                                    </div>
                                  )}

                                  {(doc.features?.feature_3_duplicate_detection || doc.processed_data?.verification_result?.duplicate_detection) && (
                                    <div className="flex items-center justify-between bg-white bg-opacity-50 p-2 rounded">
                                      <span className="font-semibold">🔄 F3 Duplicate Check:</span>
                                      <span className="text-green-700 font-bold">
                                        {doc.features?.feature_3_duplicate_detection
                                          ? (doc.features.feature_3_duplicate_detection.is_duplicate ? '⚠️ DUPLICATE' : '✓ UNIQUE')
                                          : doc.processed_data?.verification_result?.duplicate_detection?.status}
                                      </span>
                                    </div>
                                  )}

                                  {(doc.features?.feature_4_authenticity_scoring || doc.processed_data?.verification_result?.authenticity_score) && (
                                    <div className="flex items-center justify-between bg-white bg-opacity-50 p-2 rounded">
                                      <span className="font-semibold">🎯 F4 Authenticity:</span>
                                      <span className={
                                        ((doc.features?.feature_4_authenticity_scoring?.authenticity_score ?? 0) >= 70 || (doc.processed_data?.verification_result?.authenticity_score?.overall ?? 0) >= 70)
                                          ? 'text-green-700 font-bold'
                                          : ((doc.features?.feature_4_authenticity_scoring?.authenticity_score ?? 0) >= 50 || (doc.processed_data?.verification_result?.authenticity_score?.overall ?? 0) >= 50)
                                            ? 'text-yellow-700 font-bold' : 'text-red-700 font-bold'
                                      }>
                                        {doc.features?.feature_4_authenticity_scoring
                                          ? `${doc.features.feature_4_authenticity_scoring.authenticity_score}/100 (${doc.features.feature_4_authenticity_scoring.overall_status})`
                                          : `${doc.processed_data?.verification_result?.authenticity_score?.overall}/100`}
                                      </span>
                                    </div>
                                  )}

                                  {(doc.features?.feature_5_dispute_prediction || doc.processed_data?.verification_result?.dispute_risk) && (
                                    <div className="flex items-center justify-between bg-white bg-opacity-50 p-2 rounded">
                                      <span className="font-semibold">⚖️ F5 Dispute Risk:</span>
                                      <span className={
                                        (doc.features?.feature_5_dispute_prediction?.dispute_risk_level === 'LOW' || doc.processed_data?.verification_result?.dispute_risk?.risk_level === 'LOW')
                                          ? 'text-green-700 font-bold'
                                          : (doc.features?.feature_5_dispute_prediction?.dispute_risk_level === 'MEDIUM' || doc.processed_data?.verification_result?.dispute_risk?.risk_level === 'MEDIUM')
                                            ? 'text-orange-700 font-bold' : 'text-red-700 font-bold'
                                      }>
                                        {doc.features?.feature_5_dispute_prediction
                                          ? `${doc.features.feature_5_dispute_prediction.dispute_risk_level} (${Math.round(doc.features.feature_5_dispute_prediction.dispute_probability * 100)}%)`
                                          : `${doc.processed_data?.verification_result?.dispute_risk?.risk_level} (${doc.processed_data?.verification_result?.dispute_risk?.risk_score})`}
                                      </span>
                                    </div>
                                  )}
                                </div>
                              )}
                            </div>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              )}
            </div>

            {/* Charts and Graphs Section - Only for valid documents */}
            {validDocuments.length > 0 && (
              <VerificationCharts documents={validDocuments} />
            )}

            {/* Property Information Section */}
            {ownerDetails && (
              <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6 mb-8">
                <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Information</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <div className="flex items-center gap-3 mb-3">
                      <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center">
                        <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                      </div>
                      <div>
                        <h4 className="font-semibold text-blue-900">Judicial Status</h4>
                        <p className="text-sm text-blue-700">
                          {getJudicialStatusDisplay(ownerDetails.judicial_status)}
                        </p>
                      </div>
                    </div>
                  </div>

                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <div className="flex items-center gap-3 mb-3">
                      <div className="w-10 h-10 bg-green-600 rounded-full flex items-center justify-center">
                        <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                        </svg>
                      </div>
                      <div>
                        <h4 className="font-semibold text-green-900">Geo Location</h4>
                        <p className="text-sm text-green-700">
                          {ownerDetails.latitude !== null && ownerDetails.longitude !== null &&
                            ownerDetails.latitude !== 0 && ownerDetails.longitude !== 0 ?
                            `${ownerDetails.latitude.toFixed(6)}, ${ownerDetails.longitude.toFixed(6)}` :
                            'Location not available'
                          }
                        </p>
                        <p className="text-xs text-green-600 mt-1">Latitude, Longitude</p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}


            {/* Summary Section */}
            <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
              <div className="text-center">
                <h3 className="text-xl font-semibold text-gray-900 mb-4">Verification Summary</h3>
                <div className="grid grid-cols-3 gap-6 mb-6">
                  <div className="text-center p-4 bg-green-50 rounded-lg border border-green-200">
                    <div className="text-2xl font-bold text-green-700">
                      {verifiedCount}
                    </div>
                    <div className="text-sm text-green-600 font-medium">Verified Documents</div>
                  </div>
                  <div className="text-center p-4 bg-amber-50 rounded-lg border border-amber-200">
                    <div className="text-2xl font-bold text-amber-700">
                      {pendingCount}
                    </div>
                    <div className="text-sm text-amber-600 font-medium">Pending Review</div>
                  </div>
                  <div className="text-center p-4 bg-red-50 rounded-lg border border-red-200">
                    <div className="text-2xl font-bold text-red-700">
                      {rejectedCount}
                    </div>
                    <div className="text-sm text-red-600 font-medium">Rejected Documents</div>
                  </div>
                </div>
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                  <div className="flex items-start gap-3">
                    <div className="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center flex-shrink-0 mt-1">
                      <span className="text-white text-sm font-bold"></span>
                    </div>
                    <div>
                      <h4 className="font-semibold text-blue-900 mb-2">Professional Consultant Analysis</h4>
                      <p className="text-sm text-blue-800 leading-relaxed">
                        {getConsultantMessage(verifiedCount, pendingCount, rejectedCount, documents.length)}
                      </p>
                    </div>
                  </div>
                </div>
                <div className="flex justify-center gap-4">
                  <button
                    onClick={() => {
                      const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                      navigate(`/docverification?session=${sessionId}`);
                    }}
                    className="flex items-center justify-center px-6 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors"
                  >
                    <ArrowLeft className="w-5 h-5 mr-2" />
                    Back to Upload
                  </button>

                  <button
                    onClick={() => {
                      const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                      navigate(`/title-search/instant-results?session_id=${sessionId}`);
                    }}
                    className="flex items-center justify-center px-6 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                  >
                    <Shield className="w-5 h-5 mr-2" />
                    Instant Title Verification
                  </button>

                  <button
                    onClick={() => generatePDF()}
                    className="flex items-center justify-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                  >
                    <Download className="w-5 h-5 mr-2" />
                    Download Report
                  </button>

                  <button
                    onClick={() => {
                      const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                      navigate(`/ribil-scorecard-page?session=${sessionId}`);
                    }}
                    className="flex items-center justify-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                  >
                    View RIBIL Score
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </main>
    </div>
  );
}

export default function DocumentVerificationResults() {
  return (
    <ErrorBoundary>
      <DocumentVerificationResultsContent />
    </ErrorBoundary>
  );
}