import React, { useState, useEffect } from 'react';
import { BankDropdown } from '../components/loan-calculator/BankDropdown';
import { EMIResultBox } from '../components/loan-calculator/EMIResultBox';
import { BankComparisonTable } from '../components/loan-calculator/BankComparisonTable';
import { useLoanCalculator } from '../hooks/useLoanCalculator';
import { useBankRates } from '../hooks/useBankRates';
import { BANKS_DATA } from '../data/banks';

export function EMICalculator() {
  const [formData, setFormData] = useState({
    homePrice: '8000000',
    downPayment: '1600000',
    loanAmount: '6400000',
    interestRate: '8.5',
    loanTenure: '20',
    monthlyIncome: '50000',
    employmentType: 'salaried'
  });
  
  const [selectedBankId, setSelectedBankId] = useState<string | null>(null);
  const [comparisons, setComparisons] = useState<any[]>([]);
  const [activeTab, setActiveTab] = useState<'calculator' | 'comparison'>('calculator');
  
  const { calculateEMI, compareAllBanks, getBankDetails, getBankRateForTenure, loading } = useLoanCalculator();
  const { bankRates, loading: bankRatesLoading, error: bankRatesError, lastUpdated } = useBankRates('karnataka', 'bangalore');

  const [result, setResult] = useState<{
    emi: number;
    totalInterest: number;
    totalAmount: number;
    bankName?: string;
    interestRate?: number;
    paymentBreakdown: Array<{
      month: number;
      principal: number;
      interest: number;
      balance: number;
    }>;
  } | null>(null);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    const newData = { ...formData, [name]: value };
    
    // Auto-calculate loan amount when home price or down payment changes
    if (name === 'homePrice' || name === 'downPayment') {
      const homePrice = parseFloat(name === 'homePrice' ? value : formData.homePrice);
      const downPayment = parseFloat(name === 'downPayment' ? value : formData.downPayment);
      newData.loanAmount = (homePrice - downPayment).toString();
    }
    
    setFormData(newData);
  };
  
  const handleBankSelect = (bankId: string) => {
    setSelectedBankId(bankId);
    const bank = getBankDetails(bankId);
    if (bank) {
      const rate = getBankRateForTenure(bankId, parseInt(formData.loanTenure));
      setFormData(prev => ({ ...prev, interestRate: rate.toString() }));
    }
  };
  
  // Auto-update interest rate when tenure changes and bank is selected
  useEffect(() => {
    if (selectedBankId) {
      const rate = getBankRateForTenure(selectedBankId, parseInt(formData.loanTenure));
      setFormData(prev => ({ ...prev, interestRate: rate.toString() }));
    }
  }, [formData.loanTenure, selectedBankId, getBankRateForTenure]);


  
  const calculateLoan = () => {
    const principal = parseFloat(formData.loanAmount);
    const rate = parseFloat(formData.interestRate);
    const tenure = parseInt(formData.loanTenure);

    if (!principal || principal <= 0) {
      alert('Please enter a valid loan amount');
      return;
    }
    if (!rate || rate <= 0) {
      alert('Please enter a valid interest rate');
      return;
    }
    if (!tenure || tenure <= 0) {
      alert('Please select a valid loan tenure');
      return;
    }

    const calculation = calculateEMI(principal, rate, tenure);
    const bankName = selectedBankId ? getBankDetails(selectedBankId)?.name : undefined;
    
    setResult({
      ...calculation,
      bankName,
      interestRate: rate,
      paymentBreakdown: [] // Simplified for now
    });
  };
  
  const handleCompareAllBanks = () => {
    const principal = parseFloat(formData.loanAmount);
    const tenure = parseInt(formData.loanTenure);
    
    if (!principal || principal <= 0) {
      alert('Please enter a valid loan amount');
      return;
    }
    
    const bankComparisons = compareAllBanks(principal, tenure);
    setComparisons(bankComparisons);
    setActiveTab('comparison');
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-4xl mx-auto px-4">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Loan Calculator</h1>
          <p className="text-gray-600 max-w-3xl mx-auto">
            Compare home loan offers from top banks in India. Get personalized EMI calculations with real interest rates and processing fees.
          </p>

        </div>

        {activeTab === 'calculator' && (
          <div className="space-y-8">
            {/* Bank Selection */}
            <div className="bg-white rounded-lg shadow-lg p-6">
              <BankDropdown
                selectedBankId={selectedBankId}
                onBankSelect={handleBankSelect}
              />
              {bankRatesError && (
                <div className="mt-2 p-2 bg-yellow-50 border border-yellow-200 rounded text-sm text-yellow-700">
                  ⚠️ {bankRatesError} - Rates may not be current
                </div>
              )}
              {lastUpdated && (
                <div className="mt-2 text-xs text-gray-500">
                  Rates last updated: {new Date(lastUpdated).toLocaleDateString()}
                </div>
              )}
            </div>
            
            {/* Loan Calculator Form */}
            <div className="bg-white rounded-lg shadow-lg p-8">
              <div className="grid md:grid-cols-2 gap-8">
                <div>
                  <h2 className="text-xl font-semibold mb-6">Loan Details</h2>
                  
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Home Price: ₹{parseInt(formData.homePrice).toLocaleString('en-IN')}
                      </label>
                      <input
                        type="range"
                        name="homePrice"
                        min="2000000"
                        max="50000000"
                        step="100000"
                        value={formData.homePrice}
                        onChange={handleInputChange}
                        className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                      />
                      <div className="flex justify-between text-xs text-gray-500 mt-1">
                        <span>₹20L</span>
                        <span>₹5Cr</span>
                      </div>
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Down Payment: ₹{parseInt(formData.downPayment).toLocaleString('en-IN')}
                      </label>
                      <input
                        type="range"
                        name="downPayment"
                        min="200000"
                        max={Math.floor(parseInt(formData.homePrice) * 0.5).toString()}
                        step="50000"
                        value={formData.downPayment}
                        onChange={handleInputChange}
                        className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                      />
                      <div className="flex justify-between text-xs text-gray-500 mt-1">
                        <span>₹2L</span>
                        <span>50% of home price</span>
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Loan Amount: ₹{parseInt(formData.loanAmount).toLocaleString('en-IN')}
                      </label>
                      <div className="p-3 bg-gray-50 rounded-md text-gray-600">
                        Auto-calculated: Home Price - Down Payment
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Interest Rate (%)
                      </label>
                      <input
                        type="number"
                        step="0.1"
                        name="interestRate"
                        value={formData.interestRate}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        placeholder="Enter interest rate"
                        disabled={!!selectedBankId}
                      />
                      {selectedBankId && (
                        <p className="text-xs text-blue-600 mt-1">
                          Rate auto-updated based on selected bank
                        </p>
                      )}
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Loan Tenure: {formData.loanTenure} Years
                      </label>
                      <input
                        type="range"
                        name="loanTenure"
                        min="5"
                        max="30"
                        step="5"
                        value={formData.loanTenure}
                        onChange={handleInputChange}
                        className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                      />
                      <div className="flex justify-between text-xs text-gray-500 mt-1">
                        <span>5 Years</span>
                        <span>30 Years</span>
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Monthly Income: ₹{parseInt(formData.monthlyIncome || '50000').toLocaleString('en-IN')}
                      </label>
                      <input
                        type="range"
                        name="monthlyIncome"
                        min="25000"
                        max="500000"
                        step="5000"
                        value={formData.monthlyIncome || '50000'}
                        onChange={handleInputChange}
                        className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                      />
                      <div className="flex justify-between text-xs text-gray-500 mt-1">
                        <span>₹25K</span>
                        <span>₹5L</span>
                      </div>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Employment Type
                      </label>
                      <select
                        name="employmentType"
                        value={formData.employmentType || 'salaried'}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      >
                        <option value="salaried">Salaried</option>
                        <option value="self-employed">Self Employed</option>
                        <option value="business">Business Owner</option>
                      </select>
                    </div>

                    <div className="flex gap-3">
                      <button
                        onClick={calculateLoan}
                        className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-3 px-4 rounded-md transition-colors font-medium flex items-center justify-center gap-2"
                        style={{ backgroundColor: '#2563eb', color: 'white' }}
                      >
                        <span>📊</span>
                        Calculate EMI
                      </button>
                      <button
                        onClick={handleCompareAllBanks}
                        className="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-3 px-4 rounded-md transition-colors font-medium flex items-center justify-center gap-2"
                        style={{ backgroundColor: '#2563eb', color: 'white' }}
                      >
                        <span>🏦</span>
                        Compare Banks
                      </button>
                    </div>
                  </div>
                </div>

                <div>
                  <h2 className="text-xl font-semibold mb-6">EMI Results</h2>
                  
                  {result ? (
                    <EMIResultBox result={result} />
                  ) : (
                    <div className="bg-gray-50 border border-gray-200 rounded-lg p-8 text-center">
                      <p className="text-gray-500">Select a bank and calculate EMI</p>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}
        
        {activeTab === 'comparison' && (
          <div className="bg-white rounded-lg shadow-lg p-8">
            <div className="mb-6">
              <button
                onClick={() => setActiveTab('calculator')}
                className="flex items-center gap-2 text-blue-600 hover:text-blue-700 mb-4 font-medium"
              >
                <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                </svg>
                Back to Calculator
              </button>
              <h2 className="text-xl font-semibold mb-2">Bank Comparison</h2>
              <p className="text-gray-600">
                Loan Amount: ₹{parseInt(formData.loanAmount).toLocaleString('en-IN')} | 
                Tenure: {formData.loanTenure} years
              </p>
            </div>
            
            <BankComparisonTable
              comparisons={comparisons}
              onSelectBank={(bankId) => {
                setSelectedBankId(bankId);
                setActiveTab('calculator');
              }}
            />
            
            {comparisons.length === 0 && (
              <div className="text-center py-8">
                <p className="text-gray-500 mb-4">No comparison data available</p>
                <button
                  onClick={handleCompareAllBanks}
                  className="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-md transition-colors"
                >
                  Compare All Banks
                </button>
              </div>
            )}
          </div>
        )}
      </div>
    </div>
  );
}