import React, { useState, useEffect } from 'react';
import { useParams } from 'react-router-dom';
import { escrowApi } from '@/api/escrow';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/utils/toast';
import { PaymentModal } from '@/components/transactions/PaymentModal';
import { TransactionType } from '@/types/transaction';

export const EscrowDetail: React.FC = () => {
  const { id } = useParams<{ id: string }>();
  const [escrow, setEscrow] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [showPaymentModal, setShowPaymentModal] = useState(false);

  useEffect(() => {
    loadEscrow();
  }, [id]);

  const loadEscrow = async () => {
    try {
      const data = await escrowApi.getById(id!);
      setEscrow(data);
    } catch (error) {
      toast.error('Failed to load escrow');
    } finally {
      setLoading(false);
    }
  };

  const handleDeposit = async (transactionId: string) => {
    try {
      await escrowApi.deposit(id!, {
        amount: escrow.escrow_amount,
        payment_method: 'upi',
        transaction_id: transactionId
      });
      toast.success('Funds deposited to escrow successfully!');
      loadEscrow();
    } catch (error) {
      toast.error('Failed to deposit funds');
    }
  };

  if (loading) return <div className="p-8">Loading...</div>;
  if (!escrow) return <div className="p-8">Escrow not found</div>;

  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const isBuyer = escrow.buyer_id === user.user_id;
  const isSeller = escrow.seller_id === user.user_id;

  return (
    <div className="max-w-4xl mx-auto p-8">
      <Card className="p-6">
        <div className="flex justify-between items-start mb-6">
          <div>
            <h1 className="text-3xl font-bold mb-2">Escrow Account</h1>
            <p className="text-gray-600">Account ID: {escrow.escrow_number}</p>
          </div>
          <Badge className={
            escrow.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
            escrow.status === 'funded' ? 'bg-blue-100 text-blue-800' :
            escrow.status === 'held' ? 'bg-purple-100 text-purple-800' :
            escrow.status === 'released' ? 'bg-green-100 text-green-800' :
            escrow.status === 'refunded' ? 'bg-red-100 text-red-800' :
            'bg-gray-100 text-gray-800'
          }>
            {escrow.status === 'refunded' ? 'TERMINATED' : escrow.status.toUpperCase()}
          </Badge>
        </div>

        <div className="grid grid-cols-2 gap-6 mb-6">
          <div>
            <label className="text-sm text-gray-600">Escrow Amount</label>
            <p className="text-2xl font-bold text-blue-600">₹{escrow.escrow_amount?.toLocaleString()}</p>
          </div>
          <div>
            <label className="text-sm text-gray-600">Current Balance</label>
            <p className="text-2xl font-bold">₹{escrow.current_balance?.toLocaleString() || 0}</p>
          </div>
        </div>

        <div className="mb-6">
          <h3 className="font-semibold mb-2">Description</h3>
          <p className="text-gray-700">{escrow.description}</p>
        </div>

        <div className="mb-6">
          <h3 className="font-semibold mb-2">Parties</h3>
          <div className="space-y-2">
            <div className="flex items-center gap-2 p-2 bg-gray-50 rounded">
              <span className="font-medium">Buyer:</span>
              <span>{escrow.buyer_id}</span>
              {isBuyer && <Badge className="ml-auto">You</Badge>}
            </div>
            <div className="flex items-center gap-2 p-2 bg-gray-50 rounded">
              <span className="font-medium">Seller:</span>
              <span>{escrow.seller_id}</span>
              {isSeller && <Badge className="ml-auto">You</Badge>}
            </div>
          </div>
        </div>

        {isBuyer && escrow.status === 'pending' && (
          <Button 
            onClick={() => setShowPaymentModal(true)}
            className="w-full"
            style={{ backgroundColor: '#16a34a' }}
          >
            Deposit ₹{escrow.escrow_amount?.toLocaleString()} to Escrow
          </Button>
        )}

        {isBuyer && escrow.status === 'funded' && (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <p className="text-blue-800">
              ✓ Funds deposited successfully. Waiting for conditions to be met before release to seller.
            </p>
          </div>
        )}

        {isSeller && escrow.status === 'funded' && (
          <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
            <p className="text-purple-800">
              Funds are held in escrow. They will be released to you once all conditions are verified.
            </p>
          </div>
        )}

        {escrow.status === 'released' && (
          <div className="bg-green-50 border border-green-200 rounded-lg p-4">
            <p className="text-green-800">
              ✓ Escrow completed. Funds have been released to the seller.
            </p>
          </div>
        )}

        {escrow.status === 'refunded' && (
          <div className="bg-red-50 border-2 border-red-300 rounded-lg p-6 text-center">
            <div className="text-4xl mb-3">⛔</div>
            <h3 className="text-xl font-bold text-red-800 mb-2">ESCROW TERMINATED</h3>
            <p className="text-red-700 mb-2">
              This escrow account has been refunded and permanently closed.
            </p>
            <p className="text-sm text-red-600">
              To continue this transaction, a new escrow account must be created by signing the contract again.
            </p>
          </div>
        )}
      </Card>

      {showPaymentModal && (
        <PaymentModal
          isOpen={showPaymentModal}
          onClose={() => setShowPaymentModal(false)}
          amount={escrow.escrow_amount}
          propertyId={escrow.property_id}
          transactionType={TransactionType.ESCROW_DEPOSIT}
          description={`Escrow deposit for ${escrow.escrow_number}`}
          onSuccess={handleDeposit}
        />
      )}
    </div>
  );
};
