import { useEffect } from 'react';
import { useSearchParams, useNavigate } from 'react-router-dom';
import { toast } from 'sonner';

export function GoogleCallback() {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();

  useEffect(() => {
    console.log('GoogleCallback component loaded');
    console.log('Current URL:', window.location.href);
    const code = searchParams.get('code');
    const error = searchParams.get('error');
    console.log('Code:', code);
    console.log('Error:', error);

    if (error) {
      toast.error('Google login failed');
      navigate('/auth');
      return;
    }

    if (code) {
      console.log('Making API call to backend with code:', code.substring(0, 20) + '...');
      fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/google-auth/callback?code=${code}`, {
        method: 'POST'
      })
      .then(response => {
        console.log('Response status:', response.status);
        if (!response.ok) {
          return response.text().then(text => {
            console.error('Response error:', text);
            throw new Error(`HTTP ${response.status}: ${text}`);
          });
        }
        return response.json();
      })
      .then(userData => {
        console.log('Google OAuth success:', userData);
        
        // Get user type from URL state parameter
        const urlParams = new URLSearchParams(window.location.search);
        const userType = urlParams.get('state') || 'buyer';
        
        if (userType === 'seller') {
          localStorage.removeItem('agentToken');
          localStorage.removeItem('agentEmail');
          localStorage.removeItem('agentId');
          localStorage.setItem('sellerToken', userData.access_token);
          localStorage.setItem('sellerEmail', userData.email);
          localStorage.setItem('sellerId', userData.user_id);
          toast.success('Login successful!');
          navigate('/seller/dashboard', { replace: true });
        } else if (userType === 'agent') {
          localStorage.removeItem('sellerToken');
          localStorage.removeItem('sellerEmail');
          localStorage.removeItem('sellerId');
          localStorage.setItem('agentToken', userData.access_token);
          localStorage.setItem('agentEmail', userData.email);
          localStorage.setItem('agentId', userData.user_id);
          localStorage.setItem('user_id', userData.user_id);
          localStorage.setItem('access_token', userData.access_token);
          toast.success('Login successful!');
          navigate('/agent/dashboard', { replace: true });
        } else {
          // Buyer flow
          localStorage.setItem('access_token', userData.access_token);
          localStorage.setItem('refresh_token', userData.refresh_token);
          localStorage.setItem('user_id', userData.user_id);
          localStorage.setItem('user_email', userData.email);
          localStorage.setItem('login_type', 'google');
          
          const user = {
            user_id: userData.user_id,
            email: userData.email,
            name: userData.name,
            avatar_url: userData.avatar_url,
            is_new_user: userData.is_new_user
          };
          localStorage.setItem('user', JSON.stringify(user));
          
          // Check if user is admin and redirect accordingly
          try {
            const payload = JSON.parse(atob(userData.access_token.split('.')[1]));
            if (payload.is_admin === true) {
              toast.success('Admin login successful!');
              navigate('/admin/dashboard', { replace: true });
              return;
            }
          } catch (e) {
            console.warn('Could not parse token for admin check:', e);
          }
          
          toast.success('Login successful!');
          navigate('/', { replace: true });
        }
      })
      .catch(error => {
        console.error('OAuth callback error:', error);
        console.error('Error details:', error.message);
        toast.error(`Google login failed: ${error.message}`);
        navigate('/auth');
      });
    } else {
      navigate('/auth');
    }
  }, [searchParams, navigate]);

  return (
    <div style={{ 
      display: 'flex', 
      justifyContent: 'center', 
      alignItems: 'center', 
      height: '100vh',
      flexDirection: 'column',
      gap: '16px'
    }}>
      <div>Processing Google login...</div>
      <div style={{ 
        width: '32px', 
        height: '32px', 
        border: '3px solid #f3f3f3',
        borderTop: '3px solid #3498db',
        borderRadius: '50%',
        animation: 'spin 1s linear infinite'
      }} />
      <style>{`
        @keyframes spin {
          0% { transform: rotate(0deg); }
          100% { transform: rotate(360deg); }
        }
      `}</style>
    </div>
  );
}