import React, { useState, useEffect } from 'react';
import axios from 'axios';
import GrievanceForm from '../components/GrievanceForm';

interface Grievance {
  id: number;
  case_id: string;
  grievance_type: string;
  priority: string;
  status: string;
  title: string;
  created_at: string;
  property_id?: number;
}

const GrievancePage: React.FC = () => {
  const [grievances, setGrievances] = useState<Grievance[]>([]);
  const [showForm, setShowForm] = useState(false);
  const [selectedGrievance, setSelectedGrievance] = useState<any>(null);
  const [loading, setLoading] = useState(false);

  const userId = parseInt(localStorage.getItem('user_id') || '0');
  const userType = localStorage.getItem('user_type') || 'buyer';

  useEffect(() => {
    fetchGrievances();
  }, []);

  const fetchGrievances = async () => {
    setLoading(true);
    try {
      const response = await axios.get(`/api/grievances/?user_id=${userId}&user_type=${userType}`);
      setGrievances(response.data.cases);
    } catch (error) {
      console.error('Error fetching grievances:', error);
    } finally {
      setLoading(false);
    }
  };

  const viewDetails = async (caseId: string) => {
    try {
      const response = await axios.get(`/api/grievances/${caseId}`);
      setSelectedGrievance(response.data.case);
    } catch (error) {
      console.error('Error fetching details:', error);
    }
  };

  const deleteGrievance = async (caseId: string) => {
    if (!confirm('Are you sure you want to delete this grievance?')) return;
    try {
      await axios.delete(`/api/grievances/${caseId}/delete`);
      fetchGrievances();
    } catch (error) {
      console.error('Error deleting grievance:', error);
    }
  };

  const getPriorityColor = (priority: string) => {
    const colors: Record<string, string> = {
      low: 'bg-blue-100 text-blue-800',
      medium: 'bg-yellow-100 text-yellow-800',
      high: 'bg-orange-100 text-orange-800',
      critical: 'bg-red-100 text-red-800'
    };
    return colors[priority] || 'bg-gray-100 text-gray-800';
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      submitted: 'bg-gray-100 text-gray-800',
      registered: 'bg-blue-100 text-blue-800',
      under_investigation: 'bg-purple-100 text-purple-800',
      pending_resolution: 'bg-yellow-100 text-yellow-800',
      resolved: 'bg-green-100 text-green-800',
      closed: 'bg-gray-100 text-gray-800',
      escalated: 'bg-red-100 text-red-800'
    };
    return colors[status] || 'bg-gray-100 text-gray-800';
  };

  return (
    <div className="container mx-auto p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Grievance Management</h1>
        <button
          onClick={() => setShowForm(!showForm)}
          className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700"
        >
          {showForm ? 'Cancel' : 'Submit Grievance'}
        </button>
      </div>

      {showForm && (
        <div className="mb-6">
          <GrievanceForm
            userId={userId}
            userType={userType}
            onSuccess={() => {
              setShowForm(false);
              fetchGrievances();
            }}
          />
        </div>
      )}

      {loading ? (
        <div className="text-center py-8">Loading...</div>
      ) : (
        <div className="bg-white rounded-lg shadow overflow-hidden">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Case ID</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Priority</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Created</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {grievances.map((grievance) => (
                <tr key={grievance.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    {grievance.case_id}
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-900">{grievance.title}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {grievance.grievance_type.replace('_', ' ')}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`px-2 py-1 text-xs rounded-full ${getPriorityColor(grievance.priority)}`}>
                      {grievance.priority}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`px-2 py-1 text-xs rounded-full ${getStatusColor(grievance.status)}`}>
                      {grievance.status.replace('_', ' ')}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {new Date(grievance.created_at).toLocaleDateString()}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm">
                    <button
                      onClick={() => viewDetails(grievance.case_id)}
                      className="text-blue-600 hover:text-blue-900 mr-3"
                    >
                      View
                    </button>
                    <button
                      onClick={() => deleteGrievance(grievance.case_id)}
                      className="text-red-600 hover:text-red-900"
                    >
                      Delete
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {selectedGrievance && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6">
              <div className="flex justify-between items-start mb-4">
                <h2 className="text-2xl font-bold">Case Details: {selectedGrievance.case_id}</h2>
                <button onClick={() => setSelectedGrievance(null)} className="text-gray-500 hover:text-gray-700">
                  ✕
                </button>
              </div>

              <div className="grid grid-cols-2 gap-4 mb-6">
                <div>
                  <p className="text-sm text-gray-500">Status</p>
                  <span className={`inline-block px-3 py-1 text-sm rounded-full ${getStatusColor(selectedGrievance.status)}`}>
                    {selectedGrievance.status.replace('_', ' ')}
                  </span>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Priority</p>
                  <span className={`inline-block px-3 py-1 text-sm rounded-full ${getPriorityColor(selectedGrievance.priority)}`}>
                    {selectedGrievance.priority}
                  </span>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Type</p>
                  <p className="font-medium">{selectedGrievance.grievance_type.replace('_', ' ')}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-500">Created</p>
                  <p className="font-medium">{new Date(selectedGrievance.created_at).toLocaleString()}</p>
                </div>
              </div>

              <div className="mb-6">
                <h3 className="font-semibold mb-2">Title</h3>
                <p>{selectedGrievance.title}</p>
              </div>

              <div className="mb-6">
                <h3 className="font-semibold mb-2">Description</h3>
                <p className="text-gray-700">{selectedGrievance.description}</p>
              </div>

              {selectedGrievance.evidence && selectedGrievance.evidence.length > 0 && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2">Evidence Files</h3>
                  <ul className="space-y-2">
                    {selectedGrievance.evidence.map((file: any) => (
                      <li key={file.id} className="flex items-center justify-between bg-gray-50 p-3 rounded">
                        <span>{file.file_name}</span>
                        <a href={`/${file.file_path}`} download className="text-blue-600 hover:text-blue-800">
                          Download
                        </a>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {selectedGrievance.investigation_notes && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2">Investigation Notes</h3>
                  <p className="text-gray-700 bg-gray-50 p-3 rounded">{selectedGrievance.investigation_notes}</p>
                </div>
              )}

              {selectedGrievance.resolution_summary && (
                <div className="mb-6">
                  <h3 className="font-semibold mb-2">Resolution Summary</h3>
                  <p className="text-gray-700 bg-green-50 p-3 rounded">{selectedGrievance.resolution_summary}</p>
                </div>
              )}

              {selectedGrievance.timeline && selectedGrievance.timeline.length > 0 && (
                <div>
                  <h3 className="font-semibold mb-3">Timeline</h3>
                  <div className="space-y-3">
                    {selectedGrievance.timeline.map((event: any) => (
                      <div key={event.id} className="flex border-l-2 border-blue-500 pl-4">
                        <div className="flex-1">
                          <p className="font-medium">{event.action.replace('_', ' ')}</p>
                          <p className="text-sm text-gray-600">{event.description}</p>
                          <p className="text-xs text-gray-400 mt-1">
                            {new Date(event.timestamp).toLocaleString()}
                          </p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default GrievancePage;
