import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { Calendar, Clock, MapPin, User, ArrowLeft, Phone, Mail, Car, Home, FileText, AlertCircle, Bell } from 'lucide-react';
import { toast } from 'sonner';
import { VisitConfirmationModal } from '../components/itinerary/VisitConfirmationModal';
import { getPropertyDetails, createVisitBooking } from '../api/visitBookings';

export const ItineraryDetail = () => {
  const { id } = useParams();
  const navigate = useNavigate();
  const [property, setProperty] = useState<any>(null);
  const [showConfirmation, setShowConfirmation] = useState(false);
  const [formData, setFormData] = useState({
    // Basic Visit Details
    purposeOfVisit: 'site-visit',
    propertyName: '',
    propertyId: '',
    visitDate: '',
    visitStartTime: '',
    visitDuration: '60',
    
    // Visitor Information
    fullName: '',
    mobileNumber: '',
    email: '',
    numberOfVisitors: '1',
    
    // Travel & Arrival Details
    modeOfTravel: 'car',
    expectedArrivalTime: '',
    needParking: false,
    
    // Spots/Stops Inside Property
    spotsToVisit: [] as string[],
    specificBlockUnit: '',
    
    // Meeting Preferences
    wantToMeetAgent: true,
    preferredMeetingTime: '',
    meetingPurpose: '',
    preferredRepresentative: '',
    
    // Special Requirements
    accessibilityNeeds: '',
    languagePreference: 'english',
    photographyAllowed: true,
    needBrochures: false,
    
    // Notifications & Reminders
    smsReminder: true,
    whatsappReminder: true,
    notifyAgent: true,
    
    // Additional Notes
    customRequests: '',
    specificInstructions: '',
    clarifications: ''
  });

  useEffect(() => {
    const fetchProperty = async () => {
      try {
        console.log('🔍 Fetching property details for ID:', id);
        const propertyData = await getPropertyDetails(parseInt(id!));
        console.log('✅ Property data received:', propertyData);
        setProperty(propertyData);
      } catch (error) {
        console.error('❌ Error fetching property from API:', error);
        // Fallback to localStorage
        const properties = JSON.parse(localStorage.getItem('user_itineraries') || '[]');
        const found = properties.find((p: any) => p.id === parseInt(id!));
        if (found) {
          console.log('📦 Using fallback property from localStorage:', found);
          setProperty(found);
        } else {
          console.warn('⚠️ Property not found in localStorage either');
          // Create a basic property object for testing
          setProperty({
            id: parseInt(id!),
            title: `Property ${id}`,
            address: '123 Main Street',
            city: 'Bangalore',
            price: 'Price on Request',
            type: 'Residential',
            bedrooms: 2
          });
        }
      }
    };
    
    if (id) fetchProperty();
  }, [id]);

  const spotsOptions = ['Sample flat/unit', 'Clubhouse', 'Swimming Pool', 'Gym', 'Kids Area', 'Common Areas', 'Parking Area', 'Roof Terrace', 'Landscape Area'];

  const handleSpotsToggle = (spot: string) => {
    setFormData(prev => ({
      ...prev,
      spotsToVisit: prev.spotsToVisit.includes(spot)
        ? prev.spotsToVisit.filter(s => s !== spot)
        : [...prev.spotsToVisit, spot]
    }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.visitDate || !formData.visitStartTime || !formData.fullName || !formData.mobileNumber) {
      toast.error('Please fill all required fields');
      return;
    }

    setShowConfirmation(true);
  };

  const handleConfirmBooking = async () => {
    try {
      console.log('🚀 Starting booking process...');
      
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      if (!user.user_id && !user.id) {
        toast.error('Please login to book a visit');
        return;
      }
      
      const bookingData = {
        property_id: property.id,
        purpose_of_visit: formData.purposeOfVisit,
        visit_date: formData.visitDate,
        visit_start_time: formData.visitStartTime,
        visit_duration: parseInt(formData.visitDuration),
        full_name: formData.fullName,
        mobile_number: formData.mobileNumber,
        email: formData.email || '',
        number_of_visitors: formData.numberOfVisitors,
        mode_of_travel: formData.modeOfTravel,
        expected_arrival_time: formData.expectedArrivalTime || '',
        need_parking: formData.needParking,
        spots_to_visit: formData.spotsToVisit,
        specific_block_unit: formData.specificBlockUnit || '',
        want_to_meet_agent: formData.wantToMeetAgent,
        preferred_meeting_time: formData.preferredMeetingTime || '',
        meeting_purpose: formData.meetingPurpose || '',
        preferred_representative: formData.preferredRepresentative || '',
        accessibility_needs: formData.accessibilityNeeds || '',
        language_preference: formData.languagePreference,
        photography_allowed: formData.photographyAllowed,
        need_brochures: formData.needBrochures,
        sms_reminder: formData.smsReminder,
        whatsapp_reminder: formData.whatsappReminder,
        notify_agent: formData.notifyAgent,
        custom_requests: formData.customRequests || '',
        specific_instructions: formData.specificInstructions || '',
        clarifications: formData.clarifications || ''
      };

      console.log('📤 Sending booking data:', bookingData);
      
      const response = await createVisitBooking(bookingData);
      console.log('✅ Booking response:', response);
      
      // Also save to localStorage for immediate display
      const localBooking = {
        id: Date.now().toString(),
        ...bookingData,
        status: 'scheduled',
        created_at: new Date().toISOString(),
        user_id: user.user_id || user.id,
        property_owner_id: property.owner_id || property.user_id // Add property owner ID
      };
      
      const existingBookings = JSON.parse(localStorage.getItem('visit_bookings') || '[]');
      existingBookings.push(localBooking);
      localStorage.setItem('visit_bookings', JSON.stringify(existingBookings));
      
      setShowConfirmation(false);
      toast.success('Visit Booked Successfully!', {
        description: 'Your property visit has been scheduled. Check your dashboard for details.'
      });
      
      // Navigate to user dashboard inquiry management
      setTimeout(() => {
        navigate('/user/dashboard');
      }, 1500);
      
    } catch (error) {
      console.error('❌ Error creating booking:', error);
      setShowConfirmation(false);
      toast.error('Failed to book visit', {
        description: error instanceof Error ? error.message : 'Please check your connection and try again.'
      });
    }
  };



  if (!property) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <AlertCircle className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600 text-lg">Property not found</p>
          <button onClick={() => navigate('/user/dashboard?tab=itineraries')} className="mt-4 text-blue-600 hover:text-blue-700">
            Back to Dashboard
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-5xl mx-auto px-4">
        <button onClick={() => navigate('/user/dashboard?tab=itineraries')} className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-6 transition-colors">
          <ArrowLeft className="w-5 h-5" />
          Back to Dashboard
        </button>

        {/* Property Header */}
        <div className="bg-white rounded-xl shadow-sm p-6 mb-8">
          <h1 className="text-2xl font-bold text-gray-900 mb-2">{property.title}</h1>
          <div className="flex items-center gap-2 text-gray-600 mb-3">
            <MapPin className="w-4 h-4" />
            <span>{property.address}, {property.city}</span>
          </div>
          <div className="flex flex-wrap gap-4 text-sm">
            <span className="bg-blue-50 text-blue-700 px-3 py-1 rounded-full">₹{property.price || 'Price on Request'}</span>
            <span className="bg-green-50 text-green-700 px-3 py-1 rounded-full">{property.type || 'Residential'}</span>
            <span className="bg-purple-50 text-purple-700 px-3 py-1 rounded-full">{property.bedrooms || 2} BHK</span>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          {/* 1. Basic Visit Details */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <Calendar className="w-5 h-5 text-blue-600" />
              Basic Visit Details
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Purpose of Visit
                </label>
                <select
                  value={formData.purposeOfVisit}
                  onChange={(e) => setFormData({ ...formData, purposeOfVisit: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="site-visit">Site Visit</option>
                  <option value="rental-viewing">Rental Viewing</option>
                  <option value="property-tour">Property Tour</option>
                  <option value="meeting">Meeting</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Property Name / Property ID
                </label>
                <input
                  type="text"
                  value={property?.title || ''}
                  disabled
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg bg-gray-50 text-gray-600"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Preferred Visit Date *
                </label>
                <input
                  type="date"
                  required
                  value={formData.visitDate}
                  onChange={(e) => setFormData({ ...formData, visitDate: e.target.value })}
                  min={new Date().toISOString().split('T')[0]}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Preferred Visit Start Time *
                </label>
                <input
                  type="time"
                  required
                  value={formData.visitStartTime}
                  onChange={(e) => setFormData({ ...formData, visitStartTime: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Estimated Visit Duration
                </label>
                <select
                  value={formData.visitDuration}
                  onChange={(e) => setFormData({ ...formData, visitDuration: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="30">30 minutes</option>
                  <option value="60">1 hour</option>
                  <option value="90">1.5 hours</option>
                  <option value="120">2 hours</option>
                </select>
              </div>
            </div>
          </div>

          {/* 2. Visitor Information */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <User className="w-5 h-5 text-blue-600" />
              Visitor Information
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Full Name *
                </label>
                <input
                  type="text"
                  required
                  value={formData.fullName}
                  onChange={(e) => setFormData({ ...formData, fullName: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter your full name"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Mobile Number *
                </label>
                <input
                  type="tel"
                  required
                  value={formData.mobileNumber}
                  onChange={(e) => setFormData({ ...formData, mobileNumber: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="+91 XXXXX XXXXX"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Email (optional)
                </label>
                <input
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="your.email@example.com"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Number of Visitors
                </label>
                <select
                  value={formData.numberOfVisitors}
                  onChange={(e) => setFormData({ ...formData, numberOfVisitors: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="1">1 person</option>
                  <option value="2">2 people</option>
                  <option value="3">3 people</option>
                  <option value="4">4 people</option>
                  <option value="5+">5+ people</option>
                </select>
              </div>
            </div>
          </div>

          {/* 3. Travel & Arrival Details */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <Car className="w-5 h-5 text-blue-600" />
              Travel & Arrival Details
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Mode of Travel
                </label>
                <select
                  value={formData.modeOfTravel}
                  onChange={(e) => setFormData({ ...formData, modeOfTravel: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="car">Car</option>
                  <option value="bike">Bike</option>
                  <option value="public-transport">Public Transport</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Expected Arrival Time
                </label>
                <input
                  type="time"
                  value={formData.expectedArrivalTime}
                  onChange={(e) => setFormData({ ...formData, expectedArrivalTime: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div className="md:col-span-2">
                <label className="flex items-center gap-2 cursor-pointer">
                  <input
                    type="checkbox"
                    checked={formData.needParking}
                    onChange={(e) => setFormData({ ...formData, needParking: e.target.checked })}
                    className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">Need parking?</span>
                </label>
              </div>
            </div>
          </div>

          {/* 4. Spots/Stops Inside Property */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <Home className="w-5 h-5 text-blue-600" />
              Spots / Stops Inside the Property
            </h2>
            <div className="grid grid-cols-2 md:grid-cols-3 gap-3 mb-4">
              {spotsOptions.map((spot) => (
                <label key={spot} className="flex items-center gap-2 cursor-pointer p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors">
                  <input
                    type="checkbox"
                    checked={formData.spotsToVisit.includes(spot)}
                    onChange={() => handleSpotsToggle(spot)}
                    className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">{spot}</span>
                </label>
              ))}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Any specific block/tower/unit number
              </label>
              <input
                type="text"
                value={formData.specificBlockUnit}
                onChange={(e) => setFormData({ ...formData, specificBlockUnit: e.target.value })}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="e.g., Block A, Tower 2, Unit 501"
              />
            </div>
          </div>

          {/* 5. Meeting Preferences */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <User className="w-5 h-5 text-blue-600" />
              Meeting Preferences
            </h2>
            <div className="space-y-4">
              <div>
                <label className="flex items-center gap-2 cursor-pointer">
                  <input
                    type="checkbox"
                    checked={formData.wantToMeetAgent}
                    onChange={(e) => setFormData({ ...formData, wantToMeetAgent: e.target.checked })}
                    className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">Do you want to meet sales/agent/owner?</span>
                </label>
              </div>
              {formData.wantToMeetAgent && (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Preferred Meeting Time
                    </label>
                    <input
                      type="time"
                      value={formData.preferredMeetingTime}
                      onChange={(e) => setFormData({ ...formData, preferredMeetingTime: e.target.value })}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Meeting Purpose
                    </label>
                    <input
                      type="text"
                      value={formData.meetingPurpose}
                      onChange={(e) => setFormData({ ...formData, meetingPurpose: e.target.value })}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="e.g., Discuss pricing, financing options"
                    />
                  </div>
                  <div className="md:col-span-2">
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Name of preferred representative (optional)
                    </label>
                    <input
                      type="text"
                      value={formData.preferredRepresentative}
                      onChange={(e) => setFormData({ ...formData, preferredRepresentative: e.target.value })}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      placeholder="Name of specific agent or representative"
                    />
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* 6. Special Requirements */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <FileText className="w-5 h-5 text-blue-600" />
              Special Requirements
            </h2>
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Accessibility needs
                </label>
                <textarea
                  value={formData.accessibilityNeeds}
                  onChange={(e) => setFormData({ ...formData, accessibilityNeeds: e.target.value })}
                  rows={3}
                  placeholder="Wheelchair access, elevator, etc."
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Language preference
                </label>
                <select
                  value={formData.languagePreference}
                  onChange={(e) => setFormData({ ...formData, languagePreference: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="english">English</option>
                  <option value="hindi">Hindi</option>
                  <option value="kannada">Kannada</option>
                  <option value="tamil">Tamil</option>
                  <option value="telugu">Telugu</option>
                </select>
              </div>
              <div className="space-y-3">
                <label className="flex items-center gap-2 cursor-pointer">
                  <input
                    type="checkbox"
                    checked={formData.photographyAllowed}
                    onChange={(e) => setFormData({ ...formData, photographyAllowed: e.target.checked })}
                    className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">Photography allowed?</span>
                </label>
                <label className="flex items-center gap-2 cursor-pointer">
                  <input
                    type="checkbox"
                    checked={formData.needBrochures}
                    onChange={(e) => setFormData({ ...formData, needBrochures: e.target.checked })}
                    className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">Need printed brochures?</span>
                </label>
              </div>
            </div>
          </div>

          {/* 7. Notifications & Reminders */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
              <Bell className="w-5 h-5 text-blue-600" />
              Notifications & Reminders
            </h2>
            <div className="space-y-3">
              <label className="flex items-center gap-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={formData.smsReminder}
                  onChange={(e) => setFormData({ ...formData, smsReminder: e.target.checked })}
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <span className="text-sm text-gray-700">Send SMS reminder?</span>
              </label>
              <label className="flex items-center gap-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={formData.whatsappReminder}
                  onChange={(e) => setFormData({ ...formData, whatsappReminder: e.target.checked })}
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <span className="text-sm text-gray-700">Send WhatsApp reminder?</span>
              </label>
              <label className="flex items-center gap-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={formData.notifyAgent}
                  onChange={(e) => setFormData({ ...formData, notifyAgent: e.target.checked })}
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <span className="text-sm text-gray-700">Notify agent automatically?</span>
              </label>
            </div>
          </div>

          {/* 8. Additional Notes */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <h2 className="text-xl font-bold text-gray-900 mb-6">Additional Notes from User</h2>
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Any custom requests
                </label>
                <textarea
                  value={formData.customRequests}
                  onChange={(e) => setFormData({ ...formData, customRequests: e.target.value })}
                  rows={3}
                  placeholder="Any specific requests or arrangements..."
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Specific instructions
                </label>
                <textarea
                  value={formData.specificInstructions}
                  onChange={(e) => setFormData({ ...formData, specificInstructions: e.target.value })}
                  rows={3}
                  placeholder="Any specific instructions for the visit..."
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Things they want to clarify during visit
                </label>
                <textarea
                  value={formData.clarifications}
                  onChange={(e) => setFormData({ ...formData, clarifications: e.target.value })}
                  rows={3}
                  placeholder="Questions or clarifications needed during the visit..."
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
            </div>
          </div>

          {/* Submit Buttons */}
          <div className="bg-white rounded-xl shadow-sm p-6">
            <div className="flex flex-col sm:flex-row gap-4">
              <button
                type="button"
                onClick={() => navigate('/user/dashboard?tab=itineraries')}
                className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
              >
                Cancel
              </button>
              <button
                type="submit"
                className="flex-1 px-6 py-3 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors font-semibold shadow-sm"
              >
                Book Visit
              </button>
            </div>
            <p className="text-xs text-gray-500 mt-4 text-center">
              By booking this visit, you agree to our terms and conditions. You will receive a confirmation email/SMS shortly.
            </p>
          </div>
        </form>

        <VisitConfirmationModal
          isOpen={showConfirmation}
          onClose={() => setShowConfirmation(false)}
          booking={formData}
          property={property}
          onConfirm={handleConfirmBooking}
        />
      </div>
    </div>
  );
};
