import React, { useEffect, useState } from 'react';
import { ExternalLink, Calendar, FileText, Building, Info, CheckCircle } from 'lucide-react';
import { toast } from 'sonner';

export function LegalGuidelines() {
  const [isAccepted, setIsAccepted] = useState(false);

  // Check if guidelines are already accepted
  useEffect(() => {
    const accepted = localStorage.getItem('legalGuidelinesAccepted') === 'true';
    setIsAccepted(accepted);
    localStorage.setItem('hasReadGovernmentGuidelines', 'true');
  }, []);

  const handleAcceptGuidelines = (e: React.ChangeEvent<HTMLInputElement>) => {
    const checked = e.target.checked;
    setIsAccepted(checked);
    localStorage.setItem('legalGuidelinesAccepted', checked ? 'true' : 'false');
    
    if (checked) {
      toast.success('Legal guidelines accepted! You can now proceed with property listing.');
      // Trigger storage event for other tabs/windows
      window.dispatchEvent(new StorageEvent('storage', {
        key: 'legalGuidelinesAccepted',
        newValue: 'true',
        oldValue: 'false'
      }));
    }
  };

  const guidelines = [
    {
      title: "Real Estate (Regulation and Development) Act, 2016 (RERA)",
      subtitle: "Regulates the real estate sector to ensure transparency and accountability",
      date: "15/1/2024",
      category: "RERA",
      authority: "Ministry of Housing and Urban Affairs",
      priority: "HIGH",
      description: "Mandatory registration of real estate projects and agents. Restricts pre-launch sales and requires escrow of buyer payments. Provides buyer grievance redressal via Real Estate Regulatory Authority and Appellate Tribunal.",
      link: "https://services.india.gov.in/service/listing?cat_id=111&ln=en"
    },
    {
      title: "Karnataka RERA (K-RERA) Guidelines for Bangalore",
      subtitle: "State-specific RERA regulations for Karnataka and Bangalore properties",
      date: "10/1/2024",
      category: "RERA",
      authority: "Karnataka RERA Authority",
      priority: "HIGH",
      description: "Karnataka Real Estate Regulatory Authority governs all real estate projects in Bangalore. All projects above 500 sq meters or 8 apartments must be registered. Agents must also be registered under K-RERA.",
      link: "https://rera.karnataka.gov.in/home;jsessionid=7E5AEBC48A925A4AB2F22B8365A6D694?language=en"
    },
    {
      title: "Bangalore Development Authority (BDA) Regulations",
      subtitle: "Building approvals and development guidelines for Bangalore",
      date: "20/12/2023",
      category: "Planning",
      authority: "Bangalore Development Authority",
      priority: "MEDIUM",
      description: "BDA regulations for building approvals, layout sanctions, and development permissions in Bangalore. Includes guidelines for residential and commercial constructions.",
      link: "https://kbda.karnataka.gov.in/"
    },
    {
      title: "Property Tax Guidelines - BBMP",
      subtitle: "Property tax calculation and payment guidelines for Bangalore properties",
      date: "15/12/2023",
      category: "Tax",
      authority: "BBMP",
      priority: "MEDIUM",
      description: "Bruhat Bengaluru Mahanagara Palike (BBMP) property tax guidelines including Self Assessment Scheme (SAS), tax rates, and online payment procedures.",
      link: "https://bbmptax.karnataka.gov.in/"
    },
    {
      title: "Property Registration Guidelines - Karnataka",
      subtitle: "Document requirements and registration process for Bangalore properties",
      date: "5/1/2024",
      category: "Documentation",
      authority: "Department of Stamps and Registration, Karnataka",
      priority: "MEDIUM",
      description: "Guidelines for property registration in Karnataka including stamp duty rates, registration fees, and required documents. Online registration available through Kaveri portal.",
      link: "https://kaveri.karnataka.gov.in/landing-page"
    }
  ];

  const acts = [
    {
      title: "Transfer of Property Act, 1882",
      subtitle: "Governs the transfer of property through sale, mortgage, lease, gift, or exchange",
      date: "1/1/1882",
      category: "Property Transfer",
      authority: "Government of India",
      priority: "MEDIUM",
      description: "Defines rights and liabilities of buyers, sellers, and lessors; applies to immovable property."
    },
    {
      title: "Registration Act, 1908",
      subtitle: "Mandates registration of property documents to ensure legal validity",
      date: "1/1/1908",
      category: "Registration",
      authority: "Government of India",
      priority: "HIGH",
      description: "All transactions involving sale, lease (above 12 months), or gift must be registered at the Sub-Registrar's office."
    },
    {
      title: "Indian Stamp Act, 1899",
      subtitle: "Governs stamp duty payment on property transactions",
      date: "1/1/1899",
      category: "Stamp Duty",
      authority: "Government of India",
      priority: "HIGH",
      description: "Varies by state; payment required for sale deeds, conveyances, and leases."
    },
    {
      title: "Indian Contract Act, 1872",
      subtitle: "Defines enforceable contracts",
      date: "1/1/1872",
      category: "Contracts",
      authority: "Government of India",
      priority: "MEDIUM",
      description: "Applicable to sale agreements, construction contracts, and development agreements in real estate."
    },
    {
      title: "Insolvency and Bankruptcy Code (IBC), 2016",
      subtitle: "Addresses insolvency of builders and developers",
      date: "28/5/2016",
      category: "Insolvency",
      authority: "Government of India",
      priority: "MEDIUM",
      description: "Recognizes homebuyers as financial creditors, enabling them to claim refunds or compensation during insolvency proceedings."
    },
    {
      title: "Benami Transactions (Prohibition) Amendment Act, 2016",
      subtitle: "Prohibits holding property in someone else's name to conceal ownership",
      date: "1/11/2016",
      category: "Anti-Benami",
      authority: "Government of India",
      priority: "HIGH",
      description: "Confiscation and prosecution for benami (illegal/false-name) holdings."
    },
    {
      title: "Foreign Exchange Management Act (FEMA), 1999",
      subtitle: "Governs property ownership and investment by NRIs, OCIs, and foreign nationals",
      date: "1/6/2000",
      category: "Foreign Investment",
      authority: "Reserve Bank of India",
      priority: "MEDIUM",
      description: "NRIs can buy residential or commercial property but not agricultural land, plantation, or farmhouses."
    },
    {
      title: "Income Tax Act, 1961 (Real Estate Provisions)",
      subtitle: "Regulates taxation on property income, sale, and capital gains",
      date: "1/4/1962",
      category: "Taxation",
      authority: "Income Tax Department",
      priority: "HIGH",
      description: "TDS on property purchases above ₹50 lakh (Section 194-IA). Deduction on home loan interest (Section 24(b)). Capital gains exemptions under Sections 54 & 54F."
    },
    {
      title: "Indian Easements Act, 1882",
      subtitle: "Defines rights of usage, access, and passage across another person's property",
      date: "1/1/1882",
      category: "Property Rights",
      authority: "Government of India",
      priority: "MEDIUM",
      description: "Defines rights of usage, access, and passage across another person's property (e.g., right of way)."
    },
    {
      title: "Indian Succession Act, 1925",
      subtitle: "Governs inheritance and transfer of property upon death",
      date: "1/1/1925",
      category: "Inheritance",
      authority: "Government of India",
      priority: "MEDIUM",
      description: "Governs inheritance and transfer of property upon death (when personal/religious laws don't apply)."
    },
    {
      title: "Environment (Protection) Act, 1986",
      subtitle: "Regulates environmental clearance for large construction projects",
      date: "23/5/1986",
      category: "Environment",
      authority: "Ministry of Environment, Forest and Climate Change",
      priority: "MEDIUM",
      description: "Projects exceeding 20,000 sq. meters need prior environmental approval."
    }
  ];

  const importantNotice = {
    title: "Important for Property Listings",
    content: "All properties listed on NAL India must comply with RERA regulations. Projects above 500 sq meters or 8 apartments require RERA registration. Always verify RERA registration numbers before listing."
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 pt-20">
      {/* Header */}
      <div
        className="bg-gradient-to-r from-blue-500 to-blue-700"
        style={{
          background: "linear-gradient(90deg, #3B82F6 0%, #1D4ED8 100%)",
        }}
      >
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <div className="text-center">
            <h1
              className="text-4xl md:text-5xl font-bold mb-4"
              style={{ color: "#FFFFFF" }}
            >
              Government Guidelines & Real Estate Policies
            </h1>
            <p
              className="text-xl max-w-3xl mx-auto"
              style={{ color: "#FFFFFF" }}
            >
              Comprehensive legal framework governing real estate transactions
              in India
            </p>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* Important Notice */}
        <div className="mb-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Info className="w-5 h-5 text-blue-600 mt-0.5 flex-shrink-0" />
              <div>
                <h3 className="font-semibold text-blue-900 mb-2">
                  {importantNotice.title}
                </h3>
                <p className="text-blue-800 text-sm leading-relaxed">
                  {importantNotice.content}
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Guidelines Cards */}
        <div className="space-y-6">
          {guidelines.map((guideline, index) => (
            <div
              key={index}
              className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow"
            >
              {/* Header */}
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <div className="flex items-center space-x-3 mb-2">
                    <h2 className="text-xl font-bold text-gray-900">
                      {guideline.title}
                    </h2>
                  </div>
                  <p className="text-gray-600 mb-3">{guideline.subtitle}</p>
                </div>
              </div>

              {/* Meta Information */}
              <div className="flex items-center space-x-6 mb-4 text-sm text-gray-500">
                <div className="flex items-center space-x-1">
                  <Calendar className="w-4 h-4" />
                  <span>{guideline.date}</span>
                </div>
                <div className="flex items-center space-x-1">
                  <FileText className="w-4 h-4" />
                  <span>{guideline.category}</span>
                </div>
                <div className="flex items-center space-x-1">
                  <Building className="w-4 h-4" />
                  <span>{guideline.authority}</span>
                </div>
              </div>

              {/* Description */}
              <p className="text-gray-700 mb-4 leading-relaxed">
                {guideline.description}
              </p>

              {/* Read Full Guidelines Link */}
              <a
                href={guideline.link}
                target="_blank"
                rel="noopener noreferrer"
                className="inline-flex items-center text-blue-500 hover:text-blue-700 font-medium text-sm underline"
              >
                Read Full Guidelines
                <ExternalLink className="w-4 h-4 ml-1" />
              </a>
            </div>
          ))}
        </div>

        {/* Acts Section */}
        <div className="mt-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-6">
            Indian Real Estate Acts & Laws
          </h2>
          <div className="space-y-6">
            {acts.map((act, index) => (
              <div
                key={index}
                className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow"
              >
                {/* Header */}
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <div className="flex items-center space-x-3 mb-2">
                      <h2 className="text-xl font-bold text-gray-900">
                        {act.title}
                      </h2>
                    </div>
                    <p className="text-gray-600 mb-3">{act.subtitle}</p>
                  </div>
                </div>

                {/* Meta Information */}
                <div className="flex items-center space-x-6 mb-4 text-sm text-gray-500">
                  <div className="flex items-center space-x-1">
                    <Calendar className="w-4 h-4" />
                    <span>{act.date}</span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <FileText className="w-4 h-4" />
                    <span>{act.category}</span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <Building className="w-4 h-4" />
                    <span>{act.authority}</span>
                  </div>
                </div>

                {/* Description */}
                <p className="text-gray-700 leading-relaxed">
                  {act.description}
                </p>
              </div>
            ))}
          </div>
        </div>

        {/* Accept Guidelines Checkbox */}
        <div className="mt-12 bg-white border border-gray-200 rounded-lg p-4">
          <label className="flex items-center">
            <input
              type="checkbox"
              checked={isAccepted}
              onChange={handleAcceptGuidelines}
              className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
            />
            <span className="ml-3 text-sm font-medium text-gray-900">
              I have read and accept all legal guidelines for property listing
              in India
            </span>
          </label>
        </div>

        {/* Company Information */}
        <div className="mt-8 bg-gray-50 rounded-lg p-6">
          <div className="text-center">
            <h3 className="font-bold text-gray-900 mb-2">NAL India Pvt Ltd.</h3>
            <p className="text-sm text-gray-600 mb-1">
              #28, Third Floor, MCHS Layout KV Jayaram Road, Jakkur, Bengaluru,
              India
            </p>
            <p className="text-sm text-gray-600">
              Email:info@newagelandindia.com | Phone: +91-8068447416
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}