import React, { useState, useEffect } from 'react';
import { useParams, useSearchParams } from 'react-router-dom';
import { DocumentUpload } from '../components/documents/DocumentUpload';
import { DocumentManager } from '../components/documents/DocumentManager';
import { getPropertyDocuments, PropertyDocument } from '../api/documents';
import { FileText, Home, MapPin, DollarSign, Image as ImageIcon, BarChart3 } from 'lucide-react';
import { toast } from 'sonner';
import { CompetitorAnalyticsModal } from '../components/competitor-analytics/CompetitorAnalyticsModal';

interface PropertyData {
  id: string;
  title: string;
  property_type: string;
  listing_type: string;
  city: string;
  locality: string;
  price?: number;
  monthly_rent?: number;
}

export const ListPropertyPage: React.FC = () => {
  const { propertyId } = useParams<{ propertyId: string }>();
  const [searchParams] = useSearchParams();
  const [activeTab, setActiveTab] = useState('documents');
  const [propertyData, setPropertyData] = useState<PropertyData | null>(null);
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showAnalyticsModal, setShowAnalyticsModal] = useState(false);

  // Mock property data - replace with actual API call
  useEffect(() => {
    const mockPropertyData: PropertyData = {
      id: propertyId || 'new',
      title: 'Luxury 3BHK Apartment with Sea View',
      property_type: 'apartment',
      listing_type: 'sell',
      city: 'Mumbai',
      locality: 'Bandra West',
      price: 5000000
    };
    setPropertyData(mockPropertyData);
    setIsLoading(false);
  }, [propertyId]);

  // Load existing documents if editing
  useEffect(() => {
    if (propertyId && propertyId !== 'new') {
      loadDocuments();
    }
  }, [propertyId]);

  const loadDocuments = async () => {
    if (!propertyId) return;
    
    try {
      const docs = await getPropertyDocuments(propertyId);
      setDocuments(docs);
    } catch (error) {
      console.error('Error loading documents:', error);
      toast.error('Failed to load documents');
    }
  };

  const handleDocumentsChange = (newDocuments: PropertyDocument[]) => {
    setDocuments(newDocuments);
  };

  const tabs = [
    { id: 'documents', label: 'Documents', icon: FileText },
    { id: 'property', label: 'Property Details', icon: Home },
    { id: 'location', label: 'Location', icon: MapPin },
    { id: 'pricing', label: 'Pricing', icon: DollarSign },
    { id: 'media', label: 'Media', icon: ImageIcon }
  ];

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading property...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="py-6">
            <div className="flex items-center justify-between">
              <div>
                <h1 className="text-2xl font-bold text-gray-900">
                  {propertyId === 'new' ? 'List New Property' : 'Manage Property'}
                </h1>
                {propertyData && (
                  <div className="mt-2 flex items-center space-x-4 text-sm text-gray-600">
                    <span className="flex items-center">
                      <Home className="w-4 h-4 mr-1" />
                      {propertyData.property_type}
                    </span>
                    <span className="flex items-center">
                      <MapPin className="w-4 h-4 mr-1" />
                      {propertyData.locality}, {propertyData.city}
                    </span>
                    {propertyData.price && (
                      <span className="flex items-center">
                        <DollarSign className="w-4 h-4 mr-1" />
                        ₹{propertyData.price.toLocaleString()}
                      </span>
                    )}
                  </div>
                )}
              </div>
              
              <div className="flex items-center space-x-3">
                <button 
                  onClick={() => setShowAnalyticsModal(true)}
                  className="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors flex items-center space-x-2"
                >
                  <BarChart3 className="w-5 h-5" />
                  <span className="font-semibold">Market Analytics</span>
                </button>
                <button className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors">
                  Save Draft
                </button>
                <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                  {propertyId === 'new' ? 'Publish Property' : 'Update Property'}
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Sidebar Navigation */}
          <div className="lg:w-64 flex-shrink-0">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
              <div className="p-4 border-b border-gray-200">
                <h3 className="font-semibold text-gray-900">Property Management</h3>
              </div>
              <nav className="p-2">
                {tabs.map((tab) => {
                  const Icon = tab.icon;
                  return (
                    <button
                      key={tab.id}
                      onClick={() => setActiveTab(tab.id)}
                      className={`w-full flex items-center px-3 py-2 text-sm font-medium rounded-lg transition-colors ${
                        activeTab === tab.id
                          ? 'bg-blue-50 text-blue-700 border-r-2 border-blue-700'
                          : 'text-gray-600 hover:bg-gray-50 hover:text-gray-900'
                      }`}
                    >
                      <Icon className="w-4 h-4 mr-3" />
                      {tab.label}
                      {tab.id === 'documents' && documents.length > 0 && (
                        <span className="ml-auto bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">
                          {documents.length}
                        </span>
                      )}
                    </button>
                  );
                })}
              </nav>
            </div>

            {/* Property Summary Card */}
            {propertyData && (
              <div className="mt-6 bg-white rounded-lg shadow-sm border border-gray-200 p-4">
                <h4 className="font-semibold text-gray-900 mb-3">Property Summary</h4>
                <div className="space-y-2 text-sm">
                  <div className="flex justify-between">
                    <span className="text-gray-600">Type:</span>
                    <span className="font-medium capitalize">{propertyData.property_type}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Listing:</span>
                    <span className="font-medium capitalize">{propertyData.listing_type}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Location:</span>
                    <span className="font-medium">{propertyData.city}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Documents:</span>
                    <span className="font-medium">{documents.length} uploaded</span>
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Main Content */}
          <div className="flex-1">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200">
              {/* Tab Content */}
              <div className="p-6">
                {activeTab === 'documents' && (
                  <div>
                    {propertyId && propertyId !== 'new' ? (
                      <DocumentManager
                        propertyId={propertyId}
                        propertyTitle={propertyData?.title}
                        readOnly={false}
                      />
                    ) : (
                      <div>
                        <div className="mb-6">
                          <h2 className="text-xl font-semibold text-gray-900 mb-2">Property Documents</h2>
                          <p className="text-gray-600">
                            Upload and manage all property-related documents. These documents help verify 
                            your property and build trust with potential buyers.
                          </p>
                        </div>
                        
                        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                          <div className="flex items-start">
                            <svg className="w-5 h-5 text-yellow-600 mt-0.5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                            </svg>
                            <div>
                              <h3 className="text-sm font-medium text-yellow-800">Create Property First</h3>
                              <p className="text-sm text-yellow-700 mt-1">
                                You need to create a property listing first before uploading documents. 
                                Documents will be associated with your property ID.
                              </p>
                            </div>
                          </div>
                        </div>
                        
                        <DocumentUpload
                          propertyId={undefined}
                          existingDocuments={documents}
                          onDocumentsChange={handleDocumentsChange}
                          maxFileSize={10}
                          allowedTypes={['.pdf', '.doc', '.docx', '.jpg', '.jpeg', '.png']}
                        />
                      </div>
                    )}
                  </div>
                )}

                {activeTab === 'property' && (
                  <div>
                    <h2 className="text-xl font-semibold text-gray-900 mb-4">Property Details</h2>
                    <div className="bg-gray-50 rounded-lg p-8 text-center">
                      <Home className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-600">Property details form will be integrated here</p>
                      <p className="text-sm text-gray-500 mt-2">
                        This will include property specifications, features, and amenities
                      </p>
                    </div>
                  </div>
                )}

                {activeTab === 'location' && (
                  <div>
                    <h2 className="text-xl font-semibold text-gray-900 mb-4">Location Details</h2>
                    <div className="bg-gray-50 rounded-lg p-8 text-center">
                      <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-600">Location and map integration will be here</p>
                      <p className="text-sm text-gray-500 mt-2">
                        Address, coordinates, and nearby landmarks
                      </p>
                    </div>
                  </div>
                )}

                {activeTab === 'pricing' && (
                  <div>
                    <h2 className="text-xl font-semibold text-gray-900 mb-4">Pricing Information</h2>
                    <div className="bg-gray-50 rounded-lg p-8 text-center">
                      <DollarSign className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-600">Pricing details form will be integrated here</p>
                      <p className="text-sm text-gray-500 mt-2">
                        Sale price, rent, deposits, and payment terms
                      </p>
                    </div>
                  </div>
                )}

                {activeTab === 'media' && (
                  <div>
                    <h2 className="text-xl font-semibold text-gray-900 mb-4">Media Gallery</h2>
                    <div className="bg-gray-50 rounded-lg p-8 text-center">
                      <ImageIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-600">Photo and video upload will be integrated here</p>
                      <p className="text-sm text-gray-500 mt-2">
                        Property images, videos, and virtual tours
                      </p>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Competitor Analytics Modal */}
      <CompetitorAnalyticsModal
        isOpen={showAnalyticsModal}
        onClose={() => setShowAnalyticsModal(false)}
        propertyData={propertyData ? {
          id: propertyId !== 'new' ? parseInt(propertyId || '0') || undefined : undefined,
          city: propertyData.city,
          locality: propertyData.locality || '',
          property_type: propertyData.property_type,
          listing_type: propertyData.listing_type,
          property_condition: 'resale', // Default value, can be updated from property details
          bedrooms: 3, // Default or from propertyData if available
          built_up_area: 1200, // Default or from propertyData if available
          sale_price: propertyData.price || propertyData.monthly_rent,
          listed_date: new Date().toISOString().split('T')[0]
        } : undefined}
      />
    </div>
  );
};