import { validateStep, validateListingForm } from '../utils/formValidation';
import React, { useState, useEffect, useRef } from 'react';
import { useForm, FormProvider } from 'react-hook-form';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { CheckCircle, Edit, FileText, Globe, MapPin, Loader2, Upload, X, Eye, BarChart3 } from 'lucide-react';
import { api } from '../utils/api';
import { useGeocoding } from '../hooks/useGeocoding';

import { toast } from 'sonner';
import { AIDescriptionGenerator } from '../components/AIDescriptionGenerator';
import { aiDescriptionApi } from '../utils/aiDescriptionApi';
import { MediaStep } from '../components/form/MediaStep';
import { uploadPropertyDocuments, generateDocumentDownloadURL } from '../api/documents';
import { ListingFormData } from '../types/listing';
import { PropertyTagsStep } from '../components/form/PropertyTagsStep';
import { CompetitorAnalyticsModal } from '../components/competitor-analytics/CompetitorAnalyticsModal';

import '../styles/asterisk.css';

// Google Maps TypeScript declarations
declare global {
  interface Window {
    google: any;
  }
}

declare namespace google {
  namespace maps {
    class Map {
      constructor(mapDiv: HTMLElement, opts?: MapOptions);
      addListener(eventName: string, handler: Function): void;
    }
    class Marker {
      constructor(opts?: MarkerOptions);
      setPosition(position: LatLng | LatLngLiteral): void;
      getPosition(): LatLng | undefined;
      addListener(eventName: string, handler: Function): void;
    }
    class Geocoder {
      geocode(request: GeocoderRequest, callback: (results: GeocoderResult[] | null, status: GeocoderStatus) => void): void;
    }
    interface MapOptions {
      center?: LatLngLiteral;
      zoom?: number;
      mapTypeControl?: boolean;
      streetViewControl?: boolean;
      fullscreenControl?: boolean;
    }
    interface MarkerOptions {
      position?: LatLngLiteral;
      map?: Map;
      draggable?: boolean;
    }
    interface LatLngLiteral {
      lat: number;
      lng: number;
    }
    class LatLng {
      lat(): number;
      lng(): number;
    }
    interface MapMouseEvent {
      latLng: LatLng | null;
    }
    interface GeocoderRequest {
      location?: LatLngLiteral;
    }
    interface GeocoderResult {
      formatted_address: string;
    }
    type GeocoderStatus = string;
  }
}

// Validation
const validation = {
  required: (message: string) => ({
    required: message
  }),
  minLength: (min: number, message: string) => ({
    minLength: { value: min, message }
  }),
  maxLength: (max: number, message: string) => ({
    maxLength: { value: max, message }
  }),
  pattern: (pattern: RegExp, message: string) => ({
    pattern: { value: pattern, message }
  }),
  min: (min: number, message: string) => ({
    min: { value: min, message }
  })
};

// Google Maps API Key
const GOOGLE_MAPS_API_KEY = 'AIzaSyBGM7Q8jORMzDOj8GMcqjel1FSFPVzG0sI';

// Map Picker Component (Ola/Rapido Style)
const MapPicker: React.FC<{
  isOpen: boolean;
  onClose: () => void;
  onLocationSelect: (locationData: {
    lat: number;
    lng: number;
    address: string;
    city: string;
    state: string;
    pincode: string;
    locality: string;
    landmark: string;
  }) => void;
  initialLat?: number;
  initialLng?: number;
}> = ({ isOpen, onClose, onLocationSelect, initialLat = 12.9716, initialLng = 77.5946 }) => {
  const [selectedPosition, setSelectedPosition] = useState({ lat: initialLat, lng: initialLng });
  const [map, setMap] = useState<any>(null);
  const [marker, setMarker] = useState<any>(null);
  const [geocoder, setGeocoder] = useState<any>(null);
  const [addressInfo, setAddressInfo] = useState<any>(null);
  const [isLoadingAddress, setIsLoadingAddress] = useState(false);
  const mapRef = useRef<HTMLDivElement>(null);

  // Auto-fetch address when position changes using Nominatim primary, Geoapify fallback, with caching
  const fetchAddressInfo = async (position: { lat: number; lng: number }) => {
    setIsLoadingAddress(true);

    // Check cache first
    const cacheKey = `location_${position.lat.toFixed(4)}_${position.lng.toFixed(4)}`;
    const cachedData = localStorage.getItem(cacheKey);
    const cacheTime = localStorage.getItem(`${cacheKey}_time`);
    const CACHE_DURATION = 24 * 60 * 60 * 1000; // 24 hours

    if (cachedData && cacheTime && (Date.now() - parseInt(cacheTime)) < CACHE_DURATION) {
      const addressData = JSON.parse(cachedData);
      setAddressInfo(addressData);
      setIsLoadingAddress(false);
      return;
    }

    try {
      console.log('🗺️ Trying Nominatim API first for reverse geocoding...');

      // Try Nominatim API first (OpenStreetMap)
      const nominatimResponse = await fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${position.lat}&lon=${position.lng}&addressdetails=1&accept-language=en`, {
        headers: {
          'User-Agent': 'NAL-Property-App/1.0'
        }
      });

      if (nominatimResponse.ok) {
        const nominatimData = await nominatimResponse.json();

        if (nominatimData && nominatimData.address) {
          const addressData = {
            fullAddress: nominatimData.display_name || '',
            city: nominatimData.address.city || nominatimData.address.town || nominatimData.address.village || '',
            state: nominatimData.address.state || '',
            pincode: nominatimData.address.postcode || '',
            locality: nominatimData.address.suburb || nominatimData.address.neighbourhood || '',
            landmark: nominatimData.address.road || ''
          };

          // Cache the result
          localStorage.setItem(cacheKey, JSON.stringify(addressData));
          localStorage.setItem(`${cacheKey}_time`, Date.now().toString());

          setAddressInfo(addressData);
          setIsLoadingAddress(false);
          return;
        }
      }

      console.log('⚠️ Nominatim failed, trying Geoapify fallback...');

      // Fallback to Geoapify API
      const GEOAPIFY_API_KEY = '9b3807944bbe44d2a0b29a5706c15bf9';
      const geoapifyResponse = await fetch(`https://api.geoapify.com/v1/geocode/reverse?lat=${position.lat}&lon=${position.lng}&apiKey=${GEOAPIFY_API_KEY}&format=json`);

      if (geoapifyResponse.ok) {
        const geoapifyData = await geoapifyResponse.json();
        if (geoapifyData && geoapifyData.results && geoapifyData.results.length > 0) {
          const result = geoapifyData.results[0];

          const addressData = {
            fullAddress: result.formatted || '',
            city: result.city || '',
            state: result.state || '',
            pincode: result.postcode || '',
            locality: result.suburb || result.district || '',
            landmark: result.street || ''
          };

          // Cache the result
          localStorage.setItem(cacheKey, JSON.stringify(addressData));
          localStorage.setItem(`${cacheKey}_time`, Date.now().toString());

          setAddressInfo(addressData);
          setIsLoadingAddress(false);
          return;
        }
      }

      console.log('⚠️ Both Nominatim and Geoapify failed, trying Google Geocoder fallback...');

      // Final fallback to Google Geocoder
      if (geocoder) {
        geocoder.geocode({ location: position }, (results: any, status: any) => {
          if (status === 'OK' && results?.[0]) {
            const result = results[0];
            const components = result.address_components;

            let addressData = {
              fullAddress: result.formatted_address,
              city: '',
              state: '',
              pincode: '',
              locality: '',
              landmark: ''
            };

            // Parse address components
            components.forEach((component: any) => {
              const types = component.types;
              if (types.includes('locality') || types.includes('sublocality_level_1')) {
                addressData.locality = component.long_name;
              }
              if (types.includes('administrative_area_level_2')) {
                addressData.city = component.long_name;
              }
              if (types.includes('administrative_area_level_1')) {
                addressData.state = component.long_name;
              }
              if (types.includes('postal_code')) {
                addressData.pincode = component.long_name;
              }
              if (types.includes('route') || types.includes('premise')) {
                addressData.landmark = component.long_name;
              }
            });

            // Cache the result
            localStorage.setItem(cacheKey, JSON.stringify(addressData));
            localStorage.setItem(`${cacheKey}_time`, Date.now().toString());

            setAddressInfo(addressData);
          } else {
            // All geocoding services failed - silent fallback
          }
          setIsLoadingAddress(false);
        });
      } else {
        setIsLoadingAddress(false);
      }

    } catch (error) {
      console.error('Geocoding error:', error);

      // Final fallback to Google Geocoder
      if (geocoder) {
        geocoder.geocode({ location: position }, (results: any, status: any) => {
          if (status === 'OK' && results?.[0]) {
            const result = results[0];
            const components = result.address_components;

            let addressData = {
              fullAddress: result.formatted_address,
              city: '',
              state: '',
              pincode: '',
              locality: '',
              landmark: ''
            };

            components.forEach((component: any) => {
              const types = component.types;
              if (types.includes('locality') || types.includes('sublocality_level_1')) {
                addressData.locality = component.long_name;
              }
              if (types.includes('administrative_area_level_2')) {
                addressData.city = component.long_name;
              }
              if (types.includes('administrative_area_level_1')) {
                addressData.state = component.long_name;
              }
              if (types.includes('postal_code')) {
                addressData.pincode = component.long_name;
              }
              if (types.includes('route') || types.includes('premise')) {
                addressData.landmark = component.long_name;
              }
            });

            setAddressInfo(addressData);
          }
          setIsLoadingAddress(false);
        });
      } else {
        setIsLoadingAddress(false);
      }
    }
  };

  useEffect(() => {
    if (isOpen && mapRef.current && !map) {
      const googleMap = new google.maps.Map(mapRef.current, {
        center: selectedPosition,
        zoom: 16,
        mapTypeControl: false,
        streetViewControl: false,
        fullscreenControl: false
      });

      const mapMarker = new google.maps.Marker({
        position: selectedPosition,
        map: googleMap,
        draggable: true
      });

      const mapGeocoder = new google.maps.Geocoder();

      // Handle map clicks
      googleMap.addListener('click', (e: any) => {
        if (e.latLng) {
          const newPos = { lat: e.latLng.lat(), lng: e.latLng.lng() };
          setSelectedPosition(newPos);
          mapMarker.setPosition(newPos);
          fetchAddressInfo(newPos);
        }
      });

      // Handle marker drag
      mapMarker.addListener('dragend', () => {
        const pos = mapMarker.getPosition();
        if (pos) {
          const newPos = { lat: pos.lat(), lng: pos.lng() };
          setSelectedPosition(newPos);
          fetchAddressInfo(newPos);
        }
      });

      setMap(googleMap);
      setMarker(mapMarker);
      setGeocoder(mapGeocoder);

      // Fetch initial address
      setTimeout(() => fetchAddressInfo(selectedPosition), 1000);
    }
  }, [isOpen, map]);

  const handleConfirm = () => {
    // Always allow confirmation, even without address info
    onLocationSelect({
      lat: selectedPosition.lat,
      lng: selectedPosition.lng,
      address: addressInfo?.fullAddress || '',
      city: addressInfo?.city || '',
      state: addressInfo?.state || '',
      pincode: addressInfo?.pincode || '',
      locality: addressInfo?.locality || '',
      landmark: addressInfo?.landmark || ''
    });
    onClose();
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-white z-50 flex flex-col h-screen">
      {/* Header */}
      <div className="bg-white shadow-sm border-b p-4 flex items-center justify-between">
        <button onClick={onClose} className="p-2 hover:bg-gray-100 rounded-full">
          <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
          </svg>
        </button>
        <div className="text-center">
          <h3 className="text-lg font-semibold">Select Property Location</h3>
          <p className="text-sm text-gray-600">Tap on the map or drag the pin to select location</p>
        </div>
        <div className="w-10" />
      </div>

      {/* Map */}
      <div className="flex-1 relative min-h-0">
        <div ref={mapRef} className="w-full h-full min-h-[60vh]" />

        {/* Center Pin Indicator */}
        <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 pointer-events-none z-10">
          <div className="flex flex-col items-center">
            <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center shadow-lg border-2 border-white">
              <div className="w-3 h-3 bg-white rounded-full"></div>
            </div>
            <div className="w-1 h-8 bg-red-500"></div>
            <div className="bg-red-500 text-white text-xs px-2 py-1 rounded shadow-lg mt-1">
              Drag me!
            </div>
          </div>
        </div>


      </div>

      {/* Address Info Bottom Sheet */}
      <div className="bg-white border-t p-4 space-y-3 max-h-[40vh] overflow-y-auto flex-shrink-0">
        {isLoadingAddress ? (
          <div className="flex items-center justify-center py-4">
            <Loader2 className="w-5 h-5 animate-spin text-[#0056D2] mr-2" />
            <span className="text-gray-600">Getting address details...</span>
          </div>
        ) : addressInfo ? (
          <div className="space-y-3">
            <div className="flex items-start">
              <div className="w-6 h-6 bg-green-100 rounded-full flex items-center justify-center mr-3 mt-0.5">
                <svg className="w-4 h-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <div className="flex-1">
                <p className="font-medium text-gray-900 mb-2">{addressInfo.fullAddress}</p>
                <div className="grid grid-cols-2 gap-2 text-sm text-gray-600">
                  {addressInfo.locality && <p><span className="font-medium">Area:</span> {addressInfo.locality}</p>}
                  {addressInfo.city && <p><span className="font-medium">City:</span> {addressInfo.city}</p>}
                  {addressInfo.state && <p><span className="font-medium">State:</span> {addressInfo.state}</p>}
                  {addressInfo.pincode && <p><span className="font-medium">Pincode:</span> {addressInfo.pincode}</p>}
                </div>
              </div>
            </div>
          </div>
        ) : (
          <div className="text-center py-4">
            <div className="flex items-center justify-center mb-2">
              <MapPin className="w-5 h-5 text-gray-400 mr-2" />
              <span className="text-gray-500">Click on the map to select your property location</span>
            </div>
            <p className="text-xs text-gray-400">The address will be automatically detected</p>
          </div>
        )}

        <button
          onClick={handleConfirm}
          disabled={!selectedPosition}
          className="w-full py-3 bg-[#0056D2] text-white rounded-lg font-semibold hover:bg-[#0056D2]/90 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center"
        >
          <MapPin className="w-4 h-4 mr-2" />
          {addressInfo ? 'Confirm This Location' : 'Select Location'}
        </button>

        {selectedPosition && (
          <div className="mt-2 text-center text-xs text-gray-500">
            Coordinates: {selectedPosition.lat.toFixed(6)}, {selectedPosition.lng.toFixed(6)}
          </div>
        )}
      </div>
    </div>
  );
};

export const ListingForm: React.FC = () => {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isEditMode, setIsEditMode] = useState(false);
  const [editPropertyId, setEditPropertyId] = useState<string | null>(null);
  const [isLoadingProperty, setIsLoadingProperty] = useState(false);
  const [showMapPicker, setShowMapPicker] = useState(false);
  const [isGoogleMapsLoaded, setIsGoogleMapsLoaded] = useState(false);
  const [isGeneratingDescription, setIsGeneratingDescription] = useState(false);
  const [showAIDescriptionModal, setShowAIDescriptionModal] = useState(false);
  const [showCompetitorAnalyticsModal, setShowCompetitorAnalyticsModal] = useState(false);
  const [hasReadGovernmentGuidelines, setHasReadGovernmentGuidelines] = useState(false);
  const [legalGuidelinesAccepted, setLegalGuidelinesAccepted] = useState(false);
  const [lastSaved, setLastSaved] = useState<Date | null>(null);
  const autoSaveTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const draftLoadedRef = useRef(false);
  const [selectedTags, setSelectedTags] = useState<string[]>([]);
  const isSubmittingRef = useRef(false);


  const { geocodeAddress, reverseGeocode, isLoading: isGeocoding, error: geocodingError, clearError } = useGeocoding();
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const propertyType = searchParams.get('type'); // 'asset' or 'listing'

  const methods = useForm<ListingFormData>({
    mode: 'onChange',
    defaultValues: {
      ownerName: '',
      companyName: '',
      mobile: '',
      email: '',
      reraNumber: '',
      title: '',
      description: '',
      listingType: undefined,
      propertyType: undefined,
      builtUpArea: undefined,
      carpetArea: undefined,
      plotArea: undefined,
      superArea: undefined,
      bedrooms: undefined,
      bathrooms: undefined,
      balconies: undefined,
      floorNumber: '',
      totalFloors: '',
      facing: '',
      furnishing: '',
      propertyAge: '',
      possessionStatus: '',
      price: undefined,
      salePrice: undefined,
      pricePerSqFt: undefined,
      monthlyRent: undefined,
      securityDeposit: undefined,
      priceNegotiable: false,
      loanAvailability: false,
      startingBidPrice: undefined,
      minimumIncrement: undefined,
      reservePrice: undefined,
      auctionStartDate: '',
      auctionStartTime: '',
      auctionEndDate: '',
      auctionEndTime: '',
      timezone: 'Asia/Kolkata',
      urgentSaleStartDate: '',
      urgentSaleStartTime: '',
      urgentSaleEndDate: '',
      urgentSaleEndTime: '',
      actualPrice: undefined,
      city: '',
      locality: '',
      address: '',
      landmark: '',
      pincode: '',
      state: '',
      latitude: undefined,
      longitude: undefined,
      amenities: [],
      customAmenities: [],
      features: [],
      specifications: [],
      highlights: [],
      selectedMedia: [] as any[],
      primaryImage: null,
      imageUrls: [],
      virtualTourUrl: '',
      threeSixtyView: '',
      housePlan: '',
      gatedCommunity: undefined,
      contactType: 'owner',
      acceptTerms: false,
      acceptGovtGuidelines: false,
      enableAlerts: false,
      // PG/Co-living specific
      accommodationType: '',
      totalBedsAvailable: undefined,
      perBedRent: undefined,
      foodAvailability: '',
      genderPreference: '',
      pgSecurityDeposit: undefined,
      pgMaintenanceCharges: undefined,
      securityType: '',
      availabilityDate: '',
      minimumStayDuration: undefined,
      bookingAmount: undefined,
      propertyManagerContact: '',
      roomFurnishingDetails: '',
      houseRules: '',
      guestPolicy: '',
      attachedBathroom: false,
      pgCommonFacilities: [],
      images: [],
      videos: [],
      documents: [],
      requiredDocs: [],
      kycVerified: false,
      phoneVerified: false,
      emailVerified: false
    }
  });

  const { register, formState: { errors }, trigger, watch, setValue, handleSubmit, getValues, reset } = methods;

  useEffect(() => {
    if (isEditMode) return;
    const savedDraft = localStorage.getItem('listing_form_draft');
    if (savedDraft) {
      try {
        const draft = JSON.parse(savedDraft);
        Object.keys(draft.formData).forEach(key => {
          const value = draft.formData[key];
          if (value !== null && value !== undefined && value !== '') {
            setValue(key as any, value);
          }
        });
        if (draft.selectedTags) setSelectedTags(draft.selectedTags);
        if (draft.currentStep !== undefined) setCurrentStep(draft.currentStep);
        setLastSaved(new Date(draft.timestamp));
        toast.success('Draft restored!');
      } catch (e) { }
    }
  }, []);

  useEffect(() => {
    if (isEditMode) return;
    const subscription = watch((formData) => {
      if (isSubmittingRef.current) return; // Don't save if submitting
      if (autoSaveTimeoutRef.current) clearTimeout(autoSaveTimeoutRef.current);
      autoSaveTimeoutRef.current = setTimeout(() => {
        if (!isSubmittingRef.current && (formData.title || formData.ownerName)) {
          localStorage.setItem('listing_form_draft', JSON.stringify({
            formData,
            selectedTags,
            currentStep,
            timestamp: new Date().toISOString()
          }));
          setLastSaved(new Date());
        }
      }, 500);
    });
    return () => subscription.unsubscribe();
  }, [watch, isEditMode, selectedTags, currentStep]);

  useEffect(() => {
    if (isEditMode) return;
    const handleBeforeUnload = () => {
      const formData = getValues();
      if (formData.title || formData.ownerName) {
        localStorage.setItem('listing_form_draft', JSON.stringify({
          formData,
          selectedTags,
          currentStep,
          timestamp: new Date().toISOString()
        }));
      }
    };
    window.addEventListener('beforeunload', handleBeforeUnload);
    return () => window.removeEventListener('beforeunload', handleBeforeUnload);
  }, [isEditMode, selectedTags, currentStep, getValues]);

  useEffect(() => {
    if (isEditMode) return;
    const timer = setTimeout(() => {
      const formData = getValues();
      if (formData.title || formData.ownerName) {
        localStorage.setItem('listing_form_draft', JSON.stringify({
          formData,
          selectedTags,
          currentStep,
          timestamp: new Date().toISOString()
        }));
        setLastSaved(new Date());
      }
    }, 500);
    return () => clearTimeout(timer);
  }, [currentStep, isEditMode, selectedTags, getValues]);


  // Check for real-time updates from legal guidelines page
  useEffect(() => {
    const checkGuidelinesStatus = () => {
      const accepted = localStorage.getItem('legalGuidelinesAccepted') === 'true';
      setLegalGuidelinesAccepted(accepted);

      // Auto-enable the checkbox if guidelines are accepted
      if (accepted && setValue) {
        setValue('acceptGovtGuidelines', true);
      }
    };

    // Check every second for updates
    const interval = setInterval(checkGuidelinesStatus, 1000);

    return () => clearInterval(interval);
  }, [setValue]);



  // Load Google Maps API
  useEffect(() => {
    const loadGoogleMaps = () => {
      if (window.google && window.google.maps) {
        setIsGoogleMapsLoaded(true);
        return;
      }

      const script = document.createElement('script');
      script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_API_KEY}&libraries=places`;
      script.async = true;
      script.defer = true;
      script.onload = () => setIsGoogleMapsLoaded(true);
      document.head.appendChild(script);
    };

    loadGoogleMaps();
  }, []);

  // Check for location data returned from map selector
  useEffect(() => {
    const checkMapLocation = () => {
      const savedLocation = localStorage.getItem('selectedMapLocation');
      if (savedLocation) {
        try {
          const locationData = JSON.parse(savedLocation);
          setValue('latitude', locationData.lat);
          setValue('longitude', locationData.lng);
          if (locationData.address && !watch('address')) {
            setValue('address', locationData.address);
          }
          localStorage.removeItem('selectedMapLocation');
          toast.success('Location selected from map!');
        } catch (error) {
          console.error('Error parsing saved location:', error);
        }
      }
    };

    // Check when component mounts and when window gains focus (user returns from map)
    checkMapLocation();
    window.addEventListener('focus', checkMapLocation);

    return () => {
      window.removeEventListener('focus', checkMapLocation);
    };
  }, [setValue, watch]);

  // Check for edit mode and prefill user details on component mount
  useEffect(() => {
    const initializeForm = async () => {
      const editId = searchParams.get('edit');
      if (editId) {
        setIsEditMode(true);
        setEditPropertyId(editId);
        loadPropertyForEdit(editId);
      } else {
        // Prefill personal details from logged-in user or seller data
        await prefillUserDetails();
      }

      // Check if user has visited legal guidelines page
      const hasVisitedGuidelines = localStorage.getItem('hasReadGovernmentGuidelines');
      if (hasVisitedGuidelines === 'true') {
        setHasReadGovernmentGuidelines(true);
      }

      // Check if user has accepted legal guidelines
      const guidelinesAccepted = localStorage.getItem('legalGuidelinesAccepted');
      if (guidelinesAccepted === 'true') {
        setLegalGuidelinesAccepted(true);
      }

      // Listen for storage changes (when user accepts guidelines in another tab)
      const handleStorageChange = (e: StorageEvent) => {
        if (e.key === 'legalGuidelinesAccepted' && e.newValue === 'true') {
          setLegalGuidelinesAccepted(true);
        }
      };

      window.addEventListener('storage', handleStorageChange);
      return () => window.removeEventListener('storage', handleStorageChange);
    };
    
    initializeForm();
  }, [searchParams]);

  // Prefill personal details from localStorage or seller data
  const prefillUserDetails = async () => {
    try {
      // First check for asset listing data (when coming from "List for Sale" button)
      const assetListingData = localStorage.getItem('asset-listing-data');
      if (assetListingData) {
        try {
          const data = JSON.parse(assetListingData);
          console.log('Auto-filling details from asset listing data:', data);
          
          if (data.ownerName) setValue('ownerName', data.ownerName);
          if (data.email) setValue('email', data.email);
          if (data.mobile) setValue('mobile', data.mobile);
          if (data.companyName) setValue('companyName', data.companyName);
          
          // Also prefill other property details if available
          if (data.title) setValue('title', data.title);
          if (data.description) setValue('description', data.description);
          if (data.propertyType) setValue('propertyType', data.propertyType);
          if (data.listingType) setValue('listingType', data.listingType);
          if (data.address) setValue('address', data.address);
          if (data.city) setValue('city', data.city);
          if (data.state) setValue('state', data.state);
          if (data.pincode) setValue('pincode', data.pincode);
          if (data.locality) setValue('locality', data.locality);
          if (data.builtUpArea) setValue('builtUpArea', data.builtUpArea);
          if (data.bedrooms) setValue('bedrooms', data.bedrooms);
          if (data.bathrooms) setValue('bathrooms', data.bathrooms);
          if (data.salePrice) setValue('salePrice', data.salePrice);
          
          // Clear the asset listing data after using it
          localStorage.removeItem('asset-listing-data');
          
          return; // Exit early if asset listing data found
        } catch (error) {
          console.error('Error parsing asset listing data:', error);
        }
      }
      
      // Then try to get seller details from the property being listed for sale
      const sellerId = localStorage.getItem('sellerId') || searchParams.get('sellerId');
      
      if (sellerId) {
        console.log('🔍 Fetching seller details for ID:', sellerId);
        
        // Fetch seller details from properties API using the new function
        try {
          const properties = await api.fetchPropertiesBySellerId(sellerId);
          
          if (properties.length > 0) {
            // Get seller details from the first property
            const firstProperty = properties[0];
            const userName = firstProperty.fullname || firstProperty.owner_name || '';
            const userEmail = firstProperty.email || '';
            const userMobile = firstProperty.mobile || firstProperty.property_manager_contact || '';
            const userCompany = firstProperty.company_name || '';
            
            console.log('Auto-filling seller details from properties:', { userName, userEmail, userMobile, userCompany });
            
            if (userName) setValue('ownerName', userName);
            if (userEmail) setValue('email', userEmail);
            if (userMobile) setValue('mobile', userMobile);
            if (userCompany) setValue('companyName', userCompany);
            
            return; // Exit early if seller details found
          }
        } catch (error) {
          console.error('Error fetching seller details:', error);
        }
      }
      
      // Fallback to localStorage user data
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userName = user.first_name || user.name || user.full_name || user.username || '';
      const userEmail = user.email || '';
      const userMobile = user.mobile || user.phone || user.phone_number || '';
      const userCompany = user.company_name || user.company || '';

      console.log('Auto-filling user details from localStorage:', { userName, userEmail, userMobile, userCompany });

      if (userName) setValue('ownerName', userName);
      if (userEmail) setValue('email', userEmail);
      if (userMobile) setValue('mobile', userMobile);
      if (userCompany) setValue('companyName', userCompany);
    } catch (error) {
      console.error('Error prefilling user details:', error);
    }
  };

  // Load property data for editing
  const loadPropertyForEdit = async (propertyId: string) => {
    try {
      setIsLoadingProperty(true);
      const propertyData = await api.fetchPropertyById(propertyId);

      if (propertyData) {
        // Check if current user owns this property
        const currentUser = JSON.parse(localStorage.getItem('user') || '{}');
        const userEmail = currentUser.email || localStorage.getItem('user_email');
        if (!userEmail || propertyData.email !== userEmail) {
          toast.error('You can only edit your own properties');
          navigate('/');
          return;
        }

        // Map property data to form format
        const formData = mapPropertyToFormData(propertyData);
        reset(formData);
        toast.success('Property data loaded for editing');
      } else {
        toast.error('Property not found');
        navigate('/');
      }
    } catch (error) {
      console.error('Error loading property for edit:', error);
      toast.error('Failed to load property data');
      navigate('/');
    } finally {
      setIsLoadingProperty(false);
    }
  };

  // Map property data to form data structure
  const mapPropertyToFormData = (property: any): Partial<ListingFormData> => {
    return {
      // Personal Details
      ownerName: property.owner_name || '',
      companyName: property.company_name || '',
      mobile: property.mobile || '',
      email: property.email || '',

      // Property Info
      title: property.title || '',
      description: property.description || '',
      listingType: property.listing_type || undefined,
      propertyType: property.property_type || undefined,

      // Property Details
      builtUpArea: property.built_up_area || undefined,
      carpetArea: property.carpet_area || undefined,
      plotArea: property.plot_area || undefined,
      superArea: property.super_area || undefined,
      bedrooms: property.bedrooms || undefined,
      bathrooms: property.bathrooms || undefined,
      balconies: property.balconies || undefined,
      floorNumber: property.floor_number || '',
      totalFloors: property.total_floors || '',
      facing: property.facing || '',
      furnishing: property.furnishing || '',
      propertyAge: property.property_age || '',
      possessionStatus: property.possession_status || '',

      // Pricing
      price: property.price || undefined,
      salePrice: property.sale_price ? parseFloat(property.sale_price) : undefined,
      monthlyRent: property.monthly_rent ? parseFloat(property.monthly_rent) : undefined,
      securityDeposit: property.security_deposit ? parseFloat(property.security_deposit) : undefined,
      priceNegotiable: property.price_negotiable || false,
      loanAvailability: property.loan_availability || false,

      // Bidding specific
      startingBidPrice: property.starting_bidding_price || undefined,
      minimumIncrement: property.minimum_increment || undefined,
      reservePrice: property.reserve_price || undefined,
      auctionStartDate: property.auction_start_date || '',
      auctionStartTime: property.auction_start_time || '',
      auctionEndDate: property.auction_end_date || '',
      auctionEndTime: property.auction_end_time || '',

      // Urgent sale specific
      actualPrice: property.actual_price || undefined,
      urgentSaleStartDate: property.urgent_sale_start_date || '',
      urgentSaleStartTime: property.urgent_sale_start_time || '',
      urgentSaleEndDate: property.urgent_sale_end_date || '',
      urgentSaleEndTime: property.urgent_sale_end_time || '',

      // Location
      city: property.city || '',
      locality: property.locality || '',
      address: property.address || '',
      landmark: property.landmark || '',
      pincode: property.pincode || '',
      state: property.state || '',
      latitude: property.latitude || undefined,
      longitude: property.longitude || undefined,

      // Features
      amenities: property.amenities || [],
      features: property.features || [],
      highlights: property.highlights || [],
      specifications: property.specifications || [],

      // Gated Community
      gatedCommunity: property.gated_community || undefined,

      // Media
      selectedMedia: [] as any[],
      primaryImage: property.primary_image || property.photo || null,
      imageUrls: property.image_urls || [],
      virtualTourUrl: property.virtual_tour_url || '',
      threeSixtyView: property.three_sixty_view || '',
      housePlan: property.house_plan || '',

      // Contact
      contactType: property.contact_type || 'owner',

      // System
      acceptTerms: true, // Assume already accepted for existing properties
      enableAlerts: false
    };
  };





  const handleGeocode = async (landmark?: string, pincode?: string, city?: string, state?: string) => {
    clearError();

    console.log('🗺️ Geocoding request:', { landmark, pincode, city, state });

    const result = await geocodeAddress({
      landmark,
      pincode,
      city,
      state
    });

    console.log('🗺️ Geocoding result:', result);

    if (result) {
      setValue('latitude', result.latitude);
      setValue('longitude', result.longitude);
      toast.success(`Coordinates found: ${result.latitude.toFixed(4)}, ${result.longitude.toFixed(4)}`);

      // Optionally update address fields if they're empty
      if (result.address) {
        if (!city && result.address.city) {
          setValue('city', result.address.city);
        }
        if (!state && result.address.state) {
          setValue('state', result.address.state);
        }
      }
    } else {
      toast.error('Could not find coordinates for the provided location');
    }
  };

  const handleGenerateDescription = async () => {
    const currentData = getValues();

    if (!currentData.title) {
      toast.error('Please enter a property title first');
      return;
    }

    setIsGeneratingDescription(true);

    try {
      const propertyData = {
        title: currentData.title || 'Property',
        propertyType: currentData.propertyType || '',
        listingType: currentData.listingType || '',
        bedrooms: currentData.bedrooms ? String(currentData.bedrooms) : '',
        bathrooms: currentData.bathrooms ? String(currentData.bathrooms) : '',
        builtUpArea: currentData.builtUpArea ? String(currentData.builtUpArea) : '',
        city: currentData.city || '',
        locality: currentData.locality || '',
        facing: currentData.facing || '',
        furnishing: currentData.furnishing || '',
        propertyAge: currentData.propertyAge || '',
        amenities: currentData.amenities || [],
        features: currentData.features || [],
        highlights: currentData.highlights || [],
        salePrice: currentData.salePrice ? String(currentData.salePrice) : '',
        monthlyRent: currentData.monthlyRent ? String(currentData.monthlyRent) : '',
        phone: currentData.mobile || '',
        email: currentData.email || ''
      };

      const result = await aiDescriptionApi.generateDescription(propertyData);

      if (result.success && result.description) {
        setValue('description', result.description);
        toast.success('AI description generated successfully!');
      } else {
        toast.error(result.error || 'Failed to generate description');
      }

    } catch (error) {
      console.error('Error generating description:', error);
      toast.error('Failed to generate description. Please try again.');
    } finally {
      setIsGeneratingDescription(false);
    }
  };

  const onSubmit = async (data: ListingFormData) => {
    // Combine checkbox amenities with custom amenities
    const checkboxAmenities = data.amenities || [];
    const customAmenities = data.customAmenities || [];
    const allAmenities = [...checkboxAmenities, ...customAmenities];
    
    // Auto-generate tags based on property details
    const autoTags: string[] = [];

    if (data.listingType === 'sell') {
      autoTags.push('Urgent Sale');
    }

    if (data.propertyAge === '0-1' || data.propertyAge === 'under-construction') {
      autoTags.push('New Launch');
    }

    const price = data.salePrice || data.monthlyRent || 0;
    if (price > 5000000) {
      autoTags.push('Price Negotiable');
    }

    // Filter out empty image URLs
    const filteredData = {
      ...data,
      amenities: allAmenities, // Use combined amenities
      imageUrls: data.imageUrls?.filter(url => url && url.trim() !== '') || [],
      tags: autoTags,
      // Set listing_type to null for assets
      listingType: propertyType === 'asset' ? null : data.listingType
    };

    console.log('Submitting:', filteredData);

    try {
      let result;
      if (isEditMode && editPropertyId) {
        // Update existing property
        result = await api.updateListing(editPropertyId, filteredData);
        if (result.success) {
          toast.success('Property updated successfully!');
          navigate(`/property/${editPropertyId}`);
        } else {
          toast.error(`Error: ${result.error}`);
        }
      } else {
        // Create new property
        result = await api.submitListing(filteredData);
        if (result.success) {
          toast.success(`Property listed successfully! Property ID: ${result.id}`);
          api.clearDraft();
          navigate('/');
        } else {
          toast.error(`Error: ${result.error}`);
        }
      }
    } catch (error) {
      console.error('Error submitting form:', error);
      toast.error('Error submitting form. Please try again.');
    }
  };

  const steps = [
    { title: 'Basic Info', component: 'page1' },
    { title: 'Details & Pricing', component: 'page2' },
    { title: 'Location & Media', component: 'page3' },
    { title: 'Documents', component: 'page4' },
    { title: 'Review & Submit', component: 'review' }
  ];

  const nextStep = () => {
    console.log('nextStep called, current step:', currentStep, 'steps length:', steps.length);
    if (currentStep < steps.length - 1) {
      const newStep = currentStep + 1;
      console.log('Moving to step:', newStep, 'which is:', steps[newStep]?.title);
      setCurrentStep(newStep);
      window.scrollTo({ top: 0, behavior: 'smooth' });
    } else {
      console.log('Already at last step, cannot proceed');
    }
  };

  const prevStep = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  // Page 1: Basic Info Component
  const Page1BasicInfo = React.useMemo(() => () => {
    const selectedListingType = watch('listingType');
    const selectedPropertyType = watch('propertyType');

    const handleNext = async () => {
      console.log('Page1BasicInfo: Current step:', currentStep);
      const requiredFields = propertyType === 'asset' 
        ? ['ownerName', 'mobile', 'email', 'title', 'propertyType', 'description'] as const
        : ['ownerName', 'mobile', 'email', 'title', 'listingType', 'propertyType', 'description'] as const;
      
      console.log('Page1BasicInfo: Checking fields:', requiredFields);
      const isValid = await trigger(requiredFields);
      console.log('Page1BasicInfo: Validation result:', isValid);
      
      if (!isValid) {
        toast.error('Please fill all required fields');
        console.log('Page1BasicInfo: Validation failed, errors:', errors);
        return;
      }

      console.log('Page1BasicInfo: Calling nextStep, current step:', currentStep);
      nextStep();
      console.log('Page1BasicInfo: After nextStep, current step should be:', currentStep + 1);
    };



    const listingTypes = [
      { value: 'sell', label: 'Sell' },
      { value: 'rent', label: 'Rent' },
      { value: 'lease', label: 'Lease' },
      { value: 'urgent-sale', label: 'Urgent Sale' },
      { value: 'bidding', label: 'Bidding' }
    ];

    const propertyTypes = [
      { value: 'apartment', label: 'Apartment' },
      { value: 'villa', label: 'Villa' },
      { value: 'house', label: 'House' },
      { value: 'commercial', label: 'Commercial' },
      { value: 'pg-coliving', label: 'PG/Co-living' }
    ];

    return (
      <div className="space-y-8">
        {/* Personal Details */}
        <div className="bg-blue-50 rounded-lg p-5 mb-6">
          <div className="flex items-center mb-4">
            <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
              <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
              </svg>
            </div>
            <div>
              <h2 className="text-xl font-bold text-gray-900">Personal Details</h2>
              <p className="text-sm text-gray-600">Your contact information</p>
            </div>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Full Name <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('ownerName', { required: 'Name is required' })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="Enter your full name"
              />
              {errors.ownerName && <p className="mt-1 text-sm text-red-500">{errors.ownerName.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Company Name</label>
              <input
                type="text"
                {...register('companyName')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="Company name (optional)"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Mobile Number <span className="required-asterisk">*</span></label>
              <input
                type="tel"
                {...register('mobile', {
                  required: 'Mobile number is required',
                  pattern: {
                    value: /^[6-9]\d{9}$/,
                    message: 'Please enter a valid 10-digit Indian mobile number'
                  }
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="Enter 10-digit mobile number"
                maxLength={10}
              />
              {errors.mobile && <p className="mt-1 text-sm text-red-500">{errors.mobile.message}</p>}
              <p className="mt-1 text-xs text-gray-500">Format: 10 digits starting with 6-9</p>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Email Address <span className="required-asterisk">*</span></label>
              <input
                type="email"
                {...register('email', { required: 'Email is required' })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="Enter email address"
              />
              {errors.email && <p className="mt-1 text-sm text-red-500">{errors.email.message}</p>}
            </div>
          </div>

          {/* Contact Type */}
          <div className="mt-4">
            <label className="block text-sm font-medium text-gray-700 mb-2">Contact Type</label>
            <div className="flex space-x-4">
              <label className="flex items-center">
                <input
                  type="radio"
                  value="owner"
                  {...register('contactType')}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">Owner</span>
              </label>
              <label className="flex items-center">
                <input
                  type="radio"
                  value="agent"
                  {...register('contactType')}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">Agent</span>
              </label>
            </div>
          </div>
        </div>

        {/* Property Information */}
        <div className="bg-green-50 rounded-lg p-5 mb-6">
          <div className="flex items-center mb-4">
            <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
              <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z" />
              </svg>
            </div>
            <div>
              <h3 className="text-xl font-bold text-gray-900">Property Information</h3>
              <p className="text-sm text-gray-600">Title and description</p>
            </div>
          </div>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Property Title <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('title', { required: 'Property title is required' })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., Luxury 3BHK Apartment with Sea View"
              />
              {errors.title && <p className="mt-1 text-sm text-red-500">{errors.title.message}</p>}
            </div>
            <div>
              <div className="flex items-center justify-between mb-2">
                <label className="block text-sm font-medium text-gray-700">Property Description <span className="required-asterisk">*</span></label>
                <div className="flex space-x-2">
                  <button
                    type="button"
                    onClick={handleGenerateDescription}
                    disabled={isGeneratingDescription}
                    className="flex items-center px-4 py-2 text-sm font-medium text-purple-600 bg-white border border-purple-300 rounded-lg hover:bg-purple-50 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200 shadow-sm hover:shadow-md"
                  >
                    {isGeneratingDescription ? (
                      <>
                        <svg className="w-4 h-4 mr-2 animate-spin" fill="none" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        Generating...
                      </>
                    ) : (
                      <>
                        <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                        </svg>
                        Quick AI
                      </>
                    )}
                  </button>
                  <button
                    type="button"
                    onClick={() => {
                      if (!watch('title')) {
                        toast.error('Please enter a property title first');
                        return;
                      }
                      setShowAIDescriptionModal(true);
                    }}
                    className="flex items-center px-4 py-2 text-sm font-medium text-purple-600 bg-white border border-purple-300 rounded-lg hover:bg-purple-50 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200 shadow-sm hover:shadow-md"
                  >
                    <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                    </svg>
                    AI Options
                  </button>
                </div>
              </div>
              <textarea
                {...register('description', { required: 'Property description is required' })}
                rows={4}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-y"
                placeholder="Describe your property in detail... or click 'AI Generate' to create one automatically"
              />
              {errors.description && <p className="mt-1 text-sm text-red-500">{errors.description.message}</p>}
              {isGeneratingDescription && (
                <div className="mt-2 flex items-center text-sm text-blue-600">
                  <svg className="w-4 h-4 mr-2 animate-spin" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  AI is crafting a professional description for your property...
                </div>
              )}
              <div className="mt-2 text-xs text-gray-500">
                💡 Tip: Use "Quick AI" for instant generation or "AI Options" to choose from multiple styles
              </div>
            </div>
          </div>
        </div>

        {/* Listing Type - Only show for listings, not assets */}
        {propertyType !== 'asset' && (
          <div className="bg-orange-50 rounded-lg p-5 mb-6">
            <div className="flex items-center mb-4">
              <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
                <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                </svg>
              </div>
              <div>
                <h3 className="text-xl font-bold text-gray-900">Listing Type</h3>
                <p className="text-sm text-gray-600">Choose listing method</p>
              </div>
            </div>
            <input type="hidden" {...register('listingType', validation.required('Please select listing type'))} />
            <div className="grid grid-cols-2 md:grid-cols-5 gap-3">
              {listingTypes.map((type) => (
                <button
                  key={type.value}
                  type="button"
                  onClick={() => {
                    setValue('listingType', type.value as any);
                    trigger('listingType');
                  }}
                  className={`py-3 px-4 border-2 rounded-lg text-center transition-all duration-200 hover:border-[#0056D2] hover:shadow-sm ${selectedListingType === type.value
                    ? 'border-[#0056D2] bg-[#0056D2] text-white shadow-md'
                    : 'border-gray-300 text-gray-700 bg-white shadow-sm'
                    }`}
                >
                  <span className="font-semibold text-sm">{type.label}</span>
                </button>
              ))}
            </div>
            {errors.listingType && <p className="mt-2 text-sm text-red-500">{errors.listingType.message}</p>}
          </div>
        )}

        {/* Property Type */}
        <div className="bg-teal-50 rounded-lg p-5 mb-6">
          <div className="flex items-center mb-4">
            <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
              <svg className="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
              </svg>
            </div>
            <div>
              <h3 className="text-xl font-bold text-gray-900">Property Type</h3>
              <p className="text-sm text-gray-600">Select property category</p>
            </div>
          </div>
          <input type="hidden" {...register('propertyType', validation.required('Please select property type'))} />
          <div className="grid grid-cols-2 md:grid-cols-5 gap-3">
            {propertyTypes.map((type) => (
              <button
                key={type.value}
                type="button"
                onClick={() => {
                  setValue('propertyType', type.value as any);
                  trigger('propertyType');
                }}
                className={`py-3 px-3 border-2 rounded-lg text-center transition-all duration-200 hover:border-[#0056D2] hover:shadow-sm ${selectedPropertyType === type.value
                  ? 'border-[#0056D2] bg-[#0056D2] text-white shadow-md'
                  : 'border-gray-300 text-gray-700 bg-white shadow-sm'
                  }`}
              >
                <span className="font-semibold text-xs">{type.label}</span>
              </button>
            ))}
          </div>
          {errors.propertyType && <p className="mt-2 text-sm text-red-500">{errors.propertyType.message}</p>}
        </div>





        <div className="flex justify-between pt-6">
          <div></div>
          <button
            type="button"
            onClick={handleNext}
            className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg"
          >
            Next: Details & Pricing
          </button>
        </div>
      </div>
    );
  }, [errors, isGeneratingDescription, showAIDescriptionModal, currentStep]);

  // Page 2: Property Details & Pricing
  const Page2Details = React.useMemo(() => () => {
    const selectedListingType = watch('listingType');
    const selectedPropertyType = watch('propertyType');

    const handleNext = async () => {
      console.log('Page2Details: Current step:', currentStep);
      console.log('Page2Details: Calling nextStep directly');
      nextStep();
    };

    return (
      <div className="space-y-6">
        {/* Property Specifications */}
        <div className="bg-blue-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Property Specifications</h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Built-up Area (sq ft) <span className="required-asterisk">*</span></label>
              <input
                type="number"
                {...register('builtUpArea', {
                  required: 'Built-up area is required',
                  valueAsNumber: true,
                  min: { value: 1, message: 'Area must be greater than 0' }
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 1200"
              />
              {errors.builtUpArea && <p className="mt-1 text-sm text-red-500">{errors.builtUpArea.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Carpet Area (sq ft)</label>
              <input
                type="number"
                {...register('carpetArea')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 1000"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Bedrooms <span className="required-asterisk">*</span></label>
              <select
                {...register('bedrooms', {
                  required: 'Number of bedrooms is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="1">1 BHK</option>
                <option value="2">2 BHK</option>
                <option value="3">3 BHK</option>
                <option value="4">4 BHK</option>
                <option value="5">5+ BHK</option>
              </select>
              {errors.bedrooms && <p className="mt-1 text-sm text-red-500">{errors.bedrooms.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Bathrooms <span className="required-asterisk">*</span></label>
              <select
                {...register('bathrooms', {
                  required: 'Number of bathrooms is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="1">1</option>
                <option value="2">2</option>
                <option value="3">3</option>
                <option value="4">4+</option>
              </select>
              {errors.bathrooms && <p className="mt-1 text-sm text-red-500">{errors.bathrooms.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Balconies</label>
              <select
                {...register('balconies')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="0">0</option>
                <option value="1">1</option>
                <option value="2">2</option>
                <option value="3">3+</option>
              </select>
            </div>
          </div>
        </div>

        {/* Gated Community */}
        <div className="bg-yellow-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Community Type</h3>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Is this a Gated Community? <span className="required-asterisk">*</span></label>
            <div className="flex space-x-4">
              <label className="flex items-center">
                <input
                  type="radio"
                  value="yes"
                  {...register('gatedCommunity', { required: 'Please select if this is a gated community' })}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">Yes</span>
              </label>
              <label className="flex items-center">
                <input
                  type="radio"
                  value="no"
                  {...register('gatedCommunity', { required: 'Please select if this is a gated community' })}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">No</span>
              </label>
            </div>
            {errors.gatedCommunity && <p className="mt-1 text-sm text-red-500">{errors.gatedCommunity.message}</p>}
            <div className="mt-2 text-xs text-gray-500">
              Current value: {watch('gatedCommunity') || 'Not selected'}
            </div>
          </div>
        </div>

        {/* Property Features */}
        <div className="bg-green-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Property Features</h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Floor Number</label>
              <input
                type="text"
                {...register('floorNumber')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 5th Floor"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Total Floors</label>
              <input
                type="text"
                {...register('totalFloors')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 15"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Facing <span className="required-asterisk">*</span></label>
              <select
                {...register('facing', {
                  required: 'Property facing is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="north">North</option>
                <option value="south">South</option>
                <option value="east">East</option>
                <option value="west">West</option>
                <option value="north-east">North-East</option>
                <option value="north-west">North-West</option>
                <option value="south-east">South-East</option>
                <option value="south-west">South-West</option>
              </select>
              {errors.facing && <p className="mt-1 text-sm text-red-500">{errors.facing.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Furnishing <span className="required-asterisk">*</span></label>
              <select
                {...register('furnishing', {
                  required: 'Furnishing status is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="unfurnished">Unfurnished</option>
                <option value="semi-furnished">Semi-Furnished</option>
                <option value="fully-furnished">Fully Furnished</option>
              </select>
              {errors.furnishing && <p className="mt-1 text-sm text-red-500">{errors.furnishing.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Property Age <span className="required-asterisk">*</span></label>
              <select
                {...register('propertyAge', {
                  required: 'Property age is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="under-construction">Under Construction</option>
                <option value="0-1">0-1 Years</option>
                <option value="1-5">1-5 Years</option>
                <option value="5-10">5-10 Years</option>
                <option value="10+">10+ Years</option>
              </select>
              {errors.propertyAge && <p className="mt-1 text-sm text-red-500">{errors.propertyAge.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Possession Status</label>
              <select
                {...register('possessionStatus')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              >
                <option value="">Select</option>
                <option value="ready-to-move">Ready to Move</option>
                <option value="under-construction">Under Construction</option>
                <option value="new-launch">New Launch</option>
              </select>
            </div>
          </div>
        </div>



        {/* Amenities & Features */}
        <div className="bg-purple-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Amenities & Features</h3>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3 mb-4">
            {['Swimming Pool', 'Gym', 'Parking', 'Security', 'Garden', 'Elevator', 'Power Backup', 'Water Supply', 'Club House', 'Children Play Area', 'Jogging Track', 'CCTV', 'Air Conditioning', 'Balcony', 'Modular Kitchen', 'Servant Room', 'Study Room', 'Pooja Room', 'Terrace Garden', 'Intercom', 'Piped Gas', 'Maintenance Staff', 'Visitor Parking', 'Washing Machine', 'Refrigerator', 'Microwave', 'WiFi', 'Laundry', 'Kitchen Access'].map((amenity) => (
              <label key={amenity} className="flex items-center">
                <input
                  type="checkbox"
                  value={amenity}
                  {...register('amenities')}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">{amenity}</span>
              </label>
            ))}
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Additional Amenities</label>
            <input
              type="text"
              {...register('customAmenitiesText')}
              className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
              placeholder="Enter any additional amenities not listed above (comma separated)"
              onChange={(e) => {
                const customText = e.target.value;
                if (customText.trim()) {
                  const customItems = customText.split(',').map(item => item.trim()).filter(item => item);
                  setValue('customAmenities', customItems);
                } else {
                  setValue('customAmenities', []);
                }
              }}
            />
            <p className="mt-1 text-xs text-gray-500">Example: Rooftop Access, Private Garden, Solar Panels</p>
          </div>
        </div>

        {/* Property Highlights */}
        <div className="bg-pink-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Property Highlights</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            {['Prime Location', 'Vastu Compliant', 'Corner Property', 'Park Facing', 'Gated Community', 'Near Metro', 'Investment Property', 'Ready to Move'].map((highlight) => (
              <label key={highlight} className="flex items-center">
                <input
                  type="checkbox"
                  value={highlight}
                  {...register('highlights')}
                  className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                />
                <span className="ml-2 text-sm text-gray-700">{highlight}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Detailed Pricing */}
        <div className="bg-red-50 rounded-lg p-5">
          <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3 mb-4">
            <h3 className="text-xl font-bold text-gray-900">Detailed Pricing</h3>
            <button
              type="button"
              onClick={() => {
                const currentData = getValues();
                if (!currentData.city || !currentData.locality || !currentData.propertyType) {
                  toast.error('Please fill in city, locality, and property type first');
                  return;
                }
                
                if (propertyType !== 'asset' && !currentData.listingType) {
                  toast.error('Please select a listing type first');
                  return;
                }
                
                setShowCompetitorAnalyticsModal(true);
              }}
              className="px-5 py-2.5 bg-green-600 textwhite rounded-lg hover:bg-gray-700 transition-all duration-200 flex items-center space-x-2 shadow-lg hover:shadow-xl font-semibold text-sm whitespace-nowrap"
            >
              <BarChart3 className="w-5 h-5" />
              <span>Market Analysis</span>
            </button>
          </div>
          {selectedListingType === 'sell' && (
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Sale Price (₹) <span className="required-asterisk">*</span></label>
                  <input
                    type="number"
                    {...register('salePrice', {
                      required: 'Sale price is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Sale price must be greater than 0' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 5000000"
                  />
                  {errors.salePrice && <p className="mt-1 text-sm text-red-500">{errors.salePrice.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Price per sq ft (₹)</label>
                  <input
                    type="number"
                    {...register('pricePerSqFt')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 4500"
                  />
                </div>
              </div>
              <div className="flex items-center space-x-6">
                <label className="flex items-center">
                  <input
                    type="checkbox"
                    {...register('priceNegotiable')}
                    className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                  />
                  <span className="ml-2 text-sm text-gray-700">Price Negotiable</span>
                </label>
                <label className="flex items-center">
                  <input
                    type="checkbox"
                    {...register('loanAvailability')}
                    className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                  />
                  <span className="ml-2 text-sm text-gray-700">Loan Available</span>
                </label>
              </div>
            </div>
          )}
          {selectedListingType === 'rent' && (
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Monthly Rent (₹) <span className="required-asterisk">*</span></label>
                  <input
                    type="number"
                    {...register('monthlyRent', {
                      required: 'Monthly rent is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Monthly rent must be greater than 0' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 25000"
                  />
                  {errors.monthlyRent && <p className="mt-1 text-sm text-red-500">{errors.monthlyRent.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Security Deposit (₹)</label>
                  <input
                    type="number"
                    {...register('securityDeposit')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 50000"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Lease Terms</label>
                <textarea
                  {...register('leaseTerms')}
                  rows={2}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                  placeholder="Minimum lease period, terms & conditions..."
                />
              </div>
            </div>
          )}
          {selectedListingType === 'bidding' && (
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Starting Bid Price (₹) <span className="required-asterisk">*</span></label>
                  <input
                    type="number"
                    {...register('startingBidPrice', {
                      required: 'Starting bid price is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Starting bid price must be greater than 0' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 4000000"
                  />
                  {errors.startingBidPrice && <p className="mt-1 text-sm text-red-500">{errors.startingBidPrice.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Minimum Increment (₹)</label>
                  <input
                    type="number"
                    {...register('minimumIncrement')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 50000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Reserve Price (₹)</label>
                  <input
                    type="number"
                    {...register('reservePrice')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 5000000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Timezone</label>
                  <select
                    {...register('timezone')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="Asia/Kolkata">Asia/Kolkata (IST)</option>
                    <option value="Asia/Mumbai">Asia/Mumbai (IST)</option>
                  </select>
                </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Auction Start Date</label>
                  <input
                    type="date"
                    {...register('auctionStartDate')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Auction Start Time</label>
                  <input
                    type="time"
                    {...register('auctionStartTime')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Auction End Date</label>
                  <input
                    type="date"
                    {...register('auctionEndDate')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Auction End Time</label>
                  <input
                    type="time"
                    {...register('auctionEndTime')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                </div>
              </div>
            </div>
          )}
          {selectedListingType === 'lease' && (
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Lease Type <span className="text-red-600 font-bold">*</span></label>
                  <select
                    {...register('leaseType', validation.required('Lease type is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="commercial">Commercial Lease</option>
                    <option value="residential">Residential Lease</option>
                    <option value="industrial">Industrial Lease</option>
                    <option value="retail">Retail Lease</option>
                  </select>
                  {errors.leaseType && <p className="mt-1 text-sm text-red-500">{errors.leaseType.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Lease Duration <span className="text-red-600 font-bold">*</span></label>
                  <select
                    {...register('leaseDuration', validation.required('Lease duration is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="6-months">6 Months</option>
                    <option value="1-year">1 Year</option>
                    <option value="2-years">2 Years</option>
                    <option value="3-years">3 Years</option>
                    <option value="5-years">5 Years</option>
                    <option value="custom">Custom</option>
                  </select>
                  {errors.leaseDuration && <p className="mt-1 text-sm text-red-500">{errors.leaseDuration.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Monthly Lease Amount (₹) <span className="text-red-600 font-bold">*</span></label>
                  <input
                    type="number"
                    {...register('monthlyLeaseAmount', {
                      required: 'Monthly lease amount is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Amount must be greater than 0' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 50000"
                  />
                  {errors.monthlyLeaseAmount && <p className="mt-1 text-sm text-red-500">{errors.monthlyLeaseAmount.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Advance Deposit (₹)</label>
                  <input
                    type="number"
                    {...register('advanceDeposit')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 200000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Maintenance Charges (₹)</label>
                  <input
                    type="number"
                    {...register('maintenanceCharges')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 5000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Lock-in Period</label>
                  <select
                    {...register('lockInPeriod')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="no-lockin">No Lock-in</option>
                    <option value="3-months">3 Months</option>
                    <option value="6-months">6 Months</option>
                    <option value="1-year">1 Year</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Lease Start Date <span className="text-red-600 font-bold">*</span></label>
                  <input
                    type="date"
                    {...register('leaseStartDate', validation.required('Lease start date is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                  {errors.leaseStartDate && <p className="mt-1 text-sm text-red-500">{errors.leaseStartDate.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Lease End Date</label>
                  <input
                    type="date"
                    {...register('leaseEndDate')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Possession Availability <span className="text-red-600 font-bold">*</span></label>
                  <select
                    {...register('possessionAvailability', validation.required('Possession availability is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="immediate">Immediate</option>
                    <option value="within-15-days">Within 15 Days</option>
                    <option value="within-30-days">Within 30 Days</option>
                    <option value="within-60-days">Within 60 Days</option>
                    <option value="custom">Custom Date</option>
                  </select>
                  {errors.possessionAvailability && <p className="mt-1 text-sm text-red-500">{errors.possessionAvailability.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Payment Frequency</label>
                  <select
                    {...register('paymentFrequency')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="monthly">Monthly</option>
                    <option value="quarterly">Quarterly</option>
                    <option value="half-yearly">Half Yearly</option>
                    <option value="yearly">Yearly</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Maintenance Responsibility</label>
                  <select
                    {...register('maintenanceResponsibility')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="tenant">Tenant</option>
                    <option value="landlord">Landlord</option>
                    <option value="shared">Shared</option>
                  </select>
                </div>
              </div>
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Renewal Terms</label>
                  <textarea
                    {...register('renewalTerms')}
                    rows={2}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                    placeholder="Terms and conditions for lease renewal..."
                  />
                </div>
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    {...register('autoRenew')}
                    className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                  />
                  <span className="ml-2 text-sm text-gray-700">Auto-renew lease</span>
                </div>
              </div>
            </div>
          )}
          {selectedListingType === 'urgent-sale' && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Actual Price (₹) <span className="text-red-600 font-bold">*</span></label>
                <input
                  type="number"
                  {...register('actualPrice', {
                    required: 'Actual price is required',
                    valueAsNumber: true,
                    min: { value: 1, message: 'Actual price must be greater than 0' }
                  })}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  placeholder="e.g., 5000000"
                />
                {errors.actualPrice && <p className="mt-1 text-sm text-red-500">{errors.actualPrice.message}</p>}
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Urgent Sale Start Date</label>
                <input
                  type="date"
                  {...register('urgentSaleStartDate')}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Urgent Sale Start Time</label>
                <input
                  type="time"
                  {...register('urgentSaleStartTime')}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Urgent Sale End Date</label>
                <input
                  type="date"
                  {...register('urgentSaleEndDate')}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Urgent Sale End Time</label>
                <input
                  type="time"
                  {...register('urgentSaleEndTime')}
                  className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                />
              </div>
            </div>
          )}
        </div>

        {/* PG/Co-living Specific Fields */}
        {selectedPropertyType === 'pg-coliving' && (
          <div className="bg-indigo-50 rounded-lg p-5">
            <h3 className="text-xl font-bold text-gray-900 mb-4">PG/Co-living Details</h3>
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Accommodation Type <span className="text-red-600 font-bold">*</span></label>
                  <select
                    {...register('accommodationType', validation.required('Accommodation type is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="single-sharing">Single Sharing</option>
                    <option value="double-sharing">Double Sharing</option>
                    <option value="triple-sharing">Triple Sharing</option>
                    <option value="dormitory">Dormitory</option>
                  </select>
                  {errors.accommodationType && <p className="mt-1 text-sm text-red-500">{errors.accommodationType.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Total Beds Available <span className="text-red-600 font-bold">*</span></label>
                  <input
                    type="number"
                    {...register('totalBedsAvailable', {
                      required: 'Total beds available is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Must be at least 1 bed' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 10"
                  />
                  {errors.totalBedsAvailable && <p className="mt-1 text-sm text-red-500">{errors.totalBedsAvailable.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Per Bed Rent (₹) <span className="text-red-600 font-bold">*</span></label>
                  <input
                    type="number"
                    {...register('perBedRent', {
                      required: 'Per bed rent is required',
                      valueAsNumber: true,
                      min: { value: 1, message: 'Rent must be greater than 0' }
                    })}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 8000"
                  />
                  {errors.perBedRent && <p className="mt-1 text-sm text-red-500">{errors.perBedRent.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Food Availability / Meal Type</label>
                  <select
                    {...register('foodAvailability')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="no-food">No Food</option>
                    <option value="breakfast-only">Breakfast Only</option>
                    <option value="lunch-dinner">Lunch & Dinner</option>
                    <option value="all-meals">All Meals</option>
                    <option value="veg-only">Veg Only</option>
                    <option value="non-veg">Non-Veg Available</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Gender Preference <span className="text-red-600 font-bold">*</span></label>
                  <select
                    {...register('genderPreference', validation.required('Gender preference is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="male-only">Male Only</option>
                    <option value="female-only">Female Only</option>
                    <option value="co-ed">Co-ed</option>
                  </select>
                  {errors.genderPreference && <p className="mt-1 text-sm text-red-500">{errors.genderPreference.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Security Deposit (₹)</label>
                  <input
                    type="number"
                    {...register('pgSecurityDeposit')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 15000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Maintenance / Cleaning Charges (₹)</label>
                  <input
                    type="number"
                    {...register('pgMaintenanceCharges')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 2000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Security Type</label>
                  <select
                    {...register('securityType')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="24x7-security">24x7 Security</option>
                    <option value="cctv-only">CCTV Only</option>
                    <option value="biometric">Biometric Access</option>
                    <option value="basic">Basic Security</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Availability Date <span className="text-red-600 font-bold">*</span></label>
                  <input
                    type="date"
                    {...register('availabilityDate', validation.required('Availability date is required'))}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  />
                  {errors.availabilityDate && <p className="mt-1 text-sm text-red-500">{errors.availabilityDate.message}</p>}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Minimum Stay Duration</label>
                  <select
                    {...register('minimumStayDuration')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                  >
                    <option value="">Select</option>
                    <option value="no-minimum">No Minimum</option>
                    <option value="1-month">1 Month</option>
                    <option value="3-months">3 Months</option>
                    <option value="6-months">6 Months</option>
                    <option value="1-year">1 Year</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Booking Amount (₹)</label>
                  <input
                    type="number"
                    {...register('bookingAmount')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., 5000"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Manager Contact</label>
                  <input
                    type="tel"
                    {...register('propertyManagerContact')}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                    placeholder="e.g., +91 9876543210"
                  />
                </div>
              </div>
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Room Furnishing Details</label>
                  <textarea
                    {...register('roomFurnishingDetails')}
                    rows={2}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                    placeholder="Describe room furnishing details..."
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">House Rules / Curfew Timing</label>
                  <textarea
                    {...register('houseRules')}
                    rows={2}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                    placeholder="House rules, curfew timing, etc..."
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Guest Policy</label>
                  <textarea
                    {...register('guestPolicy')}
                    rows={2}
                    className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                    placeholder="Guest visiting policy and restrictions..."
                  />
                </div>
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    {...register('attachedBathroom')}
                    className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                  />
                  <span className="ml-2 text-sm text-gray-700">Attached Bathroom Available</span>
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Common Facilities</label>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                  {['WiFi', 'Laundry', 'Kitchen', 'TV/Entertainment', 'Study Room', 'Parking', 'Power Backup', 'Water Purifier', 'Refrigerator', 'Microwave', 'Washing Machine', 'Air Conditioning'].map((facility) => (
                    <label key={facility} className="flex items-center">
                      <input
                        type="checkbox"
                        value={facility}
                        {...register('pgCommonFacilities')}
                        className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                      />
                      <span className="ml-2 text-sm text-gray-700">{facility}</span>
                    </label>
                  ))}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Competitor Analytics CTA */}
        <div className="bg-gradient-to-r from-blue-50 to-purple-50 border-2 border-dashed border-blue-300 rounded-lg p-6">
          <div className="text-center">
            <div className="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full mb-4">
              <BarChart3 className="w-8 h-8 text-white" />
            </div>
            <h3 className="text-xl font-bold text-gray-900 mb-2">Get Market Insights</h3>
            <p className="text-gray-600 mb-4 max-w-md mx-auto">
              Analyze your property pricing against similar properties in your locality. 
              Get instant market insights to price your property competitively.
            </p>
            <button
              type="button"
              onClick={() => {
                const currentData = getValues();
                if (!currentData.city || !currentData.locality || !currentData.propertyType) {
                  toast.error('Please fill in city, locality, and property type first');
                  return;
                }
                
                if (propertyType !== 'asset' && !currentData.listingType) {
                  toast.error('Please select a listing type first');
                  return;
                }
                
                setShowCompetitorAnalyticsModal(true);
              }}
              className="px-8 py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors font-semibold shadow-md hover:shadow-lg"
            >
              Analyze Market Now
            </button>
            <p className="text-xs text-gray-500 mt-2">Free market analysis • Instant results</p>
          </div>
        </div>

        <div className="flex justify-between pt-6">
          <button
            type="button"
            onClick={prevStep}
            className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-semibold"
          >
            Previous
          </button>
          <button
            type="button"
            onClick={handleNext}
            className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg"
          >
            Next: Location & Media
          </button>
        </div>
      </div>
    );
  }, [errors, currentStep]);

  // Page 3: Location & Media
  const Page3LocationMedia = React.useMemo(() => () => {
    const handleNext = async () => {
      console.log('Page3LocationMedia: Current step:', currentStep);
      console.log('Page3LocationMedia: Calling nextStep directly');
      nextStep();
    };

    return (
      <div className="space-y-6">
        {/* Location Details */}
        <div className="bg-blue-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Location Details</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="md:col-span-2">
              <label className="block text-sm font-medium text-gray-700 mb-2">Property Address <span className="required-asterisk">*</span></label>
              <textarea
                {...register('address', {
                  required: 'Address is required'
                })}
                rows={3}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white resize-none"
                placeholder="Enter complete property address"
              />
              {errors.address && <p className="mt-1 text-sm text-red-500">{errors.address.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">City <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('city', {
                  required: 'City is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., Mumbai"
              />
              {errors.city && <p className="mt-1 text-sm text-red-500">{errors.city.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Pincode <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('pincode', {
                  required: 'Pincode is required',
                  pattern: {
                    value: /^[1-9][0-9]{5}$/,
                    message: 'Please enter a valid 6-digit Indian pincode'
                  }
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 400050"
                maxLength={6}
              />
              {errors.pincode && <p className="mt-1 text-sm text-red-500">{errors.pincode.message}</p>}
              <p className="mt-1 text-xs text-gray-500">Format: 6 digits</p>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Locality <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('locality', {
                  required: 'Locality is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., Bandra West"
              />
              {errors.locality && <p className="mt-1 text-sm text-red-500">{errors.locality.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">State <span className="required-asterisk">*</span></label>
              <input
                type="text"
                {...register('state', {
                  required: 'State is required'
                })}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., Maharashtra"
              />
              {errors.state && <p className="mt-1 text-sm text-red-500">{errors.state.message}</p>}
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Nearby Landmark</label>
              <input
                type="text"
                {...register('landmark')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., Near Bandra Station"
              />
            </div>
            <div className="md:col-span-2">
              <label className="block text-sm font-medium text-gray-700 mb-2">RERA Number</label>
              <input
                type="text"
                {...register('reraNumber')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., MH-REG-12345"
              />
            </div>
          </div>
        </div>



        {/* Enhanced Map Location */}
        <div className="bg-indigo-50 rounded-lg p-5">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center">
              <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
                <MapPin className="w-5 h-5 text-white" />
              </div>
              <div>
                <h3 className="text-xl font-bold text-gray-900">Map Location</h3>
                <p className="text-sm text-gray-600">Auto-filled from address/pincode or select on map</p>
              </div>
            </div>
            {isGeocoding && (
              <div className="flex items-center text-[#0056D2]">
                <Loader2 className="w-4 h-4 animate-spin mr-2" />
                <span className="text-sm">Locating...</span>
              </div>
            )}
          </div>

          {geocodingError && (
            <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
              <div className="flex items-start">
                <svg className="w-5 h-5 text-red-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                </svg>
                <div>
                  <p className="text-sm font-medium text-red-800">Location Error</p>
                  <p className="text-sm text-red-700 mt-1">{geocodingError}</p>
                </div>
              </div>
            </div>
          )}

          {/* Coordinates Display */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Latitude
                {watch('latitude') && (
                  <span className="ml-2 text-xs text-green-600 font-medium">✓ Auto-detected</span>
                )}
              </label>
              <input
                type="number"
                step="any"
                {...register('latitude')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 19.0760"
                readOnly={isGeocoding}
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Longitude
                {watch('longitude') && (
                  <span className="ml-2 text-xs text-green-600 font-medium">✓ Auto-detected</span>
                )}
              </label>
              <input
                type="number"
                step="any"
                {...register('longitude')}
                className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
                placeholder="e.g., 72.8777"
                readOnly={isGeocoding}
              />
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex flex-col sm:flex-row gap-3 mb-4">
            <button
              type="button"
              onClick={() => setShowMapPicker(true)}
              className="flex items-center justify-center px-4 py-2 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors"
            >
              <MapPin className="w-4 h-4 mr-2" />
              Select from Map
            </button>

            <button
              type="button"
              onClick={async () => {
                const address = watch('address');
                const pincode = watch('pincode');
                const city = watch('city');
                const landmark = watch('landmark');
                const state = watch('state');

                if (!address && !pincode && !city && !landmark) {
                  toast.error('Please enter address, pincode, city, or landmark first');
                  return;
                }

                toast.info('Fetching coordinates...');

                // Try backend geocoding first
                const result = await geocodeAddress({ landmark, pincode, city, state });

                if (result) {
                  setValue('latitude', result.latitude);
                  setValue('longitude', result.longitude);
                  toast.success(`Coordinates found: ${result.latitude.toFixed(4)}, ${result.longitude.toFixed(4)}`);
                } else {
                  // Fallback to free geocoding API
                  try {
                    const searchQuery = [address, landmark, city, pincode, state].filter(Boolean).join(', ');
                    console.log('🔄 Fallback geocoding for:', searchQuery);

                    const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(searchQuery)}&limit=1&addressdetails=1`);

                    if (response.ok) {
                      const data = await response.json();
                      if (data && data.length > 0) {
                        const lat = parseFloat(data[0].lat);
                        const lng = parseFloat(data[0].lon);

                        setValue('latitude', lat);
                        setValue('longitude', lng);
                        toast.success(`Coordinates found: ${lat.toFixed(4)}, ${lng.toFixed(4)}`);
                      } else {
                        toast.error('Could not find coordinates for the provided location');
                      }
                    } else {
                      toast.error('Geocoding service unavailable');
                    }
                  } catch (error) {
                    console.error('Fallback geocoding error:', error);
                    toast.error('Could not fetch coordinates');
                  }
                }
              }}
              disabled={isGeocoding}
              className="flex items-center justify-center px-4 py-2 border border-[#0056D2] text-[#0056D2] rounded-lg hover:bg-[#0056D2]/5 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isGeocoding ? (
                <Loader2 className="w-4 h-4 mr-2 animate-spin" />
              ) : (
                <Globe className="w-4 h-4 mr-2" />
              )}
              Get Coordinates
            </button>

            <button
              type="button"
              onClick={async () => {
                if (!navigator.geolocation) {
                  toast.error('Location access is not supported by this browser. Please use "Find on Map" instead.');
                  return;
                }

                // Check if we're on HTTPS or localhost for security
                if (location.protocol !== 'https:' && location.hostname !== 'localhost') {
                  toast.error('Location access requires a secure connection. Please enable HTTPS or use "Find on Map".');
                  return;
                }

                // Check if location permission is already denied
                if ('permissions' in navigator) {
                  try {
                    const permission = await navigator.permissions.query({ name: 'geolocation' });
                    if (permission.state === 'denied') {
                      toast.error('Location access is blocked. Please enable location access in your browser settings and refresh the page.');
                      return;
                    }
                  } catch (e) {
                    // Permission API not supported, continue with geolocation request
                  }
                }

                toast.info('Getting your location and address...');

                navigator.geolocation.getCurrentPosition(
                  async (position) => {
                    const lat = position.coords.latitude;
                    const lng = position.coords.longitude;

                    // Set coordinates first
                    setValue('latitude', lat);
                    setValue('longitude', lng);

                    // Use the same reverse geocoding logic as MapPicker
                    const cacheKey = `location_${lat.toFixed(4)}_${lng.toFixed(4)}`;
                    const cachedData = localStorage.getItem(cacheKey);
                    const cacheTime = localStorage.getItem(`${cacheKey}_time`);
                    const CACHE_DURATION = 24 * 60 * 60 * 1000; // 24 hours

                    if (cachedData && cacheTime && (Date.now() - parseInt(cacheTime)) < CACHE_DURATION) {
                      const addressData = JSON.parse(cachedData);
                      
                      // Auto-fill all address fields from cache
                      if (addressData.city) setValue('city', addressData.city);
                      if (addressData.state) setValue('state', addressData.state);
                      if (addressData.pincode) setValue('pincode', addressData.pincode);
                      if (addressData.locality) setValue('locality', addressData.locality);
                      if (addressData.landmark) setValue('landmark', addressData.landmark);
                      if (addressData.fullAddress) setValue('address', addressData.fullAddress);

                      toast.success('Location and address detected successfully!');
                      return;
                    }

                    try {
                      console.log('🗺️ Trying Nominatim API first for reverse geocoding...');

                      // Try Nominatim API first (OpenStreetMap)
                      const nominatimResponse = await fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&addressdetails=1&accept-language=en`, {
                        headers: {
                          'User-Agent': 'NAL-Property-App/1.0'
                        }
                      });

                      if (nominatimResponse.ok) {
                        const nominatimData = await nominatimResponse.json();

                        if (nominatimData && nominatimData.address) {
                          const addressData = {
                            fullAddress: nominatimData.display_name || '',
                            city: nominatimData.address.city || nominatimData.address.town || nominatimData.address.village || '',
                            state: nominatimData.address.state || '',
                            pincode: nominatimData.address.postcode || '',
                            locality: nominatimData.address.suburb || nominatimData.address.neighbourhood || '',
                            landmark: nominatimData.address.road || ''
                          };

                          // Cache the result
                          localStorage.setItem(cacheKey, JSON.stringify(addressData));
                          localStorage.setItem(`${cacheKey}_time`, Date.now().toString());

                          // Auto-fill all address fields
                          if (addressData.city) setValue('city', addressData.city);
                          if (addressData.state) setValue('state', addressData.state);
                          if (addressData.pincode) setValue('pincode', addressData.pincode);
                          if (addressData.locality) setValue('locality', addressData.locality);
                          if (addressData.landmark) setValue('landmark', addressData.landmark);
                          if (addressData.fullAddress) setValue('address', addressData.fullAddress);

                          toast.success('Location and address detected successfully!');
                          return;
                        }
                      }

                      console.log('⚠️ Nominatim failed, trying Geoapify fallback...');

                      // Fallback to Geoapify API
                      const GEOAPIFY_API_KEY = '9b3807944bbe44d2a0b29a5706c15bf9';
                      const geoapifyResponse = await fetch(`https://api.geoapify.com/v1/geocode/reverse?lat=${lat}&lon=${lng}&apiKey=${GEOAPIFY_API_KEY}&format=json`);

                      if (geoapifyResponse.ok) {
                        const geoapifyData = await geoapifyResponse.json();
                        if (geoapifyData && geoapifyData.results && geoapifyData.results.length > 0) {
                          const result = geoapifyData.results[0];

                          const addressData = {
                            fullAddress: result.formatted || '',
                            city: result.city || '',
                            state: result.state || '',
                            pincode: result.postcode || '',
                            locality: result.suburb || result.district || '',
                            landmark: result.street || ''
                          };

                          // Cache the result
                          localStorage.setItem(cacheKey, JSON.stringify(addressData));
                          localStorage.setItem(`${cacheKey}_time`, Date.now().toString());

                          // Auto-fill all address fields
                          if (addressData.city) setValue('city', addressData.city);
                          if (addressData.state) setValue('state', addressData.state);
                          if (addressData.pincode) setValue('pincode', addressData.pincode);
                          if (addressData.locality) setValue('locality', addressData.locality);
                          if (addressData.landmark) setValue('landmark', addressData.landmark);
                          if (addressData.fullAddress) setValue('address', addressData.fullAddress);

                          toast.success('Location and address detected successfully!');
                          return;
                        }
                      }

                      console.log('⚠️ Both Nominatim and Geoapify failed, coordinates only...');
                      toast.success('Location detected! Address could not be determined automatically.');

                    } catch (error) {
                      console.error('Reverse geocoding error:', error);
                      toast.success('Location detected! Address could not be determined automatically.');
                    }
                  },
                  (error) => {
                    console.error('Geolocation error:', error);
                    let errorMessage = '';

                    switch (error.code) {
                      case error.PERMISSION_DENIED:
                        errorMessage = 'Location access was denied. Please enable location access in your browser settings and try again, or use "Find on Map" instead.';
                        break;
                      case error.POSITION_UNAVAILABLE:
                        errorMessage = 'Your location could not be determined. Please try using "Find on Map" instead.';
                        break;
                      case error.TIMEOUT:
                        errorMessage = 'Location request timed out. Please try again or use "Find on Map".';
                        break;
                      default:
                        errorMessage = 'Unable to access your location. Please check your browser settings or use "Find on Map".';
                        break;
                    }

                    toast.error(errorMessage);
                  },
                  {
                    enableHighAccuracy: true,
                    timeout: 15000,
                    maximumAge: 300000 // 5 minutes
                  }
                );
              }}
              className="flex items-center justify-center px-4 py-2 border border-[#0056D2] text-[#0056D2] rounded-lg hover:bg-[#0056D2]/5 transition-colors"
            >
              <Globe className="w-4 h-4 mr-2" />
              Use My Location
            </button>

          </div>

          {/* Status Indicators */}
          <div className="grid grid-cols-1 gap-3">


            {(watch('latitude') && watch('longitude')) && (
              <div className="p-3 bg-green-50 border border-green-200 rounded-lg">
                <div className="flex items-start">
                  <svg className="w-5 h-5 text-green-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                  <div>
                    <p className="text-sm font-medium text-green-800">Location Confirmed</p>
                    <p className="text-sm text-green-700 mt-1">
                      Coordinates: {Number(watch('latitude')).toFixed(4)}, {Number(watch('longitude')).toFixed(4)}
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>

        {/* Map Picker Modal */}
        {isGoogleMapsLoaded && (
          <MapPicker
            isOpen={showMapPicker}
            onClose={() => setShowMapPicker(false)}
            onLocationSelect={(locationData) => {
              // Always update coordinates
              setValue('latitude', locationData.lat);
              setValue('longitude', locationData.lng);

              // Auto-fill all address fields
              if (locationData.address) setValue('address', locationData.address);
              if (locationData.city) setValue('city', locationData.city);
              if (locationData.state) setValue('state', locationData.state);
              if (locationData.pincode) setValue('pincode', locationData.pincode);
              if (locationData.locality) setValue('locality', locationData.locality);
              if (locationData.landmark) setValue('landmark', locationData.landmark);

              toast.success('Location selected and address auto-filled successfully!');
            }}
            initialLat={watch('latitude') || 12.9716}
            initialLng={watch('longitude') || 77.5946}
          />
        )}

        {/* Media Gallery Section */}
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6">
          <MediaStep
            onNext={() => { }}
            onPrev={() => { }}
            isFirst={false}
            isLast={false}
          />
        </div>



        <div className="flex justify-between pt-6">
          <button
            type="button"
            onClick={prevStep}
            className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-semibold"
          >
            Previous
          </button>
          <button
            type="button"
            onClick={handleNext}
            className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg"
          >
            Next: Documents
          </button>
        </div>
      </div>
    );
  }, [errors, isGeocoding, geocodingError, isGoogleMapsLoaded, showMapPicker, currentStep]);

  // Page 4: Documents
  const Page4Documents = React.useMemo(() => () => {
    const [uploadedDocs, setUploadedDocs] = useState<{ [key: string]: File[] }>({});
    const [uploadedDocuments, setUploadedDocuments] = useState<{ [key: string]: any[] }>({});
    const [isUploading, setIsUploading] = useState(false);
    const [passwords, setPasswords] = useState<{ [key: string]: string }>({});
    const [encryptFlags, setEncryptFlags] = useState<{ [key: string]: boolean }>({});

    const handleNext = () => {
      console.log('Page4Documents handleNext called, moving to step 5');
      nextStep();
    };

    const handleDocumentUpload = async (docType: string, files: File[]) => {
      if (!files.length) return;

      setUploadedDocs(prev => ({ 
        ...prev, 
        [docType]: [...(prev[docType] || []), ...files]
      }));
      
      toast.success(`${files.length} ${docType.replace('_', ' ')} document(s) selected`);
    };

    const removeDocument = (docType: string, index: number) => {
      setUploadedDocs(prev => ({
        ...prev,
        [docType]: prev[docType]?.filter((_, i) => i !== index) || []
      }));
    };

    const viewDocument = async (documentId: string) => {
      try {
        const result = await generateDocumentDownloadURL(documentId);
        if (result.success && result.download_url) {
          window.open(result.download_url, '_blank');
        } else {
          toast.error('Failed to generate view URL');
        }
      } catch (error) {
        toast.error('Failed to view document');
      }
    };

    const uploadAllDocuments = async () => {
      const allFiles = Object.values(uploadedDocs).flat();
      if (!allFiles.length) {
        toast.error('Please select documents to upload');
        return;
      }

      // Store documents in form data for later upload after property creation
      const documentTypes = Object.entries(uploadedDocs).flatMap(([type, files]) =>
        files.map(() => type)
      );

      setValue('documents', allFiles);
      setValue('requiredDocs', allFiles.map((file, index) => ({
        file,
        type: documentTypes[index]
      })));

      toast.success(`${allFiles.length} documents prepared for upload`);
      setUploadedDocs({});
    };

    return (
      <div className="space-y-6">
        {/* Property Documents */}
        <div className="bg-blue-50 rounded-lg p-5">
          <div className="flex items-center mb-6">
            <div className="w-8 h-8 bg-[#0056D2] rounded-lg flex items-center justify-center mr-3">
              <FileText className="w-5 h-5 text-white" />
            </div>
            <div>
              <h3 className="text-xl font-bold text-gray-900">Property Documents</h3>
              <p className="text-sm text-gray-600">Upload required property documents</p>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Property Ownership Documents */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-green-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">Property Ownership Documents</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">Sale deed, title deed, or ownership certificate</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  accept=".pdf"
                  multiple
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('ownership', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['ownership'] && uploadedDocs['ownership'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('ownership', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>

            {/* Building Approvals */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-blue-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">Building Approvals</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">Construction permit, building plan approval</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  accept=".pdf"
                  multiple
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('building_approval', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['building_approval'] && uploadedDocs['building_approval'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('building_approval', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>

            {/* NOC Certificate */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-yellow-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">NOC Certificate</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">No Objection Certificate from authorities</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  accept=".pdf"
                  multiple
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('noc', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['noc'] && uploadedDocs['noc'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('noc', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>

            {/* Property Tax Receipt */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-purple-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">Property Tax Receipt</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">Latest property tax payment receipt</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  accept=".pdf"
                  multiple
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('property_tax', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['property_tax'] && uploadedDocs['property_tax'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('property_tax', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>

            {/* Encumbrance Certificate */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-red-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">Encumbrance Certificate</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">Certificate showing property transaction history</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  accept=".pdf"
                  multiple
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('encumbrance', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['encumbrance'] && uploadedDocs['encumbrance'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('encumbrance', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>

            {/* Additional Documents */}
            <div className="bg-white rounded-lg p-4 border border-gray-200">
              <div className="flex items-center mb-3">
                <div className="w-6 h-6 bg-gray-100 rounded-full flex items-center justify-center mr-2">
                  <svg className="w-4 h-4 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                  </svg>
                </div>
                <h4 className="font-semibold text-gray-900">Additional Documents</h4>
              </div>
              <p className="text-sm text-gray-600 mb-3">Any other relevant property documents</p>

              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  multiple
                  accept=".pdf"
                  onChange={(e) => {
                    const files = Array.from(e.target.files || []);
                    handleDocumentUpload('other', files);
                    e.target.value = ''; // Reset input
                  }}
                  className="flex-1 text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-[#0056D2] file:text-white hover:file:bg-[#0056D2]/90 file:cursor-pointer cursor-pointer"
                />
                {uploadedDocs['other'] && uploadedDocs['other'].length > 0 && (
                  <label className="p-2 text-gray-500 hover:text-blue-600 cursor-pointer" title="Replace Document">
                    <Edit className="w-4 h-4" />
                    <input
                      type="file"
                      accept=".pdf"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        if (files.length > 0) {
                          handleDocumentUpload('other', files);
                          e.target.value = ''; // Reset input
                        }
                      }}
                    />
                  </label>
                )}
              </div>
            </div>
          </div>

          {/* Upload All Button - Always Visible */}
          <div className="mt-6 bg-green-50 rounded-lg p-4">
            <div className="flex items-center justify-between mb-4">
              <h4 className="font-semibold text-gray-900">Selected Documents</h4>
              <button
                onClick={uploadAllDocuments}
                disabled={isUploading}
                className="px-4 py-2 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 disabled:opacity-50 flex items-center"
              >
                {isUploading ? (
                  <>
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    Uploading...
                  </>
                ) : (
                  <>
                    <Upload className="w-4 h-4 mr-2" />
                    Upload All
                  </>
                )}
              </button>
            </div>

            {Object.keys(uploadedDocs).length > 0 ? (
              <div className="space-y-3">
                {Object.entries(uploadedDocs).map(([docType, files]) => (
                  <div key={docType} className="bg-white rounded-lg p-3 border border-gray-200">
                    <h5 className="font-medium text-gray-900 mb-2 capitalize">
                      {docType.replace('_', ' ')} ({files.length} files)
                    </h5>
                    <div className="space-y-2">
                      {files.map((fileData, index) => (
                        <div key={index} className="flex items-center justify-between text-sm">
                          <span className="text-gray-600 truncate">
                            {fileData.name}
                          </span>
                          <div className="flex items-center space-x-2">
                            <button
                              onClick={() => {
                                const url = URL.createObjectURL(fileData);
                                window.open(url, '_blank');
                              }}
                              className="text-blue-500 hover:text-blue-700"
                              title="Preview Document"
                            >
                              <Eye className="w-4 h-4" />
                            </button>
                            <button
                              onClick={() => removeDocument(docType, index)}
                              className="text-red-500 hover:text-red-700"
                            >
                              <X className="w-4 h-4" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-sm text-gray-600">No documents selected yet. Select documents above to see them here.</p>
            )}
          </div>

          {/* Uploaded Documents */}
          {Object.keys(uploadedDocuments).length > 0 && (
            <div className="mt-6 bg-blue-50 rounded-lg p-4">
              <h4 className="font-semibold text-gray-900 mb-4">Uploaded Documents</h4>
              <div className="space-y-3">
                {Object.entries(uploadedDocuments).map(([docType, docs]) => (
                  <div key={docType} className="bg-white rounded-lg p-3 border border-gray-200">
                    <h5 className="font-medium text-gray-900 mb-2 capitalize">
                      {docType.replace('_', ' ')} ({docs.length} files)
                    </h5>
                    <div className="space-y-2">
                      {docs.map((doc, index) => (
                        <div key={index} className="flex items-center justify-between text-sm">
                          <span className="text-gray-600 truncate">{doc.file_name}</span>
                          <button
                            onClick={() => viewDocument(doc.id)}
                            className="text-blue-500 hover:text-blue-700 ml-2"
                            title="View Document"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}

          <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
            <div className="flex items-start">
              <svg className="w-5 h-5 text-yellow-600 mt-0.5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
              </svg>
              <div>
                <p className="text-sm font-medium text-yellow-800">Document Guidelines</p>
                <p className="text-sm text-yellow-700 mt-1">• Accepted formats: PDF only</p>
                <p className="text-sm text-yellow-700">• Maximum file size: 10MB per document</p>
                <p className="text-sm text-yellow-700">• Ensure documents are clear and readable</p>
              </div>
            </div>
          </div>
        </div>



        <div className="flex justify-between pt-6">
          <button
            type="button"
            onClick={prevStep}
            className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-semibold"
          >
            Previous
          </button>
          <button
            type="button"
            onClick={handleNext}
            className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg"
          >
            Review & Submit
          </button>
        </div>
      </div>
    );
  }, [currentStep]);

  // Page 5: Review & Submit
  const Page5Review = React.useMemo(() => () => {
    const formData = watch();

    const handleSubmit = async () => {
      // Use comprehensive validation
      const validation = validateListingForm(formData);
      
      if (!validation.isValid) {
        validation.errors.forEach(error => toast.error(error));
        
        // Navigate to appropriate step based on missing fields
        const section1Fields = ['ownerName', 'mobile', 'email', 'title', 'description', 'listingType', 'propertyType'];
        const section2Fields = ['builtUpArea', 'bedrooms', 'bathrooms', 'facing', 'furnishing', 'propertyAge', 'gatedCommunity', 'salePrice', 'monthlyRent', 'startingBidPrice', 'actualPrice', 'accommodationType', 'totalBedsAvailable', 'perBedRent', 'genderPreference', 'availabilityDate'];
        const section3Fields = ['address', 'city', 'locality', 'state', 'pincode'];
        
        const hasSection1Errors = validation.missingFields.some(field => section1Fields.includes(field));
        const hasSection2Errors = validation.missingFields.some(field => section2Fields.includes(field));
        const hasSection3Errors = validation.missingFields.some(field => section3Fields.includes(field));
        
        if (hasSection1Errors) {
          setCurrentStep(0);
          toast.error('Please complete Basic Information section.');
        } else if (hasSection2Errors) {
          setCurrentStep(1);
          toast.error('Please complete Property Details section.');
        } else if (hasSection3Errors) {
          setCurrentStep(2);
          toast.error('Please complete Location section.');
        }
        
        window.scrollTo({ top: 0, behavior: 'smooth' });
        return;
      }

      // Validate all mandatory fields across all sections
      const mandatoryFields: (keyof ListingFormData)[] = [
        // Section 1: Basic Info
        'ownerName', 'mobile', 'email', 'title', 'description', 'listingType', 'propertyType',
        // Section 2: Property Details
        'builtUpArea', 'bedrooms', 'bathrooms', 'facing', 'furnishing', 'propertyAge', 'gatedCommunity',
        // Section 3: Location
        'address', 'city', 'locality', 'state', 'pincode',
        // Section 5: Terms
        'acceptTerms', 'acceptGovtGuidelines'
      ];

      // Add conditional mandatory fields based on listing type
      if (formData.listingType === 'sell') {
        mandatoryFields.push('salePrice');
      } else if (formData.listingType === 'rent') {
        mandatoryFields.push('monthlyRent');
      } else if (formData.listingType === 'lease') {
        mandatoryFields.push('leaseType', 'leaseDuration', 'monthlyLeaseAmount', 'leaseStartDate', 'possessionAvailability');
      } else if (formData.listingType === 'bidding') {
        mandatoryFields.push('startingBidPrice');
      } else if (formData.listingType === 'urgent-sale') {
        mandatoryFields.push('actualPrice');
      }

      // Add PG-specific mandatory fields
      if (formData.propertyType === 'pg-coliving') {
        mandatoryFields.push('accommodationType', 'totalBedsAvailable', 'perBedRent', 'genderPreference', 'availabilityDate');
      }

      const isValid = await trigger(mandatoryFields as any);

      if (!isValid) {
        // Find which section has errors and navigate to it
        const section1Fields = ['ownerName', 'mobile', 'email', 'title', 'description', 'listingType', 'propertyType'];
        const section2Fields = ['builtUpArea', 'bedrooms', 'bathrooms', 'facing', 'furnishing', 'propertyAge', 'gatedCommunity', 'salePrice', 'monthlyRent', 'startingBidPrice', 'actualPrice', 'accommodationType', 'totalBedsAvailable', 'perBedRent', 'genderPreference', 'availabilityDate'];
        const section3Fields = ['address', 'city', 'locality', 'state', 'pincode'];
        const section5Fields = ['acceptTerms', 'acceptGovtGuidelines'];

        const hasSection1Errors = section1Fields.some(field => errors[field as keyof typeof errors]);
        const hasSection2Errors = section2Fields.some(field => errors[field as keyof typeof errors]);
        const hasSection3Errors = section3Fields.some(field => errors[field as keyof typeof errors]);
        const hasSection5Errors = section5Fields.some(field => errors[field as keyof typeof errors]);

        if (hasSection1Errors) {
          setCurrentStep(0);
          toast.error('Please complete all required fields in Basic Information section.');
        } else if (hasSection2Errors) {
          setCurrentStep(1);
          toast.error('Please complete all required fields in Property Details section.');
        } else if (hasSection3Errors) {
          setCurrentStep(2);
          toast.error('Please complete all required fields in Location section.');
        } else if (hasSection5Errors) {
          toast.error('Please accept the terms and conditions to proceed.');
        } else {
          toast.error('Please complete all required fields before submitting.');
        }

        window.scrollTo({ top: 0, behavior: 'smooth' });
        return;
      }

      setIsSubmitting(true);
      try {
        // Filter out empty image URLs
        const filteredData = {
          ...formData,
          imageUrls: formData.imageUrls?.filter(url => url && url.trim() !== '') || []
        };

        if (isEditMode && editPropertyId) {
          console.log('🔄 Updating existing property:', editPropertyId);
          const result = await api.updateListing(editPropertyId, filteredData);

          if (result.success) {
            toast.success('Property updated successfully! Your changes are now pending admin approval.');
            setTimeout(() => {
              window.location.href = `/property/${editPropertyId}`;
            }, 1000);
          } else {
            toast.error(`Error: ${result.error}`);
          }
        } else {
          console.log('🆕 Creating new property');
          const result = await api.submitListing(filteredData);

          if (result.success) {
            console.log('✅ Property created with ID:', result.id);

            // Upload pending media files to Azure Blob Storage
            try {
              const { uploadPendingMediaFiles } = await import('../utils/pendingMediaUpload');
              const mediaUploadResult = await uploadPendingMediaFiles(result.id || '');

              if (mediaUploadResult.success) {
                console.log('✅ Media files uploaded successfully');
                toast.success(`Property submitted successfully! Your property is now pending admin approval. Property ID: ${result.id}`);
              } else {
                console.log('⚠️ Property created but media upload failed');
                toast.success(`Property submitted successfully! Your property is now pending admin approval. Property ID: ${result.id}`);
                toast.warning('Some media files could not be uploaded. You can add them later.');
              }
            } catch (error) {
              console.error('Error uploading media:', error);
              toast.success(`Property submitted successfully! Your property is now pending admin approval. Property ID: ${result.id}`);
              toast.warning('Media upload failed. You can add images later from property details.');
            }

            // Set submitting flag to prevent auto-save
            isSubmittingRef.current = true;

            // Clear all timeouts
            if (autoSaveTimeoutRef.current) {
              clearTimeout(autoSaveTimeoutRef.current);
            }

            // Clear ALL draft data from localStorage
            localStorage.removeItem('listing_form_draft');
            localStorage.removeItem('current_property_id');

            // Clear pending media files
            window.pendingMediaFiles = { images: [], videos: [] };

            // Reset form to blank state
            reset({
              ownerName: '',
              companyName: '',
              mobile: '',
              email: '',
              title: '',
              description: '',
              listingType: undefined,
              propertyType: undefined,
              builtUpArea: undefined,
              bedrooms: undefined,
              bathrooms: undefined,
              facing: '',
              furnishing: '',
              propertyAge: '',
              salePrice: undefined,
              monthlyRent: undefined,
              address: '',
              city: '',
              locality: '',
              pincode: '',
              state: '',
              latitude: undefined,
              longitude: undefined,
              amenities: [],
              features: [],
              specifications: [],
              highlights: [],
              gatedCommunity: undefined,
              contactType: 'owner',
              acceptTerms: false,
              acceptGovtGuidelines: false
            });

            // Reset step and tags
            setCurrentStep(0);
            setSelectedTags([]);

            // Show success message
            toast.success(`Property submitted successfully! Property ID: ${result.id}`);

            // Navigate after a short delay
            setTimeout(() => {
              navigate('/');
            }, 1500);

            api.clearDraft();
            navigate('/');
          } else {
            toast.error(`Error: ${result.error}`);
          }
        }
      } catch (error) {
        console.error('Error submitting form:', error);
        toast.error('Error submitting form. Please try again.');
      } finally {
        setIsSubmitting(false);
      }
    };

    const getStepForField = (fieldName: keyof ListingFormData): number => {
      const step0Fields = ['ownerName', 'companyName', 'mobile', 'email', 'title', 'description', 'listingType', 'propertyType', 'contactType'];
      const step1Fields = ['builtUpArea', 'carpetArea', 'plotArea', 'superArea', 'bedrooms', 'bathrooms', 'balconies', 'floorNumber', 'totalFloors', 'facing', 'furnishing', 'propertyAge', 'possessionStatus', 'gatedCommunity', 'price', 'salePrice', 'pricePerSqFt', 'monthlyRent', 'securityDeposit', 'priceNegotiable', 'loanAvailability', 'startingBidPrice', 'minimumIncrement', 'reservePrice', 'auctionStartDate', 'auctionStartTime', 'auctionEndDate', 'auctionEndTime', 'actualPrice', 'urgentSaleStartDate', 'urgentSaleStartTime', 'urgentSaleEndDate', 'urgentSaleEndTime', 'leaseType', 'leaseDuration', 'monthlyLeaseAmount', 'advanceDeposit', 'maintenanceCharges', 'lockInPeriod', 'leaseStartDate', 'leaseEndDate', 'possessionAvailability', 'paymentFrequency', 'maintenanceResponsibility', 'renewalTerms', 'autoRenew', 'accommodationType', 'totalBedsAvailable', 'perBedRent', 'foodAvailability', 'genderPreference', 'pgSecurityDeposit', 'pgMaintenanceCharges', 'securityType', 'availabilityDate', 'minimumStayDuration', 'bookingAmount', 'propertyManagerContact', 'roomFurnishingDetails', 'houseRules', 'guestPolicy', 'attachedBathroom', 'pgCommonFacilities', 'specifications', 'features', 'amenities', 'customAmenities', 'customAmenitiesText', 'highlights'];
      const step2Fields = ['address', 'city', 'locality', 'pincode', 'state', 'landmark', 'reraNumber', 'latitude', 'longitude', 'selectedMedia', 'primaryImage', 'imageUrls', 'virtualTourUrl', 'threeSixtyView', 'housePlan', 'images', 'videos'];

      if (step0Fields.includes(fieldName)) return 0;
      if (step1Fields.includes(fieldName)) return 1;
      if (step2Fields.includes(fieldName)) return 2;
      return 0;
    };

    const renderEditableField = (fieldName: keyof ListingFormData, label: string, type: 'text' | 'number' | 'email' | 'tel' | 'textarea' | 'select' | 'checkbox' = 'text', options?: string[]) => {
      const value = formData[fieldName];
      if (!value && type !== 'checkbox') return null;

      return (
        <div className="border border-gray-200 rounded-lg p-3 bg-white">
          <div className="flex items-center justify-between mb-2">
            <label className="text-sm font-medium text-gray-700">{label}</label>
            <button
              type="button"
              onClick={() => {
                const targetStep = getStepForField(fieldName);
                setCurrentStep(targetStep);
                window.scrollTo({ top: 0, behavior: 'smooth' });
              }}
              className="text-[#0056D2] hover:text-[#0056D2]/80 text-sm flex items-center"
            >
              <Edit className="w-4 h-4 mr-1" />
              Edit
            </button>
          </div>

          <div className="text-sm text-gray-600">
            {type === 'checkbox' ? (value ? 'Yes' : 'No') :
              Array.isArray(value) ? value.join(', ') :
                typeof value === 'number' && fieldName.toLowerCase().includes('price') ? `₹${value.toLocaleString()}` :
                  typeof value === 'number' && (fieldName.toLowerCase().includes('area') || fieldName.toLowerCase().includes('sqft')) ? `${value} sq ft` :
                    typeof value === 'string' || typeof value === 'number' ? value :
                      value instanceof File ? value.name :
                        typeof value === 'object' && value !== null ? JSON.stringify(value) :
                          String(value)}
          </div>
        </div>
      );
    };

    const renderEditableCheckboxGroup = (fieldName: keyof ListingFormData, label: string, allOptions: string[]) => {
      const value = formData[fieldName] as string[];
      if (!value || value.length === 0) return null;

      return (
        <div className="border border-gray-200 rounded-lg p-3 bg-white">
          <div className="flex items-center justify-between mb-2">
            <label className="text-sm font-medium text-gray-700">{label}</label>
            <button
              type="button"
              onClick={() => {
                const targetStep = getStepForField(fieldName);
                setCurrentStep(targetStep);
                window.scrollTo({ top: 0, behavior: 'smooth' });
              }}
              className="text-[#0056D2] hover:text-[#0056D2]/80 text-sm flex items-center"
            >
              <Edit className="w-4 h-4 mr-1" />
              Edit
            </button>
          </div>

          <div className="text-sm text-gray-600">
            {value.join(', ')}
          </div>
        </div>
      );
    };

    return (
      <div className="space-y-6">
        {/* Personal Details Review */}
        {(formData.ownerName || formData.mobile || formData.email || formData.companyName) && (
          <div className="bg-blue-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Personal Details</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {renderEditableField('ownerName', 'Full Name', 'text')}
              {renderEditableField('companyName', 'Company Name', 'text')}
              {renderEditableField('mobile', 'Mobile Number', 'tel')}
              {renderEditableField('email', 'Email Address', 'email')}
            </div>
          </div>
        )}

        {/* Property Information Review */}
        {(formData.title || formData.propertyType || formData.listingType || formData.description) && (
          <div className="bg-green-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Property Information</h3>
            </div>
            <div className="space-y-4">
              {renderEditableField('title', 'Property Title', 'text')}
              {renderEditableField('description', 'Description', 'textarea')}
              {renderEditableField('propertyType', 'Property Type', 'select', ['apartment', 'villa', 'house', 'commercial', 'pg-coliving'])}
              {renderEditableField('listingType', 'Listing Type', 'select', ['sell', 'rent', 'urgent-sale', 'bidding'])}
            </div>
          </div>
        )}

        {/* Property Details Review */}
        {(formData.builtUpArea || formData.bedrooms || formData.bathrooms || formData.facing || formData.furnishing || formData.propertyAge) && (
          <div className="bg-purple-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Property Details</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {renderEditableField('builtUpArea', 'Built-up Area (sq ft)', 'number')}
              {renderEditableField('carpetArea', 'Carpet Area (sq ft)', 'number')}
              {renderEditableField('superArea', 'Super Area (sq ft)', 'number')}
              {renderEditableField('plotArea', 'Plot Area (sq ft)', 'number')}
              {renderEditableField('bedrooms', 'Bedrooms', 'select', ['1', '2', '3', '4', '5'])}
              {renderEditableField('bathrooms', 'Bathrooms', 'select', ['1', '2', '3', '4'])}
              {renderEditableField('balconies', 'Balconies', 'select', ['0', '1', '2', '3'])}
              {renderEditableField('floorNumber', 'Floor Number', 'text')}
              {renderEditableField('totalFloors', 'Total Floors', 'text')}
              {renderEditableField('facing', 'Facing', 'select', ['north', 'south', 'east', 'west', 'north-east', 'north-west', 'south-east', 'south-west'])}
              {renderEditableField('furnishing', 'Furnishing', 'select', ['unfurnished', 'semi-furnished', 'fully-furnished'])}
              {renderEditableField('propertyAge', 'Property Age', 'select', ['under-construction', '0-1', '1-5', '5-10', '10+'])}
              {renderEditableField('possessionStatus', 'Possession Status', 'select', ['ready-to-move', 'under-construction', 'new-launch'])}
              {renderEditableField('gatedCommunity', 'Gated Community', 'select', ['yes', 'no'])}
            </div>
          </div>
        )}

        {/* Pricing Review */}
        {(formData.price || formData.salePrice || formData.monthlyRent || formData.securityDeposit) && (
          <div className="bg-yellow-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Pricing Details</h3>
              <button
                type="button"
                onClick={() => {
                  const currentData = getValues();
                  if (!currentData.city || !currentData.locality || !currentData.propertyType) {
                    toast.error('Please fill in city, locality, and property type to use Market Analytics');
                    return;
                  }
                  
                  if (propertyType !== 'asset' && !currentData.listingType) {
                    toast.error('Please select a listing type to use Market Analytics');
                    return;
                  }
                  
                  setShowCompetitorAnalyticsModal(true);
                }}
                className="px-4 py-2 bg-gradient-to-r from-green-500 to-blue-500 text-white rounded-lg hover:from-green-600 hover:to-blue-600 transition-all duration-200 flex items-center space-x-2 shadow-md hover:shadow-lg text-sm"
              >
                <BarChart3 className="w-4 h-4" />
                <span className="font-medium">Check Market Price</span>
              </button>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {renderEditableField('price', 'Base Price (₹)', 'number')}
              {renderEditableField('salePrice', 'Sale Price (₹)', 'number')}
              {renderEditableField('pricePerSqFt', 'Price per sq ft (₹)', 'number')}
              {renderEditableField('monthlyRent', 'Monthly Rent (₹)', 'number')}
              {renderEditableField('securityDeposit', 'Security Deposit (₹)', 'number')}
              {renderEditableField('priceNegotiable', 'Price Negotiable', 'checkbox')}
              {renderEditableField('loanAvailability', 'Loan Available', 'checkbox')}
            </div>
          </div>
        )}

        {/* Urgent Sale Details */}
        {formData.listingType === 'urgent-sale' && (formData.actualPrice || formData.urgentSaleStartDate) && (
          <div className="bg-red-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Urgent Sale Details</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {renderEditableField('actualPrice', 'Actual Price (₹)', 'number')}
              {renderEditableField('urgentSaleStartDate', 'Urgent Sale Start Date', 'text')}
              {renderEditableField('urgentSaleStartTime', 'Urgent Sale Start Time', 'text')}
              {renderEditableField('urgentSaleEndDate', 'Urgent Sale End Date', 'text')}
              {renderEditableField('urgentSaleEndTime', 'Urgent Sale End Time', 'text')}
            </div>
          </div>
        )}

        {/* Bidding Details */}
        {formData.listingType === 'bidding' && (formData.startingBidPrice || formData.reservePrice) && (
          <div className="bg-indigo-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Bidding Details</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {renderEditableField('startingBidPrice', 'Starting Bid Price (₹)', 'number')}
              {renderEditableField('minimumIncrement', 'Minimum Increment (₹)', 'number')}
              {renderEditableField('reservePrice', 'Reserve Price (₹)', 'number')}
              {renderEditableField('auctionStartDate', 'Auction Start Date', 'text')}
              {renderEditableField('auctionStartTime', 'Auction Start Time', 'text')}
              {renderEditableField('auctionEndDate', 'Auction End Date', 'text')}
              {renderEditableField('auctionEndTime', 'Auction End Time', 'text')}
            </div>
          </div>
        )}

        {/* Location Review */}
        {(formData.address || formData.city || formData.locality || formData.pincode) && (
          <div className="bg-orange-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Location Details</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {renderEditableField('address', 'Address', 'textarea')}
              {renderEditableField('city', 'City', 'text')}
              {renderEditableField('state', 'State', 'text')}
              {renderEditableField('locality', 'Locality', 'text')}
              {renderEditableField('pincode', 'Pincode', 'text')}
              {renderEditableField('landmark', 'Landmark', 'text')}
              {renderEditableField('latitude', 'Latitude', 'number')}
              {renderEditableField('longitude', 'Longitude', 'number')}
            </div>
          </div>
        )}

        {/* Features & Amenities Review */}
        {((formData.specifications && formData.specifications.length > 0) || (formData.features && formData.features.length > 0) || (formData.amenities && formData.amenities.length > 0) || (formData.highlights && formData.highlights.length > 0)) && (
          <div className="bg-teal-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Features & Amenities</h3>
            </div>
            <div className="space-y-4">
              {renderEditableCheckboxGroup('specifications', 'Property Specifications', ['Air Conditioning', 'Parking Space', 'Balcony', 'Garden', 'Swimming Pool', 'Gym', 'Security', 'Elevator', 'Power Backup'])}
              {renderEditableCheckboxGroup('features', 'Additional Features', ['Modular Kitchen', 'Servant Room', 'Study Room', 'Pooja Room', 'Terrace Garden', 'Intercom', 'Piped Gas', 'Maintenance Staff', 'Visitor Parking'])}
              {renderEditableCheckboxGroup('amenities', 'Amenities', ['Swimming Pool', 'Gym', 'Parking', 'Security', 'Garden', 'Elevator', 'Power Backup', 'Water Supply', 'Club House', 'Children Play Area', 'Jogging Track', 'CCTV'])}
              {renderEditableCheckboxGroup('highlights', 'Property Highlights', ['Prime Location', 'Vastu Compliant', 'Corner Property', 'Park Facing', 'Gated Community', 'Near Metro', 'Investment Property', 'Ready to Move'])}
            </div>
          </div>
        )}

        {/* Media Review */}
        {(formData.primaryImage || (formData.imageUrls && formData.imageUrls.some(url => url)) || formData.virtualTourUrl || formData.housePlan) && (
          <div className="bg-indigo-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Media & Virtual Tours</h3>
            </div>
            <div className="space-y-4">
              {renderEditableField('primaryImage', 'Primary Image URL', 'text')}
              {formData.imageUrls && formData.imageUrls.map((url, index) =>
                url && renderEditableField(`imageUrls.${index}` as any, `Image ${index + 1} URL`, 'text')
              )}
              {renderEditableField('virtualTourUrl', 'Virtual Tour URL', 'text')}
              {renderEditableField('housePlan', 'Floor Plan URL', 'text')}
            </div>
          </div>
        )}

        {/* Contact Information Review */}
        {formData.contactType && (
          <div className="bg-pink-50 rounded-lg p-5">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-xl font-bold text-gray-900">Contact Information</h3>
            </div>
            <div>
              {renderEditableField('contactType', 'Contact Type', 'select', ['owner', 'agent'])}
            </div>
          </div>
        )}

        {/* Agreement & Notifications */}
        <div className="bg-red-50 rounded-lg p-5">
          <h3 className="text-xl font-bold text-gray-900 mb-4">Agreement & Notifications</h3>
          <div className="space-y-6">
            <div className="flex items-start space-x-3">
              <input
                type="checkbox"
                {...register('acceptGovtGuidelines', validation.required('You must accept government guidelines and policies'))}
                className={`w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2] mt-1 flex-shrink-0 transition-opacity ${legalGuidelinesAccepted ? 'opacity-100 cursor-pointer' : 'opacity-50 cursor-not-allowed'
                  }`}
                disabled={!legalGuidelinesAccepted}
                onClick={(e) => {
                  if (!legalGuidelinesAccepted) {
                    e.preventDefault();
                    navigate('/legal-guidelines');
                  }
                }}
              />
              <div className="flex-1">
                <label className="text-sm font-medium text-gray-900 cursor-pointer">
                  I have read and accept all Government Guidelines and Policies <span className="required-asterisk">*</span>
                </label>
                <p className="text-xs text-gray-600 mt-1">
                  I confirm that I have read and understood all applicable government guidelines including RERA regulations, property registration requirements, and local authority policies. I agree to comply with all legal requirements for property listing.
                </p>
                <button
                  type="button"
                  onClick={() => navigate('/legal-guidelines')}
                  className="text-xs text-[#0056D2] hover:underline mt-2 inline-block font-medium"
                >
                  View Government Guidelines & Policies →
                </button>
                {!legalGuidelinesAccepted && (
                  <div className="mt-3 p-3 bg-amber-50 border border-amber-200 rounded-lg">
                    <p className="text-xs text-amber-800 font-medium flex items-center">
                      <svg className="w-4 h-4 mr-2 text-amber-600 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                      </svg>
                      Warning: You must first accept the legal guidelines on the Legal Guidelines page to enable this checkbox.
                    </p>
                  </div>
                )}
                {errors.acceptGovtGuidelines && <p className="text-sm text-red-500 mt-2">{errors.acceptGovtGuidelines.message}</p>}
              </div>
            </div>

            <div className="flex items-start space-x-3">
              <input
                type="checkbox"
                {...register('acceptTerms', validation.required('You must accept the terms and conditions'))}
                className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2] mt-1 flex-shrink-0"
              />
              <div className="flex-1">
                <label className="text-sm font-medium text-gray-900 cursor-pointer">
                  I accept the Terms and Conditions <span className="required-asterisk">*</span>
                </label>
                <p className="text-xs text-gray-600 mt-1">
                  By checking this box, I agree to NAL India's terms of service, privacy policy, and property listing guidelines.
                </p>
                {errors.acceptTerms && <p className="text-sm text-red-500 mt-2">{errors.acceptTerms.message}</p>}
              </div>
            </div>

            <div className="flex items-start space-x-3">
              <input
                type="checkbox"
                {...register('enableAlerts')}
                className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2] mt-1 flex-shrink-0"
              />
              <div className="flex-1">
                <label className="text-sm font-medium text-gray-900 cursor-pointer">
                  Enable Property Alerts
                </label>
                <p className="text-xs text-gray-600 mt-1">
                  Receive notifications about inquiries, views, and updates related to your property listing.
                </p>
              </div>
            </div>
          </div>
        </div>

        <div className="flex justify-between pt-6">
          <button
            type="button"
            onClick={prevStep}
            className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-semibold"
          >
            Previous
          </button>
          <div className="flex items-center space-x-3">
            <button
              type="button"
              onClick={() => {
                // Open competitor analytics modal with current form data
                const currentData = getValues();
                if (!currentData.city || !currentData.locality || !currentData.propertyType) {
                  toast.error('Please fill in city, locality, and property type to use Competitor Analytics');
                  return;
                }
                
                // For assets, we don't need listing type
                if (propertyType !== 'asset' && !currentData.listingType) {
                  toast.error('Please select a listing type to use Competitor Analytics');
                  return;
                }
                
                setShowCompetitorAnalyticsModal(true);
              }}
              className="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors flex items-center space-x-2 shadow-md hover:shadow-lg"
            >
              <BarChart3 className="w-5 h-5" />
              <span className="font-semibold">Analyze Market</span>
            </button>
            <button
              type="button"
              onClick={handleSubmit}
              disabled={isSubmitting || isLoadingProperty}
              className="px-8 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-semibold shadow-md hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isSubmitting ? (isEditMode ? 'Updating...' : 'Submitting...') :
                isLoadingProperty ? 'Loading...' :
                  isEditMode ? 'Update Property' : 'Submit Listing'}
            </button>
          </div>
        </div>
      </div>
    );
  }, [errors, legalGuidelinesAccepted, currentStep]);

  const renderCurrentStep = () => {
    console.log('renderCurrentStep: currentStep =', currentStep);
    switch (currentStep) {
      case 0: 
        console.log('Rendering Page1BasicInfo');
        return <Page1BasicInfo />;
      case 1: 
        console.log('Rendering Page2Details');
        return <Page2Details />;
      case 2: 
        console.log('Rendering Page3LocationMedia');
        return <Page3LocationMedia />;
      case 3: 
        console.log('Rendering Page4Documents');
        return <Page4Documents />;
      case 4: 
        console.log('Rendering Page5Review');
        return <Page5Review />;
      default: 
        console.log('Default: Rendering Page1BasicInfo');
        return <Page1BasicInfo />;
    }
  };

  return (
    <FormProvider {...methods}>
      <div className="min-h-screen bg-gray-50 py-6">
        <div className="max-w-4xl mx-auto px-4">
          {/* Professional Header */}
          <div className="text-center mb-8">
            <div className="inline-flex items-center justify-center w-12 h-12 bg-[#0056D2] rounded-lg mb-4">
              <svg className="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z" />
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 21l4-4 4 4" />
              </svg>
            </div>
            <h1 className="text-3xl font-bold text-gray-900 mb-2">
              {isEditMode ? 'Edit Your Property' : 'List Your Property'}
            </h1>
            <p className="text-gray-600 max-w-xl mx-auto">
              {isEditMode
                ? 'Update your property details and reach more potential buyers'
                : 'Create a professional listing to reach thousands of potential buyers'
              }
            </p>
          </div>

          {/* Progress Steps */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 mb-6">
            <div className="flex items-center justify-between">
              {steps.map((step, index) => (
                <div key={index} className="flex items-center flex-1">
                  <button
                    type="button"
                    onClick={() => {
                      setCurrentStep(index);
                      window.scrollTo({ top: 0, behavior: 'smooth' });
                    }}
                    className="flex flex-col items-center cursor-pointer hover:opacity-80 transition-opacity"
                  >
                    <div className={`flex items-center justify-center w-10 h-10 rounded-full transition-all duration-300 ${index <= currentStep
                      ? 'bg-[#0056D2] text-white shadow-md'
                      : 'border-2 border-gray-300 text-gray-500 bg-white hover:border-[#0056D2]'
                      }`}>
                      {index < currentStep ? (
                        <CheckCircle className="w-6 h-6" />
                      ) : (
                        <span className="text-sm font-semibold">{index + 1}</span>
                      )}
                    </div>
                    <span className={`mt-2 text-xs font-medium text-center transition-colors duration-300 ${index <= currentStep ? 'text-[#0056D2]' : 'text-gray-500'
                      }`}>
                      {step.title}
                    </span>
                  </button>
                  {index < steps.length - 1 && (
                    <div className={`flex-1 h-0.5 mx-3 rounded-full transition-all duration-500 ${index < currentStep ? 'bg-[#0056D2]' : 'bg-gray-200'
                      }`} />
                  )}
                </div>
              ))}
            </div>
          </div>

          {/* Form Content */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
            <div className="bg-[#0056D2] h-1"></div>
            <div className="p-6">
              {renderCurrentStep()}
            </div>
          </div>
        </div>

        {/* Floating Competitor Analytics Button */}
        {currentStep >= 1 && currentStep <= 4 && (
          <div className="fixed bottom-6 right-6 z-40">
            <button
              type="button"
              onClick={() => {
                const currentData = getValues();
                if (!currentData.city || !currentData.locality || !currentData.propertyType) {
                  toast.error('Please fill in city, locality, and property type first');
                  return;
                }
                
                if (propertyType !== 'asset' && !currentData.listingType) {
                  toast.error('Please select a listing type first');
                  return;
                }
                
                setShowCompetitorAnalyticsModal(true);
              }}
              className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white p-4 rounded-full shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-110 group"
              title="Market Analytics"
            >
              <BarChart3 className="w-6 h-6" />
              <span className="absolute -top-2 -left-2 bg-red-500 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center animate-pulse">
                !
              </span>
              <div className="absolute right-full mr-3 top-1/2 transform -translate-y-1/2 bg-gray-900 text-white px-3 py-1 rounded-lg text-sm whitespace-nowrap opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                Check Market Price
                <div className="absolute left-full top-1/2 transform -translate-y-1/2 border-4 border-transparent border-l-gray-900"></div>
              </div>
            </button>
          </div>
        )}

        {/* Competitor Analytics Modal */}
        <CompetitorAnalyticsModal
          isOpen={showCompetitorAnalyticsModal}
          onClose={() => setShowCompetitorAnalyticsModal(false)}
          propertyData={(() => {
            const currentData = getValues();
            return {
              city: currentData.city || '',
              locality: currentData.locality || '',
              property_type: currentData.propertyType || 'apartment',
              listing_type: propertyType === 'asset' ? 'sell' : (currentData.listingType || 'sell'),
              property_condition: currentData.propertyAge === 'under-construction' ? 'new' : 
                                currentData.propertyAge === '0-1' ? 'new' : 'resale',
              bedrooms: currentData.bedrooms || 2,
              built_up_area: currentData.builtUpArea || 1000,
              sale_price: currentData.salePrice || currentData.monthlyRent || currentData.actualPrice || currentData.startingBidPrice || 5000000,
              listed_date: new Date().toISOString().split('T')[0]
            };
          })()}
        />

        {/* AI Description Generator Modal */}
        <AIDescriptionGenerator
          propertyData={{
            title: watch('title') || 'Property',
            propertyType: watch('propertyType') || '',
            listingType: watch('listingType') || '',
            bedrooms: watch('bedrooms') || '',
            bathrooms: watch('bathrooms') || '',
            builtUpArea: watch('builtUpArea') || '',
            carpetArea: watch('carpetArea') || '',
            plotArea: watch('plotArea') || '',
            superArea: watch('superArea') || '',
            city: watch('city') || '',
            locality: watch('locality') || '',
            address: watch('address') || '',
            landmark: watch('landmark') || '',
            pincode: watch('pincode') || '',
            state: watch('state') || '',
            facing: watch('facing') || '',
            furnishing: watch('furnishing') || '',
            propertyAge: watch('propertyAge') || '',
            floorNumber: watch('floorNumber') || '',
            totalFloors: watch('totalFloors') || '',
            balconies: watch('balconies') || '',
            possessionStatus: watch('possessionStatus') || '',
            gatedCommunity: watch('gatedCommunity') || '',
            amenities: watch('amenities') || [],
            features: watch('features') || [],
            highlights: watch('highlights') || [],
            specifications: watch('specifications') || [],
            salePrice: watch('salePrice') || '',
            monthlyRent: watch('monthlyRent') || '',
            securityDeposit: watch('securityDeposit') || '',
            pricePerSqFt: watch('pricePerSqFt') || '',
            contactPhone: watch('mobile') || '',
            contactEmail: watch('email') || ''
          }}
          onDescriptionSelect={(description) => {
            console.log('Selected description:', description);
            setValue('description', description);
            // Force trigger validation to update the form
            trigger('description');
          }}
          isOpen={showAIDescriptionModal}
          onClose={() => setShowAIDescriptionModal(false)}
        />



      </div>


    </FormProvider>
  );
};


// Map Picker Component (Ola/Rapido Style)
