import React, { useState } from 'react';

export const LoanEligibility = () => {
  const [formData, setFormData] = useState({
    monthlyIncome: '',
    monthlyExpenses: '',
    existingEMI: '',
    loanTenure: '20',
    interestRate: '8.5'
  });

  const [result, setResult] = useState<{
    eligibleAmount: number;
    maxEMI: number;
    debtToIncomeRatio: number;
  } | null>(null);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const calculateEligibility = () => {
    const income = parseFloat(formData.monthlyIncome);
    const expenses = parseFloat(formData.monthlyExpenses);
    const existingEMI = parseFloat(formData.existingEMI) || 0;
    const tenure = parseInt(formData.loanTenure);
    const rate = parseFloat(formData.interestRate);

    if (!income || income <= 0) {
      alert('Please enter a valid monthly income');
      return;
    }
    if (!expenses || expenses <= 0) {
      alert('Please enter a valid monthly expenses amount');
      return;
    }
    if (expenses >= income) {
      alert('Monthly expenses cannot be equal to or greater than income');
      return;
    }
    if (!rate || rate <= 0) {
      alert('Please enter a valid interest rate');
      return;
    }

    const disposableIncome = income - expenses - existingEMI;
    const maxEMI = disposableIncome * 0.6; // 60% of disposable income
    const monthlyRate = rate / (12 * 100);
    const numPayments = tenure * 12;
    
    const eligibleAmount = maxEMI * ((Math.pow(1 + monthlyRate, numPayments) - 1) / (monthlyRate * Math.pow(1 + monthlyRate, numPayments)));
    const debtToIncomeRatio = ((existingEMI + maxEMI) / income) * 100;

    setResult({
      eligibleAmount: Math.round(eligibleAmount),
      maxEMI: Math.round(maxEMI),
      debtToIncomeRatio: Math.round(debtToIncomeRatio * 100) / 100
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-4xl mx-auto px-4">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Loan Eligibility Calculator</h1>
          <p className="text-gray-600 max-w-3xl mx-auto">
            Assess your maximum loan eligibility based on repayment capacity and financial health. 
            Avoid loan rejections by understanding what banks may offer before approaching lenders.
          </p>
          <div className="mt-4 p-3 bg-teal-50 rounded-lg inline-block">
            <p className="text-sm text-teal-700 font-medium">🏦 Know your DTI ratio and borrowing capacity in advance</p>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-lg p-8">
          <div className="grid md:grid-cols-2 gap-8">
            <div>
              <h2 className="text-xl font-semibold mb-6">Enter Your Details</h2>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Monthly Income: ₹{parseInt(formData.monthlyIncome || '0').toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="monthlyIncome"
                    min="25000"
                    max="500000"
                    step="5000"
                    value={formData.monthlyIncome || '25000'}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹25K</span>
                    <span>₹5L</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Monthly Expenses: ₹{parseInt(formData.monthlyExpenses || '0').toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="monthlyExpenses"
                    min="10000"
                    max="200000"
                    step="2000"
                    value={formData.monthlyExpenses || '10000'}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹10K</span>
                    <span>₹2L</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Existing EMI: ₹{parseInt(formData.existingEMI || '0').toLocaleString('en-IN')}
                  </label>
                  <input
                    type="range"
                    name="existingEMI"
                    min="0"
                    max="100000"
                    step="1000"
                    value={formData.existingEMI || '0'}
                    onChange={handleInputChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                  />
                  <div className="flex justify-between text-xs text-gray-500 mt-1">
                    <span>₹0</span>
                    <span>₹1L</span>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Loan Tenure (Years)
                  </label>
                  <select
                    name="loanTenure"
                    value={formData.loanTenure}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="10">10 Years</option>
                    <option value="15">15 Years</option>
                    <option value="20">20 Years</option>
                    <option value="25">25 Years</option>
                    <option value="30">30 Years</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Interest Rate (%)
                  </label>
                  <input
                    type="number"
                    step="0.1"
                    name="interestRate"
                    value={formData.interestRate}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                    placeholder="Enter interest rate"
                  />
                </div>

                <button
                  onClick={calculateEligibility}
                  className="w-full bg-teal-600 text-white py-3 px-4 rounded-md hover:bg-teal-700 transition-colors font-medium flex items-center justify-center gap-2"
                >
                  <span>🏦</span>
                  Check Eligibility
                </button>
              </div>
            </div>

            <div>
              <h2 className="text-xl font-semibold mb-6">Eligibility Results</h2>
              
              {result ? (
                <div className="space-y-4">
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-green-800 mb-2">Eligible Amount</h3>
                    <p className="text-2xl font-bold text-green-600">
                      ₹{result.eligibleAmount.toLocaleString('en-IN')}
                    </p>
                    <p className="text-sm text-green-500">(₹{result.eligibleAmount >= 10000000 ? (result.eligibleAmount/10000000).toFixed(2) + ' Crores' : (result.eligibleAmount/100000).toFixed(2) + ' Lakhs'})</p>
                  </div>

                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-blue-800 mb-2">Max EMI</h3>
                    <p className="text-2xl font-bold text-blue-600">
                      ₹{result.maxEMI.toLocaleString('en-IN')}
                    </p>
                    <p className="text-sm text-blue-500">(₹{(result.maxEMI/100000).toFixed(2)} Lakhs)</p>
                  </div>

                  <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-yellow-800 mb-2">DTI</h3>
                    <p className="text-2xl font-bold text-yellow-600">
                      {result.debtToIncomeRatio}%
                    </p>
                    <p className="text-sm text-yellow-700 mt-1">
                      {result.debtToIncomeRatio <= 40 ? 'Good' : result.debtToIncomeRatio <= 50 ? 'Moderate' : 'High Risk'}
                    </p>
                  </div>

                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <h3 className="text-lg font-semibold text-gray-800 mb-2">Tips</h3>
                    <ul className="text-sm text-gray-600 space-y-1">
                      <li>• Keep debt-to-income ratio below 40%</li>
                      <li>• Maintain good credit score (750+)</li>
                      <li>• Have stable employment history</li>
                      <li>• Save for down payment (20% recommended)</li>
                    </ul>
                  </div>
                </div>
              ) : (
                <div className="bg-gray-50 border border-gray-200 rounded-lg p-8 text-center">
                  <p className="text-gray-500">Enter your details to calculate loan eligibility</p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};