import React, { useState, useEffect, useRef } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { ArrowLeft, MapPin, Check, Search } from 'lucide-react';

declare global {
  interface Window {
    google: any;
  }
}

const GOOGLE_MAPS_API_KEY = 'AIzaSyBGM7Q8jORMzDOj8GMcqjel1FSFPVzG0sI';

// Fallback coordinates for common Indian cities
const CITY_COORDINATES: { [key: string]: { lat: number; lng: number; name: string } } = {
  'bangalore': { lat: 12.9716, lng: 77.5946, name: 'Bangalore, Karnataka, India' },
  'bengaluru': { lat: 12.9716, lng: 77.5946, name: 'Bengaluru, Karnataka, India' },
  'mumbai': { lat: 19.0760, lng: 72.8777, name: 'Mumbai, Maharashtra, India' },
  'delhi': { lat: 28.7041, lng: 77.1025, name: 'Delhi, India' },
  'chennai': { lat: 13.0827, lng: 80.2707, name: 'Chennai, Tamil Nadu, India' },
  'kolkata': { lat: 22.5726, lng: 88.3639, name: 'Kolkata, West Bengal, India' },
  'hyderabad': { lat: 17.3850, lng: 78.4867, name: 'Hyderabad, Telangana, India' },
  'pune': { lat: 18.5204, lng: 73.8567, name: 'Pune, Maharashtra, India' },
  'koramangala': { lat: 12.9279, lng: 77.6271, name: 'Koramangala, Bangalore, Karnataka, India' },
  'whitefield': { lat: 12.9698, lng: 77.7500, name: 'Whitefield, Bangalore, Karnataka, India' },
  'indiranagar': { lat: 12.9784, lng: 77.6408, name: 'Indiranagar, Bangalore, Karnataka, India' },
  'malleshwaram': { lat: 13.0067, lng: 77.5667, name: 'Malleshwaram, Bangalore, Karnataka, India' },
  'jayanagar': { lat: 12.9237, lng: 77.5838, name: 'Jayanagar, Bangalore, Karnataka, India' },
  'btm': { lat: 12.9166, lng: 77.6101, name: 'BTM Layout, Bangalore, Karnataka, India' },
  'hsr': { lat: 12.9116, lng: 77.6370, name: 'HSR Layout, Bangalore, Karnataka, India' },
  'electronic city': { lat: 12.8456, lng: 77.6603, name: 'Electronic City, Bangalore, Karnataka, India' },
  'mg road': { lat: 12.9759, lng: 77.6037, name: 'MG Road, Bangalore, Karnataka, India' },
  'brigade road': { lat: 12.9716, lng: 77.6033, name: 'Brigade Road, Bangalore, Karnataka, India' }
};

export const MapSelector: React.FC = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const mapRef = useRef<HTMLDivElement>(null);
  const [map, setMap] = useState<any>(null);
  const [marker, setMarker] = useState<any>(null);
  const [selectedLocation, setSelectedLocation] = useState<{lat: number, lng: number} | null>(null);
  const [address, setAddress] = useState('');
  const [isLoading, setIsLoading] = useState(true);
  const [searchInput, setSearchInput] = useState('');
  const searchInputRef = useRef<HTMLInputElement>(null);

  // Get initial coordinates from URL params
  const initialLat = parseFloat(searchParams.get('lat') || '12.9716');
  const initialLng = parseFloat(searchParams.get('lng') || '77.5946');

  useEffect(() => {
    loadGoogleMaps();
    
    // Cleanup function
    return () => {
      if ((window as any).initMap) {
        delete (window as any).initMap;
      }
    };
  }, []);

  const loadGoogleMaps = () => {
    if (window.google && window.google.maps) {
      initializeMap();
      return;
    }

    // Remove existing script if any
    const existingScript = document.querySelector('script[src*="maps.googleapis.com"]');
    if (existingScript) {
      existingScript.remove();
    }

    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_API_KEY}&libraries=places&callback=initMap`;
    script.async = true;
    script.defer = true;
    
    // Global callback function
    (window as any).initMap = () => {
      initializeMap();
    };
    
    script.onerror = () => {
      console.error('Failed to load Google Maps');
      setIsLoading(false);
    };
    
    document.head.appendChild(script);
  };

  const initializeMap = () => {
    if (!mapRef.current || !window.google) {
      console.error('Map container or Google Maps not available');
      return;
    }

    try {
      const googleMap = new window.google.maps.Map(mapRef.current, {
        center: { lat: initialLat, lng: initialLng },
        zoom: 15,
        mapTypeControl: true,
        streetViewControl: true,
        fullscreenControl: true,
        zoomControl: true,
        mapTypeControlOptions: {
          style: window.google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
          position: window.google.maps.ControlPosition.TOP_CENTER,
        },
      });

      const mapMarker = new window.google.maps.Marker({
        position: { lat: initialLat, lng: initialLng },
        map: googleMap,
        draggable: true,
        animation: window.google.maps.Animation.DROP,
        title: 'Drag me to select location',
      });

      // Handle map clicks
      googleMap.addListener('click', (e: any) => {
        if (e.latLng) {
          const newPos = { lat: e.latLng.lat(), lng: e.latLng.lng() };
          setSelectedLocation(newPos);
          mapMarker.setPosition(newPos);
          reverseGeocode(newPos.lat, newPos.lng);
        }
      });

      // Handle marker drag
      mapMarker.addListener('dragend', () => {
        const pos = mapMarker.getPosition();
        if (pos) {
          const newPos = { lat: pos.lat(), lng: pos.lng() };
          setSelectedLocation(newPos);
          reverseGeocode(newPos.lat, newPos.lng);
        }
      });

      setMap(googleMap);
      setMarker(mapMarker);
      setSelectedLocation({ lat: initialLat, lng: initialLng });
      
      // Initialize Places Autocomplete
      setTimeout(() => {
        if (searchInputRef.current) {
          const autocomplete = new window.google.maps.places.Autocomplete(searchInputRef.current, {
            types: ['geocode'],
            componentRestrictions: { country: 'IN' }
          });

          autocomplete.addListener('place_changed', () => {
            const place = autocomplete.getPlace();
            if (place.geometry && place.geometry.location) {
              const newPos = {
                lat: place.geometry.location.lat(),
                lng: place.geometry.location.lng()
              };
              
              googleMap.setCenter(newPos);
              googleMap.setZoom(15);
              mapMarker.setPosition(newPos);
              setSelectedLocation(newPos);
              setAddress(place.formatted_address || place.name || '');
            }
          });
        }
      }, 100);
      
      setIsLoading(false);
      
      // Initial reverse geocode
      reverseGeocode(initialLat, initialLng);
    } catch (error) {
      console.error('Error initializing map:', error);
      setIsLoading(false);
    }
  };

  const reverseGeocode = (lat: number, lng: number) => {
    if (!window.google) return;
    
    const geocoder = new window.google.maps.Geocoder();
    geocoder.geocode({ location: { lat, lng } }, (results: any, status: any) => {
      if (status === 'OK' && results?.[0]) {
        setAddress(results[0].formatted_address);
      } else {
        setAddress(`Location: ${lat.toFixed(6)}, ${lng.toFixed(6)}`);
      }
    });
  };

  const handleManualSearch = () => {
    if (!searchInput.trim()) {
      alert('Please enter a location to search');
      return;
    }
    
    if (!window.google) {
      alert('Google Maps is still loading. Please wait a moment.');
      return;
    }

    console.log('Searching for:', searchInput);
    const geocoder = new window.google.maps.Geocoder();
    
    // First try fallback coordinates with fuzzy matching
    const searchKey = searchInput.toLowerCase().trim();
    
    // Handle common misspellings
    const fuzzyMatch = (input: string) => {
      const corrections: { [key: string]: string } = {
        'banglore': 'bangalore',
        'bangaluru': 'bengaluru',
        'mumbay': 'mumbai',
        'dilli': 'delhi',
        'chenai': 'chennai',
        'kolkatta': 'kolkata',
        'hydrabad': 'hyderabad',
        'puna': 'pune'
      };
      return corrections[input] || input;
    };
    
    const correctedKey = fuzzyMatch(searchKey);
    
    if (CITY_COORDINATES[correctedKey]) {
      const cityData = CITY_COORDINATES[correctedKey];
      const newPos = { lat: cityData.lat, lng: cityData.lng };
      
      console.log('Using fallback coordinates for:', searchKey, newPos);
      
      if (map && marker) {
        map.setCenter(newPos);
        map.setZoom(15);
        marker.setPosition(newPos);
        setSelectedLocation(newPos);
        setAddress(cityData.name);
        setSearchInput('');
      }
      return;
    }
    
    // Try Google Geocoding as fallback
    const searchQueries = [
      searchInput,
      `${searchInput}, India`,
      `${searchInput}, Bangalore, India`,
      `${searchInput}, Karnataka, India`
    ];
    
    const trySearch = (queryIndex: number) => {
      if (queryIndex >= searchQueries.length) {
        alert(`Location "${searchInput}" not found. Try: Bangalore, Mumbai, Delhi, Chennai, Koramangala, etc.`);
        return;
      }
      
      const query = searchQueries[queryIndex];
      console.log(`Trying search ${queryIndex + 1}:`, query);
      
      geocoder.geocode({ address: query }, (results: any, status: any) => {
        console.log(`Search ${queryIndex + 1} results:`, results, 'Status:', status);
        
        if (status === 'OK' && results?.[0]) {
          const location = results[0].geometry.location;
          const newPos = {
            lat: location.lat(),
            lng: location.lng()
          };
          
          console.log('Moving map to:', newPos);
          
          if (map && marker) {
            map.setCenter(newPos);
            map.setZoom(15);
            marker.setPosition(newPos);
            setSelectedLocation(newPos);
            setAddress(results[0].formatted_address);
            setSearchInput('');
          }
        } else {
          trySearch(queryIndex + 1);
        }
      });
    };
    
    trySearch(0);
  };

  const handleConfirmLocation = () => {
    if (selectedLocation) {
      // Store location in localStorage for the form to pick up
      localStorage.setItem('selectedMapLocation', JSON.stringify({
        lat: selectedLocation.lat,
        lng: selectedLocation.lng,
        address: address
      }));
      
      // Navigate back to the form
      navigate('/list-property');
    }
  };

  return (
    <div className="min-h-screen bg-white flex flex-col">
      {/* Header */}
      <div className="bg-white shadow-sm border-b p-4">
        <div className="flex items-center justify-between mb-4">
          <button 
            onClick={() => navigate('/list-property')}
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="w-5 h-5 mr-2" />
            Back to Form
          </button>
          <h1 className="text-lg font-semibold text-gray-900">Select Property Location</h1>
          <div className="w-20"></div>
        </div>
        
        {/* Search Box */}
        <div className="flex max-w-lg mx-auto space-x-2">
          <div className="relative flex-1">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              ref={searchInputRef}
              type="text"
              value={searchInput}
              onChange={(e) => setSearchInput(e.target.value)}
              placeholder="Search for city, locality, or address..."
              className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              onKeyPress={(e) => e.key === 'Enter' && handleManualSearch()}
            />
          </div>
          <button
            onClick={handleManualSearch}
            className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0041A3] transition-colors flex items-center space-x-2 shadow-md font-medium"
          >
            <Search className="w-4 h-4" />
            <span>Search</span>
          </button>
        </div>
      </div>

      {/* Map Container */}
      <div className="flex-1 relative">
        {isLoading && (
          <div className="absolute inset-0 bg-gray-100 flex items-center justify-center z-10">
            <div className="text-center">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2"></div>
              <p className="text-gray-600">Loading map...</p>
            </div>
          </div>
        )}
        
        <div 
          ref={mapRef} 
          className="w-full h-full min-h-[400px]" 
          style={{ minHeight: '70vh' }}
        />
        
        {/* Center crosshair */}
        <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 pointer-events-none z-10">
          <MapPin className="w-8 h-8 text-red-500 drop-shadow-lg" />
        </div>
      </div>

      {/* Bottom Panel */}
      <div className="bg-white border-t p-4 space-y-4">
        {selectedLocation && (
          <div className="bg-gray-50 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <MapPin className="w-5 h-5 text-red-500 mt-0.5 flex-shrink-0" />
              <div className="flex-1">
                <p className="font-medium text-gray-900">Selected Location</p>
                <p className="text-sm text-gray-600 mt-1">{address || 'Getting address...'}</p>
                <p className="text-xs text-gray-500 mt-1">
                  Coordinates: {selectedLocation.lat.toFixed(6)}, {selectedLocation.lng.toFixed(6)}
                </p>
              </div>
            </div>
          </div>
        )}

        <div className="text-center text-sm text-gray-600 mb-4">
          <p>Tap anywhere on the map or drag the pin to select your property location</p>
        </div>

        <button
          onClick={handleConfirmLocation}
          disabled={!selectedLocation}
          className="w-full flex items-center justify-center space-x-2 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          <Check className="w-5 h-5" />
          <span className="font-medium">Confirm This Location</span>
        </button>
      </div>
    </div>
  );
};