import React from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { CheckCircle, Phone } from 'lucide-react';
import { Button } from '../components/ui/button';

export function MarketingServicesPage() {
  const navigate = useNavigate();
  const solution = {
    title: 'Marketing Services',
    description: 'Promote properties using featured listings, AI banners, WhatsApp campaigns, search boost, and agent-targeted marketing for maximum visibility and lead generation.',
    features: [
      'WhatsApp Business API integration for direct customer engagement',
      'Google Ads and Facebook campaigns optimized for Indian audiences',
      'Multi-language marketing content in Hindi, Tamil, Telugu, and regional languages',
      'Integration with popular Indian property portals like 99acres and MagicBricks',
      'Festival season campaigns during Diwali, Dussehra, and regional celebrations'
    ],
    benefits: [
      'Reach NRI buyers through targeted campaigns in USA, UK, and Gulf countries',
      'Leverage Indian social media platforms like ShareChat and regional apps',
      'Optimize for Indian search behavior and local keywords',
      'Compliance with Indian advertising standards and ASCI guidelines',
      'Cost-effective campaigns optimized for Indian market CPCs and demographics'
    ],
    useCases: [
      'Luxury project launches in Mumbai, Delhi, and Bangalore metro markets',
      'Affordable housing campaigns under PM Awas Yojana and state schemes',
      'NRI-focused marketing for investment properties in tier-1 cities',
      'Regional language campaigns for local markets in Tamil Nadu and Karnataka',
      'Festival season promotions during Akshaya Tritiya and Gudi Padwa'
    ],
    pricing: 'Starting from ₹15,000/month'
  };

  const serviceSections = [
    {
      title: 'Common Services',
      items: [
        {
          title: 'Market Analysis',
          description: 'Data-driven market research with pricing, demand, and trend insights.',
          image: '/solutions/market analysis.jpg',
          href: '/services/market-analysis'
        },
        {
          title: 'Micro Market Performance Hub',
          description: 'Top micro markets, rankings, watchlists, and performance insights.',
          image: 'https://leadsnap.com/wp-content/uploads/2024/09/aa1.png',
          href: '/top-markets'
        },
        {
          title: 'Locality Intelligence',
          description: 'Deep locality analysis with scorecards, heatmaps, and live stats.',
          image: 'https://cdn.ourcrowd.com/wp-content/uploads/2022/06/Location-Intelligence.jpg',
          href: '/advisory/locality-intelligence'
        }
      ]
    },
    {
      title: 'Buyer Services',
      items: [
        {
          title: 'Campaigns Listing',
          description: 'Manage and track active marketing campaigns.',
          image: 'https://media.istockphoto.com/id/1219580367/photo/marketing-campaign-brand-advertisement-business-strategy.jpg?s=612x612&w=0&k=20&c=AJ7xdZWOoRzHJ7Nb0EyilhBJSIzFfrqPfFiYDfX5qsE=',
          href: '/campaign-management'
        },
        {
          title: 'Future Insights',
          description: 'AI-powered infrastructure and price forecast insights.',
          image: 'https://www.nec.com/en/global/insights/article/take-your-business-into-the-future-today/images/sp_visual.jpg',
          href: '/user/dashboard?section=FutureInsights'
        },
        {
          title: 'AI Property Comparison',
          description: 'Compare multiple properties with AI-backed insights.',
          image: '/solutions/brand building.jpg',
          href: '/compare'
        },
        {
          title: 'Price Prediction',
          description: 'Predict property pricing with ML-based analysis.',
          image: 'https://rtslabs.com/wp-content/uploads/2025/07/Predictive-Analytics-in-Real-Estate.png',
          href: '/price-prediction'
        }
      ]
    },
    {
      title: 'Seller Services',
      items: [
        {
          title: 'Campaigns Management',
          description: 'Manage and track active marketing campaigns.',
          image: 'https://media.istockphoto.com/id/1219580367/photo/marketing-campaign-brand-advertisement-business-strategy.jpg?s=612x612&w=0&k=20&c=AJ7xdZWOoRzHJ7Nb0EyilhBJSIzFfrqPfFiYDfX5qsE=',
          href: '/seller/dashboard/campaigns'
        },
        {
          title: 'Property Analytics',
          description: 'Performance analytics for listings and lead activity.',
          image: 'https://datasemantics.co/wp-content/uploads/2024/09/6-Whats-Next-for-Data-Visualization.png',
          href: '/seller/dashboard/property-analytics'
        },
        {
          title: 'Marketing Tools',
          description: 'Boost visibility with paid and organic tools.',
          image: 'https://www.socialchamp.com/blog/wp-content/uploads/2024/08/AI-Social-Listening.png',
          href: '/seller/dashboard/marketing-tools'
        },
        {
          title: 'Valuation Tools',
          description: 'Valuation insights and pricing guidance.',
          image: 'https://media.licdn.com/dms/image/v2/D4D12AQFIMQGal3dz8w/article-cover_image-shrink_720_1280/article-cover_image-shrink_720_1280/0/1719953362082?e=2147483647&v=beta&t=0L_CcRf430c0KWOjYQC-ztFtLsKRMH2jA01PFzXxaoM',
          href: '/seller/dashboard/valuation'
        },
        {
          title: 'Price Prediction',
          description: 'ML price forecasting for listing strategy.',
          image: 'https://rtslabs.com/wp-content/uploads/2025/07/Predictive-Analytics-in-Real-Estate.png',
          href: '/seller/dashboard/price-prediction'
        },
        {
          title: 'Price Intelligence',
          description: 'Competitive pricing and market positioning.',
          image: 'https://media.licdn.com/dms/image/v2/D5612AQEbnSPtenDSuA/article-cover_image-shrink_600_2000/B56ZuMrzhWHYAQ-/0/1767591853194?e=2147483647&v=beta&t=yeB_8ubF9dOyzb6V2cKwAcTpCGIangKYevkOBIcgjns',
          href: '/seller/dashboard/price-intelligence'
        },
        {
          title: 'Market Insights',
          description: 'Trends, seasonality, and area-level insights.',
          image: 'https://cdn.rentechdigital.com/common_files/blogs/what-is-marketing-insight-and-why-is-it-important-swipecart-blog-1-10-3-23.png',
          href: '/seller/dashboard/market-intelligence'
        }
      ]
    },
    {
      title: 'Agent Services',
      items: [
        {
          title: 'Campaign Management',
          description: 'Create and manage campaigns for client listings.',
          image: 'https://media.istockphoto.com/id/1219580367/photo/marketing-campaign-brand-advertisement-business-strategy.jpg?s=612x612&w=0&k=20&c=AJ7xdZWOoRzHJ7Nb0EyilhBJSIzFfrqPfFiYDfX5qsE=',
          href: '/agent/dashboard/campaigns'
        },
        {
          title: 'Price Prediction',
          description: 'ML price prediction for client advising.',
          image: 'https://rtslabs.com/wp-content/uploads/2025/07/Predictive-Analytics-in-Real-Estate.png',
          href: '/agent/dashboard/price-prediction'
        },
        {
          title: 'Property Analytics',
          description: 'Analytics for property performance and demand.',
          image: 'https://datasemantics.co/wp-content/uploads/2024/09/6-Whats-Next-for-Data-Visualization.png',
          href: '/agent/dashboard/property-analytics'
        },
        {
          title: 'Analytics & Market Insights',
          description: 'Market insights to guide client decisions.',
          image: 'https://media.licdn.com/dms/image/v2/D5612AQFbdznZYDutLA/article-cover_image-shrink_600_2000/article-cover_image-shrink_600_2000/0/1705066964287?e=2147483647&v=beta&t=2NGtoBGrbEHttri04RfFwD9R-7sOdcOJXjl4zHdCJqg',
          href: '/agent/dashboard/analytics'
        }
      ]
    }
  ];

  const handleMarketingToolsNav = () => {
    const token =
      localStorage.getItem('sellerToken') ||
      localStorage.getItem('access_token') ||
      localStorage.getItem('agentToken') ||
      localStorage.getItem('token') ||
      localStorage.getItem('auth_token');

    if (!token) {
      navigate('/seller');
      return;
    }

    navigate('/seller/dashboard/marketing-tools');
  };

  const isAuthenticated = () => {
    return Boolean(
      localStorage.getItem('sellerToken') ||
      localStorage.getItem('access_token') ||
      localStorage.getItem('agentToken') ||
      localStorage.getItem('token') ||
      localStorage.getItem('auth_token')
    );
  };

  const getCurrentRole = (): 'buyer' | 'seller' | 'agent' | 'admin' | 'user' => {
    try {
      const accessToken = localStorage.getItem('access_token');
      if (accessToken) {
        const payload = JSON.parse(atob(accessToken.split('.')[1]));
        const roles: string[] = payload.roles || [];
        if (roles.includes('Admin')) return 'admin';
        if (roles.includes('Agent')) return 'agent';
        if (roles.includes('Seller')) return 'seller';
        if (roles.includes('Buyer')) return 'buyer';
      }
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const role = (user.user_role || user.role || 'user').toString().toLowerCase();
      if (role === 'admin') return 'admin';
      if (role === 'agent') return 'agent';
      if (role === 'seller') return 'seller';
      if (role === 'buyer') return 'buyer';
      return 'user';
    } catch {
      return 'user';
    }
  };

  const hasRoleAccess = (requiredRole: 'buyer' | 'seller' | 'agent') => {
    const role = getCurrentRole();
    if (role === 'admin') return true;
    if (requiredRole === 'buyer') return role === 'buyer' || role === 'user';
    return role === requiredRole;
  };

  const handleProtectedNav = (href: string, requiredRole?: 'buyer' | 'seller' | 'agent') => {
    if (!isAuthenticated()) {
      if (requiredRole) {
        alert(`Please login as ${requiredRole} to access this section.`);
      } else {
        alert('Please login to access this section.');
      }
      navigate('/login');
      return;
    }
    if (requiredRole && !hasRoleAccess(requiredRole)) {
      alert(`This section is only for ${requiredRole} accounts. Please login with a ${requiredRole} account.`);
      navigate('/login');
      return;
    }
    navigate(href);
  };


  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="text-white py-16" style={{backgroundColor: '#1e40af'}}>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-6">{solution.title}</h1>
            <p className="text-xl md:text-2xl text-purple-100 max-w-3xl mx-auto">{solution.description}</p>
            <div className="mt-8">
              <span className="inline-block bg-white/20 text-white px-6 py-3 rounded-full text-lg font-semibold">
                {solution.pricing}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Solutions Images Section */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center">
              <img 
                src="/solutions/digital marketingg.jpg"
                alt="Marketing Solutions" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Digital Marketing</h3>
              <p className="text-gray-600">Comprehensive marketing campaigns</p>
            </div>
            
            <div className="text-center">
              <img 
                src="/solutions/brand building.jpg"
                alt="Smart Technology" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Brand Building</h3>
              <p className="text-gray-600">Professional brand development</p>
            </div>
            
            <div className="text-center">
              <img 
                src="/solutions/results tracking.jpg"
                alt="Professional Services" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Results Tracking</h3>
              <p className="text-gray-600">Measurable results and analytics</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        {/* Services Overview */}
        <div className="mb-16 space-y-12">
          {serviceSections.map((section, index) => (
            <div
              key={section.title}
              className={index === 0 ? '' : 'mt-12'}
            >
              <h2 className="text-3xl font-bold text-gray-900 text-center mb-3">{section.title}</h2>
              {section.title === 'Seller Services' ? (
                <>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-8">
                    {section.items.slice(0, 3).map((item) => (
                      <button
                        key={item.title}
                        type="button"
                        onClick={() => handleProtectedNav(item.href, 'seller')}
                        className="bg-white rounded-lg p-6 shadow-sm border border-gray-200 text-center hover:shadow-md transition-shadow"
                      >
                        <img
                          src={item.image}
                          alt={item.title}
                          className="w-full h-48 object-cover rounded-lg mb-4 shadow-sm"
                          style={item.title === 'Micro Market Performance Hub' ? { backgroundColor: '#e0f2fe' } : undefined}
                        />
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">{item.title}</h3>
                        <p className="text-gray-600 text-sm">{item.description}</p>
                      </button>
                    ))}
                  </div>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-8">
                    {section.items.slice(3).map((item) => (
                      <button
                        key={item.title}
                        type="button"
                        onClick={() => handleProtectedNav(item.href, 'seller')}
                        className="bg-white rounded-lg p-6 shadow-sm border border-gray-200 text-center hover:shadow-md transition-shadow"
                      >
                        <img
                          src={item.image}
                          alt={item.title}
                          className="w-full h-48 object-cover rounded-lg mb-4 shadow-sm"
                          style={item.title === 'Micro Market Performance Hub' ? { backgroundColor: '#e0f2fe' } : undefined}
                        />
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">{item.title}</h3>
                        <p className="text-gray-600 text-sm">{item.description}</p>
                      </button>
                    ))}
                  </div>
                </>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-8">
                  {section.items.map((item) => (
                    section.title === 'Buyer Services' || section.title === 'Agent Services' ? (
                      <button
                        key={item.title}
                        type="button"
                        onClick={() => handleProtectedNav(item.href, section.title === 'Buyer Services' ? 'buyer' : 'agent')}
                        className="bg-white rounded-lg p-6 shadow-sm border border-gray-200 text-center hover:shadow-md transition-shadow"
                      >
                        <img
                          src={item.image}
                          alt={item.title}
                          className="w-full h-48 object-cover rounded-lg mb-4 shadow-sm"
                          style={item.title === 'Micro Market Performance Hub' ? { backgroundColor: '#e0f2fe' } : undefined}
                        />
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">{item.title}</h3>
                        <p className="text-gray-600 text-sm">{item.description}</p>
                      </button>
                    ) : (
                      <Link
                        key={item.title}
                        to={item.href}
                        className="bg-white rounded-lg p-6 shadow-sm border border-gray-200 text-center hover:shadow-md transition-shadow"
                      >
                        <img
                          src={item.image}
                          alt={item.title}
                          className="w-full h-48 object-cover rounded-lg mb-4 shadow-sm"
                        />
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">{item.title}</h3>
                        <p className="text-gray-600 text-sm">{item.description}</p>
                      </Link>
                    )
                  ))}
                </div>
              )}
            </div>
          ))}
        </div>


        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Features */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Features</h2>
            <ul className="space-y-4">
              {solution.features.map((feature, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{feature}</span>
                </li>
              ))}
            </ul>
          </div>

          {/* Benefits */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Benefits</h2>
            <ul className="space-y-4">
              {solution.benefits.map((benefit, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{benefit}</span>
                </li>
              ))}
            </ul>
          </div>

          {/* Use Cases */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Use Cases</h2>
            <ul className="space-y-4">
              {solution.useCases.map((useCase, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{useCase}</span>
                </li>
              ))}
            </ul>
          </div>
        </div>

        {/* CTA Section */}
        <div className="mt-16 bg-white rounded-lg p-8 shadow-sm border border-gray-200 text-center">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">Ready to Transform Your Business?</h2>
          <p className="text-gray-600 mb-8 max-w-2xl mx-auto">
            Discover how marketing services can revolutionize your real estate operations and drive growth.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
            <Button className="text-black px-8 py-3" style={{backgroundColor: '#9333ea'}}>
              <Phone className="w-4 h-4 mr-2" />
+91-8068447416
            </Button>
            <Button
              variant="outline"
              className="border-[#9333ea] text-[#9333ea] hover:bg-[#9333ea] hover:text-white px-8 py-3"
              onClick={() => {
                alert(
                  'Request Quote for Marketing Services\n\n' +
                  'Solution: Marketing Services\n' +
                  'Pricing: Starting from ₹15,000/month\n\n' +
                  'Contact us at +91-8068447416 for a detailed quote.'
                );
              }}
            >
              Request Quote
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}
