import React, { useState, useEffect } from 'react';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../components/ui/tabs';
import { MicroMarketHeatmap } from '../components/micro-markets/MicroMarketHeatmap';
import { MicroMarketLeaderboard } from '../components/micro-markets/MicroMarketLeaderboard';
import { MicroMarketWatchlist } from '../components/micro-markets/MicroMarketWatchlist';
import { InvestmentInsights } from '../components/micro-markets/InvestmentInsights';
import { Map, TrendingUp, Eye, Lightbulb, Search, MapPin, ArrowUp } from 'lucide-react';

interface MicroMarket {
  id: number;
  name: string;
  city: string;
  overall_score: number;
  price_appreciation_score: number;
  demand_strength_score: number;
  supply_stability_score: number;
  tier: string;
  avg_price_per_sqft: number;
  growth_percentage: number;
  property_count: number;
}

export function MicroMarketPerformanceHub() {
  const [activeTab, setActiveTab] = useState('heatmap');
  const [markets, setMarkets] = useState<MicroMarket[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCity, setSelectedCity] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    // Use mock data directly for now
    const mockMarkets: MicroMarket[] = [
      {
        id: 1,
        name: "Bandra West",
        city: "Mumbai",
        overall_score: 92,
        price_appreciation_score: 88,
        demand_strength_score: 95,
        supply_stability_score: 85,
        tier: "High Performing",
        avg_price_per_sqft: 45000,
        growth_percentage: 15.2,
        property_count: 1250
      },
      {
        id: 2,
        name: "Koramangala",
        city: "Bangalore",
        overall_score: 89,
        price_appreciation_score: 92,
        demand_strength_score: 88,
        supply_stability_score: 87,
        tier: "High Performing",
        avg_price_per_sqft: 12500,
        growth_percentage: 18.7,
        property_count: 980
      },
      {
        id: 3,
        name: "Gurgaon Sector 54",
        city: "Delhi",
        overall_score: 85,
        price_appreciation_score: 82,
        demand_strength_score: 90,
        supply_stability_score: 83,
        tier: "High Performing",
        avg_price_per_sqft: 8500,
        growth_percentage: 12.4,
        property_count: 750
      },
      {
        id: 4,
        name: "Whitefield",
        city: "Bangalore",
        overall_score: 78,
        price_appreciation_score: 75,
        demand_strength_score: 82,
        supply_stability_score: 77,
        tier: "Emerging",
        avg_price_per_sqft: 9200,
        growth_percentage: 14.1,
        property_count: 650
      },
      {
        id: 5,
        name: "Powai",
        city: "Mumbai",
        overall_score: 82,
        price_appreciation_score: 80,
        demand_strength_score: 85,
        supply_stability_score: 81,
        tier: "High Performing",
        avg_price_per_sqft: 18500,
        growth_percentage: 11.8,
        property_count: 890
      },
      {
        id: 6,
        name: "Hinjewadi",
        city: "Pune",
        overall_score: 76,
        price_appreciation_score: 78,
        demand_strength_score: 75,
        supply_stability_score: 75,
        tier: "Emerging",
        avg_price_per_sqft: 7800,
        growth_percentage: 16.3,
        property_count: 540
      }
    ];
    setMarkets(mockMarkets);
    setLoading(false);
  }, [selectedCity]);

  const filteredMarkets = markets.filter((market) =>
    market.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    market.city.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const getTierColor = (tier: string) => {
    switch (tier) {
      case 'High Performing': return 'bg-green-100 text-green-800';
      case 'Emerging': return 'bg-blue-100 text-blue-800';
      case 'Stable': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      <style>{`
        .market-hero {
          background: linear-gradient(rgba(30, 41, 59, 0.9), rgba(30, 41, 59, 0.8)), url('https://images.unsplash.com/photo-1460925895917-afdab827c52f?ixlib=rb-4.0.3&auto=format&fit=crop&w=2015&q=80') !important;
          background-size: cover !important;
          background-position: center !important;
          padding: 8rem 0 !important;
          color: white !important;
          text-align: center !important;
        }
        .market-hero h1 {
          font-size: 3.5rem !important;
          font-weight: 700 !important;
          margin-bottom: 1.5rem !important;
          color: white !important;
          letter-spacing: -0.025em !important;
        }
        .market-hero p {
          font-size: 1.25rem !important;
          margin-bottom: 2.5rem !important;
          color: #e2e8f0 !important;
          max-width: 50rem !important;
          margin-left: auto !important;
          margin-right: auto !important;
          line-height: 1.7 !important;
        }
        .market-pricing {
          background: rgba(255, 255, 255, 0.95) !important;
          color: #1e293b !important;
          padding: 1.5rem 3rem !important;
          border-radius: 1rem !important;
          font-weight: 600 !important;
          font-size: 1.1rem !important;
          display: inline-block !important;
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1) !important;
          backdrop-filter: blur(10px) !important;
        }
      `}</style>

      <header className="market-hero">
        <div style={{ maxWidth: '64rem', margin: '0 auto', padding: '0 1rem' }}>
          <h1>Micro Market Performance Hub</h1>
          <p>
            Monitor micro market momentum with heatmaps, rankings, watchlists, and performance insights to spot
            high-growth opportunities early.
          </p>
          <div className="market-pricing">
            Micro Market Report: ₹8,999 | Locality Performance: ₹5,999 | Custom Benchmarking: ₹18,999 | Investment Advisory: ₹29,999
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8 mt-8">

        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList className="flex w-full flex-wrap gap-2 py-2 px-2">
            <TabsTrigger value="top" className="flex items-center gap-2 px-4 py-2">
              <TrendingUp className="w-4 h-4" />
              Top Performing
            </TabsTrigger>
            <TabsTrigger value="heatmap" className="flex items-center gap-2 px-4 py-2">
              <Map className="w-4 h-4" />
              Heatmap
            </TabsTrigger>
            <TabsTrigger value="leaderboard" className="flex items-center gap-2 px-4 py-2">
              <TrendingUp className="w-4 h-4" />
              Rankings
            </TabsTrigger>
            <TabsTrigger value="watchlist" className="flex items-center gap-2 px-4 py-2">
              <Eye className="w-4 h-4" />
              Watchlist
            </TabsTrigger>
            <TabsTrigger value="insights" className="flex items-center gap-2 px-4 py-2">
              <Lightbulb className="w-4 h-4" />
              Insights
            </TabsTrigger>
          </TabsList>

          <TabsContent value="top" className="mt-6">
            <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
              <div className="flex flex-col md:flex-row gap-4">
                <div className="flex-1">
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                    <input
                      type="text"
                      placeholder="Search markets or cities..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>
                </div>
                <div className="flex gap-2">
                  <select
                    value={selectedCity}
                    onChange={(e) => setSelectedCity(e.target.value)}
                    className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="all">All Cities</option>
                    <option value="Mumbai">Mumbai</option>
                    <option value="Delhi">Delhi</option>
                    <option value="Bangalore">Bangalore</option>
                    <option value="Pune">Pune</option>
                    <option value="Hyderabad">Hyderabad</option>
                  </select>
                </div>
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {loading ? (
                Array.from({ length: 6 }).map((_, i) => (
                  <div key={i} className="bg-white rounded-lg shadow-sm p-6 animate-pulse">
                    <div className="h-4 bg-gray-200 rounded mb-4"></div>
                    <div className="h-6 bg-gray-200 rounded mb-2"></div>
                    <div className="h-4 bg-gray-200 rounded mb-4"></div>
                    <div className="space-y-2">
                      <div className="h-3 bg-gray-200 rounded"></div>
                      <div className="h-3 bg-gray-200 rounded"></div>
                    </div>
                  </div>
                ))
              ) : (
                filteredMarkets.map((market) => (
                  <div key={market.id} className="bg-white rounded-lg shadow-sm hover:shadow-md transition-shadow p-6">
                    <div className="flex justify-between items-start mb-4">
                      <div>
                        <h3 className="font-semibold text-gray-900 text-lg">{market.name}</h3>
                        <div className="flex items-center text-gray-600 text-sm mt-1">
                          <MapPin className="w-4 h-4 mr-1" />
                          {market.city}
                        </div>
                      </div>
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${getTierColor(market.tier)}`}>
                        {market.tier}
                      </span>
                    </div>

                    <div className="mb-4">
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm text-gray-600">Overall Score</span>
                        <span className="text-2xl font-bold text-blue-600">{market.overall_score}/100</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div
                          className="bg-blue-600 h-2 rounded-full"
                          style={{ width: `${market.overall_score}%` }}
                        ></div>
                      </div>
                    </div>

                    <div className="space-y-3 mb-4">
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Avg Price/sqft</span>
                        <span className="font-medium">₹{market.avg_price_per_sqft?.toLocaleString()}</span>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Growth Rate</span>
                        <div className="flex items-center">
                          <ArrowUp className="w-4 h-4 text-green-600 mr-1" />
                          <span className="font-medium text-green-600">{market.growth_percentage}%</span>
                        </div>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Properties</span>
                        <span className="font-medium">{market.property_count}</span>
                      </div>
                    </div>

                    <button className="w-full bg-blue-600 text-white py-2 px-4 rounded-lg text-sm font-medium hover:bg-blue-700 transition-colors">
                      View Market Details
                    </button>
                  </div>
                ))
              )}
            </div>

            <div className="mt-12 bg-white rounded-lg shadow-sm p-6">
              <h2 className="text-xl font-semibold text-gray-900 mb-4">Market Insights</h2>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="text-center">
                  <div className="text-3xl font-bold text-blue-600 mb-2">85+</div>
                  <div className="text-sm text-gray-600">High Performing Markets</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-green-600 mb-2">12%</div>
                  <div className="text-sm text-gray-600">Average Growth Rate</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-purple-600 mb-2">50K+</div>
                  <div className="text-sm text-gray-600">Properties Analyzed</div>
                </div>
              </div>
            </div>
          </TabsContent>

          <TabsContent value="heatmap" className="mt-6">
            <MicroMarketHeatmap />
          </TabsContent>

          <TabsContent value="leaderboard" className="mt-6">
            <MicroMarketLeaderboard />
          </TabsContent>

          <TabsContent value="watchlist" className="mt-6">
            <MicroMarketWatchlist />
          </TabsContent>

          <TabsContent value="insights" className="mt-6">
            <InvestmentInsights />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
