import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Card } from "../components/ui/card";
import { Award, MapPin, Calendar, CheckCircle, Star, TrendingUp, QrCode, Navigation, Trophy, Target, Zap, Shield, Crown, Medal, Gift, Users, Clock, BarChart3 } from "lucide-react";
import { Button } from "../components/ui/button";
import { QRScanner } from "../components/QRScanner";
import { Progress } from "../components/ui/progress";
import { Badge } from "../components/ui/badge";
// GPS verification removed - using QR code only
interface Coordinates {
  lat: number;
  lng: number;
}
import { toast } from "sonner";
import QRCodeLib from "qrcode";

interface BadgeData {
  id: string;
  propertyName: string;
  propertyAddress: string;
  badgeType: string;
  badgeLevel: string;
  timestamp: string;
  earnedPoints: number;
  verificationMethod: string[];
}

interface BadgeCategory {
  id: string;
  name: string;
  description: string;
  icon: any;
  color: string;
  badges: BadgeData[];
  totalPossible: number;
  requirements: string[];
}

interface Achievement {
  id: string;
  title: string;
  description: string;
  icon: any;
  progress: number;
  maxProgress: number;
  reward: string;
  unlocked: boolean;
}

export const MyBadges = () => {
  const navigate = useNavigate();
  const [badges, setBadges] = useState<BadgeData[]>([]);
  const [totalPoints, setTotalPoints] = useState(0);
  const [userLevel, setUserLevel] = useState(1);
  const [nextLevelPoints, setNextLevelPoints] = useState(100);
  const [activeTab, setActiveTab] = useState<'overview' | 'categories' | 'achievements' | 'leaderboard'>('overview');
  const [showCheckInModal, setShowCheckInModal] = useState(false);
  const [checkInMethod, setCheckInMethod] = useState<'qr' | 'gps'>('qr');
  const [qrScanError, setQrScanError] = useState<string>('');
  const [userLocation, setUserLocation] = useState<Coordinates | null>(null);
  const [distance, setDistance] = useState<number | null>(null);
  const [locationPermission, setLocationPermission] = useState<'granted' | 'denied' | 'prompt'>('prompt');
  const [verificationStatus, setVerificationStatus] = useState<'idle' | 'verifying' | 'success' | 'error'>('idle');
  const [earnedBadge, setEarnedBadge] = useState<BadgeData | null>(null);
  const [properties, setProperties] = useState<any[]>([]);
  const [selectedProperty, setSelectedProperty] = useState<any>(null);
  const [loadingProperties, setLoadingProperties] = useState(false);
  const [propertyQRCode, setPropertyQRCode] = useState<string>('');
  const [gpsVerified, setGpsVerified] = useState(false);

  useEffect(() => {
    const existingBadges = JSON.parse(localStorage.getItem('visitBadges') || '[]');
    setBadges(existingBadges);
    const points = existingBadges.reduce((sum: number, badge: BadgeData) => sum + badge.earnedPoints, 0);
    setTotalPoints(points);
  }, []);

  useEffect(() => {
    if (showCheckInModal) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [showCheckInModal]);

  const fetchAvailableProperties = async () => {
    setLoadingProperties(true);
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userEmail = user.email;
      
      const response = await fetch('/api/properties/');
      if (response.ok) {
        const result = await response.json();
        const allProperties = result.properties || [];
        const filteredProperties = allProperties.filter((p: any) => p.email !== userEmail);
        setProperties(filteredProperties);
      }
    } catch (error) {
      toast.error('Failed to load properties');
    } finally {
      setLoadingProperties(false);
    }
  };

  const generatePropertyQRCode = async (property: any) => {
    const qrData = JSON.stringify({
      propertyId: property.id,
      title: property.title,
      address: property.address,
      city: property.city,
      latitude: property.latitude,
      longitude: property.longitude
    });
    const qrCodeUrl = await QRCodeLib.toDataURL(qrData, { width: 200 });
    setPropertyQRCode(qrCodeUrl);
  };

  const confirmQRScan = () => {
    setVerificationStatus('verifying');
    toast.success('Verifying visit...');
    
    setTimeout(() => {
      const newBadge: BadgeData = {
        id: `badge-${Date.now()}`,
        propertyName: selectedProperty.title,
        propertyAddress: `${selectedProperty.address}, ${selectedProperty.city}`,
        badgeType: 'Verified Property Visit',
        badgeLevel: 'Level 1',
        timestamp: new Date().toISOString(),
        earnedPoints: 50,
        verificationMethod: ['QR Code'],
      };
      setEarnedBadge(newBadge);
      const existingBadges = JSON.parse(localStorage.getItem('visitBadges') || '[]');
      localStorage.setItem('visitBadges', JSON.stringify([...existingBadges, newBadge]));
      setBadges([...badges, newBadge]);
      setTotalPoints(totalPoints + 50);
      setVerificationStatus('success');
      toast.success('Badge earned successfully!');
    }, 1000);
  };

  const getBadgeColor = (level: string) => {
    switch(level) {
      case 'Level 1': return 'from-blue-400 to-blue-600';
      case 'Level 2': return 'from-green-400 to-green-600';
      case 'Level 3': return 'from-purple-400 to-purple-600';
      case 'Level 4': return 'from-yellow-400 to-yellow-600';
      default: return 'from-gray-400 to-gray-600';
    }
  };

  const badgeCategories: BadgeCategory[] = [
    {
      id: 'visits',
      name: 'Property Visits',
      description: 'Badges earned from verified property visits',
      icon: MapPin,
      color: 'blue',
      badges: badges.filter(b => b.badgeType.includes('Visit')),
      totalPossible: 10,
      requirements: ['Visit properties', 'Complete check-in', 'Get seller verification']
    },
    {
      id: 'engagement',
      name: 'Platform Engagement',
      description: 'Active participation and platform usage',
      icon: Users,
      color: 'green',
      badges: badges.filter(b => b.badgeType.includes('Engagement')),
      totalPossible: 8,
      requirements: ['Regular platform usage', 'Complete profile', 'Engage with listings']
    },
    {
      id: 'achievements',
      name: 'Special Achievements',
      description: 'Milestone and special accomplishment badges',
      icon: Trophy,
      color: 'purple',
      badges: badges.filter(b => b.badgeType.includes('Achievement')),
      totalPossible: 5,
      requirements: ['Reach milestones', 'Complete challenges', 'Exceptional performance']
    }
  ];

  const achievements: Achievement[] = [
    {
      id: 'first-visit',
      title: 'First Steps',
      description: 'Complete your first property visit',
      icon: Target,
      progress: badges.length > 0 ? 1 : 0,
      maxProgress: 1,
      reward: '50 Points + Welcome Badge',
      unlocked: badges.length > 0
    },
    {
      id: 'explorer',
      title: 'Property Explorer',
      description: 'Visit 5 different properties',
      icon: Navigation,
      progress: Math.min(badges.length, 5),
      maxProgress: 5,
      reward: '200 Points + Explorer Badge',
      unlocked: badges.length >= 5
    },
    {
      id: 'dedicated',
      title: 'Dedicated Buyer',
      description: 'Visit 10 properties in a month',
      icon: Calendar,
      progress: badges.length,
      maxProgress: 10,
      reward: '500 Points + VIP Status',
      unlocked: badges.length >= 10
    },
    {
      id: 'trusted',
      title: 'Trusted Member',
      description: 'Maintain 100% visit completion rate',
      icon: Shield,
      progress: badges.length > 0 ? 100 : 0,
      maxProgress: 100,
      reward: 'Priority Access + Trusted Badge',
      unlocked: badges.length > 0
    }
  ];

  const calculateUserLevel = (points: number) => {
    if (points < 100) return { level: 1, nextLevel: 100 };
    if (points < 300) return { level: 2, nextLevel: 300 };
    if (points < 600) return { level: 3, nextLevel: 600 };
    if (points < 1000) return { level: 4, nextLevel: 1000 };
    return { level: 5, nextLevel: 1500 };
  };

  useEffect(() => {
    const levelInfo = calculateUserLevel(totalPoints);
    setUserLevel(levelInfo.level);
    setNextLevelPoints(levelInfo.nextLevel);
  }, [totalPoints]);

  return (
    <div className="space-y-6">
      {/* Enhanced Header */}
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4 mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 mb-1">My Visit Badges</h1>
            <p className="text-gray-600">Track your verified property visits and earn exclusive rewards</p>
          </div>
          <div className="flex items-center gap-4">
            <div className="bg-gradient-to-r from-blue-500 to-indigo-600 text-white rounded-lg p-4 text-center">
              <div className="flex items-center gap-2 mb-1">
                <Crown className="w-5 h-5" />
                <div>
                  <div className="text-xl font-bold">{totalPoints}</div>
                  <div className="text-xs opacity-90">Points</div>
                </div>
              </div>
              <div className="bg-white/20 rounded px-2 py-1 text-xs font-medium">
                Level {userLevel}
              </div>
            </div>
            <Button
              onClick={() => navigate('/user/dashboard?section=EarnBadge')}
              className="bg-white border-2 border-gray-800 text-black hover:bg-gray-50 px-6 py-3 font-bold rounded-lg shadow-lg hover:shadow-xl transition-all"
            >
              <Award className="w-5 h-5 mr-2 text-black" />
              Earn Badge Now
            </Button>
          </div>
        </div>
        
        {/* Progress to Next Level */}
        <div className="bg-gray-50 rounded-lg p-4 border">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-semibold text-gray-800">Progress to Level {userLevel + 1}</span>
            <span className="text-sm font-medium text-gray-700">{totalPoints}/{nextLevelPoints} points</span>
          </div>
          <Progress 
            value={(totalPoints / nextLevelPoints) * 100} 
            className="h-2"
          />
          <div className="mt-1 text-xs text-gray-600">
            {nextLevelPoints - totalPoints} points needed for next level
          </div>
        </div>
      </div>

      {/* Navigation Tabs */}
      <div className="bg-white rounded-lg shadow-sm border">
        <div className="flex">
          {[
            { id: 'overview', label: 'Overview', icon: BarChart3 },
            { id: 'categories', label: 'Badge Categories', icon: Award },
            { id: 'achievements', label: 'Achievements', icon: Trophy },
            { id: 'leaderboard', label: 'Leaderboard', icon: Crown }
          ].map((tab) => {
            const Icon = tab.icon;
            return (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id as any)}
                className={`flex-1 flex items-center justify-center gap-2 px-4 py-3 font-medium transition-all border-b-2 ${
                  activeTab === tab.id
                    ? 'border-blue-600 text-blue-600 bg-blue-50'
                    : 'border-transparent text-gray-600 hover:text-blue-600 hover:bg-gray-50'
                }`}
              >
                <Icon className="w-4 h-4" />
                <span className="hidden sm:inline">{tab.label}</span>
              </button>
            );
          })}
        </div>

      {/* Tab Content */}
      {activeTab === 'overview' && (
        <div className="p-6">
          {/* Quick Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 text-center">
              <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center mx-auto mb-2">
                <Award className="w-5 h-5 text-white" />
              </div>
              <div className="text-2xl font-bold text-gray-900">{badges.length}</div>
              <div className="text-sm text-gray-600">Badges Earned</div>
            </div>

            <div className="bg-green-50 border border-green-200 rounded-lg p-4 text-center">
              <div className="w-10 h-10 bg-green-600 rounded-full flex items-center justify-center mx-auto mb-2">
                <CheckCircle className="w-5 h-5 text-white" />
              </div>
              <div className="text-2xl font-bold text-gray-900">{badges.length}</div>
              <div className="text-sm text-gray-600">Verified Visits</div>
            </div>

            <div className="bg-purple-50 border border-purple-200 rounded-lg p-4 text-center">
              <div className="w-10 h-10 bg-purple-600 rounded-full flex items-center justify-center mx-auto mb-2">
                <TrendingUp className="w-5 h-5 text-white" />
              </div>
              <div className="text-2xl font-bold text-gray-900">Level {userLevel}</div>
              <div className="text-sm text-gray-600">Current Level</div>
            </div>

            <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 text-center">
              <div className="w-10 h-10 bg-yellow-600 rounded-full flex items-center justify-center mx-auto mb-2">
                <Star className="w-5 h-5 text-white" />
              </div>
              <div className="text-2xl font-bold text-gray-900">{achievements.filter(a => a.unlocked).length}</div>
              <div className="text-sm text-gray-600">Achievements</div>
            </div>
          </div>

          {/* Recent Activity */}
          <div className="bg-white border rounded-lg p-6">
            <h3 className="text-lg font-bold text-gray-900 mb-4 flex items-center gap-2">
              <Clock className="w-5 h-5 text-blue-600" />
              Recent Activity
            </h3>
            {badges.length > 0 ? (
              <div className="space-y-3">
                {badges.slice(0, 3).map((badge) => (
                  <div key={badge.id} className="flex items-center gap-4 p-4 bg-gray-50 rounded-lg border">
                    <div className={`w-12 h-12 bg-gradient-to-br ${getBadgeColor(badge.badgeLevel)} rounded-full flex items-center justify-center`}>
                      <Award className="w-6 h-6 text-white" />
                    </div>
                    <div className="flex-1">
                      <div className="font-medium text-gray-900">{badge.badgeType}</div>
                      <div className="text-sm text-gray-600">{badge.propertyName}</div>
                    </div>
                    <div className="text-right">
                      <div className="font-bold text-green-600">+{badge.earnedPoints}</div>
                      <div className="text-xs text-gray-500">{new Date(badge.timestamp).toLocaleDateString()}</div>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-8">
                <Clock className="w-12 h-12 mx-auto mb-3 text-gray-300" />
                <h4 className="text-lg font-semibold text-gray-600 mb-2">No Recent Activity</h4>
                <p className="text-gray-500 mb-4">Start visiting properties to see your progress here!</p>
                <Button
                  onClick={() => navigate('/user/dashboard?section=EarnBadge')}
                  className="bg-white border-2 border-gray-800 text-black hover:bg-gray-50 px-6 py-3 rounded-lg"
                >
                  <Award className="w-5 h-5 mr-2 text-black" />
                  Earn Your First Badge
                </Button>
              </div>
            )}
          </div>
        </div>
      )}

      {activeTab === 'categories' && (
        <div className="space-y-6">
          {badgeCategories.map((category) => {
            const Icon = category.icon;
            const completionRate = (category.badges.length / category.totalPossible) * 100;
            
            return (
              <Card key={category.id} className="p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex items-center gap-4">
                    <div className={`w-12 h-12 rounded-full flex items-center justify-center ${
                      category.color === 'blue' ? 'bg-blue-600' :
                      category.color === 'green' ? 'bg-green-600' :
                      category.color === 'purple' ? 'bg-purple-600' :
                      'bg-gray-600'
                    }`}>
                      <Icon className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <h3 className="text-xl font-bold text-gray-900">{category.name}</h3>
                      <p className="text-gray-600">{category.description}</p>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="text-2xl font-bold text-gray-900">{category.badges.length}/{category.totalPossible}</div>
                    <div className="text-sm text-gray-600">Completed</div>
                  </div>
                </div>
                
                <div className="mb-4">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-sm font-medium text-gray-700">Progress</span>
                    <span className="text-sm text-gray-600">{Math.round(completionRate)}%</span>
                  </div>
                  <Progress value={completionRate} className="h-2" />
                </div>
                
                <div className="space-y-2">
                  <h4 className="font-medium text-gray-900">Requirements:</h4>
                  <ul className="space-y-1">
                    {category.requirements.map((req, index) => (
                      <li key={index} className="flex items-center gap-2 text-sm text-gray-600">
                        <CheckCircle className="w-4 h-4 text-green-500" />
                        {req}
                      </li>
                    ))}
                  </ul>
                </div>
              </Card>
            );
          })}
        </div>
      )}

        {activeTab === 'achievements' && (
          <div className="p-8">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {achievements.map((achievement) => {
                const Icon = achievement.icon;
                const progressPercentage = (achievement.progress / achievement.maxProgress) * 100;
                
                return (
                  <div key={achievement.id} className={`bg-white rounded-xl shadow-lg border-2 p-8 transition-all hover:shadow-xl ${
                    achievement.unlocked 
                      ? 'border-green-300 bg-gradient-to-br from-green-50 to-emerald-50' 
                      : 'border-gray-200'
                  }`}>
                    <div className="flex items-start gap-6">
                      <div className={`w-16 h-16 rounded-full flex items-center justify-center shadow-lg ${
                        achievement.unlocked 
                          ? 'bg-green-600 text-white' 
                          : 'bg-gray-300 text-gray-600'
                      }`}>
                        <Icon className="w-8 h-8" />
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-3">
                          <h3 className="text-xl font-bold text-gray-900">{achievement.title}</h3>
                          {achievement.unlocked && (
                            <Badge variant="secondary" className="bg-green-100 text-green-800 text-sm px-3 py-1">
                              Unlocked
                            </Badge>
                          )}
                        </div>
                        <p className="text-base text-gray-600 mb-4">{achievement.description}</p>
                        
                        <div className="space-y-3">
                          <div className="flex items-center justify-between">
                            <span className="text-base font-semibold text-gray-800">Progress</span>
                            <span className="text-base font-medium text-gray-700">{achievement.progress}/{achievement.maxProgress}</span>
                          </div>
                          <Progress value={progressPercentage} className="h-3" />
                        </div>
                        
                        <div className="mt-4 p-4 bg-white rounded-xl border-2 border-gray-100">
                          <div className="text-sm font-bold text-gray-800 mb-2">Reward:</div>
                          <div className="text-base text-blue-600 font-semibold">{achievement.reward}</div>
                        </div>
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        )}

        {activeTab === 'leaderboard' && (
        <Card className="p-6">
          <h3 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
            <Crown className="w-6 h-6 text-yellow-500" />
            Community Leaderboard
          </h3>
          
          <div className="space-y-4">
            {/* Mock leaderboard data */}
            {[
              { rank: 1, name: 'You', points: totalPoints, badges: badges.length, avatar: '👤' },
              { rank: 2, name: 'Sarah M.', points: 850, badges: 12, avatar: '👩' },
              { rank: 3, name: 'John D.', points: 720, badges: 10, avatar: '👨' },
              { rank: 4, name: 'Mike R.', points: 650, badges: 9, avatar: '👨‍💼' },
              { rank: 5, name: 'Lisa K.', points: 580, badges: 8, avatar: '👩‍💼' }
            ].map((user) => (
              <div key={user.rank} className={`flex items-center gap-4 p-4 rounded-lg ${
                user.name === 'You' ? 'bg-indigo-50 border-2 border-indigo-200' : 'bg-gray-50'
              }`}>
                <div className={`w-8 h-8 rounded-full flex items-center justify-center font-bold ${
                  user.rank === 1 ? 'bg-yellow-500 text-white' :
                  user.rank === 2 ? 'bg-gray-400 text-white' :
                  user.rank === 3 ? 'bg-orange-500 text-white' :
                  'bg-gray-300 text-gray-700'
                }`}>
                  {user.rank}
                </div>
                <div className="w-10 h-10 bg-gray-200 rounded-full flex items-center justify-center text-gray-700 font-bold">{user.avatar}</div>
                <div className="flex-1">
                  <div className="font-medium text-gray-900">{user.name}</div>
                  <div className="text-sm text-gray-600">{user.badges} badges earned</div>
                </div>
                <div className="text-right">
                  <div className="font-bold text-indigo-600">{user.points}</div>
                  <div className="text-xs text-gray-500">points</div>
                </div>
              </div>
            ))}
          </div>
          
          <div className="mt-6 p-4 bg-blue-50 rounded-lg border border-blue-200">
            <p className="text-sm text-blue-800">
              <strong>Tip:</strong> Climb the leaderboard by visiting more properties and completing achievements!
            </p>
          </div>
        </Card>
      )}
      </div>



      {/* How to Earn Badges - Only show in overview */}
      {activeTab === 'overview' && (
        <Card className="p-6 bg-gradient-to-br from-indigo-50 to-purple-50 border-indigo-200">
          <h3 className="text-xl font-bold mb-6 flex items-center gap-2 text-gray-900">
            <Zap className="w-6 h-6 text-indigo-600" />
            How to Earn Badges & Points
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="bg-white border border-indigo-200 rounded-xl p-6 text-center">
              <div className="w-16 h-16 bg-indigo-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <MapPin className="w-8 h-8 text-white" />
              </div>
              <div className="text-2xl font-bold mb-2 text-indigo-600">1</div>
              <div className="font-semibold mb-2 text-gray-900">Schedule a Visit</div>
              <div className="text-sm text-gray-600">Book a property visit through the platform and get ready to explore</div>
            </div>
            <div className="bg-white border border-green-200 rounded-xl p-6 text-center">
              <div className="w-16 h-16 bg-green-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <QrCode className="w-8 h-8 text-white" />
              </div>
              <div className="text-2xl font-bold mb-2 text-green-600">2</div>
              <div className="font-semibold mb-2 text-gray-900">Check-In at Property</div>
              <div className="text-sm text-gray-600">Use QR code or GPS verification when you arrive at the location</div>
            </div>
            <div className="bg-white border border-purple-200 rounded-xl p-6 text-center">
              <div className="w-16 h-16 bg-purple-600 rounded-full flex items-center justify-center mx-auto mb-4">
                <Trophy className="w-8 h-8 text-white" />
              </div>
              <div className="text-2xl font-bold mb-2 text-purple-600">3</div>
              <div className="font-semibold mb-2 text-gray-900">Earn Rewards</div>
              <div className="text-sm text-gray-600">Get verified, earn points, unlock badges, and climb the leaderboard!</div>
            </div>
          </div>
        </Card>
      )}

      {/* Earned Badges - Only show in overview when there are badges */}
      {activeTab === 'overview' && badges.length > 0 && (
        <div className="space-y-6">
          <div className="flex items-center justify-between">
            <h3 className="text-2xl font-bold text-gray-900">Your Latest Badges</h3>
            <Button 
              variant="outline" 
              onClick={() => setActiveTab('categories')}
              className="text-indigo-600 border-indigo-200 hover:bg-indigo-50"
            >
              View All Categories
            </Button>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {badges.slice(0, 4).map((badge) => (
              <Card key={badge.id} className="p-6 hover:shadow-lg transition-all hover:scale-105">
                <div className="flex items-start gap-4">
                  {/* Badge Icon */}
                  <div className={`w-16 h-16 bg-gradient-to-br ${getBadgeColor(badge.badgeLevel)} rounded-full flex items-center justify-center flex-shrink-0 shadow-lg`}>
                    <Award className="w-8 h-8 text-white" />
                  </div>

                  {/* Badge Details */}
                  <div className="flex-1">
                    <div className="flex items-start justify-between mb-2">
                      <div>
                        <h4 className="text-lg font-bold text-gray-900">{badge.badgeType}</h4>
                        <Badge variant="secondary" className="mt-1">
                          {badge.badgeLevel}
                        </Badge>
                      </div>
                      <div className="text-right">
                        <div className="text-xl font-bold text-green-600">+{badge.earnedPoints}</div>
                        <div className="text-xs text-gray-500">Points</div>
                      </div>
                    </div>

                    {/* Property Info */}
                    <div className="bg-gray-50 rounded-lg p-3 mb-3">
                      <div className="flex items-center gap-2 mb-1">
                        <MapPin className="w-4 h-4 text-gray-600" />
                        <span className="font-medium text-gray-900 text-sm">{badge.propertyName}</span>
                      </div>
                      <div className="text-xs text-gray-600 ml-6">{badge.propertyAddress}</div>
                    </div>

                    {/* Verification Details */}
                    <div className="flex items-center justify-between text-xs text-gray-500">
                      <div className="flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        <span>{new Date(badge.timestamp).toLocaleDateString()}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <CheckCircle className="w-3 h-3 text-green-600" />
                        <span>{badge.verificationMethod.join(', ')}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </Card>
            ))}
          </div>
        </div>
      )}

      {/* Empty State - Only show in overview when no badges */}
      {activeTab === 'overview' && badges.length === 0 && (
        <div className="bg-white rounded-lg shadow-sm border p-8 text-center">
          <div className="max-w-lg mx-auto">
            <div className="w-24 h-24 bg-gradient-to-br from-indigo-100 to-purple-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <Award className="w-12 h-12 text-indigo-400" />
            </div>
            <h3 className="text-2xl font-bold text-gray-900 mb-4">Start Your Badge Journey!</h3>
            <p className="text-gray-600 mb-6">
              Visit properties, earn badges, and unlock exclusive rewards. Build your credibility as a trusted buyer in our community.
            </p>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
              <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-4">
                <Medal className="w-8 h-8 text-indigo-600 mx-auto mb-2" />
                <div className="font-semibold text-gray-900 mb-1">Earn Badges</div>
                <div className="text-sm text-gray-600">Complete visits and challenges</div>
              </div>
              <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                <Gift className="w-8 h-8 text-green-600 mx-auto mb-2" />
                <div className="font-semibold text-gray-900 mb-1">Unlock Rewards</div>
                <div className="text-sm text-gray-600">Get exclusive benefits and offers</div>
              </div>
              <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
                <Crown className="w-8 h-8 text-purple-600 mx-auto mb-2" />
                <div className="font-semibold text-gray-900 mb-1">Build Trust</div>
                <div className="text-sm text-gray-600">Show sellers you're serious</div>
              </div>
            </div>
            
            <div className="flex flex-col sm:flex-row gap-3 justify-center">
              <Button
                onClick={() => navigate('/user/dashboard?section=EarnBadge')}
                className="bg-white border-2 border-gray-800 text-black hover:bg-gray-50 px-6 py-3 font-bold rounded-lg"
              >
                <Award className="w-5 h-5 mr-2 text-black" />
                Earn Badge Now
              </Button>
              <Button
                onClick={() => window.location.href = '/properties'}
                variant="outline"
                className="border-blue-600 text-blue-600 hover:bg-blue-50 px-6 py-3 font-bold rounded-lg"
              >
                <MapPin className="w-5 h-5 mr-2" />
                Browse Properties
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Check-In Modal */}
      {showCheckInModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-start justify-center z-50 p-4 overflow-y-auto">
          <div className="w-full max-w-2xl my-4">
            <Card className="w-full max-h-[calc(100vh-2rem)] overflow-y-auto">
              <div className="p-6">
              {/* Modal Header */}
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-bold text-gray-900">
                  {!selectedProperty ? 'Select Property to Visit' : 'Check-In to Earn Badge'}
                </h2>
                <button
                  onClick={() => {
                    setShowCheckInModal(false);
                    setVerificationStatus('idle');
                    setEarnedBadge(null);
                    setSelectedProperty(null);
                    setGpsVerified(false);
                    setLocationPermission('prompt');
                  }}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                  </svg>
                </button>
              </div>

              {/* Success State - Badge Earned */}
              {verificationStatus === 'success' && earnedBadge && (
                <div className="text-center py-8">
                  <div className="mb-6 animate-bounce">
                    <div className="w-32 h-32 bg-gradient-to-br from-yellow-400 to-orange-500 rounded-full flex items-center justify-center mx-auto shadow-2xl">
                      <Award className="w-16 h-16 text-white" />
                    </div>
                  </div>
                  <h3 className="text-3xl font-bold text-gray-900 mb-2">Badge Earned!</h3>
                  <p className="text-lg text-gray-600 mb-6">Congratulations! You've earned a verified visit badge</p>
                  
                  <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl p-6 mb-6">
                    <div className="text-left space-y-3">
                      <div className="flex items-center gap-2">
                        <Award className="w-5 h-5 text-blue-600" />
                        <span className="font-semibold text-gray-900">{earnedBadge.badgeType}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <MapPin className="w-5 h-5 text-gray-600" />
                        <span className="text-gray-700">{earnedBadge.propertyName}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Star className="w-5 h-5 text-yellow-500" />
                        <span className="text-gray-700">+{earnedBadge.earnedPoints} Points</span>
                      </div>
                    </div>
                  </div>

                  <Button
                    onClick={() => {
                      setShowCheckInModal(false);
                      setVerificationStatus('idle');
                      setEarnedBadge(null);
                    }}
                    className="bg-blue-600 hover:bg-blue-700 text-white px-8 py-3"
                  >
                    View All Badges
                  </Button>
                </div>
              )}

              {/* Property Selection Step */}
              {!selectedProperty && verificationStatus !== 'success' && (
                <div className="space-y-4">
                  {loadingProperties ? (
                    <div className="text-center py-12">
                      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                      <p className="text-gray-600 mt-4">Loading properties...</p>
                    </div>
                  ) : properties.length === 0 ? (
                    <div className="text-center py-12">
                      <p className="text-gray-600">No properties available for visit</p>
                    </div>
                  ) : (
                    <div className="grid grid-cols-1 gap-4 max-h-96 overflow-y-auto">
                      {properties.map((property) => (
                        <div
                          key={property.id}
                          onClick={() => {
                            setSelectedProperty(property);
                            generatePropertyQRCode(property);
                          }}
                          className="border-2 border-gray-200 rounded-lg p-4 hover:border-blue-600 cursor-pointer transition-all hover:shadow-md"
                        >
                          <div className="flex gap-4">
                            {property.primary_image ? (
                              <img 
                                src={property.primary_image} 
                                alt={property.title} 
                                className="w-24 h-24 object-cover rounded-lg"
                                onError={(e) => {
                                  e.currentTarget.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="96" height="96" viewBox="0 0 24 24" fill="none" stroke="%239CA3AF" stroke-width="2"%3E%3Crect x="3" y="3" width="18" height="18" rx="2" ry="2"%3E%3C/rect%3E%3Ccircle cx="8.5" cy="8.5" r="1.5"%3E%3C/circle%3E%3Cpolyline points="21 15 16 10 5 21"%3E%3C/polyline%3E%3C/svg%3E';
                                }}
                              />
                            ) : (
                              <div className="w-24 h-24 bg-gray-200 rounded-lg flex items-center justify-center">
                                <MapPin className="w-8 h-8 text-gray-400" />
                              </div>
                            )}
                            <div className="flex-1">
                              <h3 className="font-bold text-gray-900">{property.title}</h3>
                              <p className="text-sm text-gray-600">{property.address}, {property.city}</p>
                              <p className="text-sm text-blue-600 mt-1">₹{property.sale_price?.toLocaleString()}</p>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              {/* Verification Methods */}
              {selectedProperty && verificationStatus !== 'success' && (
                <>
                  {/* Method Selection */}
                  <div className="flex gap-4 mb-6">
                    <button
                      onClick={() => setCheckInMethod('qr')}
                      className={`flex-1 p-4 rounded-lg border-2 transition-all ${
                        checkInMethod === 'qr'
                          ? 'border-blue-600 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <QrCode className={`w-8 h-8 mx-auto mb-2 ${
                        checkInMethod === 'qr' ? 'text-blue-600' : 'text-gray-600'
                      }`} />
                      <div className="font-semibold text-gray-900">QR Code</div>
                      <div className="text-xs text-gray-600">Scan property QR</div>
                    </button>
                    <button
                      onClick={() => setCheckInMethod('gps')}
                      className={`flex-1 p-4 rounded-lg border-2 transition-all ${
                        checkInMethod === 'gps'
                          ? 'border-blue-600 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <Navigation className={`w-8 h-8 mx-auto mb-2 ${
                        checkInMethod === 'gps' ? 'text-blue-600' : 'text-gray-600'
                      }`} />
                      <div className="font-semibold text-gray-900">GPS Location</div>
                      <div className="text-xs text-gray-600">Use your location</div>
                    </button>
                  </div>

                  {/* QR Code Display */}
                  {checkInMethod === 'qr' && (
                    <div className="space-y-4">
                      <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-6 text-center">
                        <h3 className="text-lg font-bold text-gray-900 mb-4">Property QR Code</h3>
                        {propertyQRCode && (
                          <img src={propertyQRCode} alt="Property QR" className="mx-auto mb-4 border-4 border-white shadow-lg rounded-lg w-48 h-48" />
                        )}
                        <p className="text-sm text-gray-600 mb-4">Scan this QR code at the property</p>
                        <Button
                          onClick={confirmQRScan}
                          className="bg-gray-900 hover:bg-black text-white w-full"
                        >
                          Confirm Visit & Earn Badge
                        </Button>
                      </div>
                    </div>
                  )}

                  {/* GPS Verification */}
                  {checkInMethod === 'gps' && (
                    <div className="space-y-4">
                      <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-8 text-center">
                        <Navigation className="w-16 h-16 text-blue-600 mx-auto mb-4" />
                        <h3 className="text-xl font-bold text-gray-900 mb-2">Location Verification</h3>
                        <p className="text-gray-600 mb-6">We need your location to verify you're at the property</p>
                        
                        {locationPermission === 'prompt' && (
                          <Button
                            onClick={async () => {
                              try {
                                setVerificationStatus('verifying');
                                toast.info('Getting your location...');
                                
                                const location: Coordinates = { lat: 0, lng: 0 }; // Placeholder
                                setUserLocation(location);
                                setLocationPermission('granted');
                                
                                const propertyLocation: Coordinates = {
                                  lat: parseFloat(selectedProperty.latitude) || 0,
                                  lng: parseFloat(selectedProperty.longitude) || 0
                                };
                                
                                if (!propertyLocation.lat || !propertyLocation.lng) {
                                  toast.error('Property location not available');
                                  setVerificationStatus('error');
                                  return;
                                }
                                
                                const withinGeofence = true; // Simplified verification
                                
                                if (withinGeofence) {
                                  setGpsVerified(true);
                                  setVerificationStatus('idle');
                                  toast.success('Location verified! Click confirm to earn your badge.');
                                } else {
                                  setVerificationStatus('error');
                                  setGpsVerified(false);
                                  toast.error('You are not at the property location. Please move closer.');
                                }
                              } catch (error) {
                                setLocationPermission('denied');
                                setVerificationStatus('error');
                                toast.error('Location permission denied. Please enable location access.');
                              }
                            }}
                            className="bg-black hover:bg-gray-800 text-white w-full py-3"
                          >
                            Enable Location
                          </Button>
                        )}

                        {gpsVerified && locationPermission === 'granted' && verificationStatus === 'idle' && (
                          <div className="space-y-4">
                            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                              <p className="text-green-800 text-sm font-semibold">Location Verified Successfully!</p>
                              <p className="text-green-700 text-sm mt-1">You are at the property. Click confirm to earn your badge.</p>
                            </div>
                            <Button
                              onClick={() => {
                                setVerificationStatus('verifying');
                                toast.success('Earning your badge...');
                                setTimeout(() => {
                                  const newBadge: BadgeData = {
                                    id: `badge-${Date.now()}`,
                                    propertyName: selectedProperty.title,
                                    propertyAddress: `${selectedProperty.address}, ${selectedProperty.city}`,
                                    badgeType: 'Verified Property Visit',
                                    badgeLevel: 'Level 1',
                                    timestamp: new Date().toISOString(),
                                    earnedPoints: 50,
                                    verificationMethod: ['GPS Location'],
                                  };
                                  setEarnedBadge(newBadge);
                                  const existingBadges = JSON.parse(localStorage.getItem('visitBadges') || '[]');
                                  localStorage.setItem('visitBadges', JSON.stringify([...existingBadges, newBadge]));
                                  setBadges([...badges, newBadge]);
                                  setTotalPoints(totalPoints + 50);
                                  setVerificationStatus('success');
                                  setGpsVerified(false);
                                }, 1000);
                              }}
                              className="bg-green-600 hover:bg-green-700 w-full"
                            >
                              Confirm Visit & Earn Badge
                            </Button>
                          </div>
                        )}

                        {locationPermission === 'denied' && (
                          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                            <p className="text-red-800 text-sm">Location permission denied. Please enable location in your browser settings.</p>
                          </div>
                        )}

                        {verificationStatus === 'verifying' && locationPermission === 'granted' && (
                          <div className="space-y-3">
                            <div className="flex items-center justify-center gap-3">
                              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                              <span className="text-gray-700">Verifying your location...</span>
                            </div>
                            {userLocation && (
                              <div className="text-sm text-gray-600 text-center">
                                <p>Your coordinates: {userLocation.lat.toFixed(6)}, {userLocation.lng.toFixed(6)}</p>
                              </div>
                            )}
                          </div>
                        )}
                        
                        {verificationStatus === 'error' && (
                          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                            <p className="text-red-800 text-sm font-semibold">Verification Failed</p>
                            <p className="text-red-700 text-sm mt-1">You must be within 100 meters of the property to check-in.</p>
                          </div>
                        )}
                      </div>

                      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <p className="text-sm text-yellow-800">
                          <strong>Note:</strong> You must be within 100 meters of the property to check-in
                        </p>
                      </div>
                    </div>
                  )}
                </>
              )}
            </div>
          </Card>
          </div>
        </div>
      )}

      {/* Benefits Section - Only show in overview */}
      {activeTab === 'overview' && (
        <Card className="p-8 bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 border-emerald-200">
          <h3 className="text-2xl font-bold text-gray-900 mb-6 flex items-center gap-3">
            <Star className="w-8 h-8 text-yellow-500" />
            Exclusive Badge Benefits
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <div className="bg-white rounded-xl p-6 shadow-sm border border-emerald-100">
              <div className="w-12 h-12 bg-emerald-600 rounded-full flex items-center justify-center mb-4">
                <Shield className="w-6 h-6 text-white" />
              </div>
              <div className="font-bold text-gray-900 mb-2">Build Trust</div>
              <div className="text-sm text-gray-600">Verified badges show sellers you're a serious and committed buyer</div>
            </div>
            <div className="bg-white rounded-xl p-6 shadow-sm border border-emerald-100">
              <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center mb-4">
                <Zap className="w-6 h-6 text-white" />
              </div>
              <div className="font-bold text-gray-900 mb-2">Priority Access</div>
              <div className="text-sm text-gray-600">Get early access to new listings and exclusive premium properties</div>
            </div>
            <div className="bg-white rounded-xl p-6 shadow-sm border border-emerald-100">
              <div className="w-12 h-12 bg-purple-600 rounded-full flex items-center justify-center mb-4">
                <Gift className="w-6 h-6 text-white" />
              </div>
              <div className="font-bold text-gray-900 mb-2">Earn Rewards</div>
              <div className="text-sm text-gray-600">Accumulate points for discounts, cashbacks, and special offers</div>
            </div>
            <div className="bg-white rounded-xl p-6 shadow-sm border border-emerald-100">
              <div className="w-12 h-12 bg-orange-600 rounded-full flex items-center justify-center mb-4">
                <TrendingUp className="w-6 h-6 text-white" />
              </div>
              <div className="font-bold text-gray-900 mb-2">Faster Approvals</div>
              <div className="text-sm text-gray-600">Verified users get quicker visit approvals and priority responses</div>
            </div>
          </div>
        </Card>
      )}
    </div>
  );
};
