import React from 'react';
import { useParams, Navigate } from 'react-router-dom';
import { useNegotiation } from '../hooks/useNegotiation';
import { UserRole } from '../types/negotiation';
import { negotiationApi } from '../api/negotiation.api';
import NegotiationStatusBadge from '../components/negotiation/NegotiationStatusBadge';
import NegotiationTimeline from '../components/negotiation/NegotiationTimeline';
import AISuggestionBox from '../components/negotiation/AISuggestionBox';
import OfferComposer from '../components/negotiation/OfferComposer';
import AudioCall from '../components/negotiation/AudioCall';

interface NegotiationPageProps {
  userRole: UserRole; // This should come from your auth context/props
}

const NegotiationPage: React.FC<NegotiationPageProps> = ({ userRole }) => {
  const { negotiationId } = useParams<{ negotiationId: string }>();

  if (!negotiationId) {
    return <Navigate to="/" replace />;
  }

  const {
    timeline,
    loading,
    error,
    submitOffer,
    acceptOffer,
    rejectOffer,
    refresh,
    canAcceptOffer,
    canRejectOffer,
    canMakeOffer,
  } = useNegotiation(negotiationId, userRole);

  const handleAcceptOffer = async (eventId: string) => {
    try {
      await acceptOffer(eventId);
    } catch (error) {
      // Error is handled in the hook
      console.error('Failed to accept offer:', error);
    }
  };

  const handleRejectOffer = async (eventId: string) => {
    try {
      await rejectOffer(eventId);
    } catch (error) {
      // Error is handled in the hook
      console.error('Failed to reject offer:', error);
    }
  };

  const handleSubmitOffer = async (amount: number, message?: string) => {
    await submitOffer(amount, message);
  };

  // Get current user ID for call functionality
  const getCurrentUserId = () => {
    try {
      const token = localStorage.getItem('access_token') || sessionStorage.getItem('access_token');
      if (token) {
        const payload = JSON.parse(atob(token.split('.')[1]));
        return payload.user_id;
      }
    } catch (error) {
      console.error('Failed to get user ID from token:', error);
    }
    return null;
  };

  const currentUserId = getCurrentUserId();
  
  // Check if calls are allowed (active negotiation)
  const canMakeCall = () => {
    if (!timeline) return false;
    const currentStatus = timeline.status;
    return !['ACCEPTED', 'REJECTED', 'EXPIRED'].includes(currentStatus);
  };

  if (loading && !timeline) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading negotiation...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center max-w-md mx-auto p-6">
          <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <svg className="w-8 h-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
          </div>
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Error Loading Negotiation</h2>
          <p className="text-gray-600 mb-4">{error}</p>
          <button
            onClick={refresh}
            className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            Try Again
          </button>
        </div>
      </div>
    );
  }

  if (!timeline) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-600">Negotiation not found</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-6">
          <div className="flex items-center justify-between">
            <h1 className="text-2xl font-bold text-gray-900">
              Property Negotiation
            </h1>
            <div className="flex gap-2">
              <button
                onClick={refresh}
                disabled={loading}
                className="px-3 py-2 text-sm bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50"
              >
                {loading ? 'Refreshing...' : 'Refresh'}
              </button>
              <button
                onClick={async () => {
                  if (confirm('Clear all negotiations? This will delete all your negotiation history.')) {
                    try {
                      await negotiationApi.clearAllNegotiations();
                      alert('All negotiations cleared successfully');
                      window.location.href = '/';
                    } catch (error: any) {
                      console.error('Clear negotiations error:', error);
                      const errorMsg = error.response?.data?.detail || error.message || 'Failed to clear negotiations';
                      alert(errorMsg);
                    }
                  }
                }}
                className="px-3 py-2 text-sm bg-red-600 text-white rounded-md hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500"
              >
                Clear All
              </button>
            </div>
          </div>
          <p className="text-gray-600 mt-1">Negotiation ID: {negotiationId}</p>
        </div>

        <div className="space-y-6">
          {/* Status Badge */}
          <NegotiationStatusBadge status={timeline.status} />

          {/* Timeline */}
          <NegotiationTimeline
            events={timeline.events}
            canAcceptOffer={canAcceptOffer}
            canRejectOffer={canRejectOffer}
            onAcceptOffer={handleAcceptOffer}
            onRejectOffer={handleRejectOffer}
            loading={loading}
          />

          {/* AI Suggestions */}
          <AISuggestionBox aiInsight={timeline.ai_insight} />

          {/* Audio Call */}
          {currentUserId && (
            <AudioCall
              negotiationId={negotiationId}
              userId={currentUserId}
              canMakeCall={canMakeCall()}
              disabled={loading}
            />
          )}

          {/* Offer Composer */}
          <OfferComposer
            onSubmitOffer={handleSubmitOffer}
            canMakeOffer={canMakeOffer()}
            loading={loading}
          />
        </div>
      </div>
    </div>
  );
};

export default NegotiationPage;