import { useEffect, useState } from 'react';
import { useNavigate, useSearchParams, useParams } from 'react-router-dom';
import { adminService } from '../api/adminService';
import { toast } from 'sonner';

export const OAuthCallback = () => {
    const [searchParams] = useSearchParams();
    const navigate = useNavigate();
    const { provider } = useParams<{ provider: string }>();
    const [status, setStatus] = useState<'processing' | 'success' | 'error'>('processing');

    useEffect(() => {
        const handleCallback = async () => {
            const code = searchParams.get('code');
            const state = searchParams.get('state');
            const error = searchParams.get('error');

            if (error) {
                toast.error(`OAuth Error: ${error}`);
                setStatus('error');
                setTimeout(() => navigate('/admin/dashboard/developer/marketplace'), 3000);
                return;
            }

            if (!code || !state) {
                toast.error('Missing OAuth parameters');
                setStatus('error');
                setTimeout(() => navigate('/admin/dashboard/developer/marketplace'), 3000);
                return;
            }

            try {
                const response = await adminService.connectors.handleOAuthCallback({ code, state });
                
                if (response.data.success) {
                    toast.success(`${provider} connected successfully!`);
                    setStatus('success');
                    setTimeout(() => navigate('/admin/dashboard/developer/marketplace'), 2000);
                } else {
                    toast.error('OAuth connection failed');
                    setStatus('error');
                    setTimeout(() => navigate('/admin/dashboard/developer/marketplace'), 3000);
                }
            } catch (error: any) {
                console.error('OAuth callback error:', error);
                toast.error(error.response?.data?.detail || 'Failed to complete OAuth');
                setStatus('error');
                setTimeout(() => navigate('/admin/dashboard/developer/marketplace'), 3000);
            }
        };

        handleCallback();
    }, [searchParams, navigate, provider]);

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50">
            <div className="text-center">
                {status === 'processing' && (
                    <>
                        <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-purple-600 mx-auto mb-4"></div>
                        <h2 className="text-xl font-bold text-gray-900">Connecting {provider}...</h2>
                        <p className="text-gray-600 mt-2">Please wait while we complete the authorization</p>
                    </>
                )}
                {status === 'success' && (
                    <>
                        <div className="text-green-600 text-6xl mb-4">✓</div>
                        <h2 className="text-xl font-bold text-gray-900">Connected Successfully!</h2>
                        <p className="text-gray-600 mt-2">Redirecting to marketplace...</p>
                    </>
                )}
                {status === 'error' && (
                    <>
                        <div className="text-red-600 text-6xl mb-4">✗</div>
                        <h2 className="text-xl font-bold text-gray-900">Connection Failed</h2>
                        <p className="text-gray-600 mt-2">Redirecting back to marketplace...</p>
                    </>
                )}
            </div>
        </div>
    );
};
