import React, { useState, useEffect } from 'react';
import PipelineBoard from '../components/Pipeline/PipelineBoard';
import { pipelineService } from '../services/pipelineService';
import { toast } from 'react-hot-toast';
import { 
  TrendingUp, 
  Users, 
  Clock, 
  DollarSign, 
  AlertTriangle,
  CheckCircle,
  BarChart3,
  Calendar
} from 'lucide-react';

interface DashboardStats {
  totalLeads: number;
  activeLeads: number;
  overdueLeads: number;
  closedThisMonth: number;
  conversionRate: number;
  avgDealValue: number;
  avgTimeToClose: number;
}

const PipelineDashboard: React.FC = () => {
  const [stats, setStats] = useState<DashboardStats>({
    totalLeads: 0,
    activeLeads: 0,
    overdueLeads: 0,
    closedThisMonth: 0,
    conversionRate: 0,
    avgDealValue: 0,
    avgTimeToClose: 0
  });
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadDashboardStats();
  }, []);

  const loadDashboardStats = async () => {
    try {
      // Load pipeline data to calculate stats
      const response = await pipelineService.getPipeline({ limit: 1000 });
      const leads = response.data || [];
      
      const activeLeads = leads.filter(lead => 
        !['Closed', 'Lost / Dropped'].includes(lead.current_stage)
      );
      
      const overdueLeads = leads.filter(lead => lead.is_overdue);
      
      const closedLeads = leads.filter(lead => 
        lead.current_stage === 'Closed'
      );
      
      // Calculate conversion rate (closed / total)
      const conversionRate = leads.length > 0 ? (closedLeads.length / leads.length) * 100 : 0;
      
      // Calculate average deal value (mock calculation)
      const avgDealValue = closedLeads.length > 0 
        ? closedLeads.reduce((sum, lead) => sum + (lead.budget_max || 0), 0) / closedLeads.length
        : 0;

      setStats({
        totalLeads: leads.length,
        activeLeads: activeLeads.length,
        overdueLeads: overdueLeads.length,
        closedThisMonth: closedLeads.length,
        conversionRate,
        avgDealValue,
        avgTimeToClose: 15 // Mock value in days
      });
    } catch (error) {
      toast.error('Failed to load dashboard stats');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) {
      return `₹${(amount / 10000000).toFixed(1)}Cr`;
    } else if (amount >= 100000) {
      return `₹${(amount / 100000).toFixed(1)}L`;
    }
    return `₹${amount.toLocaleString()}`;
  };

  const StatCard: React.FC<{
    title: string;
    value: string | number;
    icon: React.ReactNode;
    color: string;
    trend?: string;
    trendUp?: boolean;
  }> = ({ title, value, icon, color, trend, trendUp }) => (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-center justify-between">
        <div>
          <p className="text-sm font-medium text-gray-600">{title}</p>
          <p className="text-2xl font-bold text-gray-900 mt-1">{value}</p>
          {trend && (
            <div className={`flex items-center gap-1 mt-2 text-sm ${
              trendUp ? 'text-green-600' : 'text-red-600'
            }`}>
              <TrendingUp size={16} className={trendUp ? '' : 'rotate-180'} />
              <span>{trend}</span>
            </div>
          )}
        </div>
        <div className={`p-3 rounded-full ${color}`}>
          {icon}
        </div>
      </div>
    </div>
  );

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Pipeline Dashboard</h1>
            <p className="text-gray-600">Monitor your sales pipeline performance</p>
          </div>
          <div className="flex items-center gap-3">
            <button
              onClick={loadDashboardStats}
              className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              <BarChart3 size={16} />
              Refresh Stats
            </button>
          </div>
        </div>
      </div>

      {/* Stats Grid */}
      <div className="p-6">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <StatCard
            title="Total Leads"
            value={stats.totalLeads}
            icon={<Users size={24} className="text-blue-600" />}
            color="bg-blue-100"
            trend="+12% from last month"
            trendUp={true}
          />
          
          <StatCard
            title="Active Leads"
            value={stats.activeLeads}
            icon={<TrendingUp size={24} className="text-green-600" />}
            color="bg-green-100"
            trend="+8% from last month"
            trendUp={true}
          />
          
          <StatCard
            title="Overdue Leads"
            value={stats.overdueLeads}
            icon={<AlertTriangle size={24} className="text-red-600" />}
            color="bg-red-100"
            trend={stats.overdueLeads > 0 ? "Needs attention" : "All on track"}
            trendUp={false}
          />
          
          <StatCard
            title="Closed This Month"
            value={stats.closedThisMonth}
            icon={<CheckCircle size={24} className="text-emerald-600" />}
            color="bg-emerald-100"
            trend="+15% from last month"
            trendUp={true}
          />
        </div>

        {/* Performance Metrics */}
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
          <StatCard
            title="Conversion Rate"
            value={`${stats.conversionRate.toFixed(1)}%`}
            icon={<BarChart3 size={24} className="text-purple-600" />}
            color="bg-purple-100"
            trend="+2.3% from last month"
            trendUp={true}
          />
          
          <StatCard
            title="Avg Deal Value"
            value={formatCurrency(stats.avgDealValue)}
            icon={<DollarSign size={24} className="text-yellow-600" />}
            color="bg-yellow-100"
            trend="+5.2% from last month"
            trendUp={true}
          />
          
          <StatCard
            title="Avg Time to Close"
            value={`${stats.avgTimeToClose} days`}
            icon={<Clock size={24} className="text-indigo-600" />}
            color="bg-indigo-100"
            trend="-3 days from last month"
            trendUp={true}
          />
        </div>

        {/* Pipeline Board */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
          <PipelineBoard />
        </div>
      </div>
    </div>
  );
};

export default PipelineDashboard;