import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { negotiationApi } from '../api/negotiation.api';
import { getPropertyById, Property } from '../api/properties';
import { BuyerMarketSnapshot } from '../components/buyer-dashboard/BuyerMarketSnapshot';
import RecommendedProperties from '../components/RecommendedProperties';

interface PropertyDetailsProps {
  propertyId: number;
}

const PropertyDetails: React.FC<PropertyDetailsProps> = ({ propertyId }) => {
  const navigate = useNavigate();
  const [property, setProperty] = useState<Property | null>(null);
  const [loading, setLoading] = useState(true);
  const [isStartingNegotiation, setIsStartingNegotiation] = useState(false);
  const [error, setError] = useState<string>('');
  const [currentUserId, setCurrentUserId] = useState<string | null>(null);

  useEffect(() => {
    const fetchProperty = async () => {
      try {
        const data = await getPropertyById(propertyId);
        setProperty(data);
      } catch (err) {
        setError('Failed to load property details');
      } finally {
        setLoading(false);
      }
    };

    const token = localStorage.getItem('token') || sessionStorage.getItem('token');
    if (token) {
      try {
        const payload = JSON.parse(atob(token.split('.')[1]));
        setCurrentUserId(payload.user_id || payload.sub);
      } catch (e) {
        console.error('Failed to parse token');
      }
    }

    fetchProperty();
  }, [propertyId]);

  const handleStartNegotiation = async () => {
    if (!property) return;

    setIsStartingNegotiation(true);
    setError('');

    try {
      const response = await negotiationApi.createNegotiation({
        property_id: propertyId,
        seller_id: '', // Backend will fetch from property owner
        agent_id: undefined,
      });

      // Navigate to negotiation (works for both new and existing)
      navigate(`/negotiations/${response.negotiation_id}`);
    } catch (error: any) {
      const errorMessage = error.response?.data?.detail || error.message || 'Failed to start negotiation';
      setError(errorMessage);
    } finally {
      setIsStartingNegotiation(false);
    }
  };

  const canNegotiate = () => {
    if (!property || !currentUserId) return { allowed: false, reason: 'Loading...' };
    
    if (!property.price_negotiable) {
      return { allowed: false, reason: 'This property is not open for negotiation' };
    }
    
    if (property.status !== 'active') {
      return { allowed: false, reason: 'This property is not available for negotiation' };
    }
    
    if (property.listed_by_id === currentUserId) {
      return { allowed: false, reason: 'You cannot negotiate on your own property' };
    }
    
    return { allowed: true, reason: '' };
  };

  const formatPrice = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(amount);
  };

  if (loading) {
    return (
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-3/4 mb-4"></div>
          <div className="h-4 bg-gray-200 rounded w-1/2 mb-8"></div>
          <div className="space-y-3">
            <div className="h-4 bg-gray-200 rounded"></div>
            <div className="h-4 bg-gray-200 rounded w-5/6"></div>
          </div>
        </div>
      </div>
    );
  }

  if (!property) {
    return (
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <p className="text-red-600">Property not found</p>
      </div>
    );
  }

  const negotiationStatus = canNegotiate();

  return (
    <div className="bg-white rounded-lg shadow-sm border p-6">
      {/* Property Header */}
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900 mb-2">
          {property.title}
        </h1>
        <p className="text-gray-600 mb-2">
          {property.locality}, {property.city}
        </p>
        {property.sale_price && (
          <p className="text-3xl font-bold text-green-600">
            {formatPrice(property.sale_price)}
          </p>
        )}
        
        {/* 🎯 BUYER MARKET SNAPSHOT - MVP Feature */}
        <div className="bg-red-100 border border-red-300 rounded-lg p-2 mb-2">
          <p className="text-red-800 text-xs">🔧 DEBUG: Market Snapshot should appear here</p>
        </div>
        {console.log('🔍 PropertyDetails: Checking BuyerMarketSnapshot conditions:', {
          sale_price: property.sale_price,
          locality: property.locality,
          city: property.city,
          shouldRender: !!(property.sale_price && property.locality)
        })}
        {property.sale_price && property.locality ? (
          <BuyerMarketSnapshot 
            propertyPrice={property.sale_price}
            locality={property.locality}
            city={property.city}
            variant="full"
          />
        ) : (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-4">
            <p className="text-yellow-800 text-sm">
              ⚠️ Market insights unavailable - Missing: {!property.sale_price ? 'price' : ''} {!property.locality ? 'locality' : ''}
            </p>
          </div>
        )}
        
        {console.log('🏠 Property data:', { sale_price: property.sale_price, locality: property.locality, city: property.city })}
        {property.price_negotiable && (
          <span className="inline-block mt-2 px-3 py-1 bg-blue-100 text-blue-800 text-sm rounded-full">
            Price Negotiable
          </span>
        )}
      </div>

      {/* Property Details Section */}
      <div className="mb-8">
        <h2 className="text-lg font-semibold text-gray-900 mb-4">Property Details</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="bg-gray-50 p-4 rounded-lg">
            <p className="text-sm text-gray-600">Property Type</p>
            <p className="font-medium capitalize">{property.property_type}</p>
          </div>
          <div className="bg-gray-50 p-4 rounded-lg">
            <p className="text-sm text-gray-600">Status</p>
            <p className="font-medium capitalize">{property.status}</p>
          </div>
          {property.bedrooms && (
            <div className="bg-gray-50 p-4 rounded-lg">
              <p className="text-sm text-gray-600">Bedrooms</p>
              <p className="font-medium">{property.bedrooms}</p>
            </div>
          )}
          {property.bathrooms && (
            <div className="bg-gray-50 p-4 rounded-lg">
              <p className="text-sm text-gray-600">Bathrooms</p>
              <p className="font-medium">{property.bathrooms}</p>
            </div>
          )}
        </div>
      </div>

      {/* Negotiation Section */}
      <div className="border-t pt-6">
        {negotiationStatus.allowed ? (
          <>
            <div className="flex items-center justify-between mb-4">
              <div>
                <h3 className="text-lg font-semibold text-gray-900">Interested in this property?</h3>
                <p className="text-gray-600">Start a negotiation to make an offer</p>
              </div>
            </div>

            {error && (
              <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-md">
                <p className="text-sm text-red-600">{error}</p>
              </div>
            )}

            <button
              onClick={handleStartNegotiation}
              disabled={isStartingNegotiation}
              className="w-full md:w-auto px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isStartingNegotiation ? (
                <span className="flex items-center justify-center">
                  <svg className="animate-spin h-4 w-4 mr-3" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" style={{ minWidth: '16px', maxWidth: '16px', minHeight: '16px', maxHeight: '16px' }}>
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  Starting Negotiation...
                </span>
              ) : (
                'Start Negotiation'
              )}
            </button>

            <div className="mt-4 p-4 bg-blue-50 rounded-lg">
              <h4 className="font-medium text-blue-900 mb-2">How it works:</h4>
              <ul className="text-sm text-blue-800 space-y-1">
                <li>• Submit your initial offer with an optional message</li>
                <li>• The seller can accept, reject, or counter your offer</li>
                <li>• Continue negotiating until you reach an agreement</li>
                <li>• Get AI-powered market insights to guide your decisions</li>
              </ul>
            </div>
          </>
        ) : (
          <div className="p-4 bg-gray-50 border border-gray-200 rounded-lg">
            <div className="flex items-start">
              <svg className="w-4 h-4 text-gray-400 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20" style={{ minWidth: '16px', maxWidth: '16px', minHeight: '16px', maxHeight: '16px' }}>
                <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
              </svg>
              <div>
                <h3 className="text-sm font-medium text-gray-900 mb-1">Negotiation Not Available</h3>
                <p className="text-sm text-gray-600">{negotiationStatus.reason}</p>
              </div>
            </div>
          </div>
        )}
      </div>
      
      {/* Recommended Properties Section */}
      <RecommendedProperties />
    </div>
  );
};

export default PropertyDetails;