import React, { useEffect, useState } from 'react';
import { useParams } from 'react-router-dom';
import PasswordProtectedUpload from '../components/PasswordProtectedUpload';
import DocumentList from '../components/DocumentList';
import { useDocuments } from '../hooks/useDocuments';

const PropertyDocuments: React.FC = () => {
  const { propertyId } = useParams<{ propertyId: string }>();
  const [selectedDocType, setSelectedDocType] = useState('deed');
  const [uploadSuccess, setUploadSuccess] = useState(false);
  
  const {
    documents,
    isLoading,
    error,
    uploadDocument,
    fetchDocuments,
    viewDocument,
    downloadDocument
  } = useDocuments();

  useEffect(() => {
    if (propertyId) {
      fetchDocuments(parseInt(propertyId));
    }
  }, [propertyId, fetchDocuments]);

  const handleUpload = async (file: File, password?: string) => {
    if (!propertyId) return;

    const result = await uploadDocument(
      file,
      parseInt(propertyId),
      selectedDocType,
      password
    );

    if (result.success) {
      setUploadSuccess(true);
      setTimeout(() => setUploadSuccess(false), 3000);
    }
  };

  const docTypes = [
    { value: 'deed', label: 'Property Deed' },
    { value: 'survey', label: 'Survey Document' },
    { value: 'tax', label: 'Tax Receipt' },
    { value: 'noc', label: 'NOC Certificate' },
    { value: 'approval', label: 'Approval Letter' },
    { value: 'other', label: 'Other Document' }
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-800 mb-2">
          Property Documents
        </h1>
        <p className="text-gray-600">
          Upload and manage documents for Property #{propertyId}
        </p>
      </div>

      {/* Upload Section */}
      <div className="bg-white rounded-lg shadow-md p-6 mb-8">
        <h2 className="text-lg font-semibold text-gray-800 mb-4">
          📤 Upload New Document
        </h2>
        
        <div className="mb-4">
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Document Type
          </label>
          <select
            value={selectedDocType}
            onChange={(e) => setSelectedDocType(e.target.value)}
            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            {docTypes.map((type) => (
              <option key={type.value} value={type.value}>
                {type.label}
              </option>
            ))}
          </select>
        </div>

        <PasswordProtectedUpload
          onUpload={handleUpload}
          isUploading={isLoading}
        />

        {uploadSuccess && (
          <div className="mt-4 p-3 bg-green-50 border border-green-200 rounded-md">
            <div className="flex items-center">
              <span className="text-green-500 mr-2">✅</span>
              <span className="text-green-700">Document uploaded successfully!</span>
            </div>
          </div>
        )}

        {error && (
          <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded-md">
            <div className="flex items-center">
              <span className="text-red-500 mr-2">❌</span>
              <span className="text-red-700">{error}</span>
            </div>
          </div>
        )}
      </div>

      {/* Documents List */}
      <div className="bg-white rounded-lg shadow-md p-6">
        {isLoading && documents.length === 0 ? (
          <div className="text-center py-8">
            <div className="animate-spin text-4xl mb-2">⏳</div>
            <p className="text-gray-500">Loading documents...</p>
          </div>
        ) : (
          <DocumentList
            documents={documents}
            onView={viewDocument}
            onDownload={downloadDocument}
          />
        )}
      </div>

      {/* Info Section */}
      <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-4">
        <h3 className="font-semibold text-blue-800 mb-2">
          🔒 Password Protection Info
        </h3>
        <ul className="text-sm text-blue-700 space-y-1">
          <li>• Enable encryption to password-protect sensitive documents</li>
          <li>• Share the password only with authorized parties</li>
          <li>• Encrypted documents show a 🔒 lock icon</li>
          <li>• Password is required to view or download encrypted documents</li>
        </ul>
      </div>
    </div>
  );
};

export default PropertyDocuments;