import React, { useState, useEffect } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { ArrowLeft, ArrowRight, TrendingUp, Shield, AlertTriangle, CheckCircle, Loader2 } from 'lucide-react';
import { PieChart, Pie, Cell, ResponsiveContainer, RadarChart, PolarGrid, PolarAngleAxis, PolarRadiusAxis, Radar, Tooltip } from 'recharts';

// Helper function to get authentication data for all user types
const getAuthData = (): { userId: string; authToken: string } | null => {
  // Check for buyer authentication (access_token)
  const userStr = localStorage.getItem('user');
  const accessToken = localStorage.getItem('access_token');
  if (userStr && accessToken) {
    const user = JSON.parse(userStr);
    const userId = user?.user_id || user?.id;
    if (userId) {
      return { userId, authToken: accessToken };
    }
  }

  // Fallback to auth_token for backward compatibility
  const authToken = localStorage.getItem('auth_token');
  if (userStr && authToken) {
    const user = JSON.parse(userStr);
    const userId = user?.user_id || user?.id;
    if (userId) {
      return { userId, authToken };
    }
  }

  // Check for seller authentication
  const sellerToken = localStorage.getItem('sellerToken');
  const sellerId = localStorage.getItem('sellerId');
  if (sellerToken && sellerId) {
    return { userId: sellerId, authToken: sellerToken };
  }

  // Check for agent authentication
  const agentToken = localStorage.getItem('agentToken');
  const agentId = localStorage.getItem('agentId');
  if (agentToken && agentId) {
    return { userId: agentId, authToken: agentToken };
  }

  return null;
};

const RibilScorecardPage: React.FC = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [ribilData, setRibilData] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
  
  useEffect(() => {
    const fetchRibilScore = async () => {
      if (!sessionId) {
        setError('No session ID found');
        setLoading(false);
        return;
      }
      
      try {
        const authData = getAuthData();
        const userId = authData?.userId || '550e8400-e29b-41d4-a716-446655440000';
        const authToken = authData?.authToken;
        
        // Try to calculate RIBIL score first
        const calculateResponse = await fetch(`http://localhost:8090/api/ribil/calculate/${sessionId}?user_id=${encodeURIComponent(userId)}&session_id=${encodeURIComponent(sessionId)}`, {
          method: 'POST',
          headers: {
            ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
            'Content-Type': 'application/json'
          }
        });
        
        if (calculateResponse.ok) {
          const calculateData = await calculateResponse.json();
          if (calculateData.success) {
            console.log('RIBIL Score calculated:', calculateData);
            setRibilData(calculateData);
            setLoading(false);
            return;
          }
        }
        
        // If calculation fails, try to get existing score
        const response = await fetch(`http://localhost:8090/api/ribil/score/${sessionId}?user_id=${encodeURIComponent(userId)}&session_id=${encodeURIComponent(sessionId)}`, {
          headers: {
            ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
            'Content-Type': 'application/json'
          }
        });
        const data = await response.json();
        
        if (data.success) {
          console.log('RIBIL Data received:', data);
          setRibilData(data);
        } else {
          setError(data.error || 'Failed to load RIBIL score');
        }
      } catch (err) {
        setError('Failed to fetch RIBIL score');
        console.error('RIBIL fetch error:', err);
      } finally {
        setLoading(false);
      }
    };
    
    fetchRibilScore();
  }, [sessionId]);
  
  const getRibilColor = (level: string) => {
    switch (level) {
      case 'EXCELLENT': return '#10b981';
      case 'GOOD': return '#3b82f6';
      case 'FAIR': return '#f59e0b';
      case 'POOR': return '#f97316';
      case 'CRITICAL': return '#ef4444';
      default: return '#6b7280';
    }
  };
  
  const getRibilGrade = (level: string) => {
    switch (level) {
      case 'EXCELLENT': return 'A+';
      case 'GOOD': return 'B+';
      case 'FAIR': return 'C+';
      case 'POOR': return 'D';
      case 'CRITICAL': return 'E';
      default: return 'N/A';
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 font-display flex items-center justify-center">
        <div className="text-center">
          <Loader2 className="w-12 h-12 animate-spin mx-auto mb-4 text-blue-600" />
          <p className="text-gray-600">Calculating RIBIL Score...</p>
        </div>
      </div>
    );
  }
  
  if (error || !ribilData) {
    return (
      <div className="min-h-screen bg-gray-50 font-display flex items-center justify-center">
        <div className="text-center max-w-md">
          <AlertTriangle className="w-12 h-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Error Loading RIBIL Score</h2>
          <p className="text-gray-600 mb-4">{error || 'No data available'}</p>
          <button
            onClick={() => navigate(`/docverification/results?session=${sessionId}`)}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            Back to Results
          </button>
        </div>
      </div>
    );
  }
  
  const score = ribilData?.data?.ribil_score || ribilData?.ribil_score || 0;
  const level = ribilData?.data?.ribil_grade || ribilData?.ribil_grade || 'UNKNOWN';
  const components = ribilData?.data?.components || ribilData?.component_scores || {};
  const recommendations = ribilData?.data?.insights || ribilData?.recommendations || {};
  const propertyDetails = ribilData?.data?.property_details || ribilData?.property_details || {};
  
  // Always show the combined card if we have RIBIL data
  const shouldShowPropertyCard = ribilData && (propertyDetails?.owner_name || propertyDetails?.survey_number || propertyDetails?.property_number || score > 0);
  const docContributions = ribilData.document_contributions || [];
  const benchmark = ribilData.benchmark_data || {};
  
  const duplicateScore = Math.round(components.duplicate_detection_score || 0);
  const duplicateLabel = duplicateScore === 0 ? 'DUPLICATE' : 'No Duplicate';
  
  const radarData = [
    { feature: 'Document Verification', score: Math.round(components.document_verification || 0), fullMark: 100 },
    { feature: 'Completeness', score: Math.round(components.document_completeness || 0), fullMark: 100 },
    { feature: 'Quality', score: Math.round(components.verification_quality || 0), fullMark: 100 }
  ];
  
  const getDocTypeDisplayName = (docType: string) => {
    const typeMap: Record<string, string> = {
      'title_deed': 'Title Deed',
      'sale_deed': 'Sale Deed',
      'encumbrance_certificate': 'EC Certificate',
      'property_card': 'Property Card',
      'survey_settlement': 'Survey Record',
      'tax_receipt': 'Tax Receipt',
      'possession_certificate': 'Possession Cert'
    };
    return typeMap[docType] || docType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
  };
  
  const downloadReport = () => {
    const reportContent = `
      RIBIL SCORECARD REPORT
      =====================
      
      Property Details:
      Owner: ${propertyDetails.owner_name || 'N/A'}
      Survey: ${propertyDetails.survey_number || 'N/A'}
      Property: ${propertyDetails.property_number || 'N/A'}
      Status: ${propertyDetails.judicial_status || 'N/A'}
      
      RIBIL Score: ${score}
      Grade: ${getRibilGrade(score)}
      Level: ${level}
      
      Component Scores:
      ${Object.entries(components).map(([k, v]) => `${k}: ${Math.round(v as number)}`).join('\n')}
      
      Recommendations:
      ${recommendations.overall_recommendation || 'N/A'}
    `;
    
    const blob = new Blob([reportContent], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `RIBIL-Report-${sessionId}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };
  
  return (
    <div className="min-h-screen bg-gray-50 font-display">
      <main className="flex-grow">
        <div className="container mx-auto max-w-6xl px-4 sm:px-6 lg:px-8 py-12">
          <div className="flex flex-col gap-8">
            {/* Header Section */}
            <div className="flex flex-col gap-3 text-center">
              <p className="text-base font-semibold text-blue-600">Step 4 of 5</p>
              <p className="text-4xl font-black tracking-tighter text-black">RIBIL Score Analysis</p>
              <p className="text-lg font-normal text-gray-600 max-w-2xl mx-auto">
                Your property trust score based on document verification results
              </p>
            </div>

            {/* Progress Steps */}
            <div className="bg-white rounded-xl shadow-lg border border-gray-200 p-8">
              <div className="flex items-center max-w-6xl mx-auto">
                {/* Step 1 - Completed */}
                <button 
                  onClick={() => navigate(`/docverification?session=${sessionId}`)}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Property Owner Details</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </button>
                
                {/* Connector 1 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>
                
                {/* Step 2 - Completed */}
                <button 
                  onClick={() => navigate(`/docverification?session=${sessionId}`)}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Document Upload</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </button>
                
                {/* Connector 2 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>
                
                {/* Step 3 - Completed */}
                <button 
                  onClick={() => navigate(`/docverification/results?session=${sessionId}`)}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-green-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    ✓
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">Information Review</p>
                  <p className="text-xs text-green-500 font-medium">Completed</p>
                </button>
                
                {/* Connector 3 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                  <div className="w-2 h-2 bg-green-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-green-300"></div>
                </div>
                
                {/* Step 4 - Active */}
                <div className="flex flex-col items-center text-center">
                  <div className="flex items-center justify-center w-12 h-12 bg-blue-500 text-white rounded-full font-bold text-sm mb-3 shadow-lg">
                    4
                  </div>
                  <p className="text-sm font-semibold text-gray-900 mb-1">RIBIL Score</p>
                  <p className="text-xs text-blue-500 font-medium">In Progress</p>
                </div>
                
                {/* Connector 4 */}
                <div className="flex-1 flex items-center mx-4">
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                  <div className="w-2 h-2 bg-gray-300 rounded-full mx-2"></div>
                  <div className="flex-1 h-0.5 bg-gray-300"></div>
                </div>
                
                {/* Step 5 - Pending */}
                <button 
                  onClick={() => navigate(`/docverification/final?session=${sessionId}`)}
                  className="flex flex-col items-center text-center hover:scale-105 transition-transform"
                >
                  <div className="flex items-center justify-center w-12 h-12 bg-white text-black rounded-full font-bold text-sm mb-3 border-2 border-gray-300">
                    5
                  </div>
                  <p className="text-sm font-medium text-black mb-1">Final Verification</p>
                  <p className="text-xs text-gray-500">Pending</p>
                </button>
              </div>
            </div>

            {/* Combined Property Information & RIBIL Score Card */}
            {shouldShowPropertyCard && (
            <div className="bg-gradient-to-br from-blue-50 to-indigo-100 rounded-xl border-2 border-blue-300 shadow-lg p-6">
              <h3 className="text-xl font-bold text-indigo-900 mb-4 flex items-center">
                <svg className="w-6 h-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
                </svg>
                Property Information
              </h3>
              <div className="space-y-3">
                {/* Property Details */}
                {propertyDetails?.owner_name && (
                  <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                    <span className="text-sm font-bold text-gray-700">Owner Name:</span>
                    <span className="text-base font-bold text-gray-900">{propertyDetails.owner_name}</span>
                  </div>
                )}
                {propertyDetails?.survey_number && (
                  <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                    <span className="text-sm font-bold text-gray-700">Survey Number:</span>
                    <span className="text-base font-bold text-gray-900">{propertyDetails.survey_number}</span>
                  </div>
                )}
                {propertyDetails?.property_number && (
                  <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                    <span className="text-sm font-bold text-gray-700">Property Number:</span>
                    <span className="text-base font-bold text-gray-900">{propertyDetails.property_number}</span>
                  </div>
                )}
                {propertyDetails?.judicial_status && (
                  <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                    <span className="text-sm font-bold text-gray-700">Judicial Status:</span>
                    <span className="text-base font-bold text-green-600">{propertyDetails.judicial_status}</span>
                  </div>
                )}
                {propertyDetails?.latitude && propertyDetails?.longitude && (
                  <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                    <span className="text-sm font-bold text-gray-700">Geo-Location:</span>
                    <span className="text-sm font-mono text-gray-900">
                      {propertyDetails.latitude.toFixed(6)}, {propertyDetails.longitude.toFixed(6)}
                    </span>
                  </div>
                )}
                
                {/* RIBIL Score Details */}
                <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                  <span className="text-sm font-bold text-gray-700">RIBIL Score:</span>
                  <span className="text-base font-bold text-blue-600">{score}</span>
                </div>
                <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                  <span className="text-sm font-bold text-gray-700">Out of:</span>
                  <span className="text-base font-bold text-gray-600">100</span>
                </div>
                <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                  <span className="text-sm font-bold text-gray-700">Grade:</span>
                  <span className="text-base font-bold text-blue-700">{getRibilGrade(level)}</span>
                </div>
                <div className="flex items-center justify-between bg-white rounded-lg p-3 shadow-md">
                  <span className="text-sm font-bold text-gray-700">Trust Level:</span>
                  <span className="text-base font-bold text-blue-600">{level}</span>
                </div>
              </div>
            </div>
            )}
            
            {/* Component Scores Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              <div className="bg-white rounded-xl border-2 border-blue-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Data Validation</h3>
                  <CheckCircle className="w-6 h-6 text-blue-600" />
                </div>
                <div className="text-3xl font-bold text-blue-600 mb-2">
                  {Math.round(components.document_verification || 0)}/100
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div 
                    className="bg-blue-600 h-3 rounded-full transition-all"
                    style={{ width: `${components.document_verification || 0}%` }}
                  />
                </div>
              </div>
              
              <div className="bg-white rounded-xl border-2 border-green-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Document Complete</h3>
                  <Shield className="w-6 h-6 text-green-600" />
                </div>
                <div className="text-3xl font-bold text-green-600 mb-2">
                  {Math.round(components.document_completeness || 0)}/100
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div 
                    className="bg-green-600 h-3 rounded-full transition-all"
                    style={{ width: `${components.document_completeness || 0}%` }}
                  />
                </div>
              </div>
              
              <div className="bg-white rounded-xl border-2 border-purple-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Quality Score</h3>
                  {Math.round(components.duplicate_detection_score || 0) === 100 ? (
                    <CheckCircle className="w-6 h-6 text-green-600" />
                  ) : (
                    <AlertTriangle className="w-6 h-6 text-orange-600" />
                  )}
                </div>
                <div className="text-3xl font-bold text-purple-600 mb-2">
                  {Math.round(components.verification_quality || 0)}/100
                </div>
                <div className="text-sm font-semibold mb-3">
                  {(() => {
                    const dupScore = Math.round(components.duplicate_detection_score || 0);
                    const totalDocs = docContributions.length;
                    const duplicateDocs = docContributions.filter((doc: any) => doc.feature_scores.duplicate_check === 0).length;
                    
                    if (dupScore === 100) {
                      return <span className="text-green-600">✓ No Duplicates</span>;
                    } else if (duplicateDocs > 0) {
                      return <span className="text-orange-600">⚠️ {duplicateDocs} Duplicate{duplicateDocs > 1 ? 's' : ''} Found</span>;
                    } else {
                      return <span className="text-gray-600">Checking...</span>;
                    }
                  })()}
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div 
                    className={`h-3 rounded-full transition-all ${
                      Math.round(components.duplicate_detection_score || 0) === 100 ? 'bg-green-600' : 'bg-orange-600'
                    }`}
                    style={{ width: `${components.duplicate_detection_score || 0}%` }}
                  />
                </div>
              </div>
              
              <div className="bg-white rounded-xl border-2 border-orange-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Authenticity</h3>
                  <Shield className="w-6 h-6 text-orange-600" />
                </div>
                <div className="text-3xl font-bold text-orange-600 mb-2">
                  {Math.round(components.authenticity_score || 0)}/100
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div 
                    className="bg-orange-600 h-3 rounded-full transition-all"
                    style={{ width: `${components.authenticity_score || 0}%` }}
                  />
                </div>
              </div>
              
              <div className="bg-white rounded-xl border-2 border-red-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Dispute Risk</h3>
                  <AlertTriangle className="w-6 h-6 text-red-600" />
                </div>
                <div className="text-3xl font-bold text-red-600 mb-2">
                  {Math.round(components.dispute_risk_score || 0)}/100
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div 
                    className="bg-red-600 h-3 rounded-full transition-all"
                    style={{ width: `${components.dispute_risk_score || 0}%` }}
                  />
                </div>
              </div>
              
              <div className="bg-white rounded-xl border-2 border-indigo-200 shadow-lg p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="font-bold text-gray-900">Overall Quality</h3>
                  <TrendingUp className="w-6 h-6 text-indigo-600" />
                </div>
                <ResponsiveContainer width="100%" height={120}>
                  <RadarChart data={radarData}>
                    <PolarGrid stroke="#c7d2fe" />
                    <PolarAngleAxis dataKey="feature" tick={{ fontSize: 10, fill: '#6366f1' }} />
                    <PolarRadiusAxis angle={90} domain={[0, 100]} tick={false} />
                    <Radar dataKey="score" stroke="#6366f1" fill="#818cf8" fillOpacity={0.6} />
                    <Tooltip 
                      contentStyle={{ 
                        backgroundColor: '#1e293b', 
                        border: 'none', 
                        borderRadius: '8px',
                        padding: '8px 12px'
                      }}
                      labelStyle={{ color: '#f1f5f9', fontWeight: 'bold' }}
                      itemStyle={{ color: '#818cf8' }}
                      formatter={(value: any) => [`${value}/100`, 'Score']}
                    />
                  </RadarChart>
                </ResponsiveContainer>
              </div>
            </div>
            
            {/* Recommendations */}
            {recommendations.overall_recommendation && (
              <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
                <h3 className="text-xl font-bold text-gray-900 mb-4 flex items-center">
                  <TrendingUp className="w-6 h-6 mr-2 text-blue-600" />
                  Professional Recommendations
                </h3>
                <div className="bg-blue-50 border-l-4 border-blue-600 p-4 mb-4">
                  <p className="text-blue-900 font-semibold">{recommendations.overall_recommendation}</p>
                </div>
                
                {recommendations.action_items && recommendations.action_items.length > 0 && (
                  <div className="mb-4">
                    <h4 className="font-semibold text-gray-900 mb-2">Action Items:</h4>
                    <ul className="space-y-2">
                      {recommendations.action_items.map((item: string, idx: number) => (
                        <li key={idx} className="flex items-start">
                          <span className="text-blue-600 mr-2">•</span>
                          <span className="text-gray-700">{item}</span>
                        </li>
                      ))}
                    </ul>
                  </div>
                )}
                

              </div>
            )}

            {/* Benchmark Comparison Chart */}
            {benchmark.average_ribil_score && (
              <div className="bg-white rounded-xl border-2 border-purple-200 shadow-lg p-6">
                <h3 className="text-xl font-bold text-gray-900 mb-4 flex items-center">
                  <TrendingUp className="w-6 h-6 mr-2 text-purple-600" />
                  Benchmark Comparison
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                  <div className="text-center p-4 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg border border-blue-200">
                    <div className="text-4xl font-black text-blue-600 mb-2">{score}</div>
                    <div className="text-sm font-semibold text-gray-700">Your Score</div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-gray-50 to-gray-100 rounded-lg border border-gray-200">
                    <div className="text-4xl font-black text-gray-600 mb-2">{benchmark.average_ribil_score}</div>
                    <div className="text-sm font-semibold text-gray-700">Average Score</div>
                  </div>
                  <div className="text-center p-4 bg-gradient-to-br from-green-50 to-emerald-50 rounded-lg border border-green-200">
                    <div className="text-4xl font-black text-green-600 mb-2">{benchmark.percentile}%</div>
                    <div className="text-sm font-semibold text-gray-700">Percentile Rank</div>
                  </div>
                </div>
                <div className="bg-gradient-to-r from-purple-50 to-pink-50 border border-purple-200 rounded-lg p-4">
                  <p className="text-sm text-purple-900">
                    <strong>Performance Analysis:</strong> Your property scores <strong>{score}</strong> compared to the market average of <strong>{benchmark.average_ribil_score}</strong>. 
                    This places you in the <strong>top {100 - benchmark.percentile}%</strong> of {benchmark.total_properties_analyzed.toLocaleString()} analyzed properties.
                    {score > benchmark.average_ribil_score ? ' 🎉 Excellent performance!' : ' Consider improving documentation quality.'}
                  </p>
                </div>
              </div>
            )}

            {/* Document-Level Contributions - Compact Cards */}
            {docContributions.length > 0 && (
              <div className="bg-white rounded-xl border-2 border-orange-200 shadow-lg p-6">
                <h3 className="text-xl font-bold text-gray-900 mb-4 flex items-center">
                  <svg className="w-6 h-6 mr-2 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  Document-Level RIBIL Contributions
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {docContributions.map((doc, idx) => (
                    <div key={idx} className={`border-2 rounded-lg p-4 shadow-md ${
                      doc.impact === 'positive' ? 'border-green-300 bg-gradient-to-br from-green-50 to-emerald-50' :
                      doc.impact === 'negative' ? 'border-red-300 bg-gradient-to-br from-red-50 to-pink-50' :
                      'border-yellow-300 bg-gradient-to-br from-yellow-50 to-amber-50'
                    }`}>
                      <div className="text-center mb-3">
                        <div className="font-bold text-sm text-gray-900 mb-1">{getDocTypeDisplayName(doc.doc_type)}</div>
                        <div className="text-xs text-gray-600 truncate" title={doc.filename}>{doc.filename}</div>
                      </div>
                      <div className="text-center mb-3">
                        <div className={`text-3xl font-black mb-1 ${
                          doc.impact === 'positive' ? 'text-green-600' :
                          doc.impact === 'negative' ? 'text-red-600' :
                          'text-yellow-600'
                        }`}>
                          {doc.contribution_score}
                        </div>
                        <div className="text-xs font-bold uppercase">
                          {doc.impact === 'positive' ? '⬆️ POSITIVE' :
                           doc.impact === 'negative' ? '⬇️ NEGATIVE' :
                           '➡️ NEUTRAL'}
                        </div>
                      </div>
                      <div className="space-y-1">
                        <div className="flex items-center justify-between bg-white rounded p-2 shadow-sm">
                          <span className="text-xs font-semibold text-gray-600">Data Validation:</span>
                          <span className="text-xs font-bold text-blue-600">{doc.feature_scores.data_validation}</span>
                        </div>
                        <div className="flex items-center justify-between bg-white rounded p-2 shadow-sm">
                          <span className="text-xs font-semibold text-gray-600">Forgery Detection:</span>
                          <span className="text-xs font-bold text-green-600">{doc.feature_scores.forgery_detection}</span>
                        </div>
                        <div className="flex items-center justify-between bg-white rounded p-2 shadow-sm">
                          <span className="text-xs font-semibold text-gray-600">Duplicate Check:</span>
                          <span className="text-xs font-bold text-purple-600">
                            {doc.feature_scores.duplicate_check}
                            {doc.feature_scores.duplicate_check === 0 && (
                              <span className="ml-1 text-orange-600">⚠️</span>
                            )}
                          </span>
                        </div>
                        <div className="flex items-center justify-between bg-white rounded p-2 shadow-sm">
                          <span className="text-xs font-semibold text-gray-600">Authenticity:</span>
                          <span className="text-xs font-bold text-orange-600">{doc.feature_scores.authenticity}</span>
                        </div>
                        <div className="flex items-center justify-between bg-white rounded p-2 shadow-sm">
                          <span className="text-xs font-semibold text-gray-600">Dispute Risk:</span>
                          <span className="text-xs font-bold text-red-600">{doc.feature_scores.dispute_risk}</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* About RIBIL Section */}
            <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">About RIBIL Scorecard</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-6">
                <div className="text-center">
                  <div className="text-2xl font-bold text-gray-900 mb-2">R</div>
                  <div className="text-sm font-semibold mb-1">Registration</div>
                  <div className="text-xs text-gray-600">Legal documents & RERA compliance</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-gray-900 mb-2">I</div>
                  <div className="text-sm font-semibold mb-1">Infrastructure</div>
                  <div className="text-xs text-gray-600">Location & amenities quality</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-gray-900 mb-2">B</div>
                  <div className="text-sm font-semibold mb-1">Builder</div>
                  <div className="text-xs text-gray-600">Reputation & track record</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-gray-900 mb-2">I</div>
                  <div className="text-sm font-semibold mb-1">Investment</div>
                  <div className="text-xs text-gray-600">ROI & market potential</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-gray-900 mb-2">L</div>
                  <div className="text-sm font-semibold mb-1">Legal</div>
                  <div className="text-xs text-gray-600">Verification & compliance</div>
                </div>
              </div>
              
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <p className="text-sm text-blue-800">
                  <strong>RIBIL Scorecard</strong> is NAL India's proprietary trust index that evaluates 
                  property credibility and investment value. Higher scores indicate better documentation, 
                  legal compliance, and investment potential.
                </p>
              </div>
            </div>

            {/* Navigation Buttons */}
            <div className="flex justify-between items-center pt-4">
              <button 
                onClick={() => {
                  const sessionId = searchParams.get('session') || localStorage.getItem('verification-session-id');
                  navigate(`/docverification/results?session=${sessionId}`);
                }}
                className="flex items-center justify-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors shadow-md"
              >
                <ArrowLeft className="w-5 h-5 mr-2" />
                Back to Results
              </button>
              
              <button 
                onClick={downloadReport}
                className="flex items-center justify-center px-6 py-3 bg-green-500 text-white rounded-lg hover:bg-green-600 transition-colors shadow-md"
              >
                <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                Download Report
              </button>
              
              <button 
                onClick={() => navigate(`/docverification/final?session=${sessionId}`)}
                className="flex items-center justify-center px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors shadow-md"
              >
                <span>Proceed to Final Verification</span>
                <ArrowRight className="ml-2 w-5 h-5" />
              </button>
            </div>
          </div>
        </div>
      </main>
    </div>
  );
};

export default RibilScorecardPage;