import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useForm } from 'react-hook-form';
import { ArrowLeft, Plus, X, Upload, Image, Video, Eye } from 'lucide-react';
import { ListingFormData } from '../types/listing';
import { validation } from '../utils/validation';
import { fileUtils } from '../utils/fileUtils';
import { AddressAutocomplete } from '../components/AddressAutocomplete';
import { GoogleMap } from '../components/GoogleMap';

const categories = [
  'Residential Apartment',
  'Independent House',
  'Villa',
  'Commercial Office',
  'Retail Space',
  'Warehouse',
  'Land/Plot'
];

const cities = [
  'Mumbai', 'Delhi', 'Bangalore', 'Chennai', 'Kolkata', 'Hyderabad',
  'Pune', 'Ahmedabad', 'Surat', 'Jaipur', 'Lucknow', 'Kanpur'
];

const commonSpecifications = [
  'Air Conditioning',
  'Parking Space',
  'Balcony',
  'Furnished',
  'Semi-Furnished',
  'Unfurnished',
  'Elevator',
  'Security',
  'Power Backup',
  'Water Supply'
];

const imageCategoryOptions = [
  'Exterior',
  'Living Room',
  'Kitchen',
  'Bedroom',
  'Bathroom',
  'Balcony',
  'Other'
];

export const SinglePageListingForm: React.FC = () => {
  const navigate = useNavigate();
  const [customFeature, setCustomFeature] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('Exterior');
  const [previewImage, setPreviewImage] = useState<string | null>(null);
  const [specifications, setSpecifications] = useState<string[]>([]);
  const [features, setFeatures] = useState<string[]>([]);
  const [images, setImages] = useState<File[]>([]);
  const [videos, setVideos] = useState<File[]>([]);
  const [imageCategories, setImageCategories] = useState<{[key: string]: File[]}>({});

  const { register, handleSubmit, formState: { errors }, setValue, watch } = useForm<ListingFormData>({
    defaultValues: {
      title: '',
      category: '',
      description: '',
      price: 0,
      specifications: [],
      features: [],
      address: '',
      city: '',
      coordinates: undefined,
      pincode: '',
      locality: '',
      landmark: '',
      propertyAge: '',
      floorNumber: '',
      totalFloors: '',
      facing: '',
      images: [],
      videos: [],
      imageCategories: {}
    }
  });

  const address = watch('address') || '';
  const coordinates = watch('coordinates');

  const handleLocationSelect = (location: { lat: number; lng: number; address: string }) => {
    setValue('address', location.address);
    setValue('coordinates', { lat: location.lat, lng: location.lng });
  };

  const handleMapClick = (location: { lat: number; lng: number }) => {
    setValue('coordinates', location);
  };

  const handleSpecificationChange = (spec: string, checked: boolean) => {
    if (checked) {
      setSpecifications([...specifications, spec]);
    } else {
      setSpecifications(specifications.filter(s => s !== spec));
    }
  };

  const addCustomFeature = () => {
    if (customFeature.trim()) {
      setFeatures([...features, customFeature.trim()]);
      setCustomFeature('');
    }
  };

  const removeFeature = (index: number) => {
    setFeatures(features.filter((_, i) => i !== index));
  };

  const handleImageUpload = (files: FileList | null) => {
    if (!files) return;
    const fileArray = Array.from(files);
    const newImages = [...images, ...fileArray];
    setImages(newImages);
    
    const currentCategoryImages = imageCategories[selectedCategory] || [];
    setImageCategories({
      ...imageCategories,
      [selectedCategory]: [...currentCategoryImages, ...fileArray]
    });
  };

  const handleVideoUpload = (files: FileList | null) => {
    if (!files) return;
    const fileArray = Array.from(files);
    setVideos([...videos, ...fileArray]);
  };

  const removeImage = (index: number) => {
    const imageToRemove = images[index];
    const newImages = images.filter((_, i) => i !== index);
    setImages(newImages);

    const updatedCategories = { ...imageCategories };
    Object.keys(updatedCategories).forEach(category => {
      updatedCategories[category] = updatedCategories[category].filter(img => img !== imageToRemove);
    });
    setImageCategories(updatedCategories);
  };

  const removeVideo = (index: number) => {
    setVideos(videos.filter((_, i) => i !== index));
  };

  const previewFile = async (file: File) => {
    if (fileUtils.isImageFile(file)) {
      const preview = await fileUtils.createFilePreview(file);
      setPreviewImage(preview);
    }
  };

  const onSubmit = (data: ListingFormData) => {
    const formData = {
      ...data,
      specifications,
      features,
      images,
      videos,
      imageCategories
    };
    console.log('Form submitted:', formData);
    alert('Property listing submitted successfully!');
    navigate('/');
  };

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto p-6">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <button
            onClick={() => navigate('/')}
            className="flex items-center text-gray-600 hover:text-gray-900 transition-colors"
          >
            <ArrowLeft className="w-5 h-5 mr-2" />
            Back to Properties
          </button>
        </div>

        {/* Form Content */}
        <div className="bg-white rounded-lg border border-gray-200 p-8 shadow-sm">
          <form onSubmit={handleSubmit(onSubmit)} className="space-y-8">
            
            {/* Basic Information */}
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-6">Basic Information</h2>
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Title *
                  </label>
                  <input
                    type="text"
                    {...register('title', validation.required('Property title is required'))}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    placeholder="e.g., Luxury 3BHK Apartment in Prime Location"
                  />
                  {errors.title && (
                    <p className="mt-1 text-sm text-red-600">{errors.title.message}</p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Category *
                  </label>
                  <select
                    {...register('category', validation.required('Please select a category'))}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  >
                    <option value="">Select Category</option>
                    {categories.map(category => (
                      <option key={category} value={category}>{category}</option>
                    ))}
                  </select>
                  {errors.category && (
                    <p className="mt-1 text-sm text-red-600">{errors.category.message}</p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Description *
                  </label>
                  <textarea
                    {...register('description', {
                      ...validation.required('Description is required'),
                      ...validation.minLength(50, 'Description must be at least 50 characters')
                    })}
                    rows={6}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent resize-none"
                    placeholder="Describe your property in detail. Include key features, amenities, and what makes it special..."
                  />
                  {errors.description && (
                    <p className="mt-1 text-sm text-red-600">{errors.description.message}</p>
                  )}
                  <p className="mt-1 text-sm text-gray-500">Minimum 50 characters required</p>
                </div>
              </div>
            </div>

            {/* Property Details */}
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-6">Property Details</h2>
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">
                    Property Specifications
                  </label>
                  <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                    {commonSpecifications.map(spec => (
                      <label key={spec} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={specifications.includes(spec)}
                          onChange={(e) => handleSpecificationChange(spec, e.target.checked)}
                          className="w-4 h-4 text-[#0056D2] border-gray-300 rounded focus:ring-[#0056D2]"
                        />
                        <span className="text-sm text-gray-700">{spec}</span>
                      </label>
                    ))}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">
                    Additional Features
                  </label>
                  <div className="flex gap-2 mb-4">
                    <input
                      type="text"
                      value={customFeature}
                      onChange={(e) => setCustomFeature(e.target.value)}
                      placeholder="Add custom feature..."
                      className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                      onKeyPress={(e) => e.key === 'Enter' && addCustomFeature()}
                    />
                    <button
                      type="button"
                      onClick={addCustomFeature}
                      className="px-4 py-2 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors"
                    >
                      <Plus className="w-4 h-4" />
                    </button>
                  </div>

                  {features.length > 0 && (
                    <div className="flex flex-wrap gap-2">
                      {features.map((feature, index) => (
                        <span
                          key={index}
                          className="inline-flex items-center px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm"
                        >
                          {feature}
                          <button
                            type="button"
                            onClick={() => removeFeature(index)}
                            className="ml-2 text-blue-600 hover:text-blue-800"
                          >
                            <X className="w-3 h-3" />
                          </button>
                        </span>
                      ))}
                    </div>
                  )}
                </div>

                {/* Detailed Pricing */}
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Detailed Pricing</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    {/* Property Price */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Property Price (₹) *
                      </label>
                      <input
                        type="number"
                        {...register('price', {
                          ...validation.required('Property price is required'),
                          ...validation.min(1, 'Price must be greater than 0')
                        })}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                        placeholder="e.g., 8500000"
                      />
                      {errors.price && (
                        <p className="mt-1 text-sm text-red-600">{errors.price.message}</p>
                      )}
                    </div>

                    {/* Price per Sq Ft */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Price per Sq Ft (₹)
                      </label>
                      <input
                        type="number"
                        {...register('pricePerSqFt')}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                        placeholder="e.g., 5500"
                      />
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Location */}
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-6">Property Location</h2>
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Property Address *
                  </label>
                  <AddressAutocomplete
                    value={address}
                    onChange={(value) => setValue('address', value)}
                    onLocationSelect={handleLocationSelect}
                    placeholder="Start typing your property address..."
                  />
                  {errors.address && (
                    <p className="mt-1 text-sm text-red-600">{errors.address.message}</p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    City *
                  </label>
                  <select
                    {...register('city', validation.required('City is required'))}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                  >
                    <option value="">Select City</option>
                    {cities.map(city => (
                      <option key={city} value={city}>{city}</option>
                    ))}
                  </select>
                  {errors.city && (
                    <p className="mt-1 text-sm text-red-600">{errors.city.message}</p>
                  )}
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Location on Map
                  </label>
                  <div className="border border-gray-300 rounded-lg overflow-hidden">
                    <GoogleMap
                      address={address}
                      coordinates={coordinates || { lat: 12.9716, lng: 77.5946 }}
                      onLocationSelect={handleMapClick}
                      height="300px"
                    />
                  </div>
                  {coordinates && (
                    <p className="mt-2 text-sm text-green-600">
                      📍 Location set: {coordinates.lat.toFixed(4)}, {coordinates.lng.toFixed(4)}
                    </p>
                  )}
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Pincode *
                    </label>
                    <input
                      type="text"
                      {...register('pincode', validation.required('Pincode is required'))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                      placeholder="e.g., 560001"
                    />
                    {errors.pincode && (
                      <p className="mt-1 text-sm text-red-600">{errors.pincode.message}</p>
                    )}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Locality *
                    </label>
                    <input
                      type="text"
                      {...register('locality', validation.required('Locality is required'))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                      placeholder="e.g., Koramangala"
                    />
                    {errors.locality && (
                      <p className="mt-1 text-sm text-red-600">{errors.locality.message}</p>
                    )}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Nearby Landmark
                  </label>
                  <input
                    type="text"
                    {...register('landmark')}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    placeholder="e.g., Near Metro Station, Shopping Mall"
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Property Age *
                    </label>
                    <select
                      {...register('propertyAge', validation.required('Property age is required'))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    >
                      <option value="">Select Age</option>
                      <option value="Under Construction">Under Construction</option>
                      <option value="0-1 years">0-1 years</option>
                      <option value="1-5 years">1-5 years</option>
                      <option value="5-10 years">5-10 years</option>
                      <option value="10+ years">10+ years</option>
                    </select>
                    {errors.propertyAge && (
                      <p className="mt-1 text-sm text-red-600">{errors.propertyAge.message}</p>
                    )}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Facing *
                    </label>
                    <select
                      {...register('facing', validation.required('Facing direction is required'))}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    >
                      <option value="">Select Facing</option>
                      <option value="North">North</option>
                      <option value="South">South</option>
                      <option value="East">East</option>
                      <option value="West">West</option>
                      <option value="North-East">North-East</option>
                      <option value="North-West">North-West</option>
                      <option value="South-East">South-East</option>
                      <option value="South-West">South-West</option>
                    </select>
                    {errors.facing && (
                      <p className="mt-1 text-sm text-red-600">{errors.facing.message}</p>
                    )}
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Floor Number
                    </label>
                    <input
                      type="text"
                      {...register('floorNumber')}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                      placeholder="e.g., 3rd Floor, Ground Floor"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Total Floors
                    </label>
                    <input
                      type="text"
                      {...register('totalFloors')}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                      placeholder="e.g., 10 floors"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Media Upload */}
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-6">Property Media</h2>
              <div className="space-y-8">
                {/* Image URL Section */}
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Primary Image URL</h3>
                  <div className="mb-6">
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Image URL (for testing)
                    </label>
                    <input
                      type="url"
                      {...register('primaryImage')}
                      placeholder="https://example.com/image.jpg"
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    />
                    <p className="text-sm text-gray-500 mt-1">
                      Enter a direct image URL for quick testing
                    </p>
                  </div>
                </div>

                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Images</h3>
                  
                  <div className="mb-4">
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Select Category for Upload
                    </label>
                    <select
                      value={selectedCategory}
                      onChange={(e) => setSelectedCategory(e.target.value)}
                      className="w-full md:w-auto px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
                    >
                      {imageCategoryOptions.map(category => (
                        <option key={category} value={category}>{category}</option>
                      ))}
                    </select>
                  </div>

                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center hover:border-[#0056D2] transition-colors">
                    <input
                      type="file"
                      multiple
                      accept="image/*"
                      onChange={(e) => handleImageUpload(e.target.files)}
                      className="hidden"
                      id="images-upload"
                    />
                    <label htmlFor="images-upload" className="cursor-pointer">
                      <Image className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                      <div className="text-lg font-medium text-gray-900 mb-2">
                        Upload to {selectedCategory}
                      </div>
                      <div className="text-gray-600">
                        Drag and drop images here, or click to browse
                      </div>
                      <div className="text-sm text-gray-500 mt-2">
                        JPG, PNG, WebP up to 5MB each
                      </div>
                    </label>
                  </div>

                  {Object.keys(imageCategories).map(category => {
                    const categoryImages = imageCategories[category] || [];
                    if (categoryImages.length === 0) return null;

                    return (
                      <div key={category} className="mt-6">
                        <h4 className="font-medium text-gray-900 mb-3">
                          {category} ({categoryImages.length})
                        </h4>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                          {categoryImages.map((file, index) => {
                            const globalIndex = images.indexOf(file);
                            return (
                              <div key={index} className="relative group">
                                <img
                                  src={URL.createObjectURL(file)}
                                  alt={`${category} ${index + 1}`}
                                  className="w-full h-32 object-cover rounded-lg"
                                />
                                <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-50 transition-opacity rounded-lg flex items-center justify-center">
                                  <div className="opacity-0 group-hover:opacity-100 flex gap-2">
                                    <button
                                      type="button"
                                      onClick={() => previewFile(file)}
                                      className="p-2 bg-white rounded-full text-gray-700 hover:text-[#0056D2]"
                                    >
                                      <Eye className="w-4 h-4" />
                                    </button>
                                    <button
                                      type="button"
                                      onClick={() => removeImage(globalIndex)}
                                      className="p-2 bg-white rounded-full text-red-600 hover:text-red-800"
                                    >
                                      <X className="w-4 h-4" />
                                    </button>
                                  </div>
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      </div>
                    );
                  })}
                </div>

                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Videos</h3>
                  
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center hover:border-[#0056D2] transition-colors">
                    <input
                      type="file"
                      multiple
                      accept="video/*"
                      onChange={(e) => handleVideoUpload(e.target.files)}
                      className="hidden"
                      id="videos-upload"
                    />
                    <label htmlFor="videos-upload" className="cursor-pointer">
                      <Video className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                      <div className="text-lg font-medium text-gray-900 mb-2">
                        Upload Property Videos
                      </div>
                      <div className="text-gray-600">
                        Add video tours to showcase your property
                      </div>
                      <div className="text-sm text-gray-500 mt-2">
                        MP4, MOV, AVI up to 50MB each
                      </div>
                    </label>
                  </div>

                  {videos.length > 0 && (
                    <div className="mt-6">
                      <h4 className="font-medium text-gray-900 mb-3">
                        Uploaded Videos ({videos.length})
                      </h4>
                      <div className="space-y-3">
                        {videos.map((file, index) => (
                          <div key={index} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border">
                            <div className="flex items-center">
                              <Video className="w-8 h-8 text-[#0056D2] mr-3" />
                              <div>
                                <div className="font-medium text-gray-900">{file.name}</div>
                                <div className="text-sm text-gray-600">
                                  {fileUtils.formatFileSize(file.size)}
                                </div>
                              </div>
                            </div>
                            <button
                              type="button"
                              onClick={() => removeVideo(index)}
                              className="text-red-600 hover:text-red-800 p-1"
                            >
                              <X className="w-5 h-5" />
                            </button>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </div>

            {/* Submit Button */}
            <div className="pt-6">
              <button
                type="submit"
                className="w-full px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium text-lg"
              >
                Submit Property Listing
              </button>
            </div>
          </form>
        </div>

        {/* Image Preview Modal */}
        {previewImage && (
          <div className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50">
            <div className="relative max-w-4xl max-h-full p-4">
              <img src={previewImage} alt="Preview" className="max-w-full max-h-full object-contain" />
              <button
                onClick={() => setPreviewImage(null)}
                className="absolute top-4 right-4 text-white hover:text-gray-300"
              >
                <X className="w-8 h-8" />
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};