import React, { useState, useEffect, useCallback } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { GoogleMap } from '../components/GoogleMap';
import { Badge } from '../components/ui/badge';
import { Loader2, AlertCircle, Search, Filter, Eye, Lightbulb, CheckCircle, Palette, IndianRupee, Home, Maximize, Tag, TrendingUp, MapPin, BarChart3 } from 'lucide-react';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { MarketPositioning } from '../types/marketPositioning';
import { enrichPropertiesWithMarketData } from '../utils/marketPositioning';
import { MarketPositionBadge } from '../components/MarketPositionBadge';
import { GrowthZoneType, LocalityGrowthZone } from '../types/investmentGrowth';
import { SaturationLevel, ProjectPhase } from '../types/supplyPipeline';
import { FloodRiskData, UrbanGrowthData, InfrastructureData } from '../types/gee';
import { geeService } from '../services/geeService';
import { FloodRiskCard } from '../components/gee/FloodRiskCard';
import { UrbanGrowthCard } from '../components/gee/UrbanGrowthCard';
import { InfrastructureCard } from '../components/gee/InfrastructureCard';
import { Droplets, Building2, Navigation } from 'lucide-react';

interface PropertyData {
  id: number;
  title: string;
  latitude: number;
  longitude: number;
  sale_price?: number;
  monthly_rent?: number;
  listing_type: string;
  property_type: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  locality: string;
  city: string;
  primary_image?: string;
  amenities?: string[];
  market_positioning?: MarketPositioning | null;
  floodData?: FloodRiskData;
  urbanGrowthData?: UrbanGrowthData;
  infrastructureData?: InfrastructureData;
}

interface MapBounds {
  north: number;
  south: number;
  east: number;
  west: number;
}

interface SmartVisualizationProps {
  visualizationMode: 'price' | 'type' | 'size' | 'listing' | 'market' | 'growth' | 'supply' | 'flood' | 'urban' | 'infra';
  onModeChange: (mode: 'price' | 'type' | 'size' | 'listing' | 'market' | 'growth' | 'supply' | 'flood' | 'urban' | 'infra') => void;
  propertyStats: {
    totalProperties: number;
    avgPrice: number;
    priceRange: [number, number];
    typeDistribution: Record<string, number>;
    listingDistribution: Record<string, number>;
  };
  filteredProperties: PropertyData[];
  onVisualizationFilter: (allSelections: Record<string, string[]>) => void;
}

const SmartVisualization: React.FC<SmartVisualizationProps> = ({
  visualizationMode,
  onModeChange,
  propertyStats,
  filteredProperties,
  onVisualizationFilter
}) => {
  const [allModeSelections, setAllModeSelections] = useState<Record<string, string[]>>({
    price: [],
    type: [],
    size: [],
    listing: [],
    market: [],
    growth: [],
    supply: [],
    flood: [],
    urban: [],
    infra: []
  });
  
  const selectedCategories = allModeSelections[visualizationMode] || [];
  const visualizationModes = [
    {
      id: 'price' as const,
      label: 'Price Heatmap',
      icon: IndianRupee,
      description: 'Color-coded by property prices',
      color: 'bg-red-500'
    },
    {
      id: 'type' as const,
      label: 'Property Type',
      icon: Home,
      description: 'Grouped by property categories',
      color: 'bg-blue-500'
    },
    {
      id: 'size' as const,
      label: 'Size Mapping',
      icon: Maximize,
      description: 'Sized by built-up area',
      color: 'bg-green-500'
    },
    {
      id: 'listing' as const,
      label: 'Listing Type',
      icon: Tag,
      description: 'Categorized by listing status',
      color: 'bg-purple-500'
    },
    {
      id: 'market' as const,
      label: 'Market Position',
      icon: TrendingUp,
      description: 'Value analysis & pricing',
      color: 'bg-emerald-500'
    },
    {
      id: 'growth' as const,
      label: 'Growth Zones',
      icon: BarChart3,
      description: 'Investment potential areas',
      color: 'bg-orange-500'
    },
    {
      id: 'supply' as const,
      label: 'Supply Pipeline',
      icon: MapPin,
      description: 'Upcoming supply & saturation',
      color: 'bg-purple-500'
    },
    {
      id: 'flood' as const,
      label: 'Flood Risk',
      icon: Droplets,
      description: 'Flood risk assessment',
      color: 'bg-cyan-500'
    },
    {
      id: 'urban' as const,
      label: 'Urban Growth',
      icon: Building2,
      description: 'Development & investment potential',
      color: 'bg-indigo-500'
    },
    {
      id: 'infra' as const,
      label: 'Infrastructure',
      icon: Navigation,
      description: 'Accessibility & connectivity',
      color: 'bg-teal-500'
    }
  ];

  const getLegendItems = () => {
    switch (visualizationMode) {
      case 'price':
        const priceOver1Cr = filteredProperties.filter(p => (p.sale_price || p.monthly_rent || 0) >= 10000000).length;
        const price50LTo1Cr = filteredProperties.filter(p => {
          const price = p.sale_price || p.monthly_rent || 0;
          return price >= 5000000 && price < 10000000;
        }).length;
        const price20LTo50L = filteredProperties.filter(p => {
          const price = p.sale_price || p.monthly_rent || 0;
          return price >= 2000000 && price < 5000000;
        }).length;
        const priceUnder20L = filteredProperties.filter(p => (p.sale_price || p.monthly_rent || 0) < 2000000).length;
        
        return [
          { color: '#dc2626', label: '₹1Cr+', count: priceOver1Cr },
          { color: '#ea580c', label: '₹50L-1Cr', count: price50LTo1Cr },
          { color: '#ca8a04', label: '₹20L-50L', count: price20LTo50L },
          { color: '#16a34a', label: 'Under ₹20L', count: priceUnder20L }
        ];
      case 'type':
        return [
          { color: '#3b82f6', label: 'Apartment', count: propertyStats.typeDistribution.apartment || 0 },
          { color: '#8b5cf6', label: 'Villa', count: propertyStats.typeDistribution.villa || 0 },
          { color: '#06b6d4', label: 'Independent House', count: propertyStats.typeDistribution['independent-house'] || 0 },
          { color: '#f59e0b', label: 'Commercial', count: propertyStats.typeDistribution.commercial || 0 }
        ];
      case 'listing':
        return [
          { color: '#dc2626', label: 'Urgent Sale', count: propertyStats.listingDistribution['urgent-sale'] || 0 },
          { color: '#7c3aed', label: 'Auction', count: propertyStats.listingDistribution.bidding || 0 },
          { color: '#059669', label: 'For Rent', count: propertyStats.listingDistribution.rent || 0 },
          { color: '#2563eb', label: 'For Sale', count: propertyStats.listingDistribution.sell || 0 }
        ];
      case 'size':
        const size2000Plus = filteredProperties.filter(p => (p.built_up_area || 0) >= 2000).length;
        const size1200To2000 = filteredProperties.filter(p => {
          const area = p.built_up_area || 0;
          return area >= 1200 && area < 2000;
        }).length;
        const size800To1200 = filteredProperties.filter(p => {
          const area = p.built_up_area || 0;
          return area >= 800 && area < 1200;
        }).length;
        const sizeUnder800 = filteredProperties.filter(p => (p.built_up_area || 0) < 800 && (p.built_up_area || 0) > 0).length;
        
        return [
          { color: '#dc2626', label: '2000+ sq.ft', count: size2000Plus },
          { color: '#ea580c', label: '1200-2000 sq.ft', count: size1200To2000 },
          { color: '#ca8a04', label: '800-1200 sq.ft', count: size800To1200 },
          { color: '#16a34a', label: 'Under 800 sq.ft', count: sizeUnder800 }
        ];
      case 'market':
        const highValue = filteredProperties.filter(p => p.market_positioning?.market_position_tag === 'HIGH_VALUE').length;
        const smartBuy = filteredProperties.filter(p => p.market_positioning?.market_position_tag === 'SMART_BUY').length;
        const balanced = filteredProperties.filter(p => p.market_positioning?.market_position_tag === 'BALANCED').length;
        const overpriced = filteredProperties.filter(p => p.market_positioning?.market_position_tag === 'OVERPRICED').length;
        const extremeOverpriced = filteredProperties.filter(p => p.market_positioning?.market_position_tag === 'EXTREME_OVERPRICED').length;
        
        console.log('🏷️ Market Position Counts:', { highValue, smartBuy, balanced, overpriced, extremeOverpriced, total: filteredProperties.length });
        
        return [
          { color: '#10b981', label: 'High Value', count: highValue },
          { color: '#3b82f6', label: 'Smart Buy', count: smartBuy },
          { color: '#6b7280', label: 'Fairly Priced', count: balanced },
          { color: '#f59e0b', label: 'Overpriced', count: overpriced },
          { color: '#ef4444', label: 'Extremely Overpriced', count: extremeOverpriced }
        ];
      case 'growth':
        // Mock growth zone data - will be replaced with real API data
        const emerging = Math.floor(filteredProperties.length * 0.25);
        const stable = Math.floor(filteredProperties.length * 0.35);
        const saturated = Math.floor(filteredProperties.length * 0.25);
        const declining = Math.floor(filteredProperties.length * 0.15);
        
        return [
          { color: '#22c55e', label: 'Emerging', count: emerging },
          { color: '#3b82f6', label: 'Stable', count: stable },
          { color: '#f59e0b', label: 'Saturated', count: saturated },
          { color: '#ef4444', label: 'Declining', count: declining }
        ];
      case 'supply':
        // Mock supply pipeline data - will be replaced with real API data
        const undersupplied = Math.floor(filteredProperties.length * 0.2);
        const balancedSupply = Math.floor(filteredProperties.length * 0.4);
        const oversupplied = Math.floor(filteredProperties.length * 0.25);
        const saturatedSupply = Math.floor(filteredProperties.length * 0.15);
        
        return [
          { color: '#ef4444', label: 'Undersupplied', count: undersupplied },
          { color: '#22c55e', label: 'Balanced Supply', count: balancedSupply },
          { color: '#f59e0b', label: 'Oversupplied', count: oversupplied },
          { color: '#f97316', label: 'Saturated', count: saturatedSupply }
        ];
      case 'flood':
        const lowRisk = filteredProperties.filter(p => p.floodData && p.floodData.score <= 3).length;
        const mediumRisk = filteredProperties.filter(p => p.floodData && p.floodData.score > 3 && p.floodData.score <= 6).length;
        const highRisk = filteredProperties.filter(p => p.floodData && p.floodData.score > 6).length;
        const noData = filteredProperties.filter(p => !p.floodData).length;
        
        return [
          { color: '#10b981', label: 'Low Risk (0-3)', count: lowRisk },
          { color: '#f59e0b', label: 'Medium Risk (4-6)', count: mediumRisk },
          { color: '#ef4444', label: 'High Risk (7-10)', count: highRisk },
          { color: '#9ca3af', label: 'No Data', count: noData }
        ];
      case 'urban':
        const emergingGrowth = filteredProperties.filter(p => p.urbanGrowthData && p.urbanGrowthData.developmentLevel === 'emerging').length;
        const developingGrowth = filteredProperties.filter(p => p.urbanGrowthData && p.urbanGrowthData.developmentLevel === 'developing').length;
        const matureGrowth = filteredProperties.filter(p => p.urbanGrowthData && p.urbanGrowthData.developmentLevel === 'mature').length;
        const saturatedGrowth = filteredProperties.filter(p => p.urbanGrowthData && p.urbanGrowthData.developmentLevel === 'saturated').length;
        
        return [
          { color: '#10b981', label: 'Emerging (High Potential)', count: emergingGrowth },
          { color: '#3b82f6', label: 'Developing', count: developingGrowth },
          { color: '#f59e0b', label: 'Mature', count: matureGrowth },
          { color: '#6b7280', label: 'Saturated', count: saturatedGrowth }
        ];
      case 'infra':
        const excellentConn = filteredProperties.filter(p => p.infrastructureData && p.infrastructureData.connectivityLevel === 'excellent').length;
        const goodConn = filteredProperties.filter(p => p.infrastructureData && p.infrastructureData.connectivityLevel === 'good').length;
        const avgConn = filteredProperties.filter(p => p.infrastructureData && p.infrastructureData.connectivityLevel === 'average').length;
        const poorConn = filteredProperties.filter(p => p.infrastructureData && p.infrastructureData.connectivityLevel === 'poor').length;
        
        return [
          { color: '#10b981', label: 'Excellent Connectivity', count: excellentConn },
          { color: '#3b82f6', label: 'Good Connectivity', count: goodConn },
          { color: '#f59e0b', label: 'Average Connectivity', count: avgConn },
          { color: '#ef4444', label: 'Poor Connectivity', count: poorConn }
        ];
      default:
        return [];
    }
  };

  return (
    <Card className="h-full">
      <CardHeader className="pb-3">
        <CardTitle className="flex items-center gap-2">
          <Palette className="w-5 h-5" />
          Smart Visualization
        </CardTitle>
        <div className="text-sm text-gray-600">
          Choose how to visualize property data
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-2 gap-3">
          {visualizationModes.map((mode) => {
            const Icon = mode.icon;
            return (
              <Button
                key={mode.id}
                variant={visualizationMode === mode.id ? "default" : "outline"}
                size="lg"
                className="h-auto p-4 flex flex-col items-center gap-3"
                onClick={() => onModeChange(mode.id)}
              >
                <Icon className="w-6 h-6" />
                <div className="text-center">
                  <div className="font-medium text-sm">{mode.label}</div>
                  <div className="text-xs opacity-70">{mode.description}</div>
                </div>
              </Button>
            );
          })}
        </div>
        <div className="space-y-3">
          <div className="flex items-center gap-2">
            <BarChart3 className="w-5 h-5" />
            <span className="font-medium text-base">Select Categories</span>
          </div>
          <div className="space-y-2">
            {getLegendItems().map((item, index) => {
              const isSelected = selectedCategories.includes(item.label);
              return (
                <button
                  key={index}
                  onClick={() => {
                    const newSelections = { ...allModeSelections };
                    if (isSelected) {
                      newSelections[visualizationMode] = newSelections[visualizationMode].filter(cat => cat !== item.label);
                    } else {
                      newSelections[visualizationMode] = [...newSelections[visualizationMode], item.label];
                    }
                    setAllModeSelections(newSelections);
                    onVisualizationFilter(newSelections);
                  }}
                  className={`w-full flex items-center justify-between p-3 rounded-lg border transition-all ${
                    isSelected 
                      ? 'bg-blue-50 border-blue-300 shadow-sm' 
                      : 'bg-white border-gray-200 hover:bg-gray-50'
                  }`}
                >
                  <div className="flex items-center gap-3">
                    <div
                      className={`w-5 h-5 rounded-full border-2 ${
                        isSelected ? 'border-blue-500' : 'border-gray-300'
                      }`}
                      style={{ backgroundColor: item.color }}
                    />
                    <span className="text-base font-medium">{item.label}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    {item.count > 0 && (
                      <Badge variant="secondary" className="text-sm">
                        {item.count}
                      </Badge>
                    )}
                    {isSelected && (
                      <CheckCircle className="w-5 h-5 text-blue-500" />
                    )}
                  </div>
                </button>
              );
            })}
          </div>
        </div>
        <div className="border-t pt-4">
          <div className="flex gap-3">
            <Button 
              variant="outline" 
              className="flex-1 bg-gray-100 hover:bg-gray-200 text-gray-700 border-gray-300"
              onClick={() => {
                setAllModeSelections({
                  price: [],
                  type: [],
                  size: [],
                  listing: [],
                  market: [],
                  growth: [],
                  supply: [],
                  flood: [],
                  urban: [],
                  infra: []
                });
                onVisualizationFilter({
                  price: [],
                  type: [],
                  size: [],
                  listing: [],
                  market: [],
                  growth: [],
                  supply: [],
                  flood: [],
                  urban: [],
                  infra: []
                });
              }}
            >
              Clear
            </Button>
            <Button 
              className="flex-1 bg-blue-600 hover:bg-blue-700 text-white"
              onClick={() => {
                onVisualizationFilter(allModeSelections);
              }}
            >
              Apply
            </Button>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export const SmartLandMapsPage: React.FC = () => {
  const [properties, setProperties] = useState<PropertyData[]>([]);
  const [filteredProperties, setFilteredProperties] = useState<PropertyData[]>([]);
  const [selectedProperty, setSelectedProperty] = useState<PropertyData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchQuery, setSearchQuery] = useState('');
  const [visualizationMode, setVisualizationMode] = useState<'price' | 'type' | 'size' | 'listing' | 'market' | 'growth' | 'supply' | 'flood' | 'urban' | 'infra'>('price');
  const [visualizationFilters, setVisualizationFilters] = useState<string[]>([]);
  const [visualizedProperties, setVisualizedProperties] = useState<PropertyData[]>([]);
  const [finalDisplayProperties, setFinalDisplayProperties] = useState<PropertyData[]>([]);
  const [showFilters, setShowFilters] = useState(true);
  const [mapBounds, setMapBounds] = useState<MapBounds | null>(null);
  const [hasSearched, setHasSearched] = useState(false);
  const [hasFiltered, setHasFiltered] = useState(false);
  const [clearClicked, setClearClicked] = useState(false);
  const [applyClicked, setApplyClicked] = useState(false);
  const [loadingFloodData, setLoadingFloodData] = useState(false);
  const [showFloodCard, setShowFloodCard] = useState(false);
  const [loadingUrbanData, setLoadingUrbanData] = useState(false);
  const [showUrbanCard, setShowUrbanCard] = useState(false);
  const [loadingInfraData, setLoadingInfraData] = useState(false);
  const [showInfraCard, setShowInfraCard] = useState(false);
  const [filters, setFilters] = useState({
    priceRange: [0, 50000000] as [number, number],
    propertyType: 'all',
    bedrooms: 'any',
    listingType: 'all',
    marketPosition: 'all',
    growthZone: 'all',
    floodRisk: 'all',
    urbanGrowth: 'all',
    infrastructure: 'all'
  });

  // Generate coordinates for properties without lat/lng (same as Properties page)
  const generateCoordinates = (city: string) => {
    const cityCoordinates = {
      'Mumbai': { lat: 19.0760, lng: 72.8777 },
      'Delhi': { lat: 28.7041, lng: 77.1025 },
      'Bangalore': { lat: 12.9716, lng: 77.5946 },
      'Chennai': { lat: 13.0827, lng: 80.2707 },
      'Hyderabad': { lat: 17.3850, lng: 78.4867 },
      'Pune': { lat: 18.5204, lng: 73.8567 },
      'Kolkata': { lat: 22.5726, lng: 88.3639 },
      'Ahmedabad': { lat: 23.0225, lng: 72.5714 },
      'Jaipur': { lat: 26.9124, lng: 75.7873 },
      'Kochi': { lat: 9.9312, lng: 76.2673 }
    };
    
    const cityCoord = cityCoordinates[city as keyof typeof cityCoordinates];
    if (cityCoord) {
      return {
        latitude: cityCoord.lat + (Math.random() - 0.5) * 0.1,
        longitude: cityCoord.lng + (Math.random() - 0.5) * 0.1
      };
    }
    return {
      latitude: 12.9716 + (Math.random() - 0.5) * 0.1,
      longitude: 77.5946 + (Math.random() - 0.5) * 0.1
    };
  };

  // Load all properties using the same API as Properties page
  const loadProperties = useCallback(async () => {
    setLoading(true);
    setError(null);

    try {
      // Use the same API as Properties page
      const { api } = await import('../utils/api');
      const allProperties = await api.fetchPropertiesWithFilters({});
      console.log('📊 All properties loaded:', allProperties.length);
      
      // Transform to match PropertyData interface
      const propertyData: PropertyData[] = allProperties.map((prop: any) => {
        const coords = prop.latitude && prop.longitude 
          ? { latitude: prop.latitude, longitude: prop.longitude }
          : generateCoordinates(prop.city);
        
        return {
          id: prop.id,
          title: prop.title,
          latitude: coords.latitude,
          longitude: coords.longitude,
          sale_price: prop.sale_price,
          monthly_rent: prop.monthly_rent,
          listing_type: prop.listing_type,
          property_type: prop.property_type,
          bedrooms: prop.bedrooms,
          bathrooms: prop.bathrooms,
          built_up_area: prop.built_up_area,
          locality: prop.locality,
          city: prop.city,
          primary_image: prop.primary_image || prop.photo,
          amenities: prop.amenities || []
        };
      });
      
      console.log('🔧 Enriching properties with market positioning...');
      const enrichedProperties = enrichPropertiesWithMarketData(propertyData);
      
      // Enrich with flood risk data
      console.log('🌊 Fetching flood risk data...');
      setLoadingFloodData(true);
      const propertiesWithFlood = await enrichWithFloodData(enrichedProperties);
      setLoadingFloodData(false);
      
      // Enrich with urban growth data
      console.log('🏗️ Fetching urban growth data...');
      setLoadingUrbanData(true);
      const propertiesWithUrban = await enrichWithUrbanGrowthData(propertiesWithFlood);
      setLoadingUrbanData(false);
      
      // Enrich with infrastructure data
      console.log('🛣️ Fetching infrastructure data...');
      setLoadingInfraData(true);
      const propertiesWithInfra = await enrichWithInfrastructureData(propertiesWithUrban);
      setLoadingInfraData(false);
      
      setProperties(propertiesWithInfra);
      setFilteredProperties(propertiesWithInfra);
    } catch (err) {
      console.error('Failed to load properties:', err);
      setError('Failed to load properties. Please try again.');
    } finally {
      setLoading(false);
    }
  }, []);

  // Enrich properties with flood risk data
  const enrichWithFloodData = async (properties: PropertyData[]): Promise<PropertyData[]> => {
    try {
      const locations = properties.map(p => ({ lat: p.latitude, lng: p.longitude }));
      const floodDataArray = await geeService.batchGetFloodRisk(locations);
      
      return properties.map((prop, index) => ({
        ...prop,
        floodData: floodDataArray[index]
      }));
    } catch (error) {
      console.warn('Failed to fetch flood data:', error);
      return properties;
    }
  };

  // Enrich properties with urban growth data
  const enrichWithUrbanGrowthData = async (properties: PropertyData[]): Promise<PropertyData[]> => {
    try {
      const locations = properties.map(p => ({ lat: p.latitude, lng: p.longitude }));
      const urbanDataArray = await geeService.batchGetUrbanGrowth(locations);
      
      return properties.map((prop, index) => ({
        ...prop,
        urbanGrowthData: urbanDataArray[index]
      }));
    } catch (error) {
      console.warn('Failed to fetch urban growth data:', error);
      return properties;
    }
  };

  // Enrich properties with infrastructure data
  const enrichWithInfrastructureData = async (properties: PropertyData[]): Promise<PropertyData[]> => {
    try {
      const locations = properties.map(p => ({ 
        lat: p.latitude, 
        lng: p.longitude,
        city: p.city 
      }));
      const infraDataArray = await geeService.batchGetInfrastructure(locations);
      
      return properties.map((prop, index) => ({
        ...prop,
        infrastructureData: infraDataArray[index]
      }));
    } catch (error) {
      console.warn('Failed to fetch infrastructure data:', error);
      return properties;
    }
  };

  // Calculate map center based on filtered properties
  const calculateMapCenter = (filteredProps: PropertyData[]) => {
    if (filteredProps.length === 0) {
      return { lat: 20.5937, lng: 78.9629 }; // Default India center
    }
    
    const avgLat = filteredProps.reduce((sum, prop) => sum + prop.latitude, 0) / filteredProps.length;
    const avgLng = filteredProps.reduce((sum, prop) => sum + prop.longitude, 0) / filteredProps.length;
    
    return { lat: avgLat, lng: avgLng };
  };

  // Handle search - focuses on city and locality
  const handleSearch = (query: string) => {
    setSearchQuery(query);
    if (query.trim()) {
      setHasSearched(true);
    }
  };

  // Quick filter handlers
  const handleQuickFilter = (type: string, value: string) => {
    const quickFilters = {
      priceRange: [0, 50000000],
      propertyType: type === 'propertyType' ? value : 'all',
      bedrooms: type === 'bedrooms' ? value : 'any',
      listingType: type === 'listingType' ? value : 'all',
      amenities: []
    };
    
    if (type === 'price') {
      if (value === 'under50L') quickFilters.priceRange = [0, 5000000];
      if (value === 'under1Cr') quickFilters.priceRange = [0, 10000000];
    }
    
    handleFiltersChange(quickFilters);
  };

  // Popular searches
  const popularSearches = ['Mumbai', 'Bengaluru', 'Delhi', 'Pune', 'Chennai'];
  const quickFilters = [
    { label: 'Under ₹50L', type: 'price', value: 'under50L' },
    { label: 'Under ₹1Cr', type: 'price', value: 'under1Cr' },
    { label: '2-3BHK', type: 'bedrooms', value: '2' },
    { label: 'Apartments', type: 'propertyType', value: 'apartment' }
  ];

  // Handle filters - applies ALL active filters together
  const handleFiltersChange = useCallback((newFilters: any) => {
    // Merge with current filters state
    const activeFilters = { ...filters, ...newFilters };
    console.log('🔍 Applying filters:', activeFilters);
    console.log('📊 Total properties before filtering:', properties.length);
    
    setHasFiltered(true);
    let filtered = [...properties]; // Start with all properties
    
    // Apply search query first - prioritize city and locality
    if (searchQuery.trim()) {
      const query = searchQuery.toLowerCase();
      filtered = filtered.filter(prop => 
        (prop.city && prop.city.toLowerCase().includes(query)) ||
        (prop.locality && prop.locality.toLowerCase().includes(query))
      );
      console.log('📊 After city/locality search filter:', filtered.length);
    }
    
    // Apply price range filter
    if (activeFilters.priceRange && Array.isArray(activeFilters.priceRange)) {
      filtered = filtered.filter(prop => {
        const price = prop.sale_price || prop.monthly_rent || 0;
        const inRange = price >= activeFilters.priceRange[0] && price <= activeFilters.priceRange[1];
        return inRange;
      });
      console.log('📊 After price filter:', filtered.length);
    }
    
    // Apply property type filter
    if (activeFilters.propertyType && activeFilters.propertyType !== 'all') {
      console.log('🔎 Looking for property type:', activeFilters.propertyType);
      console.log('🔎 Available types in current set:', [...new Set(filtered.map(p => p.property_type))]);
      
      filtered = filtered.filter(prop => {
        const matches = prop.property_type?.toLowerCase() === activeFilters.propertyType.toLowerCase();
        if (matches) {
          console.log('✅ Match found:', prop.title, 'Type:', prop.property_type);
        }
        return matches;
      });
      console.log('📊 After property type filter:', filtered.length);
    }
    
    // Apply bedrooms filter
    if (activeFilters.bedrooms && activeFilters.bedrooms !== 'any') {
      filtered = filtered.filter(prop => {
        if (activeFilters.bedrooms === '4+') {
          return (prop.bedrooms || 0) >= 4;
        } else {
          const targetBedrooms = parseInt(activeFilters.bedrooms);
          return prop.bedrooms === targetBedrooms;
        }
      });
      console.log('📊 After bedrooms filter:', filtered.length);
    }
    
    // Apply listing type filter
    if (activeFilters.listingType && activeFilters.listingType !== 'all') {
      filtered = filtered.filter(prop => {
        const matches = prop.listing_type?.toLowerCase() === activeFilters.listingType.toLowerCase();
        return matches;
      });
      console.log('📊 After listing type filter:', filtered.length);
    }
    
    // Apply market position filter
    if (activeFilters.marketPosition && activeFilters.marketPosition !== 'all') {
      console.log('🏷️ Filtering by market position:', activeFilters.marketPosition);
      console.log('🏷️ Properties before market filter:', filtered.length);
      console.log('🏷️ Properties with market_positioning field:', filtered.filter(p => p.market_positioning).length);
      
      // Log sample properties to see their market tags
      const sampleProps = filtered.slice(0, 5);
      console.log('🏷️ Sample properties market tags:', sampleProps.map(p => ({
        title: p.title,
        tag: p.market_positioning?.market_position_tag,
        hasMarketData: !!p.market_positioning
      })));
      
      if (activeFilters.marketPosition === 'hide_extreme') {
        filtered = filtered.filter(prop => 
          prop.market_positioning?.market_position_tag !== 'EXTREME_OVERPRICED'
        );
      } else {
        const beforeCount = filtered.length;
        filtered = filtered.filter(prop => {
          const tag = prop.market_positioning?.market_position_tag;
          const matches = tag === activeFilters.marketPosition;
          if (matches) {
            console.log('✅ Market match found:', prop.title, 'Tag:', tag);
          }
          return matches;
        });
        console.log(`🏷️ Filtered from ${beforeCount} to ${filtered.length} properties`);
      }
      console.log('📊 After market position filter:', filtered.length);
    }
    
    // Apply amenities filter (property must have at least one of the selected amenities)
    if (activeFilters.amenities && Array.isArray(activeFilters.amenities) && activeFilters.amenities.length > 0) {
      filtered = filtered.filter(prop => {
        if (!prop.amenities || prop.amenities.length === 0) return false;
        
        return activeFilters.amenities.some((selectedAmenity: string) => 
          prop.amenities?.some(propAmenity => 
            propAmenity.toLowerCase().includes(selectedAmenity.toLowerCase())
          )
        );
      });
      console.log('📊 After amenities filter:', filtered.length);
    }
    
    // Apply flood risk filter
    if (activeFilters.floodRisk && activeFilters.floodRisk !== 'all') {
      console.log('🌊 Filtering by flood risk:', activeFilters.floodRisk);
      filtered = filtered.filter(prop => {
        if (!prop.floodData) return false;
        
        const score = prop.floodData.score;
        if (activeFilters.floodRisk === 'low') return score <= 3;
        if (activeFilters.floodRisk === 'medium') return score > 3 && score <= 6;
        if (activeFilters.floodRisk === 'high') return score > 6;
        if (activeFilters.floodRisk === 'low_medium') return score <= 6;
        return true;
      });
      console.log('📊 After flood risk filter:', filtered.length);
    }
    
    // Apply urban growth filter
    if (activeFilters.urbanGrowth && activeFilters.urbanGrowth !== 'all') {
      console.log('🏗️ Filtering by urban growth:', activeFilters.urbanGrowth);
      filtered = filtered.filter(prop => {
        if (!prop.urbanGrowthData) return false;
        
        const level = prop.urbanGrowthData.developmentLevel;
        if (activeFilters.urbanGrowth === 'emerging') return level === 'emerging';
        if (activeFilters.urbanGrowth === 'developing') return level === 'developing';
        if (activeFilters.urbanGrowth === 'mature') return level === 'mature';
        if (activeFilters.urbanGrowth === 'saturated') return level === 'saturated';
        if (activeFilters.urbanGrowth === 'high_potential') return level === 'emerging' || level === 'developing';
        return true;
      });
      console.log('📊 After urban growth filter:', filtered.length);
    }
    
    // Apply infrastructure filter
    if (activeFilters.infrastructure && activeFilters.infrastructure !== 'all') {
      console.log('🛣️ Filtering by infrastructure:', activeFilters.infrastructure);
      filtered = filtered.filter(prop => {
        if (!prop.infrastructureData) return false;
        
        const level = prop.infrastructureData.connectivityLevel;
        if (activeFilters.infrastructure === 'excellent') return level === 'excellent';
        if (activeFilters.infrastructure === 'good') return level === 'good';
        if (activeFilters.infrastructure === 'average') return level === 'average';
        if (activeFilters.infrastructure === 'poor') return level === 'poor';
        if (activeFilters.infrastructure === 'good_excellent') return level === 'excellent' || level === 'good';
        return true;
      });
      console.log('📊 After infrastructure filter:', filtered.length);
    }
    
    console.log('✅ Final filtered properties:', filtered.length);
    setFilteredProperties(filtered);
  }, [properties, searchQuery, filters]);

  // Calculate property statistics
  const validPrices = properties.map(p => p.sale_price || p.monthly_rent || 0).filter(p => p > 0);
  const propertyStats = {
    totalProperties: filteredProperties.length,
    avgPrice: filteredProperties.reduce((sum, prop) => sum + (prop.sale_price || prop.monthly_rent || 0), 0) / filteredProperties.length || 0,
    priceRange: [0, 50000000] as [number, number], // Fixed range: 0L to 5cr
    typeDistribution: properties.reduce((acc, prop) => {
      acc[prop.property_type] = (acc[prop.property_type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>),
    listingDistribution: properties.reduce((acc, prop) => {
      acc[prop.listing_type] = (acc[prop.listing_type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>)
  };

  // Load initial data
  useEffect(() => {
    loadProperties();
  }, [loadProperties]);
  
  // Apply filters when search query changes
  useEffect(() => {
    handleFiltersChange({});
  }, [searchQuery, handleFiltersChange]);

  // Update final display properties when filteredProperties change
  useEffect(() => {
    if (visualizedProperties.length === 0) {
      setFinalDisplayProperties(filteredProperties);
    }
  }, [filteredProperties, visualizedProperties]);

  return (
    <div className="min-h-screen bg-white flex flex-col">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 px-6 py-6">
        <div className="max-w-7xl mx-auto">
          <div className="text-center mb-6">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Smart Land Maps</h1>
            <p className="text-gray-600">
              Interactive property visualization with advanced mapping technology
            </p>
          </div>

          {/* Search Bar */}
          <div className="flex justify-center mb-6">
            <div className="relative w-full max-w-md">
              <Input
                placeholder="Search Cities, localities..."
                value={searchQuery}
                onChange={(e) => handleSearch(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && handleFiltersChange({})}
                className="pl-4 pr-20 h-12 text-lg"
              />
              <Button
                onClick={() => handleFiltersChange({})}
                className="absolute right-2 top-1/2 transform -translate-y-1/2 h-8 px-4 text-sm"
              >
                Search
              </Button>
            </div>
          </div>


        </div>
      </div>

      {/* Error Banner */}
      {error && (
        <div className="bg-red-50 border-l-4 border-red-400 p-4 mx-6 mt-4 rounded">
          <div className="flex">
            <AlertCircle className="h-5 w-5 text-red-400" />
            <div className="ml-3">
              <p className="text-sm text-red-700">{error}</p>
            </div>
          </div>
        </div>
      )}

      {/* Filters Section */}
      <div className="bg-gray-50 border-b">
        <div className="max-w-7xl mx-auto p-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Smart Filters */}
            <div className="bg-white rounded-lg p-6 shadow-sm border">
              <h3 className="text-lg font-semibold mb-6">Smart Filters</h3>
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Price Range</label>
                  <div className="relative">
                    <input 
                      type="range" 
                      min="0" 
                      max="50000000" 
                      value={filters.priceRange[1]}
                      onChange={(e) => setFilters(prev => ({ ...prev, priceRange: [0, parseInt(e.target.value)] as [number, number] }))}
                      className="w-full h-2 bg-blue-600 rounded-lg appearance-none cursor-pointer" 
                    />
                    <div className="flex justify-between text-xs text-gray-500 mt-2">
                      <span>₹0.0L</span>
                      <span>₹{(filters.priceRange[1] / 10000000).toFixed(1)}cr</span>
                    </div>
                  </div>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Property Type</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.propertyType}
                    onChange={(e) => setFilters(prev => ({ ...prev, propertyType: e.target.value }))}
                  >
                    <option value="all">All Types</option>
                    <option value="apartment">Apartment</option>
                    <option value="villa">Villa</option>
                    <option value="house">House</option>
                    <option value="commercial">Commercial</option>
                    <option value="pg-coliving">PG/Co-living</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Bedrooms</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.bedrooms}
                    onChange={(e) => setFilters(prev => ({ ...prev, bedrooms: e.target.value }))}
                  >
                    <option value="any">Any</option>
                    <option value="1">1 BHK</option>
                    <option value="2">2 BHK</option>
                    <option value="3">3 BHK</option>
                    <option value="4">4+ BHK</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Listing Type</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.listingType}
                    onChange={(e) => setFilters(prev => ({ ...prev, listingType: e.target.value }))}
                  >
                    <option value="all">All</option>
                    <option value="sell">For Sale</option>
                    <option value="rent">For Rent</option>
                    <option value="lease">Lease</option>
                    <option value="urgent-sale">Urgent Sale</option>
                    <option value="bidding">Bidding</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Market Position</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.marketPosition}
                    onChange={(e) => setFilters(prev => ({ ...prev, marketPosition: e.target.value }))}
                  >
                    <option value="all">All Properties</option>
                    <option value="HIGH_VALUE">🟢 High Value / Great Deals</option>
                    <option value="SMART_BUY">🔵 Smart Buy</option>
                    <option value="BALANCED">⚪ Fairly Priced</option>
                    <option value="OVERPRICED">🟠 Above Market</option>
                    <option value="hide_extreme">🚫 Hide Extremely Overpriced</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Growth Zone</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.growthZone || 'all'}
                    onChange={(e) => setFilters(prev => ({ ...prev, growthZone: e.target.value }))}
                  >
                    <option value="all">All Zones</option>
                    <option value="EMERGING">Emerging (High Growth)</option>
                    <option value="STABLE">Stable Growth</option>
                    <option value="SATURATED">Saturated Market</option>
                    <option value="DECLINING">Declining Areas</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Flood Risk</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.floodRisk || 'all'}
                    onChange={(e) => setFilters(prev => ({ ...prev, floodRisk: e.target.value }))}
                  >
                    <option value="all">All Risk Levels</option>
                    <option value="low">🟢 Low Risk (0-3)</option>
                    <option value="medium">🟡 Medium Risk (4-6)</option>
                    <option value="high">🔴 High Risk (7-10)</option>
                    <option value="low_medium">✅ Low to Medium Only</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Urban Growth</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.urbanGrowth || 'all'}
                    onChange={(e) => setFilters(prev => ({ ...prev, urbanGrowth: e.target.value }))}
                  >
                    <option value="all">All Development Levels</option>
                    <option value="emerging">Emerging (High Growth)</option>
                    <option value="developing">Developing</option>
                    <option value="mature">Mature</option>
                    <option value="saturated">Saturated</option>
                    <option value="high_potential">High Potential Only</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Infrastructure</label>
                  <select 
                    className="w-full p-3 border border-gray-300 rounded-lg bg-white"
                    value={filters.infrastructure || 'all'}
                    onChange={(e) => setFilters(prev => ({ ...prev, infrastructure: e.target.value }))}
                  >
                    <option value="all">All Connectivity Levels</option>
                    <option value="excellent">Excellent Connectivity</option>
                    <option value="good">Good Connectivity</option>
                    <option value="average">Average Connectivity</option>
                    <option value="poor">Poor Connectivity</option>
                    <option value="good_excellent">Good & Excellent Only</option>
                  </select>
                </div>
                <div className="flex gap-3 pt-4">
                  <Button 
                    variant="outline" 
                    className={`flex-1 ${clearClicked ? 'bg-violet-500 text-white border-violet-500' : 'bg-violet-100 hover:bg-violet-200 text-violet-700 border-violet-300'}`}
                    onClick={() => {
                      setClearClicked(true);
                      setTimeout(() => setClearClicked(false), 200);
                      setFilters({
                        priceRange: [0, 50000000],
                        propertyType: 'all',
                        bedrooms: 'any',
                        listingType: 'all',
                        marketPosition: 'all',
                        growthZone: 'all',
                        floodRisk: 'all',
                        urbanGrowth: 'all',
                        infrastructure: 'all'
                      });
                      handleFiltersChange({
                        priceRange: [0, 50000000],
                        propertyType: 'all',
                        bedrooms: 'any',
                        listingType: 'all',
                        marketPosition: 'all',
                        growthZone: 'all',
                        floodRisk: 'all',
                        urbanGrowth: 'all',
                        infrastructure: 'all'
                      });
                    }}
                  >
                    Clear Filters
                  </Button>
                  <Button 
                    className={`flex-1 ${applyClicked ? 'bg-violet-500 text-white border-violet-500' : 'bg-blue-600 hover:bg-blue-700 text-black'}`}
                    style={{ borderColor: applyClicked ? '#8b5cf6' : '#000000', borderWidth: '1px' }}
                    onClick={() => {
                      setApplyClicked(true);
                      setTimeout(() => setApplyClicked(false), 200);
                      console.log('🔍 Applying filters:', filters);
                      handleFiltersChange(filters);
                    }}
                  >
                    Apply Filters
                  </Button>
                </div>
              </div>
            </div>
            
            {/* Smart Visualization Component */}
            <SmartVisualization 
              visualizationMode={visualizationMode}
              onModeChange={setVisualizationMode}
              propertyStats={propertyStats}
              filteredProperties={filteredProperties}
              onVisualizationFilter={(allSelections) => {
                setVisualizationFilters(allSelections);
                
                // Check if any selections exist across all modes
                const hasAnySelections = Object.values(allSelections).some(selections => selections.length > 0);
                
                if (!hasAnySelections) {
                  setVisualizedProperties([]);
                  setFinalDisplayProperties(filteredProperties);
                  return;
                }
                
                // Filter properties that match ALL selected criteria across modes
                const filtered = filteredProperties.filter(property => {
                  // Check each mode's selections
                  const priceMatch = allSelections.price.length === 0 || allSelections.price.some(category => {
                    const price = property.sale_price || property.monthly_rent || 0;
                    if (category === '₹1Cr+') return price >= 10000000;
                    if (category === '₹50L-1Cr') return price >= 5000000 && price < 10000000;
                    if (category === '₹20L-50L') return price >= 2000000 && price < 5000000;
                    if (category === 'Under ₹20L') return price < 2000000;
                    return false;
                  });
                  
                  const typeMatch = allSelections.type.length === 0 || allSelections.type.some(category => {
                    if (category === 'Apartment') return property.property_type?.toLowerCase() === 'apartment';
                    if (category === 'Villa') return property.property_type?.toLowerCase() === 'villa';
                    if (category === 'Independent House') return property.property_type?.toLowerCase() === 'independent-house';
                    if (category === 'Commercial') return property.property_type?.toLowerCase() === 'commercial';
                    return false;
                  });
                  
                  const listingMatch = allSelections.listing.length === 0 || allSelections.listing.some(category => {
                    if (category === 'Urgent Sale') return property.listing_type?.toLowerCase() === 'urgent-sale';
                    if (category === 'Auction') return property.listing_type?.toLowerCase() === 'bidding';
                    if (category === 'For Rent') return property.listing_type?.toLowerCase() === 'rent';
                    if (category === 'For Sale') return property.listing_type?.toLowerCase() === 'sell';
                    return false;
                  });
                  
                  const sizeMatch = allSelections.size.length === 0 || allSelections.size.some(category => {
                    const area = property.built_up_area || 0;
                    if (category === '2000+ sq.ft') return area >= 2000;
                    if (category === '1200-2000 sq.ft') return area >= 1200 && area < 2000;
                    if (category === '800-1200 sq.ft') return area >= 800 && area < 1200;
                    if (category === 'Under 800 sq.ft') return area < 800 && area > 0;
                    return false;
                  });
                  
                  const marketMatch = allSelections.market.length === 0 || allSelections.market.some(category => {
                    const tag = property.market_positioning?.market_position_tag;
                    if (category === 'High Value') return tag === 'HIGH_VALUE';
                    if (category === 'Smart Buy') return tag === 'SMART_BUY';
                    if (category === 'Fairly Priced') return tag === 'BALANCED';
                    if (category === 'Overpriced') return tag === 'OVERPRICED';
                    if (category === 'Extremely Overpriced') return tag === 'EXTREME_OVERPRICED';
                    return false;
                  });
                  
                  const growthMatch = allSelections.growth.length === 0 || allSelections.growth.some(category => {
                    // Mock growth zone matching - will be replaced with real data
                    const localityHash = (property.locality || '').toLowerCase().charCodeAt(0) % 4;
                    if (category === 'Emerging') return localityHash === 0;
                    if (category === 'Stable') return localityHash === 1;
                    if (category === 'Saturated') return localityHash === 2;
                    if (category === 'Declining') return localityHash === 3;
                    return false;
                  });
                  
                  const supplyMatch = allSelections.supply.length === 0 || allSelections.supply.some(category => {
                    // Mock supply matching - will be replaced with real data
                    const localityHash = (property.locality || '').toLowerCase().charCodeAt(0) % 4;
                    if (category === 'Undersupplied') return localityHash === 0;
                    if (category === 'Balanced Supply') return localityHash === 1;
                    if (category === 'Oversupplied') return localityHash === 2;
                    if (category === 'Saturated') return localityHash === 3;
                    return false;
                  });
                  
                  const urbanMatch = allSelections.urban.length === 0 || allSelections.urban.some(category => {
                    const level = property.urbanGrowthData?.developmentLevel;
                    if (category === 'Emerging (High Potential)') return level === 'emerging';
                    if (category === 'Developing') return level === 'developing';
                    if (category === 'Mature') return level === 'mature';
                    if (category === 'Saturated') return level === 'saturated';
                    return false;
                  });
                  
                  const infraMatch = allSelections.infra.length === 0 || allSelections.infra.some(category => {
                    const level = property.infrastructureData?.connectivityLevel;
                    if (category === 'Excellent Connectivity') return level === 'excellent';
                    if (category === 'Good Connectivity') return level === 'good';
                    if (category === 'Average Connectivity') return level === 'average';
                    if (category === 'Poor Connectivity') return level === 'poor';
                    return false;
                  });
                  
                  // Property must match ALL modes that have selections
                  return priceMatch && typeMatch && listingMatch && sizeMatch && marketMatch && growthMatch && supplyMatch && urbanMatch && infraMatch;
                });
                
                console.log('🎯 Visualization Filter Debug:', {
                  allSelections,
                  hasAnySelections,
                  totalProperties: filteredProperties.length,
                  filteredCount: filtered.length,
                  sampleProperty: filteredProperties[0] ? {
                    title: filteredProperties[0].title,
                    price: filteredProperties[0].sale_price || filteredProperties[0].monthly_rent,
                    type: filteredProperties[0].property_type,
                    listing: filteredProperties[0].listing_type,
                    area: filteredProperties[0].built_up_area,
                    marketTag: filteredProperties[0].market_positioning?.market_position_tag
                  } : null
                });
                
                setVisualizedProperties(filtered);
                setFinalDisplayProperties(filtered);
              }}
            />
          </div>
        </div>
      </div>

      {/* Map Section */}
      <div className="flex-1 px-6 pb-6">
        <div className="max-w-7xl mx-auto">
          <div className="h-[600px] relative rounded-lg overflow-hidden shadow-lg border">
            <div className="absolute top-4 left-4 z-20 bg-white/90 backdrop-blur-sm px-3 py-2 rounded-lg shadow-md">
              <h3 className="text-sm font-semibold text-gray-800">Smart Property Locations</h3>
            </div>
            <GoogleMap 
              properties={finalDisplayProperties.map(prop => ({
                id: prop.id,
                title: prop.title,
                location: `${prop.locality}, ${prop.city}`,
                price: prop.sale_price ? `₹${(prop.sale_price / 100000).toFixed(1)}L` : 
                       prop.monthly_rent ? `₹${prop.monthly_rent.toLocaleString()}/month` : 'Price on Request',
                latitude: prop.latitude,
                longitude: prop.longitude,
                imageUrl: prop.primary_image || null,
                listing_type: prop.listing_type
              }))}
              center={calculateMapCenter(filteredProperties)}
              onPropertySelect={(property) => {
                const fullProperty = filteredProperties.find(p => p.id === property.id);
                if (fullProperty) setSelectedProperty(fullProperty);
              }}
            />

            {/* Loading Overlay */}
            {loading && (
              <div className="absolute inset-0 bg-black bg-opacity-20 flex items-center justify-center z-[1000]">
                <div className="bg-white rounded-lg p-4 flex items-center space-x-3 shadow-lg">
                  <Loader2 className="w-5 h-5 animate-spin text-blue-600" />
                  <span className="text-sm font-medium">Loading smart map data...</span>
                </div>
              </div>
            )}

            {/* Map Stats */}
            <div className="absolute bottom-4 left-4 bg-blue-600 text-white px-4 py-3 rounded-lg shadow-lg border border-blue-700 z-[1000]">
              <div className="text-sm font-semibold mb-1">
                <Eye className="w-4 h-4 inline mr-1" />
                Showing: {finalDisplayProperties.length} properties
                {hasSearched && !hasFiltered && finalDisplayProperties.length > 10 && (
                  <div className="text-xs text-orange-200 mt-1">💡 Use filters to narrow down</div>
                )}
              </div>
              <div className="text-xs text-blue-100">
                Mode: {visualizationMode.charAt(0).toUpperCase() + visualizationMode.slice(1)}
              </div>
              {loadingFloodData && (
                <div className="text-xs text-blue-100 mt-1 flex items-center gap-1">
                  <Loader2 className="w-3 h-3 animate-spin" />
                  Loading flood data...
                </div>
              )}
              {loadingUrbanData && (
                <div className="text-xs text-blue-100 mt-1 flex items-center gap-1">
                  <Loader2 className="w-3 h-3 animate-spin" />
                  Loading urban growth data...
                </div>
              )}
              {loadingInfraData && (
                <div className="text-xs text-blue-100 mt-1 flex items-center gap-1">
                  <Loader2 className="w-3 h-3 animate-spin" />
                  Loading infrastructure data...
                </div>
              )}
            </div>
            
            {/* Flood Risk Toggle Button */}
            {visualizationMode === 'flood' && (
              <div className="absolute top-4 right-4 z-[1000]">
                <Button
                  onClick={() => setShowFloodCard(!showFloodCard)}
                  className="bg-cyan-600 hover:bg-cyan-700 text-white shadow-lg"
                  size="sm"
                >
                  <Droplets className="w-4 h-4 mr-2" />
                  {showFloodCard ? 'Hide' : 'Show'} Flood Info
                </Button>
              </div>
            )}
            
            {/* Urban Growth Toggle Button */}
            {visualizationMode === 'urban' && (
              <div className="absolute top-4 right-4 z-[1000]">
                <Button
                  onClick={() => setShowUrbanCard(!showUrbanCard)}
                  className="bg-indigo-600 hover:bg-indigo-700 text-white shadow-lg"
                  size="sm"
                >
                  <Building2 className="w-4 h-4 mr-2" />
                  {showUrbanCard ? 'Hide' : 'Show'} Growth Info
                </Button>
              </div>
            )}
            
            {/* Infrastructure Toggle Button */}
            {visualizationMode === 'infra' && (
              <div className="absolute top-4 right-4 z-[1000]">
                <Button
                  onClick={() => setShowInfraCard(!showInfraCard)}
                  className="bg-teal-600 hover:bg-teal-700 text-white shadow-lg"
                  size="sm"
                >
                  <Navigation className="w-4 h-4 mr-2" />
                  {showInfraCard ? 'Hide' : 'Show'} Infrastructure Info
                </Button>
              </div>
            )}
            
            {/* Flood Risk Card Overlay */}
            {showFloodCard && selectedProperty && selectedProperty.floodData && (
              <div className="absolute top-20 right-4 z-[1000] w-96 max-w-[90vw]">
                <FloodRiskCard 
                  floodData={selectedProperty.floodData}
                  propertyTitle={selectedProperty.title}
                />
              </div>
            )}
            
            {/* Urban Growth Card Overlay */}
            {showUrbanCard && selectedProperty && selectedProperty.urbanGrowthData && (
              <div className="absolute top-20 right-4 z-[1000] w-96 max-w-[90vw]">
                <UrbanGrowthCard 
                  urbanGrowthData={selectedProperty.urbanGrowthData}
                  propertyTitle={selectedProperty.title}
                />
              </div>
            )}
            
            {/* Infrastructure Card Overlay */}
            {showInfraCard && selectedProperty && selectedProperty.infrastructureData && (
              <div className="absolute top-20 right-4 z-[1000] w-96 max-w-[90vw]">
                <InfrastructureCard 
                  infrastructureData={selectedProperty.infrastructureData}
                  propertyTitle={selectedProperty.title}
                />
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Smart Land Maps Information Section */}
      <div className="bg-gradient-to-br from-blue-50 to-indigo-100 py-16">
        <div className="max-w-7xl mx-auto px-6">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Why Choose Smart Land Maps?</h2>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Experience the future of property discovery with our AI-powered mapping technology that transforms how you explore real estate opportunities.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12">
            <Card className="p-6 text-center hover:shadow-lg transition-shadow">
              <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Search className="w-8 h-8 text-blue-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">Smart Search</h3>
              <p className="text-gray-600">
                Advanced AI algorithms help you find properties that match your exact requirements with intelligent filtering and location-based recommendations.
              </p>
            </Card>
            
            <Card className="p-6 text-center hover:shadow-lg transition-shadow">
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Eye className="w-8 h-8 text-green-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">Visual Intelligence</h3>
              <p className="text-gray-600">
                Interactive maps with color-coded markers provide instant insights into property types, prices, and availability across different neighborhoods.
              </p>
            </Card>
            
            <Card className="p-6 text-center hover:shadow-lg transition-shadow">
              <div className="w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Filter className="w-8 h-8 text-purple-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">Smart Filters</h3>
              <p className="text-gray-600">
                Comprehensive filtering system allows you to narrow down properties by price range, type, size, amenities, and listing status in real-time.
              </p>
            </Card>
          </div>
          
          <div className="bg-white rounded-2xl p-8 shadow-lg">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 items-center">
              <div>
                <h3 className="text-2xl font-bold text-gray-900 mb-4">Discover Properties Like Never Before</h3>
                <div className="space-y-4">
                  <div className="flex items-start gap-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full mt-2"></div>
                    <p className="text-gray-700">Real-time property data with accurate pricing and availability</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full mt-2"></div>
                    <p className="text-gray-700">Interactive map navigation with satellite and street view options</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full mt-2"></div>
                    <p className="text-gray-700">Detailed property information with high-quality images</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full mt-2"></div>
                    <p className="text-gray-700">Smart recommendations based on your search preferences</p>
                  </div>
                </div>
              </div>
              <div className="text-center">
                <div className="bg-white rounded-2xl p-8 border-2 border-blue-200 shadow-lg">
                  <h4 className="text-xl font-semibold mb-4 text-black">Start Exploring Today</h4>
                  <p className="mb-6 text-black">
                    Join thousands of users who have found their perfect property using Smart Land Maps
                  </p>
                  <div className="grid grid-cols-2 gap-4 text-center">
                    <div>
                      <div className="text-2xl font-bold text-black">{properties.length}+</div>
                      <div className="text-sm text-black">Properties</div>
                    </div>
                    <div>
                      <div className="text-2xl font-bold text-black">10+</div>
                      <div className="text-sm text-black">Cities</div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>


    </div>
  );
};