import React, { useState } from 'react';
import { motion } from 'framer-motion';

const STAMP_DUTY_RATES = {
  'Maharashtra': { rate: 0.05, registration: 0.01 },
  'Karnataka': { rate: 0.06, registration: 0.01 },
  'Tamil Nadu': { rate: 0.07, registration: 0.01 },
  'Delhi': { rate: 0.06, registration: 0.01 },
  'Gujarat': { rate: 0.055, registration: 0.01 },
  'Rajasthan': { rate: 0.055, registration: 0.01 },
  'Uttar Pradesh': { rate: 0.07, registration: 0.01 },
  'West Bengal': { rate: 0.06, registration: 0.01 }
};

export const StampDutyCalculator: React.FC = () => {
  const [propertyValue, setPropertyValue] = useState<string>('');
  const [selectedState, setSelectedState] = useState<string>('');
  const [propertyType, setPropertyType] = useState<string>('residential');
  const [results, setResults] = useState<any>(null);

  const calculateDuty = () => {
    if (!propertyValue || parseFloat(propertyValue) <= 0) {
      alert('Please enter a valid property value');
      return;
    }
    if (!selectedState) {
      alert('Please select a state');
      return;
    }

    const value = parseFloat(propertyValue);
    const rates = STAMP_DUTY_RATES[selectedState as keyof typeof STAMP_DUTY_RATES];
    
    if (!rates) return;

    const stampDuty = value * rates.rate;
    const registrationFee = value * rates.registration;
    const total = stampDuty + registrationFee;

    setResults({
      stampDuty,
      registrationFee,
      total,
      propertyValue: value
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 py-20">
      <div className="max-w-4xl mx-auto px-4">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="bg-white rounded-lg shadow-lg p-8"
        >
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Stamp Duty Calculator</h1>
          <p className="text-gray-600 max-w-3xl mx-auto mb-6">
            Calculate accurate stamp duty and registration charges based on your property value and state regulations. 
            Essential for budgeting additional costs beyond property price and loan amount.
          </p>
          <div className="mb-8 p-3 bg-blue-50 rounded-lg inline-block">
            <p className="text-sm text-blue-700 font-medium">📜 Factor in legal costs for complete property purchase planning</p>
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Property Value: ₹{parseInt(propertyValue || '0').toLocaleString('en-IN')}
                </label>
                <input
                  type="range"
                  min="1000000"
                  max="100000000"
                  step="500000"
                  value={propertyValue || '1000000'}
                  onChange={(e) => setPropertyValue(e.target.value)}
                  className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                />
                <div className="flex justify-between text-xs text-gray-500 mt-1">
                  <span>₹10L</span>
                  <span>₹10Cr</span>
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Select State
                </label>
                <select
                  value={selectedState}
                  onChange={(e) => setSelectedState(e.target.value)}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="">Choose State</option>
                  {Object.keys(STAMP_DUTY_RATES).map(state => (
                    <option key={state} value={state}>{state}</option>
                  ))}
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Property Type
                </label>
                <select
                  value={propertyType}
                  onChange={(e) => setPropertyType(e.target.value)}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  <option value="residential">Residential</option>
                  <option value="commercial">Commercial</option>
                </select>
              </div>

              <button
                onClick={calculateDuty}
                className="w-full bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors font-medium flex items-center justify-center gap-2"
              >
                <span>📜</span>
                Calculate Stamp Duty
              </button>
            </div>

            {results && (
              <motion.div
                initial={{ opacity: 0, x: 20 }}
                animate={{ opacity: 1, x: 0 }}
                className="bg-blue-50 p-6 rounded-lg"
              >
                <h3 className="text-xl font-semibold text-gray-900 mb-4">Calculation Results</h3>
                
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-gray-600">Property Value:</span>
                    <span className="font-semibold">₹{results.propertyValue.toLocaleString()}</span>
                    <span className="text-xs text-gray-500">(₹{results.propertyValue >= 10000000 ? (results.propertyValue/10000000).toFixed(2) + ' Cr' : (results.propertyValue/100000).toFixed(2) + ' L'})</span>
                  </div>
                  
                  <div className="flex justify-between">
                    <span className="text-gray-600">Stamp Duty:</span>
                    <div className="text-right">
                      <span className="font-semibold">₹{results.stampDuty.toLocaleString()}</span>
                      <span className="text-xs text-gray-500 block">(₹{results.stampDuty >= 10000000 ? (results.stampDuty/10000000).toFixed(2) + ' Cr' : (results.stampDuty/100000).toFixed(2) + ' L'})</span>
                    </div>
                  </div>
                  
                  <div className="flex justify-between">
                    <span className="text-gray-600">Registration Fee:</span>
                    <div className="text-right">
                      <span className="font-semibold">₹{results.registrationFee.toLocaleString()}</span>
                      <span className="text-xs text-gray-500 block">(₹{results.registrationFee >= 10000000 ? (results.registrationFee/10000000).toFixed(2) + ' Cr' : (results.registrationFee/100000).toFixed(2) + ' L'})</span>
                    </div>
                  </div>
                  
                  <div className="border-t pt-3 flex justify-between text-lg">
                    <span className="font-semibold">Total Amount:</span>
                    <div className="text-right">
                      <span className="font-bold text-blue-600">₹{results.total.toLocaleString()}</span>
                      <span className="text-xs text-blue-500 block">(₹{results.total >= 10000000 ? (results.total/10000000).toFixed(2) + ' Cr' : (results.total/100000).toFixed(2) + ' L'})</span>
                    </div>
                  </div>
                </div>

                <div className="mt-6 pt-4 border-t">
                  <p className="text-sm text-gray-600 mb-3">Pay online through official portals:</p>
                  <div className="space-y-2">
                    <a
                      href="https://www.shcilestamp.com/"
                      target="_blank"
                      rel="noopener noreferrer"
                      className="block bg-green-600 text-white text-center py-2 px-4 rounded hover:bg-green-700 transition-colors"
                    >
                      Pay via e-Stamp Portal
                    </a>
                    <a
                      href="https://igrsup.gov.in/"
                      target="_blank"
                      rel="noopener noreferrer"
                      className="block bg-blue-600 text-white text-center py-2 px-4 rounded hover:bg-blue-700 transition-colors"
                    >
                      State Registration Portal
                    </a>
                  </div>
                </div>
              </motion.div>
            )}
          </div>

          <div className="mt-8 p-4 bg-yellow-50 rounded-lg">
            <p className="text-sm text-yellow-800">
              <strong>Note:</strong> These are approximate calculations. Actual rates may vary based on property location, 
              type, and current government policies. Please verify with local authorities.
            </p>
          </div>
        </motion.div>
      </div>
    </div>
  );
};