import React, { useState, useEffect } from 'react';
import axios from 'axios';

interface Ticket {
  id: number;
  ticket_id: string;
  category: string;
  priority: string;
  status: string;
  subject: string;
  property_id?: number;
  created_at: string;
  responses_count: number;
}

interface TicketDetail {
  id: number;
  ticket_id: string;
  category: string;
  priority: string;
  status: string;
  subject: string;
  description: string;
  property_id?: number;
  created_at: string;
  responses: Array<{
    id: number;
    responder_name: string;
    responder_type: string;
    message: string;
    created_at: string;
  }>;
  attachments: Array<{
    id: number;
    file_name: string;
    file_path: string;
  }>;
}

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000';

const SupportTickets: React.FC = () => {
  const [tickets, setTickets] = useState<Ticket[]>([]);
  const [showForm, setShowForm] = useState(false);
  const [selectedTicket, setSelectedTicket] = useState<TicketDetail | null>(null);
  const [loading, setLoading] = useState(false);

  const [formData, setFormData] = useState({
    category: 'property_inquiry',
    priority: 'medium',
    subject: '',
    description: '',
    property_id: ''
  });
  const [files, setFiles] = useState<FileList | null>(null);

  const userId = parseInt(localStorage.getItem('user_id') || sessionStorage.getItem('user_id') || '0');
  const userType = localStorage.getItem('user_type') || sessionStorage.getItem('user_type') || 'buyer';

  useEffect(() => {
    fetchTickets();
  }, []);

  const fetchTickets = async () => {
    try {
      const response = await axios.get(`${API_URL}/support/tickets`, {
        params: { user_id: userId, user_type: userType }
      });
      setTickets(response.data);
    } catch (error) {
      console.error('Error fetching tickets:', error);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    const formDataToSend = new FormData();
    formDataToSend.append('user_id', userId.toString());
    formDataToSend.append('user_type', userType);
    formDataToSend.append('category', formData.category);
    formDataToSend.append('priority', formData.priority);
    formDataToSend.append('subject', formData.subject);
    formDataToSend.append('description', formData.description);
    if (formData.property_id) {
      formDataToSend.append('property_id', formData.property_id);
    }

    if (files) {
      Array.from(files).forEach(file => {
        formDataToSend.append('files', file);
      });
    }

    try {
      await axios.post(`${API_URL}/support/tickets/create`, formDataToSend, {
        headers: { 'Content-Type': 'multipart/form-data' }
      });
      alert('Ticket created successfully!');
      setShowForm(false);
      setFormData({ category: 'property_inquiry', priority: 'medium', subject: '', description: '', property_id: '' });
      setFiles(null);
      fetchTickets();
    } catch (error) {
      console.error('Error creating ticket:', error);
      alert('Failed to create ticket');
    } finally {
      setLoading(false);
    }
  };

  const viewTicketDetails = async (ticketId: string) => {
    try {
      const response = await axios.get(`${API_URL}/support/tickets/${ticketId}`);
      setSelectedTicket(response.data);
    } catch (error) {
      console.error('Error fetching ticket details:', error);
    }
  };

  const deleteTicket = async (ticketId: string) => {
    if (!confirm('Are you sure you want to delete this ticket?')) return;

    try {
      await axios.delete(`${API_URL}/support/tickets/${ticketId}/delete`);
      alert('Ticket deleted successfully');
      fetchTickets();
    } catch (error) {
      console.error('Error deleting ticket:', error);
      alert('Failed to delete ticket');
    }
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      open: '#3b82f6',
      in_progress: '#f59e0b',
      resolved: '#10b981',
      closed: '#6b7280'
    };
    return colors[status] || '#6b7280';
  };

  const getCategoryLabel = (category: string) => {
    const labels: Record<string, string> = {
      property_inquiry: 'Property Inquiry',
      viewing_request: 'Viewing Request',
      documentation: 'Documentation',
      payment: 'Payment',
      maintenance: 'Maintenance',
      general: 'General'
    };
    return labels[category] || category;
  };

  return (
    <div style={{ padding: '20px', maxWidth: '1200px', margin: '0 auto' }}>
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '20px' }}>
        <h1>Support Tickets</h1>
        <button
          onClick={() => setShowForm(!showForm)}
          style={{
            padding: '10px 20px',
            backgroundColor: '#3b82f6',
            color: 'white',
            border: 'none',
            borderRadius: '5px',
            cursor: 'pointer'
          }}
        >
          {showForm ? 'Cancel' : 'Create Ticket'}
        </button>
      </div>

      {showForm && (
        <div style={{ backgroundColor: '#f9fafb', padding: '20px', borderRadius: '8px', marginBottom: '20px' }}>
          <h2>Create New Support Ticket</h2>
          <form onSubmit={handleSubmit}>
            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Category *</label>
              <select
                value={formData.category}
                onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                style={{ width: '100%', padding: '8px', borderRadius: '4px', border: '1px solid #d1d5db' }}
                required
              >
                <option value="property_inquiry">Property Inquiry</option>
                <option value="viewing_request">Viewing Request</option>
                <option value="documentation">Documentation</option>
                <option value="payment">Payment</option>
                <option value="maintenance">Maintenance</option>
                <option value="general">General</option>
              </select>
            </div>

            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Priority *</label>
              <select
                value={formData.priority}
                onChange={(e) => setFormData({ ...formData, priority: e.target.value })}
                style={{ width: '100%', padding: '8px', borderRadius: '4px', border: '1px solid #d1d5db' }}
                required
              >
                <option value="low">Low</option>
                <option value="medium">Medium</option>
                <option value="high">High</option>
                <option value="critical">Critical</option>
              </select>
            </div>

            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Property ID (Optional)</label>
              <input
                type="number"
                value={formData.property_id}
                onChange={(e) => setFormData({ ...formData, property_id: e.target.value })}
                style={{ width: '100%', padding: '8px', borderRadius: '4px', border: '1px solid #d1d5db' }}
                placeholder="Enter property ID if related to specific property"
              />
            </div>

            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Subject *</label>
              <input
                type="text"
                value={formData.subject}
                onChange={(e) => setFormData({ ...formData, subject: e.target.value })}
                style={{ width: '100%', padding: '8px', borderRadius: '4px', border: '1px solid #d1d5db' }}
                placeholder="Brief description of your issue"
                required
              />
            </div>

            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Description *</label>
              <textarea
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                style={{ width: '100%', padding: '8px', borderRadius: '4px', border: '1px solid #d1d5db', minHeight: '100px' }}
                placeholder="Provide detailed information about your issue"
                required
              />
            </div>

            <div style={{ marginBottom: '15px' }}>
              <label style={{ display: 'block', marginBottom: '5px' }}>Attachments (Optional)</label>
              <input
                type="file"
                multiple
                onChange={(e) => setFiles(e.target.files)}
                style={{ width: '100%', padding: '8px' }}
              />
            </div>

            <button
              type="submit"
              disabled={loading}
              style={{
                padding: '10px 20px',
                backgroundColor: loading ? '#9ca3af' : '#10b981',
                color: 'white',
                border: 'none',
                borderRadius: '5px',
                cursor: loading ? 'not-allowed' : 'pointer'
              }}
            >
              {loading ? 'Creating...' : 'Submit Ticket'}
            </button>
          </form>
        </div>
      )}

      <div style={{ backgroundColor: 'white', borderRadius: '8px', overflow: 'hidden', boxShadow: '0 1px 3px rgba(0,0,0,0.1)' }}>
        <table style={{ width: '100%', borderCollapse: 'collapse' }}>
          <thead style={{ backgroundColor: '#f3f4f6' }}>
            <tr>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Ticket ID</th>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Subject</th>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Category</th>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Status</th>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Created</th>
              <th style={{ padding: '12px', textAlign: 'left', borderBottom: '1px solid #e5e7eb' }}>Actions</th>
            </tr>
          </thead>
          <tbody>
            {tickets.map((ticket) => (
              <tr key={ticket.id} style={{ borderBottom: '1px solid #e5e7eb' }}>
                <td style={{ padding: '12px' }}>{ticket.ticket_id}</td>
                <td style={{ padding: '12px' }}>{ticket.subject}</td>
                <td style={{ padding: '12px' }}>{getCategoryLabel(ticket.category)}</td>
                <td style={{ padding: '12px' }}>
                  <span style={{
                    padding: '4px 8px',
                    borderRadius: '4px',
                    backgroundColor: getStatusColor(ticket.status) + '20',
                    color: getStatusColor(ticket.status),
                    fontSize: '12px',
                    fontWeight: '500'
                  }}>
                    {ticket.status.replace('_', ' ').toUpperCase()}
                  </span>
                </td>
                <td style={{ padding: '12px' }}>{new Date(ticket.created_at).toLocaleDateString()}</td>
                <td style={{ padding: '12px' }}>
                  <button
                    onClick={() => viewTicketDetails(ticket.ticket_id)}
                    style={{
                      padding: '5px 10px',
                      backgroundColor: '#3b82f6',
                      color: 'white',
                      border: 'none',
                      borderRadius: '4px',
                      cursor: 'pointer',
                      marginRight: '5px'
                    }}
                  >
                    View
                  </button>
                  <button
                    onClick={() => deleteTicket(ticket.ticket_id)}
                    style={{
                      padding: '5px 10px',
                      backgroundColor: '#ef4444',
                      color: 'white',
                      border: 'none',
                      borderRadius: '4px',
                      cursor: 'pointer'
                    }}
                  >
                    Delete
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {tickets.length === 0 && (
          <div style={{ padding: '40px', textAlign: 'center', color: '#6b7280' }}>
            No tickets found. Create your first support ticket!
          </div>
        )}
      </div>

      {selectedTicket && (
        <div style={{
          position: 'fixed',
          top: 0,
          left: 0,
          right: 0,
          bottom: 0,
          backgroundColor: 'rgba(0,0,0,0.5)',
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
          zIndex: 1000
        }}>
          <div style={{
            backgroundColor: 'white',
            borderRadius: '8px',
            padding: '30px',
            maxWidth: '800px',
            width: '90%',
            maxHeight: '80vh',
            overflow: 'auto'
          }}>
            <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '20px' }}>
              <h2>Ticket Details</h2>
              <button
                onClick={() => setSelectedTicket(null)}
                style={{
                  padding: '5px 15px',
                  backgroundColor: '#6b7280',
                  color: 'white',
                  border: 'none',
                  borderRadius: '4px',
                  cursor: 'pointer'
                }}
              >
                Close
              </button>
            </div>

            <div style={{ marginBottom: '20px' }}>
              <p><strong>Ticket ID:</strong> {selectedTicket.ticket_id}</p>
              <p><strong>Subject:</strong> {selectedTicket.subject}</p>
              <p><strong>Category:</strong> {getCategoryLabel(selectedTicket.category)}</p>
              <p><strong>Status:</strong> <span style={{ color: getStatusColor(selectedTicket.status) }}>{selectedTicket.status.toUpperCase()}</span></p>
              <p><strong>Priority:</strong> {selectedTicket.priority.toUpperCase()}</p>
              {selectedTicket.property_id && <p><strong>Property ID:</strong> {selectedTicket.property_id}</p>}
              <p><strong>Created:</strong> {new Date(selectedTicket.created_at).toLocaleString()}</p>
              <p><strong>Description:</strong></p>
              <p style={{ backgroundColor: '#f9fafb', padding: '10px', borderRadius: '4px' }}>{selectedTicket.description}</p>
            </div>

            {selectedTicket.attachments.length > 0 && (
              <div style={{ marginBottom: '20px' }}>
                <h3>Attachments</h3>
                {selectedTicket.attachments.map((att) => (
                  <div key={att.id} style={{ marginBottom: '5px' }}>
                    <a href={`${API_URL}/${att.file_path}`} target="_blank" rel="noopener noreferrer">
                      {att.file_name}
                    </a>
                  </div>
                ))}
              </div>
            )}

            <div>
              <h3>Conversation</h3>
              {selectedTicket.responses.length === 0 ? (
                <p style={{ color: '#6b7280' }}>No responses yet</p>
              ) : (
                selectedTicket.responses.map((response) => (
                  <div key={response.id} style={{
                    backgroundColor: response.responder_type === 'admin' ? '#eff6ff' : '#f9fafb',
                    padding: '15px',
                    borderRadius: '8px',
                    marginBottom: '10px'
                  }}>
                    <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '5px' }}>
                      <strong>{response.responder_name} ({response.responder_type})</strong>
                      <span style={{ fontSize: '12px', color: '#6b7280' }}>
                        {new Date(response.created_at).toLocaleString()}
                      </span>
                    </div>
                    <p>{response.message}</p>
                  </div>
                ))
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default SupportTickets;
