import React, { useEffect, useState } from 'react';
import { useLocation, useNavigate, useSearchParams } from 'react-router-dom';
import { 
  Shield, 
  FileText, 
  AlertTriangle, 
  CheckCircle, 
  XCircle,
  Download,
  Calendar,
  Users,
  MapPin,
  TrendingUp,
  AlertCircle,
  ArrowLeft,
  Loader
} from 'lucide-react';
import { titleSearchService } from '../services/titleSearchService';

export function TitleSearchInstantResults() {
  const location = useLocation();
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [results, setResults] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  
  const sessionId = searchParams.get('session_id') || location.state?.sessionId;

  useEffect(() => {
    if (sessionId) {
      fetchInstantResults();
    } else {
      setError('No session ID provided');
      setLoading(false);
    }
  }, [sessionId]);

  const fetchInstantResults = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await titleSearchService.instantTitleVerificationBySession(sessionId);
      
      if (response.success) {
        setResults(response);
      } else {
        setError('Failed to get instant verification results');
      }
    } catch (err) {
      console.error('Error fetching instant results:', err);
      setError(err.message || 'Failed to fetch results');
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <Loader className="w-12 h-12 text-blue-500 mx-auto mb-4 animate-spin" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Processing Instant Verification</h2>
          <p className="text-gray-600">Analyzing your documents using OCR and AI...</p>
        </div>
      </div>
    );
  }

  if (error || !results) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <XCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Verification Failed</h2>
          <p className="text-gray-600 mb-4">{error || 'No instant verification results available.'}</p>
          <button
            onClick={() => navigate('/services/title-search')}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            Back to Title Search
          </button>
        </div>
      </div>
    );
  }

  const instantScore = results.instant_title_score || 0;
  const riskLevel = results.risk_level || 'High';
  const propertyDetails = results.property_details || {};
  const documentAnalyses = results.document_analyses || [];
  const propertyHistory = results.property_history || {};

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-yellow-600';
    return 'text-red-600';
  };

  const getRiskBadge = (risk: string) => {
    const riskLower = risk.toLowerCase();
    if (riskLower === 'low') return 'bg-green-100 text-green-800';
    if (riskLower === 'medium') return 'bg-yellow-100 text-yellow-800';
    return 'bg-red-100 text-red-800';
  };

  const getDocumentIcon = (docType: string, status: string) => {
    if (status === 'completed') {
      return <CheckCircle className="w-4 h-4 text-green-500" />;
    } else if (status === 'failed') {
      return <XCircle className="w-4 h-4 text-red-500" />;
    } else {
      return <AlertCircle className="w-4 h-4 text-yellow-500" />;
    }
  };

  const getDocumentDisplayName = (docType: string) => {
    const names = {
      'titleDeed': 'Title Deed',
      'saleDeed': 'Sale Deed',
      'encumbrance': 'Encumbrance Certificate',
      'khataPatta': 'Khata/Patta',
      'taxReceipts': 'Tax Receipts',
      'noc': 'NOC Documents'
    };
    return names[docType] || docType;
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <button
                onClick={() => navigate('/services/title-search')}
                className="flex items-center gap-2 text-gray-600 hover:text-gray-900"
              >
                <ArrowLeft className="w-5 h-5" />
                Back
              </button>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Instant Title Verification Results</h1>
                <p className="text-gray-600 mt-1">
                  Property ID: {propertyDetails.property_id || 'N/A'}
                </p>
              </div>
            </div>
            <button
              onClick={() => window.print()}
              className="flex items-center gap-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
            >
              <Download className="w-4 h-4" />
              Download Report
            </button>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Key Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Instant Verification Score</p>
                <p className={`text-3xl font-bold ${getScoreColor(instantScore)}`}>
                  {instantScore}%
                </p>
              </div>
              <Shield className={`w-8 h-8 ${getScoreColor(instantScore)}`} />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Risk Level</p>
                <div className="mt-2">
                  <span className={`px-3 py-1 rounded-full text-sm font-medium ${getRiskBadge(riskLevel)}`}>
                    {riskLevel} Risk
                  </span>
                </div>
              </div>
              <AlertTriangle className="w-8 h-8 text-orange-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Documents Analyzed</p>
                <p className="text-3xl font-bold text-gray-900">
                  {results.documents_analyzed || 0}
                </p>
              </div>
              <FileText className="w-8 h-8 text-green-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Processing Time</p>
                <p className="text-3xl font-bold text-gray-900">{results.processing_time || '5m'}</p>
              </div>
              <CheckCircle className="w-8 h-8 text-blue-500" />
            </div>
          </div>
        </div>

        {/* Results Summary */}
        <div className="bg-white rounded-xl shadow-sm border mb-8 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Verification Summary</h3>
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
            <p className="text-blue-800">{results.recommendation || 'Analysis completed successfully.'}</p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Document Analysis */}
            <div>
              <h4 className="font-medium text-gray-900 mb-3">Document Analysis</h4>
              <div className="space-y-3">
                {documentAnalyses.map((analysis, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <span className="text-sm font-medium">{getDocumentDisplayName(analysis.document_type)}</span>
                    <div className="flex items-center gap-2">
                      <span className="text-sm text-gray-600">
                        {Math.round(analysis.verification_score || 0)}%
                      </span>
                      {getDocumentIcon(analysis.document_type, analysis.verification_status)}
                    </div>
                  </div>
                ))}
                
                {documentAnalyses.length === 0 && (
                  <div className="text-sm text-gray-500 text-center py-4">
                    No document analyses available
                  </div>
                )}
              </div>
            </div>

            {/* Property Information */}
            <div>
              <h4 className="font-medium text-gray-900 mb-3">Property Information</h4>
              <div className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Property ID</span>
                  <span className="text-sm font-medium">{propertyDetails.property_id || 'N/A'}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Survey Number</span>
                  <span className="text-sm font-medium">{propertyDetails.survey_number || 'N/A'}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Owner Name</span>
                  <span className="text-sm font-medium">{propertyDetails.owner_name || 'N/A'}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Verification Type</span>
                  <span className="text-sm font-medium">Instant Analysis</span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Property History */}
        {propertyHistory && propertyHistory.ownership_transfers && propertyHistory.ownership_transfers.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border mb-8 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Calendar className="w-5 h-5" />
              Property Ownership History
            </h3>
            
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
              <div className="text-center">
                <p className="text-2xl font-bold text-blue-600">
                  {propertyHistory.timeline_analysis?.total_transfers || 0}
                </p>
                <p className="text-sm text-gray-600">Ownership Transfers</p>
              </div>
              <div className="text-center">
                <p className="text-2xl font-bold text-green-600">
                  {propertyHistory.timeline_analysis?.total_transactions || 0}
                </p>
                <p className="text-sm text-gray-600">Transactions</p>
              </div>
              <div className="text-center">
                <p className="text-2xl font-bold text-purple-600">
                  {propertyHistory.timeline_analysis?.timeline_span_years || 0}
                </p>
                <p className="text-sm text-gray-600">Years of History</p>
              </div>
              <div className="text-center">
                <p className="text-2xl font-bold text-orange-600">
                  {propertyHistory.verification_metrics?.verification_percentage || 0}%
                </p>
                <p className="text-sm text-gray-600">Verified Records</p>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-medium text-gray-900 flex items-center gap-2">
                <Users className="w-4 h-4" />
                Ownership Transfer Timeline
              </h4>
              {propertyHistory.ownership_transfers.map((transfer: any, index: number) => (
                <div key={index} className="border rounded-lg p-4">
                  <div className="flex items-center justify-between mb-2">
                    <div className="flex items-center gap-2">
                      <span className="text-sm font-medium text-gray-900">
                        {getDocumentDisplayName(transfer.document_type)}
                      </span>
                      <span className="text-xs text-gray-500">
                        {new Date(transfer.transfer_date).toLocaleDateString()}
                      </span>
                    </div>
                    <div className="flex items-center gap-2">
                      <span className="text-xs text-gray-500">
                        Score: {Math.round(transfer.authenticity_score || 0)}%
                      </span>
                      {getDocumentIcon(transfer.document_type, transfer.verification_status)}
                    </div>
                  </div>
                  <div className="text-sm text-gray-600">
                    <span className="font-medium">{transfer.from_owner}</span>
                    <span className="mx-2">→</span>
                    <span className="font-medium">{transfer.to_owner}</span>
                  </div>
                  {transfer.survey_numbers && transfer.survey_numbers.length > 0 && (
                    <div className="text-xs text-gray-500 mt-1">
                      Survey: {transfer.survey_numbers.join(', ')}
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Document Issues */}
        {documentAnalyses.some(doc => doc.issues_found && doc.issues_found.length > 0) && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-xl p-6 mb-8">
            <h3 className="text-lg font-semibold text-yellow-900 mb-4 flex items-center gap-2">
              <AlertTriangle className="w-5 h-5" />
              Issues Identified
            </h3>
            <div className="space-y-3">
              {documentAnalyses.map((doc, index) => (
                doc.issues_found && doc.issues_found.length > 0 && (
                  <div key={index} className="bg-white rounded-lg p-4">
                    <h4 className="font-medium text-gray-900 mb-2">
                      {getDocumentDisplayName(doc.document_type)}
                    </h4>
                    <ul className="list-disc list-inside space-y-1">
                      {doc.issues_found.map((issue, issueIndex) => (
                        <li key={issueIndex} className="text-sm text-yellow-800">{issue}</li>
                      ))}
                    </ul>
                  </div>
                )
              ))}
            </div>
          </div>
        )}

        {/* Next Steps */}
        <div className={`${instantScore >= 70 ? 'bg-green-50 border-green-200' : 'bg-yellow-50 border-yellow-200'} border rounded-xl p-6`}>
          <h3 className={`text-lg font-semibold ${instantScore >= 70 ? 'text-green-900' : 'text-yellow-900'} mb-4`}>
            Recommended Next Steps
          </h3>
          <div className={`space-y-3 ${instantScore >= 70 ? 'text-green-800' : 'text-yellow-800'}`}>
            <div className="flex items-start gap-2">
              <CheckCircle className={`w-4 h-4 ${instantScore >= 70 ? 'text-green-600' : 'text-yellow-600'} mt-0.5`} />
              <span>Review the instant verification results carefully</span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className={`w-4 h-4 ${instantScore >= 70 ? 'text-green-600' : 'text-yellow-600'} mt-0.5`} />
              <span>
                {instantScore >= 85 
                  ? 'Proceed with transaction - excellent verification results'
                  : instantScore >= 70
                  ? 'Consider additional verification for complete assurance'
                  : 'Obtain comprehensive title search before proceeding'
                }
              </span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className={`w-4 h-4 ${instantScore >= 70 ? 'text-green-600' : 'text-yellow-600'} mt-0.5`} />
              <span>
                {instantScore >= 70
                  ? 'Consult with legal professionals for final approval'
                  : 'Mandatory legal consultation required before any transaction'
                }
              </span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className={`w-4 h-4 ${instantScore >= 70 ? 'text-green-600' : 'text-yellow-600'} mt-0.5`} />
              <span>Verify all extracted information with original documents</span>
            </div>
          </div>
          
          <div className="mt-6 flex gap-4">
            <button
              onClick={() => navigate('/services/title-search')}
              className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium"
            >
              Order Comprehensive Search
            </button>
            <button
              onClick={() => navigate('/services/title-search')}
              className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 font-medium"
            >
              New Verification
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}