import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { 
  Shield, 
  FileText, 
  AlertTriangle, 
  CheckCircle, 
  Clock, 
  XCircle,
  Download,
  Eye,
  Calendar,
  Users,
  MapPin,
  Scale,
  TrendingUp,
  AlertCircle
} from 'lucide-react';
import { titleSearchService, TitleSearchResult, PropertyHistoryResult } from '../services/titleSearchService';

export function TitleSearchResults() {
  const { searchId } = useParams<{ searchId: string }>();
  const navigate = useNavigate();
  
  const [titleResults, setTitleResults] = useState<TitleSearchResult | null>(null);
  const [propertyHistory, setPropertyHistory] = useState<PropertyHistoryResult | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'overview' | 'history' | 'documents' | 'recommendations'>('overview');

  useEffect(() => {
    if (searchId) {
      fetchTitleSearchResults();
    }
  }, [searchId]);

  const fetchTitleSearchResults = async () => {
    try {
      setLoading(true);
      
      // Fetch title search results
      const titleData = await titleSearchService.getTitleSearchResults(searchId!);
      setTitleResults(titleData);
      
      // Fetch property history if available
      if (titleData.success && titleData.report_data?.search_details?.property_id) {
        try {
          const historyData = await titleSearchService.getPropertyHistory(
            titleData.report_data.search_details.property_id
          );
          setPropertyHistory(historyData);
        } catch (historyError) {
          console.warn('Property history not available:', historyError);
        }
      }
      
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to load results');
    } finally {
      setLoading(false);
    }
  };

  const getRiskLevelBadge = (riskLevel: string) => {
    const colors = titleSearchService.getRiskLevelColor(riskLevel);
    return (
      <span className={`px-3 py-1 rounded-full text-sm font-medium ${colors}`}>
        {riskLevel.charAt(0).toUpperCase() + riskLevel.slice(1)} Risk
      </span>
    );
  };

  const getTitleStatusBadge = (titleStatus: string) => {
    const colors = titleSearchService.getTitleStatusColor(titleStatus);
    return (
      <span className={`px-3 py-1 rounded-full text-sm font-medium ${colors}`}>
        {titleStatus.charAt(0).toUpperCase() + titleStatus.slice(1)}
      </span>
    );
  };

  const getScoreColor = (score: number) => {
    if (score >= 85) return 'text-green-600';
    if (score >= 70) return 'text-yellow-600';
    if (score >= 50) return 'text-orange-600';
    return 'text-red-600';
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading title search results...</p>
        </div>
      </div>
    );
  }

  if (error || !titleResults) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <XCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Failed to Load Results</h2>
          <p className="text-gray-600 mb-4">{error}</p>
          <button
            onClick={() => navigate('/services/title-search')}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            Back to Title Search
          </button>
        </div>
      </div>
    );
  }

  const reportData = titleResults.report_data;
  const searchDetails = reportData?.search_details;
  const analysisSummary = reportData?.analysis_summary;

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Title Search Results</h1>
              <p className="text-gray-600 mt-1">
                Property ID: {searchDetails?.property_id} | Survey No: {searchDetails?.survey_number}
              </p>
            </div>
            <div className="flex items-center gap-4">
              <button
                onClick={() => window.print()}
                className="flex items-center gap-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                <Download className="w-4 h-4" />
                Download Report
              </button>
              <button
                onClick={() => navigate('/services/title-search')}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                New Search
              </button>
            </div>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Key Metrics Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Title Integrity Score</p>
                <p className={`text-3xl font-bold ${getScoreColor(titleResults.title_integrity_score)}`}>
                  {titleResults.title_integrity_score}%
                </p>
              </div>
              <Shield className={`w-8 h-8 ${getScoreColor(titleResults.title_integrity_score)}`} />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Risk Level</p>
                <div className="mt-2">
                  {getRiskLevelBadge(titleResults.risk_level)}
                </div>
              </div>
              <AlertTriangle className="w-8 h-8 text-orange-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Title Status</p>
                <div className="mt-2">
                  {getTitleStatusBadge(titleResults.title_status)}
                </div>
              </div>
              <Scale className="w-8 h-8 text-blue-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Documents Analyzed</p>
                <p className="text-3xl font-bold text-gray-900">
                  {reportData?.document_analyses?.length || 0}
                </p>
              </div>
              <FileText className="w-8 h-8 text-green-500" />
            </div>
          </div>
        </div>

        {/* Navigation Tabs */}
        <div className="bg-white rounded-xl shadow-sm border mb-8">
          <div className="border-b border-gray-200">
            <nav className="flex space-x-8 px-6">
              {[
                { id: 'overview', label: 'Overview', icon: Eye },
                { id: 'history', label: 'Property History', icon: Calendar },
                { id: 'documents', label: 'Document Analysis', icon: FileText },
                { id: 'recommendations', label: 'Recommendations', icon: CheckCircle }
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id as any)}
                  className={`flex items-center gap-2 py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === tab.id
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  <tab.icon className="w-4 h-4" />
                  {tab.label}
                </button>
              ))}
            </nav>
          </div>

          <div className="p-6">
            {/* Overview Tab */}
            {activeTab === 'overview' && (
              <div className="space-y-6">
                {/* Executive Summary */}
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
                  <h3 className="text-lg font-semibold text-blue-900 mb-3">Executive Summary</h3>
                  <p className="text-blue-800 leading-relaxed">
                    {reportData?.executive_summary || titleSearchService.generateRecommendation(
                      titleResults.title_integrity_score, 
                      titleResults.risk_level
                    )}
                  </p>
                </div>

                {/* Property Details */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="bg-gray-50 rounded-lg p-6">
                    <h4 className="font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <MapPin className="w-5 h-5" />
                      Property Information
                    </h4>
                    <div className="space-y-3">
                      <div>
                        <p className="text-sm text-gray-600">Property ID</p>
                        <p className="font-medium">{searchDetails?.property_id}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Survey Number</p>
                        <p className="font-medium">{searchDetails?.survey_number || 'Not specified'}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Owner Name</p>
                        <p className="font-medium">{searchDetails?.owner_name || 'Not specified'}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-600">Search Type</p>
                        <p className="font-medium capitalize">{searchDetails?.search_type}</p>
                      </div>
                    </div>
                  </div>

                  <div className="bg-gray-50 rounded-lg p-6">
                    <h4 className="font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <TrendingUp className="w-5 h-5" />
                      Analysis Scores
                    </h4>
                    <div className="space-y-3">
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Authenticity</span>
                        <span className="font-medium">{analysisSummary?.avg_authenticity?.toFixed(1) || 'N/A'}%</span>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Consistency</span>
                        <span className="font-medium">{analysisSummary?.avg_consistency?.toFixed(1) || 'N/A'}%</span>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Completeness</span>
                        <span className="font-medium">{analysisSummary?.avg_completeness?.toFixed(1) || 'N/A'}%</span>
                      </div>
                      <div className="pt-2 border-t border-gray-300">
                        <div className="flex justify-between items-center">
                          <span className="text-sm font-medium text-gray-900">Overall Score</span>
                          <span className={`font-bold ${getScoreColor(titleResults.title_integrity_score)}`}>
                            {titleResults.title_integrity_score}%
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Property History Tab */}
            {activeTab === 'history' && (
              <div className="space-y-6">
                {propertyHistory ? (
                  <>
                    {/* Timeline Overview */}
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h4 className="font-semibold text-gray-900 mb-4">Timeline Overview</h4>
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                        <div className="text-center">
                          <p className="text-2xl font-bold text-blue-600">
                            {propertyHistory.timeline_analysis?.total_transfers || 0}
                          </p>
                          <p className="text-sm text-gray-600">Ownership Transfers</p>
                        </div>
                        <div className="text-center">
                          <p className="text-2xl font-bold text-green-600">
                            {propertyHistory.timeline_analysis?.total_transactions || 0}
                          </p>
                          <p className="text-sm text-gray-600">Transactions</p>
                        </div>
                        <div className="text-center">
                          <p className="text-2xl font-bold text-purple-600">
                            {propertyHistory.timeline_analysis?.timeline_span_years || 0}
                          </p>
                          <p className="text-sm text-gray-600">Years of History</p>
                        </div>
                        <div className="text-center">
                          <p className="text-2xl font-bold text-orange-600">
                            {propertyHistory.verification_metrics?.verification_percentage || 0}%
                          </p>
                          <p className="text-sm text-gray-600">Verified Records</p>
                        </div>
                      </div>
                    </div>

                    {/* Ownership Transfers */}
                    <div>
                      <h4 className="font-semibold text-gray-900 mb-4 flex items-center gap-2">
                        <Users className="w-5 h-5" />
                        Ownership Transfer History
                      </h4>
                      <div className="space-y-4">
                        {propertyHistory.ownership_transfers?.map((transfer, index) => (
                          <div key={index} className="bg-white border rounded-lg p-4">
                            <div className="flex items-center justify-between mb-2">
                              <div className="flex items-center gap-2">
                                <span className="text-sm font-medium text-gray-900">
                                  {transfer.document_type}
                                </span>
                                <span className="text-xs text-gray-500">
                                  {titleSearchService.formatDate(transfer.transfer_date)}
                                </span>
                              </div>
                              <div className="flex items-center gap-2">
                                <span className="text-xs text-gray-500">
                                  Score: {transfer.authenticity_score}%
                                </span>
                                <span className="text-lg">
                                  {titleSearchService.getVerificationStatusIcon(transfer.verification_status)}
                                </span>
                              </div>
                            </div>
                            <div className="text-sm text-gray-600">
                              <span className="font-medium">{transfer.from_owner}</span>
                              <span className="mx-2">→</span>
                              <span className="font-medium">{transfer.to_owner}</span>
                            </div>
                            {transfer.survey_numbers?.length > 0 && (
                              <div className="text-xs text-gray-500 mt-1">
                                Survey: {transfer.survey_numbers.join(', ')}
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    </div>

                    {/* Risk Indicators */}
                    {propertyHistory.timeline_analysis?.risk_indicators?.length > 0 && (
                      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6">
                        <h4 className="font-semibold text-yellow-900 mb-3 flex items-center gap-2">
                          <AlertCircle className="w-5 h-5" />
                          Risk Indicators
                        </h4>
                        <ul className="space-y-2">
                          {propertyHistory.timeline_analysis.risk_indicators.map((risk, index) => (
                            <li key={index} className="text-sm text-yellow-800 flex items-start gap-2">
                              <span className="text-yellow-600 mt-0.5">•</span>
                              {risk}
                            </li>
                          ))}
                        </ul>
                      </div>
                    )}
                  </>
                ) : (
                  <div className="text-center py-12">
                    <Calendar className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 mb-2">Property History Not Available</h3>
                    <p className="text-gray-600">
                      Property history analysis is not available for this search.
                    </p>
                  </div>
                )}
              </div>
            )}

            {/* Document Analysis Tab */}
            {activeTab === 'documents' && (
              <div className="space-y-6">
                <h4 className="font-semibold text-gray-900">Document Analysis Results</h4>
                <div className="grid gap-4">
                  {reportData?.document_analyses?.map((doc: any, index: number) => (
                    <div key={index} className="bg-white border rounded-lg p-6">
                      <div className="flex items-center justify-between mb-4">
                        <h5 className="font-medium text-gray-900 capitalize">
                          {doc.document_type?.replace(/([A-Z])/g, ' $1').trim()}
                        </h5>
                        <div className="flex items-center gap-4">
                          <span className="text-sm text-gray-500">
                            Overall: {((doc.authenticity_score + doc.consistency_score + doc.completeness_score) / 3).toFixed(0)}%
                          </span>
                        </div>
                      </div>
                      
                      <div className="grid grid-cols-3 gap-4">
                        <div className="text-center">
                          <p className="text-lg font-semibold text-blue-600">{doc.authenticity_score}%</p>
                          <p className="text-xs text-gray-600">Authenticity</p>
                        </div>
                        <div className="text-center">
                          <p className="text-lg font-semibold text-green-600">{doc.consistency_score}%</p>
                          <p className="text-xs text-gray-600">Consistency</p>
                        </div>
                        <div className="text-center">
                          <p className="text-lg font-semibold text-purple-600">{doc.completeness_score}%</p>
                          <p className="text-xs text-gray-600">Completeness</p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Recommendations Tab */}
            {activeTab === 'recommendations' && (
              <div className="space-y-6">
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
                  <h4 className="font-semibold text-blue-900 mb-3">Professional Recommendations</h4>
                  <div className="text-blue-800 leading-relaxed">
                    {reportData?.recommendations ? (
                      <div className="space-y-2">
                        {reportData.recommendations.split(' | ').map((rec: string, index: number) => (
                          <div key={index} className="flex items-start gap-2">
                            <span className="text-blue-600 mt-1">•</span>
                            <span>{rec}</span>
                          </div>
                        ))}
                      </div>
                    ) : (
                      <p>{titleSearchService.generateRecommendation(titleResults.title_integrity_score, titleResults.risk_level)}</p>
                    )}
                  </div>
                </div>

                {reportData?.risk_assessment && (
                  <div className="bg-orange-50 border border-orange-200 rounded-lg p-6">
                    <h4 className="font-semibold text-orange-900 mb-3">Risk Assessment</h4>
                    <div className="text-orange-800 leading-relaxed">
                      <div className="space-y-2">
                        {reportData.risk_assessment.split(' | ').map((risk: string, index: number) => (
                          <div key={index} className="flex items-start gap-2">
                            <AlertTriangle className="w-4 h-4 text-orange-600 mt-0.5 flex-shrink-0" />
                            <span>{risk}</span>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                )}

                <div className="bg-green-50 border border-green-200 rounded-lg p-6">
                  <h4 className="font-semibold text-green-900 mb-3">Next Steps</h4>
                  <div className="space-y-3 text-green-800">
                    <div className="flex items-start gap-2">
                      <CheckCircle className="w-4 h-4 text-green-600 mt-0.5" />
                      <span>Review all document analysis results carefully</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <CheckCircle className="w-4 h-4 text-green-600 mt-0.5" />
                      <span>Consult with legal professionals for high-risk items</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <CheckCircle className="w-4 h-4 text-green-600 mt-0.5" />
                      <span>Obtain additional documentation if recommended</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <CheckCircle className="w-4 h-4 text-green-600 mt-0.5" />
                      <span>Consider title insurance for added protection</span>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}