import React, { useState } from 'react';
import { CheckCircle, Phone, Mail, Zap, X, Calculator, Download, TrendingUp } from 'lucide-react';
import { Button } from '../components/ui/button';
import { ValuationFlow } from '../components/ValuationFlow';
import valuationApi from '../services/valuationApi';
import { generateValuationPDF } from '../utils/generateValuationPDF';

export function ValuationReportsPage() {
  const [showQuoteModal, setShowQuoteModal] = useState(false);
  const [showValuationFlow, setShowValuationFlow] = useState(false);
  const [showCalculator, setShowCalculator] = useState(false);
  const [loading, setLoading] = useState(false);
  const [result, setResult] = useState<any>(null);
  const [formData, setFormData] = useState({
    location: '',
    property_type: 'apartment',
    area: '',
    bedrooms: 2,
    bathrooms: 2,
    property_age: 0,
    floor_number: 1,
    total_floors: 10,
    furnishing: 'unfurnished',
    facing: 'north',
    contact_email: '',
    contact_phone: ''
  });

  const handleCalculate = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      const response = await valuationApi.calculateValuation({
        ...formData,
        area: parseInt(formData.area),
        amenities: []
      });
      if (response.success && response.data) {
        setResult(response.data);
      }
    } catch (error) {
      console.error('Valuation error:', error);
    } finally {
      setLoading(false);
    }
  };
  
  const solution = {
    title: 'Get Valuation Reports',
    description: 'Get instant AI-powered property valuation reports with estimated market value, price per sq ft, comparable properties analysis, and confidence scoring - completely free.',
    features: [
      'IBBI-registered valuer certification with government seal and signature',
      'Circle rate comparison with state government ready reckoner values',
      'Compliance with Indian Accounting Standards (Ind AS) for asset valuation',
      'Integration with state registrar office transaction data for comparables',
      'Multi-language reports in Hindi, English, and regional languages'
    ],
    benefits: [
      'Accepted by all RBI-regulated banks for home loan processing',
      'Valid for Income Tax Act Section 50C capital gains calculations',
      'Compliant with SEBI guidelines for REIT and real estate fund investments',
      'Recognized by insurance companies for property insurance coverage',
      'Court-admissible valuations for family partition and legal disputes'
    ],
    useCases: [
      'Home loan applications with SBI, HDFC Bank, and ICICI Bank',
      'Property insurance with ICICI Lombard and HDFC ERGO',
      'Income tax filing for capital gains under Section 54 exemption',
      'Family partition and inheritance under Hindu Succession Act',
      'Corporate asset valuation for balance sheet compliance'
    ],
    pricing: 'Free Instant Valuation | Professional Certified Reports from ₹999'
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="text-white py-16" style={{backgroundColor: '#1e40af'}}>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-6">{solution.title}</h1>
            <p className="text-xl md:text-2xl text-indigo-100 max-w-3xl mx-auto">Get instant AI-powered property valuation reports with estimated market value, price per sq ft, comparable properties analysis, and confidence scoring - completely free.</p>
            <div className="mt-8">
              <span className="inline-block bg-white/20 text-white px-6 py-3 rounded-full text-lg font-semibold">
                {solution.pricing}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Solutions Images Section */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center">
              <img 
                src="https://images.unsplash.com/photo-1560518883-ce09059eeffa?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80"
                alt="Property Valuation" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Property Valuation</h3>
              <p className="text-gray-600">Property Assessment</p>
              <p className="text-gray-500 text-sm">Accurate property valuations</p>
            </div>
            
            <div className="text-center">
              <img 
                src="https://images.unsplash.com/photo-1551288049-bebda4e38f71?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80"
                alt="Smart Technology" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Smart Technology</h3>
              <p className="text-gray-600">Market Analysis</p>
              <p className="text-gray-500 text-sm">Real-time market comparisons</p>
            </div>
            
            <div className="text-center">
              <img 
                src="https://images.unsplash.com/photo-1554224155-6726b3ff858f?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80"
                alt="Professional Services" 
                className="w-full h-48 object-cover rounded-lg mb-4 shadow-md"
              />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Professional Services</h3>
              <p className="text-gray-600">Professional Reports</p>
              <p className="text-gray-500 text-sm">Certified professional reports</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Features */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Features</h2>
            <ul className="space-y-4">
              {solution.features.map((feature, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{feature}</span>
                </li>
              ))}
            </ul>
          </div>

          {/* Benefits */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Benefits</h2>
            <ul className="space-y-4">
              {solution.benefits.map((benefit, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-blue-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{benefit}</span>
                </li>
              ))}
            </ul>
          </div>

          {/* Use Cases */}
          <div className="bg-white rounded-lg p-8 shadow-sm border border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Use Cases</h2>
            <ul className="space-y-4">
              {solution.useCases.map((useCase, index) => (
                <li key={index} className="flex items-start gap-3">
                  <CheckCircle className="w-5 h-5 text-purple-500 mt-0.5 flex-shrink-0" />
                  <span className="text-gray-700">{useCase}</span>
                </li>
              ))}
            </ul>
          </div>
        </div>

        {/* Quick Calculator Section */}
        <div className="mt-16 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-2xl p-8 shadow-lg border border-blue-100">
          <div className="text-center mb-8">
            <Calculator className="w-12 h-12 text-[#4f46e5] mx-auto mb-4" />
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Quick Valuation Calculator</h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Get an instant estimate of your property's worth in seconds
            </p>
          </div>

          {!showCalculator ? (
            <div className="text-center">
              <Button 
                className="bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600 text-white px-8 py-3 text-lg font-semibold"
                onClick={() => setShowCalculator(true)}
              >
                <Calculator className="w-5 h-5 mr-2" />
                Start Free Calculation
              </Button>
            </div>
          ) : (
            <div className="max-w-4xl mx-auto">
              {!result ? (
                <form onSubmit={handleCalculate} className="bg-white rounded-xl p-6 shadow-md">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Location/City</label>
                      <input
                        type="text"
                        required
                        value={formData.location}
                        onChange={(e) => setFormData({...formData, location: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                        placeholder="e.g., Koramangala, Bangalore"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
                      <select
                        value={formData.property_type}
                        onChange={(e) => setFormData({...formData, property_type: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                      >
                        <option value="apartment">Apartment</option>
                        <option value="villa">Villa</option>
                        <option value="plot">Plot</option>
                        <option value="commercial">Commercial</option>
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Area (sq ft)</label>
                      <input
                        type="number"
                        required
                        value={formData.area}
                        onChange={(e) => setFormData({...formData, area: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                        placeholder="1000"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Bedrooms</label>
                      <select
                        value={formData.bedrooms}
                        onChange={(e) => setFormData({...formData, bedrooms: parseInt(e.target.value)})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                      >
                        <option value="1">1 BHK</option>
                        <option value="2">2 BHK</option>
                        <option value="3">3 BHK</option>
                        <option value="4">4 BHK</option>
                        <option value="5">5+ BHK</option>
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Property Age (years)</label>
                      <input
                        type="number"
                        value={formData.property_age}
                        onChange={(e) => setFormData({...formData, property_age: parseInt(e.target.value)})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                        placeholder="0"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Furnishing</label>
                      <select
                        value={formData.furnishing}
                        onChange={(e) => setFormData({...formData, furnishing: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#4f46e5] focus:border-transparent"
                      >
                        <option value="unfurnished">Unfurnished</option>
                        <option value="semi_furnished">Semi-Furnished</option>
                        <option value="fully_furnished">Fully Furnished</option>
                      </select>
                    </div>
                  </div>
                  <div className="mt-6 flex gap-4">
                    <Button 
                      type="submit" 
                      disabled={loading}
                      className="flex-1 bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600 text-white py-3 font-semibold"
                    >
                      {loading ? 'Calculating...' : 'Calculate Value'}
                    </Button>
                    <Button 
                      type="button"
                      variant="outline"
                      onClick={() => setShowCalculator(false)}
                      className="px-6"
                    >
                      Cancel
                    </Button>
                  </div>
                </form>
              ) : (
                <div className="bg-white rounded-xl p-8 shadow-md">
                  <div className="text-center mb-6">
                    <div className="inline-flex items-center justify-center w-16 h-16 bg-green-100 rounded-full mb-4">
                      <CheckCircle className="w-8 h-8 text-green-600" />
                    </div>
                    <h3 className="text-2xl font-bold text-gray-900 mb-2">Valuation Complete!</h3>
                    <p className="text-gray-600">Here's your property valuation report</p>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    <div className="bg-gradient-to-br from-blue-50 to-blue-100 p-6 rounded-xl">
                      <p className="text-sm text-blue-600 font-medium mb-2">Estimated Value</p>
                      <p className="text-3xl font-bold text-blue-900">₹{(result.estimated_value / 100000).toFixed(2)}L</p>
                      <p className="text-sm text-blue-600 mt-2">₹{result.price_per_sqft.toFixed(0)}/sq ft</p>
                    </div>
                    <div className="bg-gradient-to-br from-purple-50 to-purple-100 p-6 rounded-xl">
                      <p className="text-sm text-purple-600 font-medium mb-2">Value Range</p>
                      <p className="text-lg font-bold text-purple-900">
                        ₹{(result.valuation_range_min / 100000).toFixed(2)}L - ₹{(result.valuation_range_max / 100000).toFixed(2)}L
                      </p>
                      <p className="text-sm text-purple-600 mt-2">Confidence: {result.confidence_score}%</p>
                    </div>
                  </div>

                  {result.comparables && result.comparables.length > 0 && (
                    <div className="mb-6">
                      <h4 className="text-lg font-semibold text-gray-900 mb-4">Comparable Properties</h4>
                      <div className="space-y-3">
                        {result.comparables.map((comp: any, idx: number) => (
                          <div key={idx} className="bg-gray-50 p-4 rounded-lg flex justify-between items-center">
                            <div>
                              <p className="font-medium text-gray-900">{comp.address}</p>
                              <p className="text-sm text-gray-600">{comp.area} sq ft • {comp.distance_km} km away</p>
                            </div>
                            <div className="text-right">
                              <p className="font-bold text-gray-900">₹{(comp.price / 100000).toFixed(2)}L</p>
                              <p className="text-sm text-gray-600">₹{comp.price_per_sqft}/sq ft</p>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}

                  <div className="flex gap-4">
                    <Button 
                      onClick={() => {
                        setResult(null);
                        setFormData({
                          location: '',
                          property_type: 'apartment',
                          area: '',
                          bedrooms: 2,
                          bathrooms: 2,
                          property_age: 0,
                          floor_number: 1,
                          total_floors: 10,
                          furnishing: 'unfurnished',
                          facing: 'north',
                          contact_email: '',
                          contact_phone: ''
                        });
                      }}
                      className="flex-1 bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600 text-white font-semibold"
                    >
                      New Calculation
                    </Button>
                    <Button 
                      variant="outline"
                      onClick={async () => {
                        await generateValuationPDF({
                          formData,
                          result
                        });
                      }}
                      className="flex-1 border-2 border-purple-500 text-purple-600 hover:bg-purple-50 font-semibold"
                    >
                      <Download className="w-4 h-4 mr-2" />
                      Download Report (Free)
                    </Button>
                  </div>
                </div>
              )}
            </div>
          )}
        </div>

        {/* CTA Section */}
        <div className="mt-16 bg-white rounded-lg p-8 shadow-sm border border-gray-200 text-center">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">Need a Professional Report?</h2>
          <p className="text-gray-600 mb-8 max-w-2xl mx-auto">
            Get a certified valuation report with IBBI registration, accepted by banks and legal authorities.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
            <Button 
              className="bg-[#4f46e5] hover:bg-[#4338ca] text-white px-8 py-3"
              onClick={() => setShowValuationFlow(true)}
            >
              <Phone className="w-4 h-4 mr-2" />
              Start Professional Valuation
            </Button>
            <Button 
              variant="outline" 
              className="border-[#4f46e5] text-[#4f46e5] hover:bg-[#4f46e5] hover:text-white px-8 py-3"
              onClick={() => setShowQuoteModal(true)}
            >
              <Mail className="w-4 h-4 mr-2" />
              Request Quote
            </Button>
          </div>
        </div>
      </div>
      
      {/* Quote Modal */}
      {showQuoteModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-md p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold">Request Quote</h3>
              <Button variant="ghost" size="sm" onClick={() => setShowQuoteModal(false)}>
                <X className="w-4 h-4" />
              </Button>
            </div>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-gray-600 mb-2">Service: {solution.title}</p>
                <p className="text-sm text-gray-600 mb-4">Pricing: {solution.pricing}</p>
              </div>
              <div className="bg-blue-50 p-4 rounded-lg">
                <p className="text-sm text-blue-800 mb-2">Contact us for a detailed quote:</p>
                <p className="text-sm font-medium text-blue-900">Phone: +91-8068447416</p>
                <p className="text-sm font-medium text-blue-900">Email: support@newageland.com</p>
              </div>
              <Button 
                className="w-full" 
                onClick={() => setShowQuoteModal(false)}
              >
                Close
              </Button>
            </div>
          </div>
        </div>
      )}
      
      <ValuationFlow 
        isOpen={showValuationFlow}
        onClose={() => setShowValuationFlow(false)}
        onComplete={(result) => {
          console.log('Valuation completed:', result);
          setShowValuationFlow(false);
        }}
      />
    </div>
  );
}