import { useState } from 'react';
import { Card } from '../components/ui/card';
import { QRScanner } from '../components/QRScanner';
import { Award, CheckCircle, XCircle, MapPin, Calendar, User, Shield } from 'lucide-react';
import { Button } from '../components/ui/button';

interface BadgeVerificationResult {
  isValid: boolean;
  badgeId: string;
  propertyName: string;
  propertyAddress: string;
  visitorName: string;
  visitDate: string;
  verificationMethod: string[];
  earnedPoints: number;
}

export const VerifyBadge = () => {
  const [isScanning, setIsScanning] = useState(false);
  const [verificationResult, setVerificationResult] = useState<BadgeVerificationResult | null>(null);
  const [isVerifying, setIsVerifying] = useState(false);

  const handleScanSuccess = (decodedText: string) => {
    setIsVerifying(true);
    
    // Simulate API call to verify badge
    setTimeout(() => {
      // Mock verification result
      const result: BadgeVerificationResult = {
        isValid: true,
        badgeId: decodedText,
        propertyName: 'Luxury Villa - Marine Drive',
        propertyAddress: 'Marine Drive, Mumbai, Maharashtra',
        visitorName: 'John Doe',
        visitDate: new Date().toLocaleDateString(),
        verificationMethod: ['QR Code', 'GPS Location'],
        earnedPoints: 50,
      };
      
      setVerificationResult(result);
      setIsVerifying(false);
      setIsScanning(false);
    }, 1500);
  };

  const resetVerification = () => {
    setVerificationResult(null);
    setIsScanning(false);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-12 px-4">
      <div className="max-w-2xl mx-auto">
        {/* Header */}
        <div className="text-center mb-8">
          <div className="inline-flex items-center justify-center w-20 h-20 bg-blue-600 rounded-full mb-4">
            <Shield className="w-10 h-10 text-white" />
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-2">Badge Verification</h1>
          <p className="text-lg text-gray-600">Scan any visit badge to verify its authenticity</p>
        </div>

        {/* Verification Result */}
        {verificationResult ? (
          <Card className="p-8">
            {/* Status Header */}
            <div className="text-center mb-6">
              {verificationResult.isValid ? (
                <>
                  <div className="w-24 h-24 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle className="w-12 h-12 text-green-600" />
                  </div>
                  <h2 className="text-3xl font-bold text-green-600 mb-2">✓ Verified Badge</h2>
                  <p className="text-gray-600">This badge is authentic and verified by NAL Platform</p>
                </>
              ) : (
                <>
                  <div className="w-24 h-24 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <XCircle className="w-12 h-12 text-red-600" />
                  </div>
                  <h2 className="text-3xl font-bold text-red-600 mb-2">✗ Invalid Badge</h2>
                  <p className="text-gray-600">This badge could not be verified</p>
                </>
              )}
            </div>

            {/* Badge Details */}
            {verificationResult.isValid && (
              <div className="space-y-4 mb-6">
                <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl p-6 border-2 border-blue-200">
                  <h3 className="font-bold text-gray-900 mb-4 flex items-center gap-2">
                    <Award className="w-5 h-5 text-blue-600" />
                    Visit Details
                  </h3>
                  
                  <div className="space-y-3">
                    <div className="flex items-start gap-3">
                      <MapPin className="w-5 h-5 text-gray-600 mt-0.5" />
                      <div>
                        <div className="font-semibold text-gray-900">{verificationResult.propertyName}</div>
                        <div className="text-sm text-gray-600">{verificationResult.propertyAddress}</div>
                      </div>
                    </div>

                    <div className="flex items-center gap-3">
                      <User className="w-5 h-5 text-gray-600" />
                      <div>
                        <div className="text-sm text-gray-600">Visitor</div>
                        <div className="font-semibold text-gray-900">{verificationResult.visitorName}</div>
                      </div>
                    </div>

                    <div className="flex items-center gap-3">
                      <Calendar className="w-5 h-5 text-gray-600" />
                      <div>
                        <div className="text-sm text-gray-600">Visit Date</div>
                        <div className="font-semibold text-gray-900">{verificationResult.visitDate}</div>
                      </div>
                    </div>

                    <div className="flex items-center gap-3">
                      <CheckCircle className="w-5 h-5 text-green-600" />
                      <div>
                        <div className="text-sm text-gray-600">Verification Method</div>
                        <div className="font-semibold text-gray-900">{verificationResult.verificationMethod.join(', ')}</div>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Verification Stamp */}
                <div className="bg-gradient-to-r from-green-600 to-emerald-600 text-white rounded-lg p-4 text-center">
                  <div className="flex items-center justify-center gap-2 mb-1">
                    <Shield className="w-5 h-5" />
                    <span className="font-bold">Verified by NAL Platform</span>
                  </div>
                  <p className="text-sm opacity-90">Badge ID: {verificationResult.badgeId.substring(0, 16)}...</p>
                </div>
              </div>
            )}

            {/* Actions */}
            <div className="flex gap-3">
              <Button
                onClick={resetVerification}
                className="flex-1 bg-blue-600 hover:bg-blue-700 text-white"
              >
                Scan Another Badge
              </Button>
              <Button
                onClick={() => window.location.href = '/'}
                variant="outline"
                className="flex-1"
              >
                Go Home
              </Button>
            </div>
          </Card>
        ) : (
          /* Scanner Interface */
          <Card className="p-8">
            {isVerifying ? (
              <div className="text-center py-12">
                <div className="animate-spin rounded-full h-16 w-16 border-b-4 border-blue-600 mx-auto mb-4"></div>
                <p className="text-lg text-gray-700 font-semibold">Verifying badge...</p>
                <p className="text-sm text-gray-600 mt-2">Please wait while we check the authenticity</p>
              </div>
            ) : !isScanning ? (
              <div className="text-center py-12">
                <div className="w-32 h-32 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-6">
                  <Award className="w-16 h-16 text-blue-600" />
                </div>
                <h3 className="text-2xl font-bold text-gray-900 mb-3">Scan Badge QR Code</h3>
                <p className="text-gray-600 mb-6">Point your camera at the badge QR code to verify its authenticity</p>
                <Button
                  onClick={() => setIsScanning(true)}
                  className="bg-blue-600 hover:bg-blue-700 text-white px-8 py-3 text-lg"
                >
                  Start Scanning
                </Button>
              </div>
            ) : (
              <div className="space-y-4">
                <QRScanner
                  onScanSuccess={handleScanSuccess}
                  onScanError={(error) => alert(error)}
                />
                <Button
                  onClick={() => setIsScanning(false)}
                  variant="outline"
                  className="w-full"
                >
                  Cancel
                </Button>
              </div>
            )}
          </Card>
        )}

        {/* Info Section */}
        <div className="mt-8 text-center">
          <p className="text-sm text-gray-600">
            All badges are cryptographically signed and verified through our secure platform
          </p>
        </div>
      </div>
    </div>
  );
};
