import React, { useState, useEffect } from 'react';
import { getWishlist } from '../utils/wishlistApi';
import { PropertyCard } from '../components/PropertyCard';

interface WishlistProperty {
  id: string;
  property_id: string;
  property: {
    id: number;
    title: string;
    locality: string;
    city: string;
    bedrooms: number;
    bathrooms: number;
    built_up_area: number;
    sale_price: string;
    photo: string;
    listing_type: string;
  };
}

export const Wishlist: React.FC = () => {
  const [wishlistItems, setWishlistItems] = useState<WishlistProperty[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchWishlist();
  }, []);

  const fetchWishlist = async () => {
    try {
      const items = await getWishlist();
      setWishlistItems(items as any);
    } catch (error) {
      console.error('Error fetching wishlist:', error);
    } finally {
      setLoading(false);
    }
  };

  const formatPrice = (price: string) => {
    const numPrice = parseFloat(price);
    if (isNaN(numPrice)) return 'Price on request';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 py-8">
        <div className="max-w-7xl mx-auto px-4">
          <div className="text-center">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-500">Loading your saved properties...</p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-2">My Wishlist</h1>
          <p className="text-gray-600">Properties you've saved for later</p>
        </div>

        {wishlistItems.length === 0 ? (
          <div className="text-center py-16">
            <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
            </svg>
            <h3 className="text-xl font-semibold text-gray-900 mb-2">No saved properties</h3>
            <p className="text-gray-600 mb-6">Start exploring properties and save your favorites!</p>
            <a 
              href="/properties" 
              className="inline-flex items-center px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              Browse Properties
            </a>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {wishlistItems.map((item) => (
              <PropertyCard
                key={item.id}
                id={item.property.id}
                title={item.property.title}
                location={`${item.property.locality}, ${item.property.city}`}
                beds={item.property.bedrooms}
                baths={item.property.bathrooms}
                area={`${item.property.built_up_area} sq.ft`}
                price={formatPrice(item.property.sale_price)}
                imageUrl={item.property.photo}
                isUrgent={item.property.listing_type === 'urgent-sale'}
                isBidding={item.property.listing_type === 'bidding'}
                isFavorited={true}
              />
            ))}
          </div>
        )}
      </div>
    </div>
  );
};