
import React, { useState, useEffect } from 'react';
import { adminService } from '../../../api/adminService';
import {
    Plus,
    Trash2,
    Edit2,
    RefreshCw,
    BarChart2,
    Copy,
    Check,
    Ban,
    CheckCircle,
    AlertCircle,
    Shield,
    Mail,
    Code,
    Download,
    Search,
    Lock
} from 'lucide-react';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow
} from "@/components/ui/table";
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog";
import { toast } from "sonner";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
} from 'recharts';

interface APIKey {
    id: number;
    key_prefix: string;  // Backend returns key_prefix, not full key
    name: string;
    created_by: string;  // UUID of admin who created the key
    status: string;  // 'active' | 'revoked' | 'expired'
    created_at: string;
    rate_limit: number;  // Backend uses rate_limit, not rate_limit_per_hour
    allowed_ips: string[] | null;
    allowed_endpoints: string[] | null;  // Array: ['properties','buyers','sellers']
    last_used_at: string | null;
    request_count: number;
    scopes: string[] | null;
    expires_at: string | null;
}

interface APIKeyStats {
    key_id: number;
    key_name: string;
    is_active: boolean;
    rate_limit: number;
    allowed_ips: string;
    total_lifetime_requests: number;
    last_used_at: string | null;
    created_at: string;
    last_30_days_requests: number;
    current_hour_requests: number;
    remaining_requests_this_hour: number;
    graph_data?: any[];
}

const AdminAPIKeys = () => {
    const [apiKeys, setApiKeys] = useState<APIKey[]>([]);
    const [loading, setLoading] = useState(true);
    const [summary, setSummary] = useState<any>(null);
    const [searchQuery, setSearchQuery] = useState('');
    // useToast from sonner is different than ui/use-toast. adjusting.
    // Actually typically toast() is a function exported directly or via hook.
    // The previous code imported useToast from "@/hooks/use-toast" but destination uses "sonner".
    // I will use toast from "sonner".

    // Create Modal State
    const [createOpen, setCreateOpen] = useState(false);
    const [newKeyData, setNewKeyData] = useState({
        name: '',
        rate_limit: 1000,  // Requests per minute
        allowed_ips: '',
        allowed_endpoints: 'properties,buyers,sellers',  // NAL Public API endpoints
        scopes: ['read'],  // Default read scope
        expires_in_days: 365  // 1 year expiry
    });
    const [createdKey, setCreatedKey] = useState<string | null>(null);

    // Edit Modal State
    const [editOpen, setEditOpen] = useState(false);
    const [editingKey, setEditingKey] = useState<APIKey | null>(null);

    // Stats Modal State
    const [statsOpen, setStatsOpen] = useState(false);
    const [currentStats, setCurrentStats] = useState<APIKeyStats | null>(null);

    // Stats Chart Data
    const [statsData, setStatsData] = useState<any[]>([]);
    const [regeneratedKey, setRegeneratedKey] = useState<string | null>(null);
    const [showRegenerated, setShowRegenerated] = useState(false);

    // Feature 9: Secure Bundle State
    const [bundleModalOpen, setBundleModalOpen] = useState(false);
    const [bundleEmail, setBundleEmail] = useState('');
    const [selectedKeyForBundle, setSelectedKeyForBundle] = useState<string>(''); // Key ID as string for Select
    const [sendingBundle, setSendingBundle] = useState(false);
    const [bundleSent, setBundleSent] = useState(false);

    useEffect(() => {
        fetchKeys();
        fetchSummary();
    }, []);

    const fetchKeys = async () => {
        try {
            setLoading(true);
            const res = await adminService.apiKeys.getAll();
            setApiKeys(res.data.results || []);
        } catch (error) {
            console.error("Failed to fetch API keys", error);
            toast.error("Failed to load API keys");
        } finally {
            setLoading(false);
        }
    };

    const fetchSummary = async () => {
        try {
            const res = await adminService.apiKeys.getSummary();
            setSummary(res.data);
        } catch (error) {
            console.error("Failed to fetch summary", error);
        }
    };

    const handleCreate = async () => {
        if (!newKeyData.name) {
            toast.error("Name is required");
            return;
        }

        try {
            const res = await adminService.apiKeys.create(newKeyData);
            setCreatedKey(res.data.api_key);  // Backend returns 'api_key' field
            toast.success("API Key created successfully");
            fetchKeys();
            fetchSummary();
            // Don't close immediately, let user copy key
        } catch (error) {
            toast.error("Failed to create API Key");
        }
    };

    const handleUpdate = async () => {
        if (!editingKey) return;
        try {
            const payload = {
                name: editingKey.name,
                rate_limit: editingKey.rate_limit,
                allowed_ips: editingKey.allowed_ips
            };
            await adminService.apiKeys.update(editingKey.id, payload);
            toast.success("API Key updated");
            setEditOpen(false);
            fetchKeys();
        } catch (error) {
            toast.error("Failed to update API Key");
        }
    };

    const handleDelete = async (id: number) => {
        if (!confirm("Are you sure you want to delete this API Key? This actions cannot be undone.")) return;
        try {
            await adminService.apiKeys.delete(id);
            toast.success("API Key deleted");
            fetchKeys();
            fetchSummary();
        } catch (error) {
            toast.error("Failed to delete API Key");
        }
    };

    const handleRegenerate = async (id: number) => {
        if (!confirm("Regenerate Key? The old key will stop working immediately.")) return;
        try {
            const res = await adminService.apiKeys.regenerate(id);
            setRegeneratedKey(res.data.data.key_value);
            setShowRegenerated(true);
            fetchKeys();
        } catch (error) {
            toast.error("Failed to regenerate key");
        }
    };

    const toggleStatus = async (key: APIKey) => {
        try {
            if (key.status === 'active') {
                await adminService.apiKeys.deactivate(key.id);
                toast.success(`${key.name} is now inactive`);
            } else {
                await adminService.apiKeys.activate(key.id);
                toast.success(`${key.name} is now active`);
            }
            fetchKeys();
            fetchSummary();
        } catch (error) {
            toast.error("Failed to change status");
        }
    };

    const viewStats = async (id: number) => {
        try {
            const res = await adminService.apiKeys.getStats(id);
            setCurrentStats(res.data);


            // Use real graph data from backend
            setStatsData(res.data.graph_data || []);

            setStatsOpen(true);
        } catch (error) {
            toast.error("Failed to load stats");
        }
    };

    const handleSendBundle = async () => {
        if (!bundleEmail || !selectedKeyForBundle) {
            toast.error("Email and API Key are required");
            return;
        }

        try {
            setSendingBundle(true);
            const res = await adminService.apiKeys.sendBundle(bundleEmail, parseInt(selectedKeyForBundle));
            setBundleSent(true);
            toast.success("Bundle Sent! The Developer Kit has been emailed successfully.");
        } catch (error: any) {
            toast.error(error.response?.data?.message || "Could not send the bundle.");
            console.error(error);
        } finally {
            setSendingBundle(false);
        }
    };

    const handleDownloadBundle = async () => {
        try {
            const res = await adminService.apiKeys.downloadBundle();
            const url = window.URL.createObjectURL(new Blob([res.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', 'NAL_Developer_Kit.zip');
            document.body.appendChild(link);
            link.click();
            link.remove();
            window.URL.revokeObjectURL(url);
            toast.success('NAL_Developer_Kit.zip downloaded. Add your API Key to config.js.');
        } catch (error: any) {
            toast.error(error.response?.data?.message || "Could not download the bundle.");
            console.error(error);
        }
    };

    return (
        <div className="max-w-7xl mx-auto pb-12 px-4 sm:px-6 lg:px-8">
            {/* SEGMENT 1: Header & Summary */}
            <div className="pt-8 pb-3 space-y-6">
                <div className="flex flex-col justify-center items-center text-center space-y-2">
                    <div>
                        <h1 className="text-3xl font-extrabold text-black tracking-tight">API Keys Management</h1>
                        <p className="text-gray-600 mt-2 font-medium">Manage access keys for the Public API Framework</p>
                    </div>
                </div>

                {/* Summary Cards */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <Card className="shadow-sm hover:shadow-md transition-shadow">
                        <CardHeader className="pb-2">
                            <CardTitle className="text-sm font-medium text-gray-500">Total Keys</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{summary?.total_keys || 0}</div>
                        </CardContent>
                    </Card>
                    <Card className="shadow-sm hover:shadow-md transition-shadow">
                        <CardHeader className="pb-2">
                            <CardTitle className="text-sm font-medium text-green-500">Active Keys</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{summary?.active_keys || 0}</div>
                        </CardContent>
                    </Card>
                    <Card className="shadow-sm hover:shadow-md transition-shadow">
                        <CardHeader className="pb-2">
                            <CardTitle className="text-sm font-medium text-red-500">Inactive Keys</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-red-600">{summary?.inactive_keys || 0}</div>
                        </CardContent>
                    </Card>
                    <Card className="shadow-sm hover:shadow-md transition-shadow">
                        <CardHeader className="pb-2">
                            <CardTitle className="text-sm font-medium text-[#0056D2]">Total Requests</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-[#0056D2]">{summary?.total_requests_all_time || 0}</div>
                        </CardContent>
                    </Card>
                </div>
            </div>

            {/* SEGMENT 2: Developer Tooling */}
            <div className="py-3">
            <Card className="bg-white border border-blue-100 shadow-sm hover:shadow-md transition-all w-full overflow-hidden">
                <div className="flex flex-col md:flex-row items-center justify-between p-4 gap-4">
                    <div className="flex items-center gap-4 flex-1">
                        <div className="bg-[#eff6ff] p-3 rounded-lg shrink-0 border border-blue-50">
                            <Code className="w-6 h-6 text-[#0056D2]" />
                        </div>
                        <div>
                            <div className="flex items-center gap-2">
                                <h3 className="text-lg font-bold text-gray-900">
                                    Developer Kit
                                </h3>
                                <Badge className="bg-[#eff6ff] text-[#0056D2] border-blue-100 hover:bg-blue-50 text-xs px-2 py-0 h-5">v1.0.2</Badge>
                            </div>
                            <p className="text-gray-500 text-sm">
                                Access the complete web dashboard and export API bundles.
                            </p>
                        </div>
                    </div>

                    <div className="flex items-center gap-3 shrink-0">
                        <div className="h-8 w-px bg-gray-200 hidden md:block mx-2"></div>
                        <Button
                            variant="outline"
                            className="bg-white text-gray-700 border border-gray-200 hover:border-blue-300 hover:bg-blue-50 hover:text-[#0056D2] font-semibold h-9 shadow-sm"
                            onClick={handleDownloadBundle}
                        >
                            <Download className="mr-2 h-4 w-4" /> Download
                        </Button>
                        <Button
                            className="bg-[#0056D2] hover:bg-blue-700 text-white border-none font-bold shadow-sm shadow-blue-200 h-9 px-4 rounded-lg"
                            onClick={() => {
                                setBundleEmail('');
                                setBundleSent(false);
                                setBundleModalOpen(true);
                            }}
                        >
                            <Mail className="mr-2 h-4 w-4" /> Email Bundle
                        </Button>
                    </div>
                </div>
            </Card>
            </div>

            {/* Action Bar */}
            <div className="py-3">
            <div className="flex justify-between items-center bg-white p-4 rounded-lg border border-gray-100 gap-4">
                <Input
                    placeholder="Search API keys..."
                    className="max-w-sm"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                />
                <Dialog open={createOpen} onOpenChange={(open) => {
                    setCreateOpen(open);
                    if (!open) setCreatedKey(null); // Reset created key on close
                }}>
                    <DialogTrigger asChild>
                        <Button className="bg-[#0056D2] hover:bg-blue-700 text-white">
                            <Plus className="mr-2" /> Create New Key
                        </Button>
                    </DialogTrigger>
                    <DialogContent className="sm:max-w-[500px]">
                        <DialogHeader>
                            <DialogTitle>Create New API Key</DialogTitle>
                            <DialogDescription>
                                Generate a new key for an external application.
                            </DialogDescription>
                        </DialogHeader>

                        {!createdKey ? (
                            <div className="grid gap-4 py-4">
                                <div className="grid grid-cols-4 items-center gap-4">
                                    <Label htmlFor="name" className="text-right">Name</Label>
                                    <Input
                                        id="name"
                                        value={newKeyData.name}
                                        onChange={(e) => setNewKeyData({ ...newKeyData, name: e.target.value })}
                                        className="col-span-3"
                                        placeholder="e.g. My App Integration"
                                    />
                                </div>
                                <div className="grid grid-cols-4 items-center gap-4">
                                    <Label htmlFor="rate" className="text-right">Rate Limit</Label>
                                    <Input
                                        id="rate"
                                        type="number"
                                        value={newKeyData.rate_limit}
                                        onChange={(e) => setNewKeyData({ ...newKeyData, rate_limit: parseInt(e.target.value) })}
                                        className="col-span-3"
                                    />
                                </div>
                                <div className="grid grid-cols-4 items-start gap-4">
                                    <div className="text-right pt-2">
                                        <Label htmlFor="ips">Allowed IPs</Label>
                                        <div className="text-[11px] text-gray-500 font-normal mt-0.5">(Optional)</div>
                                    </div>
                                    <Textarea
                                        id="ips"
                                        value={newKeyData.allowed_ips}
                                        onChange={(e) => setNewKeyData({ ...newKeyData, allowed_ips: e.target.value })}
                                        className="col-span-3"
                                        placeholder="Comma separated IPs (e.g. 192.168.1.1). Leave empty for any IP."
                                    />
                                </div>
                                <div className="grid grid-cols-4 items-start gap-4">
                                    <div className="text-right pt-2">
                                        <Label>Allowed Endpoints</Label>
                                        <div className="text-[11px] text-gray-500 font-normal mt-0.5">(Select access)</div>
                                    </div>
                                    <div className="col-span-3 grid grid-cols-2 gap-2">
                                        {['properties', 'buyers', 'sellers'].map((endpoint) => (
                                            <label key={endpoint} className="flex items-center gap-2 p-2 border rounded cursor-pointer hover:bg-gray-50">
                                                <input
                                                    type="checkbox"
                                                    checked={newKeyData.allowed_endpoints.includes(endpoint)}
                                                    onChange={(e) => {
                                                        const current = newKeyData.allowed_endpoints.split(',').filter(e => e);
                                                        if (e.target.checked) {
                                                            current.push(endpoint);
                                                        } else {
                                                            const idx = current.indexOf(endpoint);
                                                            if (idx > -1) current.splice(idx, 1);
                                                        }
                                                        setNewKeyData({ ...newKeyData, allowed_endpoints: current.join(',') });
                                                    }}
                                                    className="w-4 h-4 text-[#0056D2]"
                                                />
                                                <span className="capitalize font-medium">{endpoint}</span>
                                            </label>
                                        ))}
                                    </div>
                                </div>
                            </div>
                        ) : (
                            <div className="py-6 space-y-4">
                                <div className="p-6 bg-[#f0f9ff] border border-blue-100 rounded-xl text-center shadow-sm">
                                    <div className="bg-white p-3 rounded-full w-fit mx-auto mb-3 shadow-sm border border-blue-50">
                                        <CheckCircle className="text-[#0056D2] w-10 h-10" />
                                    </div>
                                    <h3 className="text-xl font-bold text-[#0056D2]">API Key Created!</h3>
                                    <p className="text-sm text-blue-600/80 mb-6 font-medium">Copy this key now. It won't be shown again.</p>

                                    <div className="flex items-center gap-2 bg-white p-3 rounded-lg border border-blue-200 shadow-inner">
                                        <code className="flex-1 font-mono text-sm break-all font-bold text-gray-800">{createdKey}</code>
                                        <Button variant="ghost" size="sm" className="hover:bg-blue-50 text-[#0056D2]" onClick={() => {
                                            navigator.clipboard.writeText(createdKey);
                                            // toast.success("Key copied to clipboard");
                                        }}>
                                            <Copy className="w-4 h-4" />
                                        </Button>
                                    </div>
                                </div>
                            </div>
                        )}

                        <DialogFooter>
                            {!createdKey ? (
                                <Button onClick={handleCreate} className="bg-[#0056D2] hover:bg-blue-700 text-white">Generate Key</Button>
                            ) : (
                                <Button onClick={() => setCreateOpen(false)}>Close</Button>
                            )}
                        </DialogFooter>
                    </DialogContent>
                </Dialog>

                {/* Developer Guide Link */}
                <Button
                    variant="outline"
                    className="ml-2 text-gray-600 border-gray-300"
                    onClick={() => {
                        const baseUrl = window.location.origin.includes('localhost')
                            ? 'http://localhost:8090'
                            : window.location.origin;
                        window.open(`${baseUrl}/api/v1/devtools/developer-guide/`, '_blank');
                    }}
                >
                    <AlertCircle className="mr-2" /> Developer Guide
                </Button>
            </div>
            </div>

            {/* Keys Table */}
            <div className="py-3">
            <Card className="shadow-none border-gray-200">
                <CardContent className="p-0">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead className="text-center">Namespace / Name</TableHead>
                                <TableHead className="text-center">Key Prefix</TableHead>
                                <TableHead className="text-center">Status</TableHead>
                                <TableHead className="text-center">Rate Limit</TableHead>
                                <TableHead className="text-center">Last Used</TableHead>
                                <TableHead className="text-center">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {apiKeys.filter(key => 
                                key.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                                key.key_prefix.toLowerCase().includes(searchQuery.toLowerCase()) ||
                                key.id.toString().includes(searchQuery)
                            ).map((key) => (
                                <TableRow key={key.id}>
                                    <TableCell className="font-medium text-center">
                                        <div className="flex flex-col items-center">
                                            <div>{key.name}</div>
                                            <div className="text-xs text-gray-500">ID: {key.id}</div>
                                        </div>
                                    </TableCell>
                                    <TableCell className="text-center">
                                        <div className="flex items-center gap-2 justify-center">
                                            <code className="bg-gray-100 px-2 py-1 rounded text-xs font-mono">
                                                {key.key_prefix}...
                                            </code>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                className="h-7 w-7 hover:bg-blue-50"
                                                title="Copy full API key from vault"
                                                onClick={async () => {
                                                    const loadingToast = toast.loading("Retrieving key from vault...");
                                                    try {
                                                        const res = await adminService.apiKeys.getRawKey(key.id);
                                                        const rawKey = res.data.raw_key || res.data.api_key;
                                                        if (rawKey) {
                                                            await navigator.clipboard.writeText(rawKey);
                                                            toast.dismiss(loadingToast);
                                                            toast.success("✓ Full API key copied to clipboard!", {
                                                                description: `${rawKey.substring(0, 15)}...`,
                                                                duration: 3000
                                                            });
                                                        } else {
                                                            toast.dismiss(loadingToast);
                                                            toast.error("Could not retrieve API key");
                                                        }
                                                    } catch (error: any) {
                                                        toast.dismiss(loadingToast);
                                                        toast.error("Failed to retrieve key from vault", {
                                                            description: error.response?.data?.detail || "Please try again"
                                                        });
                                                        console.error("Vault retrieval error:", error);
                                                    }
                                                }}
                                            >
                                                <Copy className="h-3 w-3 text-[#0056D2]" />
                                            </Button>
                                        </div>
                                    </TableCell>
                                    <TableCell className="text-center">
                                        <div className="flex flex-col gap-1 items-center justify-center">
                                            {key.status === 'active' ? (
                                                <Badge className="bg-green-100 text-green-800 hover:bg-green-100 border-none">Active</Badge>
                                            ) : key.status === 'revoked' ? (
                                                <Badge variant="destructive" className="bg-red-100 text-red-800 hover:bg-red-100 border-none">Revoked</Badge>
                                            ) : (
                                                <Badge variant="secondary" className="bg-yellow-100 text-yellow-800 hover:bg-yellow-100 border-none">Expired</Badge>
                                            )}
                                            <Badge variant="outline" className="bg-blue-50 text-[#0056D2] border-blue-100 text-[10px] py-0">
                                                <Lock size={8} className="mr-1" /> SECURED
                                            </Badge>
                                        </div>
                                    </TableCell>
                                    <TableCell className="text-center">{key.rate_limit}/hr</TableCell>
                                    <TableCell className="text-center">
                                        {key.last_used_at ? new Date(key.last_used_at).toLocaleDateString() : 'Never'}
                                    </TableCell>
                                    <TableCell className="text-center">
                                        <div className="flex justify-center gap-2">
                                            <Button variant="ghost" size="icon" title="View Stats" onClick={() => viewStats(key.id)}>
                                                <BarChart2 className="h-4 w-4 text-gray-500" />
                                            </Button>
                                            <Button variant="ghost" size="icon" title="Edit" onClick={() => {
                                                setEditingKey(key);
                                                setEditOpen(true);
                                            }}>
                                                <Edit2 className="h-4 w-4 text-[#0056D2]" />
                                            </Button>
                                            <Button variant="ghost" size="icon" title="Regenerate" onClick={() => handleRegenerate(key.id)}>
                                                <RefreshCw className="h-4 w-4 text-[#0056D2]" />
                                            </Button>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                title={key.status === 'active' ? "Deactivate" : "Activate"}
                                                onClick={() => toggleStatus(key)}
                                            >
                                                {key.status === 'active' ? (
                                                    <Ban className="h-4 w-4 text-red-500" />
                                                ) : (
                                                    <Check className="h-4 w-4 text-green-500" />
                                                )}
                                            </Button>
                                            <Button variant="ghost" size="icon" title="Delete" onClick={() => handleDelete(key.id)}>
                                                <Trash2 className="h-4 w-4 text-red-500" />
                                            </Button>
                                        </div>
                                    </TableCell>
                                </TableRow>
                            ))}
                            {apiKeys.filter(key => 
                                key.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                                key.key_prefix.toLowerCase().includes(searchQuery.toLowerCase()) ||
                                key.id.toString().includes(searchQuery)
                            ).length === 0 && (
                                <TableRow>
                                    <TableCell colSpan={6} className="text-center py-8 text-gray-500">
                                        No API keys found. Create one to get started.
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
            </div>

            {/* Edit Modal */}
            <Dialog open={editOpen} onOpenChange={setEditOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Edit API Key</DialogTitle>
                    </DialogHeader>
                    {editingKey && (
                        <div className="grid gap-4 py-4">
                            <div className="grid grid-cols-4 items-center gap-4">
                                <Label htmlFor="edit-name" className="text-right">Name</Label>
                                <Input
                                    id="edit-name"
                                    value={editingKey.name}
                                    onChange={(e) => setEditingKey({ ...editingKey, name: e.target.value })}
                                    className="col-span-3"
                                />
                            </div>
                            <div className="grid grid-cols-4 items-center gap-4">
                                <Label htmlFor="edit-rate" className="text-right">Rate Limit</Label>
                                <Input
                                    id="edit-rate"
                                    type="number"
                                    value={editingKey.rate_limit}
                                    onChange={(e) => setEditingKey({ ...editingKey, rate_limit: parseInt(e.target.value) })}
                                    className="col-span-3"
                                />
                            </div>
                            <div className="grid grid-cols-4 items-start gap-4">
                                <div className="text-right pt-2">
                                    <Label htmlFor="edit-ips">Allowed IPs</Label>
                                    <div className="text-[11px] text-gray-500 font-normal mt-0.5">(Optional)</div>
                                </div>
                                <Textarea
                                    id="edit-ips"
                                    value={Array.isArray(editingKey.allowed_ips) ? editingKey.allowed_ips.join(', ') : (editingKey.allowed_ips || '')}
                                    onChange={(e) => setEditingKey({ ...editingKey, allowed_ips: e.target.value.split(',').map(ip => ip.trim()).filter(Boolean) })}
                                    className="col-span-3"
                                />
                            </div>
                        </div>
                    )}
                    <DialogFooter>
                        <Button onClick={handleUpdate} className="bg-[#0056D2] hover:bg-blue-700 text-white">Save Changes</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            {/* Stats Modal */}
            <Dialog open={statsOpen} onOpenChange={setStatsOpen}>
                <DialogContent className="max-w-3xl">
                    <DialogHeader>
                        <DialogTitle>Usage Statistics</DialogTitle>
                        <DialogDescription>Usage data for {currentStats?.key_name}</DialogDescription>
                    </DialogHeader>

                    {currentStats && (
                        <div className="space-y-6">
                            <div className="grid grid-cols-3 gap-4">
                                <div className="p-4 bg-gray-50 rounded-lg">
                                    <div className="text-sm text-gray-500">Requests this Hour</div>
                                    <div className="text-2xl font-bold flex items-baseline gap-2">
                                        {currentStats.current_hour_requests}
                                        <span className="text-xs font-normal text-gray-400">/ {currentStats.rate_limit}</span>
                                    </div>
                                </div>
                                <div className="p-4 bg-gray-50 rounded-lg">
                                    <div className="text-sm text-gray-500">Last 30 Days</div>
                                    <div className="text-2xl font-bold">{currentStats.last_30_days_requests}</div>
                                </div>
                                <div className="p-4 bg-gray-50 rounded-lg">
                                    <div className="text-sm text-gray-500">Lifetime Total</div>
                                    <div className="text-2xl font-bold">{currentStats.total_lifetime_requests}</div>
                                </div>
                            </div>

                            <div className="h-[300px] w-full bg-white p-4 border rounded-lg">
                                <h4 className="text-sm font-medium mb-4 text-gray-500">Daily Requests (Mock Data)</h4>
                                <ResponsiveContainer width="100%" height="100%">
                                    <BarChart data={statsData}>
                                        <CartesianGrid strokeDasharray="3 3" vertical={false} />
                                        <XAxis dataKey="day" hide />
                                        <YAxis />
                                        <Tooltip />
                                        <Bar dataKey="requests" fill="#8884d8" radius={[4, 4, 0, 0]} />
                                    </BarChart>
                                </ResponsiveContainer>
                            </div>

                            <div className="text-xs text-center text-gray-400">
                                Created on: {new Date(currentStats.created_at).toLocaleDateString()}
                            </div>
                        </div>
                    )}
                </DialogContent>
            </Dialog>

            {/* Regenerated Key Modal */}
            <Dialog open={showRegenerated} onOpenChange={setShowRegenerated}>
                <DialogContent className="sm:max-w-[425px]">
                    <DialogHeader>
                        <DialogTitle className="flex items-center gap-2">
                            <Shield className="text-[#0056D2]" /> Key Regenerated
                        </DialogTitle>
                        <DialogDescription>
                            Copy your new key now. You won't be able to see it again!
                        </DialogDescription>
                    </DialogHeader>
                    <div className="bg-gray-50 p-4 rounded-lg flex items-center justify-between border">
                        <code className="text-xs font-mono break-all pr-4">
                            {regeneratedKey}
                        </code>
                        <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => {
                                if (regeneratedKey) {
                                    navigator.clipboard.writeText(regeneratedKey);
                                    // toast.success("API Key copied to clipboard");
                                }
                            }}
                        >
                            <Copy />
                        </Button>
                    </div>
                    <DialogFooter>
                        <Button onClick={() => setShowRegenerated(false)} className="w-full">
                            I've saved it
                        </Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            {/* Feature 9: Secure Bundle Modal */}
            <Dialog open={bundleModalOpen} onOpenChange={setBundleModalOpen}>
                <DialogContent className="w-[520px] max-w-[90vw]">
                    <DialogHeader>
                        <DialogTitle className="flex items-center gap-2">
                            <Shield className="text-[#0056D2]" /> Email Developer Kit
                        </DialogTitle>
                        <DialogDescription>
                            Send the Developer Kit bundle with the API key pre-configured.
                        </DialogDescription>
                    </DialogHeader>

                    {!bundleSent ? (
                        <div className="grid gap-4 py-4">
                            <div className="space-y-2">
                                <Label>Developer Email</Label>
                                <Input
                                    placeholder="dev@partner.com"
                                    value={bundleEmail}
                                    onChange={(e) => setBundleEmail(e.target.value)}
                                />
                            </div>
                            <div className="space-y-2">
                                <Label>Select API Key to Include</Label>
                                <p className="text-xs text-gray-500">Click on a key below to select it</p>
                                <div className="border rounded-md">
                                    <div className="bg-gray-50 px-3 py-2 border-b text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Available Keys ({apiKeys.filter(k => k.status === 'active').length})
                                    </div>
                                    <ScrollArea className="h-[180px]">
                                        <div className="p-1 space-y-1">
                                            {apiKeys.filter(k => k.status === 'active').length === 0 ? (
                                                <div className="text-center py-8 text-gray-500 text-sm">
                                                    No active API keys found.
                                                </div>
                                            ) : (
                                                apiKeys.filter(k => k.status === 'active').map((key) => (
                                                    <div
                                                        key={key.id}
                                                        onClick={() => setSelectedKeyForBundle(key.id.toString())}
                                                        className={`
                                                            flex items-center justify-between p-3 rounded cursor-pointer transition-all text-sm
                                                            ${selectedKeyForBundle === key.id.toString()
                                                                ? 'bg-[#0056D2] text-white border-2 border-blue-700 shadow-lg'
                                                                : 'bg-white hover:bg-gray-50 border border-gray-200'}
                                                        `}
                                                    >
                                                        <div className="flex flex-col">
                                                            <span className="font-medium">{key.name}</span>
                                                            <code className={`text-[10px] font-mono mt-0.5 ${selectedKeyForBundle === key.id.toString()
                                                                ? 'text-blue-100'
                                                                : 'text-gray-500'
                                                                }`}>
                                                                {key.key_prefix}...
                                                            </code>
                                                        </div>
                                                    </div>
                                                ))
                                            )}
                                        </div>
                                    </ScrollArea>
                                </div>
                            </div>
                        </div>
                    ) : (
                        <div className="py-8 text-center space-y-4">
                            <div className="h-16 w-16 bg-green-100 text-green-600 rounded-full flex items-center justify-center mx-auto">
                                <Mail className="h-8 w-8" />
                            </div>
                            <h3 className="text-xl font-bold text-gray-900">Bundle Sent Successfully!</h3>
                            <p className="text-gray-500 max-w-xs mx-auto">
                                We've emailed a secure, self-destructing link to {bundleEmail}.
                            </p>
                            <Button onClick={() => setBundleModalOpen(false)} variant="outline">
                                Close
                            </Button>
                        </div>
                    )}

                    {!bundleSent && (
                        <DialogFooter>
                            <Button
                                onClick={handleSendBundle}
                                disabled={sendingBundle || !bundleEmail || !selectedKeyForBundle}
                                className="bg-orange-600 hover:bg-orange-700 text-white disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {sendingBundle ? 'Sending...' : 'Send Secure Bundle'}
                            </Button>
                        </DialogFooter>
                    )}
                </DialogContent>
            </Dialog>
        </div>
    );
};

export default AdminAPIKeys;
