import React, { useEffect, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { adminService } from '../../../api/adminService';
import { Loader2, CheckCircle, XCircle } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

const ConnectorCallback = () => {
    const [searchParams] = useSearchParams();
    const navigate = useNavigate();
    const [status, setStatus] = useState<'loading' | 'success' | 'error'>('loading');
    const [message, setMessage] = useState('Processing OAuth callback...');

    useEffect(() => {
        handleCallback();
    }, []);

    const handleCallback = async () => {
        try {
            const code = searchParams.get('code');
            const state = searchParams.get('state');
            const error = searchParams.get('error');

            if (error) {
                setStatus('error');
                setMessage(`OAuth error: ${error}`);
                setTimeout(() => navigate('/admin/dashboard/developer/connectors'), 3000);
                return;
            }

            if (!code || !state) {
                setStatus('error');
                setMessage('Missing authorization code or state');
                setTimeout(() => navigate('/admin/dashboard/developer/connectors'), 3000);
                return;
            }

            const response = await adminService.connectors.handleCallback({ code, state });

            if (response.data.success) {
                setStatus('success');
                setMessage('Successfully connected! Redirecting...');
                setTimeout(() => navigate('/admin/dashboard/developer/connectors'), 2000);
            } else {
                setStatus('error');
                setMessage('Failed to complete OAuth flow');
                setTimeout(() => navigate('/admin/dashboard/developer/connectors'), 3000);
            }
        } catch (error: any) {
            console.error('OAuth callback error:', error);
            setStatus('error');
            setMessage(error.response?.data?.detail || 'Failed to process OAuth callback');
            setTimeout(() => navigate('/admin/dashboard/developer/connectors'), 3000);
        }
    };

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50">
            <Card className="w-full max-w-md">
                <CardHeader>
                    <CardTitle className="text-center">OAuth Connection</CardTitle>
                </CardHeader>
                <CardContent className="flex flex-col items-center space-y-4">
                    {status === 'loading' && (
                        <>
                            <Loader2 className="h-12 w-12 animate-spin text-purple-600" />
                            <p className="text-gray-600">{message}</p>
                        </>
                    )}
                    {status === 'success' && (
                        <>
                            <CheckCircle className="h-12 w-12 text-green-600" />
                            <p className="text-green-600 font-medium">{message}</p>
                        </>
                    )}
                    {status === 'error' && (
                        <>
                            <XCircle className="h-12 w-12 text-red-600" />
                            <p className="text-red-600 font-medium">{message}</p>
                        </>
                    )}
                </CardContent>
            </Card>
        </div>
    );
};

export default ConnectorCallback;
