
import React, { useState, useEffect } from 'react';
import { useConnectorAPI } from '../../../hooks/useConnectorAPI';
import { ConnectorCard } from '../../../components/admin-developer/connectors/ConnectorCard';
import { SyncJobMonitor } from '../../../components/admin-developer/connectors/SyncJobMonitor';
import { AddConnectorModal } from '../../../components/admin-developer/connectors/AddConnectorModal';
import { adminService } from '../../../api/adminService';
import { RefreshCw, AlertCircle } from 'lucide-react';
import { Button } from "@/components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

const Connectors = () => {
    const { connectors, loading, error, refetch } = useConnectorAPI();
    const [activeTab, setActiveTab] = useState('connectors');
    const [isAddModalOpen, setIsAddModalOpen] = useState(false);

    // NAL Property Platform Providers (from Marketplace catalog)
    const allProviders = [
        { id: 'salesforce-crm', name: 'Salesforce CRM', description: 'Sync Leads, Contacts, and Opportunity data' },
        { id: 'zoho-crm', name: 'Zoho CRM', description: 'Sync Sellers, Agents and Deal stages' },
        { id: 'google-workspace', name: 'Google Workspace', description: 'Sync contacts, calendar events and drive documents' },
        { id: 'hubspot', name: 'HubSpot', description: 'Inbound marketing and email automation' },
        { id: 'mailchimp', name: 'Mailchimp', description: 'Email marketing automation and audience management' },
        { id: 'slack', name: 'Slack', description: 'Team notifications for offers and property alerts' },
        { id: 'zillow', name: 'Zillow', description: 'Syndicate property listings' },
        { id: 'docusign', name: 'DocuSign', description: 'Track digital sale agreements' },
        { id: 'quickbooks', name: 'QuickBooks', description: 'Generate invoices and sync transactions' },
        { id: 'calendly', name: 'Calendly', description: 'Book property tours and sync availability' }
    ];

    useEffect(() => {
        // No need to fetch configs separately - connectors come from marketplace
    }, []);

    const handleDeleteConfig = async (id: number) => {
        if (confirm("Remove this integration?")) {
            await adminService.integrations.uninstall(id);
            refetch();
        }
    };

    return (
        <div className="space-y-13">
            <div className="flex justify-between items-start">
                <div className="flex flex-col items-center text-center flex-1 mr-[1.5cm]">
                    <h1 className="text-3xl font-extrabold text-black tracking-tight">Third-Party Connectors</h1>
                    <p className="text-gray-600 mt-2 font-medium">Manage secure OAuth integrations and data sync.</p>
                </div>
                <Button onClick={() => { refetch(); }} variant="outline" className="font-bold border-gray-200">
                    <RefreshCw className={`w-4 h-4 mr-2 ${loading ? 'animate-spin' : ''}`} /> Refresh
                </Button>
            </div>

            {error && (
                <div className="bg-red-50 text-red-600 p-4 rounded-xl flex items-center gap-2">
                    <AlertCircle className="w-5 h-5" /> {error}
                </div>
            )}

            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full mt-13">
                <TabsList className="bg-white rounded-lg mb-6 inline-flex shadow-none" style={{ border: '2px solid black', padding: '2px' }}>
                    <TabsTrigger value="connectors" className="px-6 py-2 rounded-md transition-all shadow-none" style={{ backgroundColor: activeTab === 'connectors' ? '#28a745' : 'transparent', color: activeTab === 'connectors' ? 'white' : 'inherit', margin: '0' }}>Connectors</TabsTrigger>
                    <TabsTrigger value="sync" className="px-6 py-2 rounded-md transition-all shadow-none" style={{ backgroundColor: activeTab === 'sync' ? '#28a745' : 'transparent', color: activeTab === 'sync' ? 'white' : 'inherit', margin: '0' }}>Sync Jobs</TabsTrigger>
                </TabsList>

                <TabsContent value="connectors">
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        {connectors.length === 0 ? (
                            <div className="col-span-full py-20 text-center bg-gray-50 rounded-3xl border-2 border-dashed border-gray-200">
                                <h3 className="text-xl font-bold text-gray-900">No Connectors Found</h3>
                                <p className="text-gray-500 max-w-sm mx-auto mt-2">Install apps from the Marketplace to see them here.</p>
                            </div>
                        ) : (
                            connectors.map(connector => {
                                const provider = allProviders.find(p => p.id === connector.provider_slug) || {
                                    id: connector.provider_slug,
                                    name: connector.provider_name,
                                    description: 'Integration'
                                };
                                return (
                                    <ConnectorCard
                                        key={connector.id}
                                        provider={provider}
                                        connector={connector}
                                        config={null}
                                        onRefresh={refetch}
                                        onConfigure={() => setIsAddModalOpen(true)}
                                        onDeleteConfig={handleDeleteConfig}
                                        onViewSync={() => setActiveTab('sync')}
                                        onConnectSuccess={() => setActiveTab('sync')}
                                    />
                                );
                            })
                        )}
                    </div>
                </TabsContent>

                <TabsContent value="sync">
                    <div className="bg-white rounded-3xl border border-gray-200 p-8 shadow-sm">
                        <h3 className="text-xl font-bold text-gray-900 mb-6 border-b border-gray-100 pb-4">Sync Job History</h3>
                        <SyncJobMonitor />
                    </div>
                </TabsContent>
            </Tabs>

            <AddConnectorModal
                open={isAddModalOpen}
                onClose={() => setIsAddModalOpen(false)}
                onSuccess={refetch}
            />
        </div>
    );
};

export default Connectors;
