
import React, { useState, useEffect } from 'react';
import { adminService } from '../../../api/adminService';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { ShieldCheck, Key, Lock, RefreshCw, AlertTriangle, FileText, CheckCircle, Clock, Info } from 'lucide-react';
import { toast } from "sonner";

const Vault = () => {
    const [health, setHealth] = useState<any>(null);
    const [auditLogs, setAuditLogs] = useState<any[]>([]);
    const [loading, setLoading] = useState(false);
    const [rotating, setRotating] = useState(false);

    const fetchData = async () => {
        setLoading(true);
        try {
            const [healthRes, logsRes] = await Promise.all([
                adminService.integrations.getVaultHealth(),
                adminService.integrations.getAuditLogs(0)
            ]);
            const healthData = healthRes.data;
            console.log('Vault Health Data:', healthData); // Debug: Check API response
            setHealth(healthData);
            setAuditLogs(logsRes.data?.results || logsRes.data || []);

            // Auto-refresh if active keys is 0
            if (healthData.active_master_key_version === 0) {
                toast.info("No active master key detected. Initializing vault...");
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            }
        } catch (error) {
            console.error("Vault Fetch Error:", error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const handleRotateKeys = async () => {
        if (!confirm("Are you sure? This will rotate the master encryption keys. This is a critical security operation.")) return;

        try {
            setRotating(true);
            await adminService.integrations.rotateVaultKeys();
            toast.success("Master keys rotated successfully");
            fetchData();
        } catch (error) {
            toast.error("Failed to rotate keys. Check system logs.");
        } finally {
            setRotating(false);
        }
    };

    return (
        <div className="space-y-3">
            <div className="flex justify-between items-start">
                <div className="flex flex-col items-center text-center flex-1 mr-[1.5cm]">
                    <h1 className="text-3xl font-extrabold text-black tracking-tight flex items-center gap-3">
                        <Lock className="h-8 w-8 text-[#0056D2]" /> Secrets Vault
                    </h1>
                    <p className="text-gray-600 mt-2 font-medium">Secure storage management for API keys and OAuth credentials.</p>
                </div>
                <Button onClick={fetchData} variant="outline" className="font-bold border-gray-200">
                    <RefreshCw className={`w-4 h-4 mr-2 ${loading ? 'animate-spin' : ''}`} /> Refresh
                </Button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                <Card className="border-gray-200 shadow-sm bg-white">
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <ShieldCheck className="h-5 w-5 text-green-600" /> Vault Status
                        </CardTitle>
                        <CardDescription>Encryption engine health</CardDescription>
                    </CardHeader>
                    <CardContent>
                        {health ? (
                            <div className="space-y-4">
                                <div className="flex justify-between items-center border-b border-gray-100 pb-3">
                                    <span className="text-sm text-gray-500 font-medium">Status</span>
                                    <Badge className={`${health.status === 'healthy' || health.status === 'Healthy' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'} border-none`}>
                                        {health.status?.toUpperCase()}
                                    </Badge>
                                </div>
                                <div className="flex justify-between items-center border-b border-gray-100 pb-3">
                                    <span className="text-sm text-gray-500 font-medium">Active Keys</span>
                                    <span className="text-lg font-extrabold text-green-600">{health.active_master_key_version > 0 ? 1 : 0}</span>
                                </div>
                                <div className="flex justify-between items-center border-b border-gray-100 pb-3">
                                    <span className="text-sm text-gray-500 font-medium">Algorithm</span>
                                    <span className="font-mono text-sm font-bold text-gray-800">AES-256-GCM</span>
                                </div>
                            </div>
                        ) : (
                            <div className="text-center py-6 text-gray-400">Loading details...</div>
                        )}
                    </CardContent>
                </Card>

                <Card className="border-gray-200 shadow-sm bg-white border-l-4 border-l-[#0056D2]">
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <Key className="h-5 w-5 text-[#0056D2]" /> Master Key Rotation
                        </CardTitle>
                        <CardDescription>Security Lifecycle Management</CardDescription>
                    </CardHeader>
                    <CardContent>
                        <Alert className="bg-blue-50 border-blue-100 mb-6">
                            <Info className="h-4 w-4 text-blue-600" />
                            <AlertTitle className="text-blue-900 font-bold">Recommended Action</AlertTitle>
                            <AlertDescription className="text-blue-700 text-xs">
                                Rotate master keys every 90 days. Last rotation: <span className="font-mono font-bold">
                                    {(() => {
                                        const rotationLog = auditLogs.find(log => 
                                            log.action?.toLowerCase().includes('rotate') || 
                                            log.action?.toLowerCase().includes('key') && log.action?.toLowerCase().includes('created')
                                        );
                                        return rotationLog ? new Date(rotationLog.created_at || rotationLog.timestamp).toLocaleDateString() : 'Never';
                                    })()}
                                </span>
                            </AlertDescription>
                        </Alert>
                        <Button
                            onClick={handleRotateKeys}
                            disabled={rotating}
                            className="w-full text-white font-bold py-6 rounded-xl"
                            style={{ backgroundColor: '#0056D2' }}
                        >
                            {rotating ? 'Rotating Keys...' : 'Rotate Master Keys Now'}
                        </Button>
                    </CardContent>
                </Card>
            </div>

            <Card className="border-gray-200 shadow-sm bg-white">
                <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                        <FileText className="h-5 w-5 text-gray-500" /> Access & Audit Logs
                    </CardTitle>
                </CardHeader>
                <CardContent>
                    <Table>
                        <TableHeader className="bg-gray-50">
                            <TableRow>
                                <TableHead className="text-black">Timestamp</TableHead>
                                <TableHead className="text-black">Action</TableHead>
                                <TableHead className="text-black">Resource</TableHead>
                                <TableHead className="text-black">Status</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {auditLogs.length === 0 ? (
                                <TableRow>
                                    <TableCell colSpan={4} className="text-center py-8 text-black">No recent security events found.</TableCell>
                                </TableRow>
                            ) : (
                                auditLogs.slice(0, 20).map((log: any) => (
                                    <TableRow key={log.id} className="hover:bg-gray-50/50">
                                        <TableCell className="text-black">
                                            {new Date(log.created_at || log.timestamp).toLocaleString()}
                                        </TableCell>
                                        <TableCell className="text-black">{log.action}</TableCell>
                                        <TableCell className="text-black">{log.resource_type}::{log.resource_id}</TableCell>
                                        <TableCell>
                                            <Badge 
                                                variant="outline" 
                                                className="border-none hover:bg-transparent"
                                                style={{
                                                    backgroundColor: log.status === 'success' ? 'rgba(34, 197, 94, 0.2)' : 'rgba(239, 68, 68, 0.2)',
                                                    color: log.status === 'success' ? '#15803d' : '#991b1b'
                                                }}
                                            >
                                                {log.status}
                                            </Badge>
                                        </TableCell>
                                    </TableRow>
                                ))
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
        </div>
    );
};

export default Vault;
