
import React, { useState, useEffect } from 'react';
import { adminService } from '../../../api/adminService';
import { Plus, Trash2, Edit2, Zap, History, Check, Ban, CheckCircle, XCircle, Clock, BarChart2, AlertCircle } from 'lucide-react';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow
} from "@/components/ui/table";
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog";
// import { useToast } from "@/hooks/use-toast"; // Assuming this exists, otherwise use sonner
import { toast } from "sonner"; // Using sonner directly as per APIKeys
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";

interface WebhookEndpoint {
    id: number;
    name?: string;           // Falls back to description
    description?: string;    // Backend field
    url: string;
    event_types: string | string[];  // Can be string or array from backend
    status: 'active' | 'inactive';   // Backend uses status, not is_active
    is_active?: boolean;     // Computed from status
    secret?: string;
    created_at: string;
    created_by_name?: string;
    user_id?: string;
}

// Helper to normalize endpoint data from backend
const normalizeEndpoint = (item: any): WebhookEndpoint => ({
    ...item,
    name: item.name || item.description || 'Unnamed Webhook',
    is_active: item.status === 'active',
    // Always convert to string for consistency
    event_types: Array.isArray(item.event_types)
        ? item.event_types.join(',')
        : (item.event_types || '')
});

// Helper to get event types as array (handles both string and string[])
const getEventTypesArray = (eventTypes: string | string[] | undefined): string[] => {
    if (!eventTypes) return [];
    if (Array.isArray(eventTypes)) return eventTypes;
    return eventTypes.split(',').filter(e => e.trim());
};

interface WebhookDelivery {
    id: number;
    webhook_event_type?: string;  // Legacy name
    event_type?: string;          // Backend field
    webhook_status?: 'pending' | 'delivered' | 'failed' | 'success';  // Legacy
    status?: 'pending' | 'success' | 'failed';  // Backend field
    webhook_response_code?: number;  // Legacy
    response_code?: number;          // Backend field
    webhook_retry_count?: number;    // Legacy
    attempt_count?: number;          // Backend field
    webhook_delivered_at?: string | null;  // Legacy
    created_at: string;
    message?: string;
    response_body?: string;  // Backend field
}

const EVENT_TYPES = [
    { id: 'buyer.joined', label: 'Buyer Joined' },
    { id: 'buyer.purchased_property', label: 'Buyer Purchased Property' },
    { id: 'seller.property_listed', label: 'Seller Property Listed' },
    { id: 'seller.property_sold', label: 'Seller Property Sold' },
    { id: 'property.updated', label: 'Property Updated' },
    { id: 'property.deleted', label: 'Property Deleted' },
];

const AdminWebhooks = () => {
    const [endpoints, setEndpoints] = useState<WebhookEndpoint[]>([]);
    const [loading, setLoading] = useState(true);
    const [summary, setSummary] = useState<any>(null);
    const [searchQuery, setSearchQuery] = useState('');
    // const { toast } = useToast();

    // Create/Edit Modal State
    const [modalOpen, setModalOpen] = useState(false);
    const [isEditing, setIsEditing] = useState(false);
    const [formData, setFormData] = useState({
        id: 0,
        name: '',
        url: '',
        event_types: [] as string[],
        is_active: true
    });

    // Deliveries Modal State
    const [deliveriesOpen, setDeliveriesOpen] = useState(false);
    const [currentEndpointId, setCurrentEndpointId] = useState<number | null>(null);
    const [deliveries, setDeliveries] = useState<WebhookDelivery[]>([]);

    // Monitor State
    const [monitorOpen, setMonitorOpen] = useState(false);
    const [monitorStats, setMonitorStats] = useState<any>(null);

    const openMonitor = async () => {
        try {
            const res = await adminService.webhooks.monitorStats();
            setMonitorStats(res.data);
            setMonitorOpen(true);
        } catch (error) {
            // toast({ title: "Error", description: "Failed to load webhook stats", variant: "destructive" });
            toast.error("Failed to load webhook stats");
        }
    };

    useEffect(() => {
        fetchEndpoints();
        fetchSummary();
    }, []);

    const fetchEndpoints = async () => {
        try {
            setLoading(true);
            const res = await adminService.webhooks.getAll();
            const rawEndpoints = res.data.results || [];
            // Normalize each endpoint to handle backend field differences
            setEndpoints(rawEndpoints.map(normalizeEndpoint));
        } catch (error) {
            // toast({ title: "Error", description: "Failed to load webhooks", variant: "destructive" });
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    const fetchSummary = async () => {
        try {
            const res = await adminService.webhooks.getSummary();
            setSummary(res.data);
        } catch (error) {
            console.error("Failed to fetch summary", error);
        }
    };

    const handleSave = async () => {
        if (!formData.name || !formData.url) {
            // toast({ title: "Validation Error", description: "Name and URL are required", variant: "destructive" });
            toast.error("Name and URL are required");
            return;
        }

        // Ensure URL starts with http
        if (!formData.url.startsWith('http')) {
            // toast({ title: "Validation Error", description: "URL must start with http:// or https://", variant: "destructive" });
            toast.error("URL must start with http:// or https://");
            return;
        }

        const payload = {
            description: formData.name,  // Backend uses 'description' field
            url: formData.url,
            event_types: formData.event_types.join(','), // Backend expects comma-separated string
            status: formData.is_active ? 'active' : 'inactive'  // Backend uses 'status' field
        };

        try {
            if (isEditing) {
                await adminService.webhooks.update(formData.id, payload);
                // toast({ title: "Success", description: "Webhook updated" });
                toast.success("Webhook updated");
            } else {
                await adminService.webhooks.create(payload);
                // toast({ title: "Success", description: "Webhook created" });
                toast.success("Webhook created");
            }
            setModalOpen(false);
            fetchEndpoints();
            fetchSummary();
        } catch (error) {
            // toast({ title: "Error", description: `Failed to ${isEditing ? 'update' : 'create'} webhook`, variant: "destructive" });
            toast.error(`Failed to ${isEditing ? 'update' : 'create'} webhook`);
        }
    };

    const handleDelete = async (id: number) => {
        if (!confirm("Delete this webhook endpoint?")) return;
        try {
            await adminService.webhooks.delete(id);
            // toast({ title: "Success", description: "Webhook deleted" });
            toast.success("Webhook deleted");
            fetchEndpoints();
            fetchSummary();
        } catch (error) {
            // toast({ title: "Error", description: "Failed to delete webhook", variant: "destructive" });
            toast.error("Failed to delete webhook");
        }
    };

    const handleTest = async (id: number) => {
        try {
            const res = await adminService.webhooks.test(id);
            // toast({ title: "Test Sent", description: "Check delivery logs for result" });
            toast.success("Test Sent. Check delivery logs for result.");
            fetchSummary(); // Update delivery counts
        } catch (error) {
            // toast({ title: "Error", description: "Failed to send test webhook", variant: "destructive" });
            toast.error("Failed to send test webhook");
        }
    };

    const viewDeliveries = async (id: number) => {
        setCurrentEndpointId(id);
        setDeliveriesOpen(true);
        setDeliveries([]); // Clear previous
        try {
            const res = await adminService.webhooks.getDeliveries(id, { limit: 20 });
            setDeliveries(res.data.deliveries || []);
        } catch (error) {
            // toast({ title: "Error", description: "Failed to load deliveries", variant: "destructive" });
            toast.error("Failed to load deliveries");
        }
    };

    const toggleStatus = async (item: WebhookEndpoint) => {
        try {
            // Backend uses 'status' field with 'active'/'inactive' values
            const newStatus = item.status === 'active' ? 'inactive' : 'active';
            await adminService.webhooks.update(item.id, { status: newStatus });
            toast.success(`Webhook ${item.status === 'active' ? 'deactivated' : 'activated'}`);
            fetchEndpoints();
        } catch (error) {
            console.error(error);
            fetchEndpoints();
        }
    };

    const openCreateModal = () => {
        setIsEditing(false);
        setFormData({ id: 0, name: '', url: '', event_types: [], is_active: true });
        setModalOpen(true);
    };

    const openEditModal = (item: WebhookEndpoint) => {
        setIsEditing(true);
        // Use helper to get event types as array
        const eventTypesArr = getEventTypesArray(item.event_types);
        setFormData({
            id: item.id,
            name: item.name || item.description || '',
            url: item.url,
            event_types: eventTypesArr,
            is_active: item.status === 'active'
        });
        setModalOpen(true);
    };

    const toggleEventType = (id: string) => {
        const current = formData.event_types;
        if (current.includes(id)) {
            setFormData({ ...formData, event_types: current.filter(e => e !== id) });
        } else {
            setFormData({ ...formData, event_types: [...current, id] });
        }
    };

    return (
        <div className="max-w-7xl mx-auto pb-12 px-4 sm:px-6 lg:px-8">
            {/* Header */}
            <div className="pt-8 pb-3 space-y-6">
                <div className="flex flex-col justify-center items-center text-center space-y-2">
                    <div>
                        <h1 className="text-3xl font-extrabold text-black tracking-tight">Webhooks</h1>
                        <p className="text-gray-600 mt-2 font-medium">Manage outbound event notifications to external systems</p>
                    </div>
                </div>

                {/* Summary Cards */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
                <Card>
                    <CardHeader className="pb-2">
                        <CardTitle className="text-sm font-medium text-gray-500">Total Webhooks</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{summary?.endpoints?.total || 0}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="pb-2">
                        <CardTitle className="text-sm font-medium text-green-500">Successful Deliveries</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-green-600">{summary?.deliveries?.successful || 0}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="pb-2">
                        <CardTitle className="text-sm font-medium text-red-500">Failed Deliveries</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-red-600">{summary?.deliveries?.failed || 0}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="pb-2">
                        <CardTitle className="text-sm font-medium text-blue-500">Last 24 Hours</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold text-blue-600">{summary?.deliveries?.last_24_hours || 0}</div>
                    </CardContent>
                </Card>
            </div>
            </div>

            {/* Action Bar */}
            <div className="py-3">
            <div className="flex justify-between items-center bg-white p-4 rounded-lg border border-gray-100 gap-4">
                <Input
                    placeholder="Search webhooks..."
                    className="max-w-sm"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                />
                <div className="flex gap-2">
                    <Button onClick={openMonitor} variant="outline" className="text-blue-600 border-blue-200 hover:bg-blue-50">
                        <BarChart2 className="mr-2" /> Delivery Stats
                    </Button>
                    <Button onClick={openCreateModal} className="bg-[#0056D2] hover:bg-blue-700 text-white">
                        <Plus className="mr-2" /> Create Webhook
                    </Button>
                </div>
            </div>
            </div>

            {/* Webhooks Table */}
            <div className="py-3">
            <Card className="shadow-none border-gray-200">
                <CardContent className="p-0">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Name</TableHead>
                                <TableHead>Target URL</TableHead>
                                <TableHead>Events</TableHead>
                                <TableHead>Status</TableHead>
                                <TableHead className="text-right">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {endpoints.filter(item => {
                                const query = searchQuery.toLowerCase();
                                return (
                                    item.name?.toLowerCase().includes(query) ||
                                    item.url?.toLowerCase().includes(query) ||
                                    item.id.toString().includes(query)
                                );
                            }).map((item) => (
                                <TableRow key={item.id}>
                                    <TableCell className="font-medium">{item.name}</TableCell>
                                    <TableCell className="max-w-[200px] truncate" title={item.url}>
                                        {item.url}
                                    </TableCell>
                                    <TableCell>
                                        {(() => {
                                            const eventTypesArr = getEventTypesArray(item.event_types);
                                            return eventTypesArr.length > 0 ? (
                                                <div className="flex flex-wrap gap-1">
                                                    {eventTypesArr.slice(0, 2).map(e => (
                                                        <Badge key={e} variant="outline" className="text-xs">{e}</Badge>
                                                    ))}
                                                    {eventTypesArr.length > 2 && (
                                                        <Badge variant="outline" className="text-xs">+{eventTypesArr.length - 2}</Badge>
                                                    )}
                                                </div>
                                            ) : (
                                                <Badge className="bg-blue-100 text-blue-800 border-none">All Events</Badge>
                                            );
                                        })()}
                                    </TableCell>
                                    <TableCell>
                                        {item.is_active ? (
                                            <Badge className="bg-green-100 text-green-800 border-none justify-center">Active</Badge>
                                        ) : (
                                            <Badge variant="destructive" className="bg-red-100 text-red-800 border-none justify-center">Inactive</Badge>
                                        )}
                                    </TableCell>
                                    <TableCell className="text-right">
                                        <div className="flex justify-end gap-2">
                                            <Button variant="ghost" size="icon" title="Test Payload" onClick={() => handleTest(item.id)}>
                                                <Zap className="h-4 w-4 text-yellow-500" />
                                            </Button>
                                            <Button variant="ghost" size="icon" title="View Deliveries" onClick={() => viewDeliveries(item.id)}>
                                                <History className="h-4 w-4 text-gray-500" />
                                            </Button>
                                            <Button variant="ghost" size="icon" title="Edit" onClick={() => openEditModal(item)}>
                                                <Edit2 className="h-4 w-4 text-blue-500" />
                                            </Button>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                title={item.is_active ? "Deactivate" : "Activate"}
                                                onClick={() => toggleStatus(item)}
                                            >
                                                {item.is_active ? <Ban className="h-4 w-4 text-red-500" /> : <Check className="h-4 w-4 text-green-500" />}
                                            </Button>
                                            <Button variant="ghost" size="icon" title="Delete" onClick={() => handleDelete(item.id)}>
                                                <Trash2 className="h-4 w-4 text-red-500" />
                                            </Button>
                                        </div>
                                    </TableCell>
                                </TableRow>
                            ))}
                            {endpoints.length === 0 && (
                                <TableRow>
                                    <TableCell colSpan={5} className="text-center py-8 text-gray-500">No webhooks configured.</TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </CardContent>
            </Card>
            </div>

            {/* Create/Edit Modal */}
            <Dialog open={modalOpen} onOpenChange={setModalOpen}>
                <DialogContent className="max-w-[520px]">
                    <DialogHeader>
                        <DialogTitle>{isEditing ? 'Edit Webhook' : 'Create Webhook'}</DialogTitle>
                        <DialogDescription>Configure where events should be sent.</DialogDescription>
                    </DialogHeader>
                    <div className="space-y-4 py-4">
                        <div className="space-y-2">
                            <Label htmlFor="name">Name</Label>
                            <Input
                                id="name"
                                value={formData.name}
                                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                placeholder="e.g. Analytics System"
                            />
                        </div>
                        <div className="space-y-2">
                            <Label htmlFor="url">Target URL</Label>
                            <Input
                                id="url"
                                value={formData.url}
                                onChange={(e) => setFormData({ ...formData, url: e.target.value })}
                                placeholder="https://api.example.com/hooks"
                            />
                        </div>
                        <div className="space-y-2">
                            <Label>Events</Label>
                            <div className="border rounded-md p-3 space-y-2 max-h-[200px] overflow-y-auto">
                                <div className="text-xs text-gray-500 mb-2">Leave all unchecked to subscribe to ALL events.</div>
                                {EVENT_TYPES.map((type) => (
                                    <div key={type.id} className="flex items-center space-x-2">
                                        <Checkbox
                                            id={type.id}
                                            checked={formData.event_types.includes(type.id)}
                                            onCheckedChange={() => toggleEventType(type.id)}
                                        />
                                        <label htmlFor={type.id} className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70">
                                            {type.label} <span className="text-xs text-gray-400">({type.id})</span>
                                        </label>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                    <DialogFooter>
                        <Button onClick={handleSave} className="bg-[#0056D2] hover:bg-blue-700 text-white">{isEditing ? 'Update' : 'Create'}</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            {/* Deliveries Modal */}
            <Dialog open={deliveriesOpen} onOpenChange={setDeliveriesOpen}>
                <DialogContent className="w-[650px] max-w-[90vw]">
                    <DialogHeader>
                        <DialogTitle>Recent Deliveries</DialogTitle>
                    </DialogHeader>
                    <div className="max-h-[400px] overflow-y-auto">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Status</TableHead>
                                    <TableHead>Event</TableHead>
                                    <TableHead>Time</TableHead>
                                    <TableHead>Message</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {deliveries.map((log) => {
                                    // Support both legacy and backend field names
                                    const logStatus = log.status || log.webhook_status;
                                    const logEventType = log.event_type || log.webhook_event_type;
                                    const logMessage = log.response_body || log.message || '';

                                    return (
                                        <TableRow key={log.id} className="hover:bg-gray-50">
                                            <TableCell>
                                                <div className="flex items-center gap-2">
                                                    {(logStatus === 'success' || logStatus === 'delivered') ? (
                                                        <span className="flex items-center text-green-600 font-bold text-[10px] tracking-wider uppercase">
                                                            <CheckCircle className="mr-1 h-3 w-3" /> SUCCESS
                                                        </span>
                                                    ) : logStatus === 'failed' ? (
                                                        <span className="flex items-center text-red-600 font-bold text-[10px] tracking-wider uppercase">
                                                            <XCircle className="mr-1 h-3 w-3" /> FAILED
                                                        </span>
                                                    ) : (
                                                        <span className="flex items-center text-gray-500 font-bold text-[10px] tracking-wider uppercase">
                                                            <Clock className="mr-1 h-3 w-3" /> PENDING
                                                        </span>
                                                    )}
                                                </div>
                                            </TableCell>
                                            <TableCell className="font-mono text-[10px] text-[#0056D2] bg-blue-50 px-2 py-0.5 rounded border border-blue-100">
                                                {logEventType}
                                            </TableCell>
                                            <TableCell className="text-[10px] text-gray-500 whitespace-nowrap">
                                                {new Date(log.created_at).toLocaleString()}
                                            </TableCell>
                                            <TableCell className="text-[11px] text-gray-600 max-w-[300px] truncate" title={logMessage}>
                                                {logMessage}
                                            </TableCell>
                                        </TableRow>
                                    );
                                })}
                                {deliveries.length === 0 && (
                                    <TableRow>
                                        <TableCell colSpan={5} className="text-center py-8 text-gray-500">No delivery logs found.</TableCell>
                                    </TableRow>
                                )}
                            </TableBody>
                        </Table>
                    </div>
                </DialogContent>
            </Dialog>

            {/* Monitor Stats Dialog */}
            <Dialog open={monitorOpen} onOpenChange={setMonitorOpen}>
                <DialogContent className="w-[650px] max-w-[90vw]">
                    <DialogHeader>
                        <DialogTitle>Delivery Stats</DialogTitle>
                    </DialogHeader>
                    <div className="max-h-[400px] overflow-y-auto space-y-4 py-4">
                        <div className="grid grid-cols-3 gap-4">
                            <Card className="border-blue-200 bg-blue-50">
                                <CardContent className="pt-4 text-center">
                                    <div className="text-2xl font-bold text-blue-600">{monitorStats?.global_stats?.total || 0}</div>
                                    <div className="text-xs font-medium text-blue-700 mt-1">Total Signals</div>
                                </CardContent>
                            </Card>
                            <Card className="border-green-200 bg-green-50">
                                <CardContent className="pt-4 text-center">
                                    <div className="text-2xl font-bold text-green-600">{monitorStats?.global_stats?.success_ratio || 0}%</div>
                                    <div className="text-xs font-medium text-green-700 mt-1">Success Rate</div>
                                </CardContent>
                            </Card>
                            <Card className="border-red-200 bg-red-50">
                                <CardContent className="pt-4 text-center">
                                    <div className="text-2xl font-bold text-red-600">{monitorStats?.global_stats?.failed || 0}</div>
                                    <div className="text-xs font-medium text-red-700 mt-1">Total Failures</div>
                                </CardContent>
                            </Card>
                        </div>
                        <div>
                            <h4 className="text-sm font-semibold mb-2 flex items-center gap-2">
                                <AlertCircle className="text-red-500 h-4 w-4" /> Recent Failures
                            </h4>
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead className="w-[120px]">Time</TableHead>
                                        <TableHead>Target</TableHead>
                                        <TableHead>Error</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {monitorStats?.recent_failures?.map((fail: any) => (
                                        <TableRow key={fail.id}>
                                            <TableCell className="text-xs text-gray-600">
                                                {new Date(fail.created_at).toLocaleString()}
                                            </TableCell>
                                            <TableCell className="text-xs font-medium truncate max-w-[150px]" title={fail.webhook_endpoint__url}>
                                                {fail.webhook_endpoint__url}
                                            </TableCell>
                                            <TableCell className="text-xs text-red-600 truncate" title={fail.message}>
                                                {fail.message}
                                            </TableCell>
                                        </TableRow>
                                    ))}
                                    {(!monitorStats?.recent_failures || monitorStats.recent_failures.length === 0) && (
                                        <TableRow>
                                            <TableCell colSpan={3} className="text-center py-6 text-gray-500 text-sm">
                                                All systems operational. No recent failures.
                                            </TableCell>
                                        </TableRow>
                                    )}
                                </TableBody>
                            </Table>
                        </div>
                    </div>
                </DialogContent>
            </Dialog>


        </div>
    );
};

export default AdminWebhooks;
