import React, { useState } from 'react';
import { Search, FileText, Download, MapPin, Shield, Receipt, Home, Users, Eye, CheckCircle, Building, Globe, Gavel, CreditCard, FileCheck, AlertTriangle, TrendingUp, Clock, Star, Award, Loader2 } from 'lucide-react';
import { toast } from 'sonner';
import jsPDF from 'jspdf';

// Simple PDF utilities (inline to avoid import issues)
const generatePropertyDossierPDF = async (property: any, selectedSections: any[]): Promise<string> => {
  await new Promise(resolve => setTimeout(resolve, 2000));
  return `data:application/pdf;base64,${btoa(JSON.stringify({ property, selectedSections }))}`;
};

const estimatePDFSize = (selectedSections: any[]): string => {
  const baseSize = 0.5;
  const sectionSize = selectedSections.length * 0.3;
  const totalSize = baseSize + sectionSize;
  return `${totalSize.toFixed(1)} MB`;
};

// Mock property data for fallback
const mockProperties = [
  {
    id: 'PROP-2024-001',
    surveyNo: 'SY/123/2024',
    address: '123 Marine Drive, Bandra West, Mumbai - 400050',
    type: 'Residential',
    area: '1200 sq ft',
    owner: 'Rajesh Kumar'
  },
  {
    id: 'PROP-2024-002', 
    surveyNo: 'SY/456/2024',
    address: '456 MG Road, Koramangala, Bangalore - 560034',
    type: 'Commercial',
    area: '2500 sq ft',
    owner: 'Anita Patel'
  }
];

const dossierSections = [
  {
    id: 'identification',
    title: 'Property Identification',
    description: 'Property ID/Survey Number, Address, Type (Residential/Commercial/Land), Area/Size',
    icon: Home,
    included: true,
    details: ['Unique Property ID', 'Survey Number', 'Complete Address', 'Property Classification', 'Total Area & Built-up Area']
  },
  {
    id: 'ownership',
    title: 'Ownership Details',
    description: 'Current owner(s), Ownership type, Date of ownership, Ownership history (optional)',
    icon: Users,
    included: true,
    details: ['Current Owner Information', 'Joint/Individual Ownership', 'Ownership Transfer History', 'Power of Attorney Details']
  },
  {
    id: 'legal',
    title: 'Legal Information',
    description: 'Sale deed details, Registration references, Encumbrance status, Litigation status (if any)',
    icon: Shield,
    included: false,
    details: ['Sale Deed Verification', 'Registration Certificate', 'Encumbrance Certificate', 'Litigation Check', 'Legal Clearance Status']
  },
  {
    id: 'tax',
    title: 'Tax & Financial Details',
    description: 'Property tax status, Last payment date, Outstanding dues',
    icon: Receipt,
    included: false,
    details: ['Property Tax Records', 'Payment History', 'Outstanding Dues', 'Tax Assessment Value', 'Utility Bill Status']
  },
  {
    id: 'location',
    title: 'Location & Infrastructure',
    description: 'Geographic location, Boundaries, Zoning classification, Nearby infrastructure',
    icon: MapPin,
    included: false,
    details: ['GPS Coordinates', 'Boundary Survey', 'Zoning Information', 'Infrastructure Access', 'Connectivity Details']
  },
  {
    id: 'documents',
    title: 'Documents & Attachments',
    description: 'Deed copies, Certificates, Approved plans, Official notices',
    icon: FileText,
    included: false,
    details: ['Sale Deed Copy', 'Title Certificate', 'Approved Building Plans', 'NOC Certificates', 'Official Notices']
  }
];

const userTypes = [
  {
    type: 'Property Buyer',
    icon: Home,
    need: 'Due diligence',
    description: 'Complete verification before purchase decision',
    benefits: ['Avoid fraud', 'Verify ownership', 'Check legal status', 'Assess true value']
  },
  {
    type: 'NRI',
    icon: Globe,
    need: 'Remote verification',
    description: 'Verify property details from abroad',
    benefits: ['Remote due diligence', 'Trusted verification', 'Complete documentation', 'Investment security']
  },
  {
    type: 'Bank / Lender',
    icon: Building,
    need: 'Loan approval',
    description: 'Comprehensive property assessment for lending',
    benefits: ['Risk assessment', 'Collateral verification', 'Legal clearance', 'Market valuation']
  },
  {
    type: 'Legal Advisor',
    icon: Gavel,
    need: 'Ownership checks',
    description: 'Legal verification and compliance check',
    benefits: ['Title verification', 'Legal compliance', 'Dispute identification', 'Documentation review']
  },
  {
    type: 'Govt Officer',
    icon: Award,
    need: 'Official verification',
    description: 'Government verification and compliance',
    benefits: ['Regulatory compliance', 'Official records', 'Tax verification', 'Zoning compliance']
  }
];

export function CustomPropertyDossier() {
  const [step, setStep] = useState(1);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedProperty, setSelectedProperty] = useState(null);
  const [sections, setSections] = useState(dossierSections);
  const [isGenerating, setIsGenerating] = useState(false);
  const [isSearching, setIsSearching] = useState(false);
  const [searchResults, setSearchResults] = useState([]);
  const [searchError, setSearchError] = useState('');
  const [activeTab, setActiveTab] = useState('overview');

  const tabs = [
    { id: 'overview', label: 'Services Overview', icon: FileCheck },
    { id: 'generate', label: 'Generate Dossier', icon: FileText },
    { id: 'faq', label: 'FAQ', icon: AlertTriangle }
  ];

  const searchPropertyById = async (propertyId) => {
    if (!propertyId || !propertyId.trim()) {
      setSearchResults([]);
      setSearchError('');
      return;
    }

    setIsSearching(true);
    setSearchError('');
    
    try {
      const apiBase = import.meta.env.VITE_API_BASE || 'http://localhost:8000/api';
      const token = localStorage.getItem('token') || localStorage.getItem('access_token') || '';
      
      const response = await fetch(`${apiBase}/properties/${encodeURIComponent(propertyId)}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          ...(token && { 'Authorization': `Bearer ${token}` })
        }
      });

      if (response.ok) {
        const property = await response.json();
        setSearchResults([{
          id: property.id || propertyId,
          surveyNo: property.survey_number || 'N/A',
          address: `${property.address || ''}, ${property.city || ''}, ${property.state || ''} - ${property.pincode || ''}`.replace(/^[,\s-]+|[,\s-]+$/g, ''),
          type: property.property_type || 'Residential',
          area: `${property.built_up_area || property.carpet_area || 'N/A'} sq ft`,
          owner: (property.listed_by?.first_name || '') + ' ' + (property.listed_by?.last_name || '') || 'N/A'
        }]);
      } else if (response.status === 404) {
        setSearchResults([]);
        setSearchError('Property not found. Please check the Property ID and try again.');
      } else {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
    } catch (error) {
      console.error('Error searching property:', error);
      setSearchResults([]);
      setSearchError('Error searching property. Please try again.');
      toast.error('Failed to search property');
    } finally {
      setIsSearching(false);
    }
  };

  const handleSearchChange = (value) => {
    setSearchTerm(value);
    if (window.searchTimeout) {
      clearTimeout(window.searchTimeout);
    }
    window.searchTimeout = setTimeout(() => {
      searchPropertyById(value);
    }, 500);
  };

  const handlePropertySelect = (property) => {
    setSelectedProperty(property);
    setStep(2);
  };

  const handleSectionToggle = (sectionId) => {
    if (sectionId === 'identification' || sectionId === 'ownership') {
      return;
    }
    setSections(sections.map(section =>
      section.id === sectionId ? { ...section, included: !section.included } : section
    ));
  };

  const handleGenerateDossier = async () => {
    setIsGenerating(true);
    try {
      const pdfUrl = await generatePropertyDossierPDF(selectedProperty, selectedSections);
      setIsGenerating(false);
      setStep(3);
    } catch (error) {
      console.error('Error generating dossier:', error);
      setIsGenerating(false);
      alert('Error generating dossier. Please try again.');
    }
  };

  const handleDownloadPDF = async () => {
    try {
      const pdf = new jsPDF();
      const pageWidth = pdf.internal.pageSize.width;
      const pageHeight = pdf.internal.pageSize.height;

      // Function to add NAL watermark on every page
      const addWatermark = () => {
        pdf.saveGraphicsState();
        pdf.setGState(new pdf.GState({ opacity: 0.08 }));
        pdf.setTextColor(128, 128, 128);
        pdf.setFontSize(60);
        pdf.setFont("helvetica", "bold");
        pdf.text("NAL INDIA", pageWidth / 2, pageHeight / 2, {
          align: "center",
          angle: 45,
        });
        pdf.restoreGraphicsState();
        pdf.setTextColor(0, 0, 0);
      };

      // Load NAL logo
      const loadLogo = () => {
        return new Promise((resolve) => {
          const img = new Image();
          img.onload = () => resolve(img);
          img.onerror = () => resolve(null);
          img.src = "/favicon.png";
        });
      };

      const logoImg = await loadLogo();

      // Function to add header with logo and title
      const addHeader = () => {
        pdf.setFillColor(65, 105, 225);
        pdf.rect(0, 0, pageWidth, 50, "F");

        if (logoImg) {
          pdf.addImage(logoImg, "PNG", 15, 10, 30, 30);
        }

        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(24);
        pdf.setFont("helvetica", "bold");
        pdf.text("PROPERTY DOSSIER REPORT", 55, 30);
        pdf.setTextColor(0, 0, 0);
      };

      // Function to add footer
      const addFooter = () => {
        pdf.setFillColor(65, 105, 225);
        pdf.rect(0, pageHeight - 20, pageWidth, 20, "F");
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(10);
        pdf.setFont("helvetica", "bold");
        pdf.text("NAL INDIA - New Age Land", 15, pageHeight - 10);
        pdf.setFont("helvetica", "normal");
        pdf.text("www.nglindia.com | info@nglindia.com | Professional Real Estate Services", 15, pageHeight - 4);
        pdf.setTextColor(0, 0, 0);
      };

      // Page 1: Property Details
      addHeader();
      addWatermark();
      
      let yPos = 70;
      
      // Property Information
      pdf.setFillColor(240, 240, 240);
      pdf.rect(15, yPos - 5, pageWidth - 30, 12, "F");
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(16);
      pdf.setFont("helvetica", "bold");
      pdf.text("PROPERTY INFORMATION", 20, yPos + 3);
      yPos += 20;
      
      const propertyDetails = [
        ["Property ID", selectedProperty?.id || 'N/A'],
        ["Survey Number", selectedProperty?.surveyNo || 'N/A'],
        ["Property Address", selectedProperty?.address || 'N/A'],
        ["Property Type", selectedProperty?.type || 'N/A'],
        ["Total Area", selectedProperty?.area || 'N/A'],
        ["Current Owner", selectedProperty?.owner || 'N/A'],
        ["Report Generated On", new Date().toLocaleString()],
        ["Dossier Status", "Complete & Verified"],
        ["Total Sections Included", `${selectedSections.length} sections`]
      ];
      
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(11);
      propertyDetails.forEach(([label, value]) => {
        pdf.setFont("helvetica", "bold");
        pdf.text(`${label}:`, 20, yPos);
        pdf.setFont("helvetica", "normal");
        const valueLines = pdf.splitTextToSize(value, pageWidth - 90);
        pdf.text(valueLines, 90, yPos);
        yPos += Math.max(8, valueLines.length * 8);
      });
      
      yPos += 15;
      
      // What is Property Dossier section
      if (yPos > pageHeight - 100) {
        addFooter();
        pdf.addPage();
        addHeader();
        addWatermark();
        yPos = 70;
      }
      
      pdf.setFillColor(248, 249, 250);
      pdf.rect(15, yPos - 5, pageWidth - 30, 12, "F");
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(16);
      pdf.setFont("helvetica", "bold");
      pdf.text("WHAT IS PROPERTY DOSSIER?", 20, yPos + 3);
      yPos += 20;
      
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(10);
      pdf.setFont("helvetica", "normal");
      const dossierInfo = "A Property Dossier is a comprehensive document that contains all essential information about a property. It serves as a complete verification report for buyers, banks, legal advisors, and government officials. This dossier includes property identification, ownership details, legal status, tax information, location data, and supporting documents - all verified from official sources.";
      const dossierLines = pdf.splitTextToSize(dossierInfo, pageWidth - 40);
      pdf.text(dossierLines, 20, yPos);
      yPos += dossierLines.length * 5 + 15;
      
      // Benefits section
      if (yPos > pageHeight - 80) {
        addFooter();
        pdf.addPage();
        addHeader();
        addWatermark();
        yPos = 70;
      }
      
      pdf.setFillColor(240, 248, 255);
      pdf.rect(15, yPos - 5, pageWidth - 30, 12, "F");
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(14);
      pdf.setFont("helvetica", "bold");
      pdf.text("KEY BENEFITS", 20, yPos + 3);
      yPos += 18;
      
      const benefits = [
        "Complete due diligence for property buyers",
        "Bank-ready documentation for loan approvals",
        "Legal verification and compliance check",
        "Risk assessment and fraud prevention",
        "Official record verification from government sources",
        "Time-saving comprehensive property analysis"
      ];
      
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(10);
      pdf.setFont("helvetica", "normal");
      benefits.forEach((benefit) => {
        if (yPos > pageHeight - 40) {
          addFooter();
          pdf.addPage();
          addHeader();
          addWatermark();
          yPos = 70;
        }
        pdf.setTextColor(0, 0, 0);
        pdf.text(`• ${benefit}`, 25, yPos);
        yPos += 7;
      });
      
      yPos += 15;
      
      // Sections Details
      if (yPos > pageHeight - 80) {
        addFooter();
        pdf.addPage();
        addHeader();
        addWatermark();
        yPos = 70;
      }
      
      pdf.setFillColor(240, 240, 240);
      pdf.rect(15, yPos - 5, pageWidth - 30, 12, "F");
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(16);
      pdf.setFont("helvetica", "bold");
      pdf.text("INCLUDED DOSSIER SECTIONS", 20, yPos + 3);
      yPos += 20;
      
      selectedSections.forEach((section, index) => {
        if (yPos > pageHeight - 100) {
          addFooter();
          pdf.addPage();
          addHeader();
          addWatermark();
          yPos = 70;
        }
        
        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(14);
        pdf.setFont("helvetica", "bold");
        pdf.text(`${index + 1}. ${section.title}`, 20, yPos);
        yPos += 10;
        
        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(10);
        pdf.setFont("helvetica", "normal");
        const descLines = pdf.splitTextToSize(section.description, pageWidth - 40);
        pdf.text(descLines, 25, yPos);
        yPos += descLines.length * 5 + 5;
        
        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(9);
        pdf.setFont("helvetica", "bold");
        pdf.text("Verification Components:", 25, yPos);
        yPos += 6;
        
        pdf.setTextColor(0, 0, 0);
        pdf.setFont("helvetica", "normal");
        section.details.forEach((detail) => {
          if (yPos > pageHeight - 40) {
            addFooter();
            pdf.addPage();
            addHeader();
            addWatermark();
            yPos = 70;
          }
          pdf.setTextColor(0, 0, 0);
          pdf.text(`• ${detail}`, 30, yPos);
          yPos += 5;
        });
        yPos += 12;
      });
      
      // Usage Information
      if (yPos > pageHeight - 80) {
        addFooter();
        pdf.addPage();
        addHeader();
        addWatermark();
        yPos = 70;
      }
      
      pdf.setFillColor(255, 248, 220);
      pdf.rect(15, yPos - 5, pageWidth - 30, 12, "F");
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(14);
      pdf.setFont("helvetica", "bold");
      pdf.text("HOW TO USE THIS DOSSIER", 20, yPos + 3);
      yPos += 18;
      
      const usageInfo = [
        "Present to banks for loan application processing",
        "Share with legal advisors for property verification",
        "Use for due diligence before property purchase",
        "Submit to government offices for official procedures",
        "Keep as permanent record of property documentation"
      ];
      
      pdf.setTextColor(0, 0, 0);
      pdf.setFontSize(10);
      pdf.setFont("helvetica", "normal");
      usageInfo.forEach((info) => {
        if (yPos > pageHeight - 40) {
          addFooter();
          pdf.addPage();
          addHeader();
          addWatermark();
          yPos = 70;
        }
        pdf.setTextColor(0, 0, 0);
        pdf.text(`• ${info}`, 25, yPos);
        yPos += 7;
      });
      
      addFooter();
      
      // Add footer to all pages
      const totalPages = pdf.internal.getNumberOfPages();
      for (let i = 1; i <= totalPages; i++) {
        pdf.setPage(i);
        addFooter();
      }
      
      // Save PDF
      const filename = `Property_Dossier_${selectedProperty?.id || 'Report'}_${new Date().toISOString().split('T')[0]}.pdf`;
      pdf.save(filename);
      
      toast.success(`Property Dossier downloaded successfully!`);
      
    } catch (error) {
      console.error('Error generating PDF:', error);
      toast.error('Failed to generate PDF. Please try again.');
    }
  };

  const selectedSections = sections.filter(s => s.included);

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="text-center mb-12">
          <div className="w-20 h-20 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-6">
            <FileText className="w-10 h-10 text-blue-600" />
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-4">Custom Property Dossier</h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto mb-8">
            Generate comprehensive, customized property reports with verified information. 
            Perfect for due diligence, loan approvals, and investment decisions.
          </p>
        </div>

        {/* Tab Navigation */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 mb-8">
          <div className="flex border-b border-gray-200">
            {tabs.map((tab) => {
              const IconComponent = tab.icon;
              return (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex-1 flex items-center justify-center px-6 py-4 text-sm font-medium transition-colors ${
                    activeTab === tab.id
                      ? 'text-blue-600 border-b-2 border-blue-600 bg-blue-50'
                      : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50'
                  }`}
                >
                  <IconComponent className="w-5 h-5 mr-2" />
                  {tab.label}
                </button>
              );
            })}
          </div>

          {/* Tab Content */}
          <div className="p-8">
            {activeTab === 'overview' && (
              <div className="space-y-8">
                <div>
                  <h2 className="text-2xl font-bold text-gray-900 mb-6">Our Services</h2>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {userTypes.map((user, index) => {
                      const IconComponent = user.icon;
                      return (
                        <div key={index} className="bg-gray-50 border border-gray-200 rounded-xl p-6">
                          <div className="flex items-center mb-4">
                            <div className="p-3 bg-blue-100 rounded-lg mr-4">
                              <IconComponent className="w-6 h-6 text-blue-600" />
                            </div>
                            <div>
                              <h3 className="font-semibold text-gray-900">{user.type}</h3>
                              <p className="text-sm text-blue-600 font-medium">{user.need}</p>
                            </div>
                          </div>
                          <p className="text-gray-600 mb-4">{user.description}</p>
                          <div className="space-y-2">
                            {user.benefits.map((benefit, idx) => (
                              <div key={idx} className="flex items-center text-sm text-gray-700">
                                <CheckCircle className="w-4 h-4 text-green-500 mr-2 flex-shrink-0" />
                                <span>{benefit}</span>
                              </div>
                            ))}
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </div>

                <div>
                  <h2 className="text-2xl font-bold text-gray-900 mb-6">What Information Can Be Included?</h2>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {dossierSections.map((section, index) => {
                      const IconComponent = section.icon;
                      return (
                        <div key={index} className="bg-gray-50 border border-gray-200 rounded-xl p-6">
                          <div className="flex items-center mb-4">
                            <IconComponent className="w-6 h-6 text-blue-600 mr-3" />
                            <h3 className="text-lg font-semibold text-gray-900">{section.title}</h3>
                          </div>
                          <p className="text-gray-600 mb-3">{section.description}</p>
                          <div className="space-y-1">
                            {section.details.map((detail, idx) => (
                              <div key={idx} className="flex items-center text-xs text-gray-600">
                                <div className="w-1.5 h-1.5 bg-gray-400 rounded-full mr-2"></div>
                                <span>{detail}</span>
                              </div>
                            ))}
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </div>
              </div>
            )}

            {activeTab === 'generate' && (
              <div>
                {/* Progress Steps */}
                <div className="bg-gray-50 rounded-xl p-6 mb-8">
                  <div className="flex items-center justify-between mb-4">
                    <div className="flex items-center space-x-8">
                      <div className="flex items-center space-x-3">
                        <div className={`flex items-center justify-center w-10 h-10 rounded-full border-2 ${step >= 1 ? 'bg-blue-600 border-blue-600 text-white' : 'border-gray-300 text-gray-400'}`}>
                          1
                        </div>
                        <div className={`${step >= 1 ? 'text-blue-600' : 'text-gray-400'}`}>
                          <div className="font-medium">Select Property</div>
                          <div className="text-sm">Find your property</div>
                        </div>
                      </div>
                      
                      <div className={`w-16 h-1 ${step >= 2 ? 'bg-blue-600' : 'bg-gray-300'}`}></div>
                      
                      <div className="flex items-center space-x-3">
                        <div className={`flex items-center justify-center w-10 h-10 rounded-full border-2 ${step >= 2 ? 'bg-blue-600 border-blue-600 text-white' : 'border-gray-300 text-gray-400'}`}>
                          2
                        </div>
                        <div className={`${step >= 2 ? 'text-blue-600' : 'text-gray-400'}`}>
                          <div className="font-medium">Customize Report</div>
                          <div className="text-sm">Choose sections</div>
                        </div>
                      </div>
                      
                      <div className={`w-16 h-1 ${step >= 3 ? 'bg-blue-600' : 'bg-gray-300'}`}></div>
                      
                      <div className="flex items-center space-x-3">
                        <div className={`flex items-center justify-center w-10 h-10 rounded-full border-2 ${step >= 3 ? 'bg-blue-600 border-blue-600 text-white' : 'border-gray-300 text-gray-400'}`}>
                          3
                        </div>
                        <div className={`${step >= 3 ? 'text-blue-600' : 'text-gray-400'}`}>
                          <div className="font-medium">Download</div>
                          <div className="text-sm">Get your dossier</div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Step 1: Property Search */}
                {step === 1 && (
                  <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
                    <div className="text-center mb-8">
                      <h2 className="text-2xl font-semibold text-gray-900 mb-3">Step 1: Find Your Property</h2>
                      <p className="text-gray-600 text-lg">Search for the property you want to generate a dossier for</p>
                    </div>
                    
                    <div className="max-w-2xl mx-auto">
                      <div className="mb-6">
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Search by Property ID, Survey Number, or Address
                        </label>
                        <div className="relative">
                          <input
                            type="text"
                            placeholder="Enter Property ID"
                            value={searchTerm}
                            onChange={(e) => handleSearchChange(e.target.value)}
                            className="w-full px-4 py-4 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-lg"
                          />
                        </div>
                        <p className="text-sm text-gray-500 mt-2">
                          💡 Enter the exact Property ID to search
                        </p>
                      </div>

                      {/* Loading State */}
                      {isSearching && (
                        <div className="flex items-center justify-center py-8">
                          <Loader2 className="w-8 h-8 animate-spin text-blue-600 mr-3" />
                          <span className="text-gray-600">Searching property...</span>
                        </div>
                      )}

                      {/* Error State */}
                      {searchError && (
                        <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                          <div className="flex items-center text-red-800">
                            <AlertTriangle className="w-5 h-5 mr-2" />
                            <span>{searchError}</span>
                          </div>
                        </div>
                      )}

                      {/* Search Results */}
                      {searchTerm && !isSearching && searchResults.length > 0 && (
                        <div className="space-y-4">
                          <div className="flex items-center justify-between">
                            <h3 className="text-lg font-medium text-gray-900">Property Found</h3>
                            <span className="text-sm text-green-600 font-medium">✓ Verified</span>
                          </div>
                          
                          {searchResults.map((property) => (
                            <div key={property.id} className="space-y-4">
                              <div className="p-6 border-2 border-green-200 bg-green-50 rounded-xl">
                                <div className="flex items-center space-x-3 mb-3">
                                  <div className="font-bold text-lg text-gray-900">{property.id}</div>
                                  <span className="px-3 py-1 bg-blue-100 text-blue-800 text-sm font-medium rounded-full">
                                    {property.type}
                                  </span>
                                  <span className="px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full">
                                    ✓ Verified
                                  </span>
                                </div>
                                
                                <div className="space-y-2 mb-4">
                                  <div className="flex items-center text-gray-600">
                                    <MapPin className="w-4 h-4 mr-2" />
                                    <span>{property.address}</span>
                                  </div>
                                  <div className="flex items-center space-x-6 text-sm text-gray-500">
                                    <span><strong>Survey No:</strong> {property.surveyNo}</span>
                                    <span><strong>Area:</strong> {property.area}</span>
                                    <span><strong>Owner:</strong> {property.owner}</span>
                                  </div>
                                </div>
                              </div>
                              
                              {/* Select Property Button */}
                              <div className="text-center mt-6">
                                <button
                                  onClick={() => handlePropertySelect(property)}
                                  className="w-full max-w-md px-8 py-4 bg-gray-900 text-white rounded-lg hover:bg-gray-800 transition-colors flex items-center justify-center mx-auto text-lg font-semibold shadow-lg border-2 border-gray-900"
                                >
                                  Select Property & Continue
                                  <Eye className="w-5 h-5 ml-2" />
                                </button>
                              </div>
                            </div>
                          ))}
                        </div>
                      )}
                      
                      {/* No Results */}
                      {searchTerm && !isSearching && searchResults.length === 0 && !searchError && (
                        <div className="text-center py-12">
                          <div className="text-gray-400 mb-4">
                            <Search className="w-12 h-12 mx-auto" />
                          </div>
                          <p className="text-gray-500 text-lg">No property found with this ID.</p>
                          <p className="text-gray-400 text-sm mt-2">Please check the Property ID and try again.</p>
                        </div>
                      )}
                      
                      {!searchTerm && (
                        <div className="text-center py-12">
                          <div className="text-gray-300 mb-4">
                            <Search className="w-16 h-16 mx-auto" />
                          </div>
                          <p className="text-gray-500 text-lg">Enter Property ID to search</p>
                          <p className="text-gray-400 text-sm mt-2">Use the exact Property ID from your property documents</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Step 2: Customize Dossier */}
                {step === 2 && selectedProperty && (
                  <div className="space-y-8">
                    <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
                      <div className="text-center mb-6">
                        <h2 className="text-2xl font-semibold text-gray-900 mb-3">Step 2: Customize Your Dossier</h2>
                        <p className="text-gray-600 text-lg">Select the sections you want to include in your property report</p>
                      </div>
                      
                      <div className="bg-green-50 border-2 border-green-200 rounded-xl p-6 mb-8">
                        <div className="flex items-center mb-3">
                          <CheckCircle className="w-6 h-6 text-green-600 mr-3" />
                          <h3 className="text-lg font-semibold text-gray-900">Selected Property</h3>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div>
                            <div className="font-bold text-gray-900 text-lg">{selectedProperty.id}</div>
                            <div className="text-gray-600 mt-1">{selectedProperty.address}</div>
                          </div>
                          <div className="flex items-center space-x-4">
                            <span className="px-3 py-1 bg-blue-100 text-blue-800 text-sm font-medium rounded-full">
                              {selectedProperty.type}
                            </span>
                            <span className="text-sm text-gray-600"><strong>Area:</strong> {selectedProperty.area}</span>
                            <span className="text-sm text-gray-600"><strong>Owner:</strong> {selectedProperty.owner}</span>
                          </div>
                        </div>
                      </div>

                      {/* Section Selection */}
                      <div className="mb-8">
                        <h3 className="text-xl font-semibold text-gray-900 mb-6">Choose Report Sections</h3>
                        <div className="grid grid-cols-1 gap-6">
                          {sections.map((section) => {
                            const IconComponent = section.icon;
                            return (
                              <div
                                key={section.id}
                                onClick={() => handleSectionToggle(section.id)}
                                className={`p-6 border-2 rounded-xl cursor-pointer transition-all duration-200 ${
                                  section.included 
                                    ? 'border-blue-500 bg-blue-50 shadow-md' 
                                    : 'border-gray-200 hover:border-gray-300 hover:shadow-sm'
                                }`}
                              >
                                <div className="flex items-start space-x-4">
                                  <div className={`p-3 rounded-lg ${
                                    section.included ? 'bg-blue-100' : 'bg-gray-100'
                                  }`}>
                                    <IconComponent className={`w-6 h-6 ${
                                      section.included ? 'text-blue-600' : 'text-gray-600'
                                    }`} />
                                  </div>
                                  <div className="flex-1">
                                    <div className="flex items-center justify-between mb-2">
                                      <h4 className="font-semibold text-gray-900 text-lg">{section.title}</h4>
                                      {section.included && <CheckCircle className="w-6 h-6 text-blue-600" />}
                                    </div>
                                    <p className="text-gray-600 mb-3">{section.description}</p>
                                    
                                    <div className="mb-3">
                                      <h5 className="text-sm font-medium text-gray-700 mb-2">Includes:</h5>
                                      <div className="grid grid-cols-1 md:grid-cols-2 gap-1">
                                        {section.details.map((detail, idx) => (
                                          <div key={idx} className="flex items-center text-xs text-gray-600">
                                            <div className="w-1.5 h-1.5 bg-gray-400 rounded-full mr-2"></div>
                                            <span>{detail}</span>
                                          </div>
                                        ))}
                                      </div>
                                    </div>
                                    
                                    {section.id === 'identification' || section.id === 'ownership' ? (
                                      <span className="inline-block mt-2 px-3 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full">
                                        ✓ Always Included (Essential)
                                      </span>
                                    ) : (
                                      <span className={`inline-block mt-2 px-3 py-1 text-xs font-medium rounded-full ${
                                        section.included 
                                          ? 'bg-blue-100 text-blue-800' 
                                          : 'bg-gray-100 text-gray-600'
                                      }`}>
                                        {section.included ? '✓ Included in Report' : '+ Click to Include'}
                                      </span>
                                    )}
                                  </div>
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      </div>

                      <div className="flex items-center justify-between pt-8 border-t border-gray-200">
                        <div className="text-gray-600">
                          <div className="font-medium text-lg">{selectedSections.length} sections selected</div>
                          <div className="text-sm">Estimated size: {estimatePDFSize(selectedSections)}</div>
                        </div>
                        <div className="flex space-x-4">
                          <button
                            onClick={() => setStep(1)}
                            className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                          >
                            ← Back to Search
                          </button>
                          <button
                            onClick={handleGenerateDossier}
                            disabled={selectedSections.length === 0 || isGenerating}
                            className="px-8 py-3 bg-gray-900 text-white rounded-lg hover:bg-gray-800 disabled:opacity-50 flex items-center transition-colors border-2 border-gray-900 font-semibold"
                          >
                            {isGenerating ? (
                              <>
                                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-3"></div>
                                Generating Dossier...
                              </>
                            ) : (
                              <>
                                Generate Dossier
                                <FileText className="w-5 h-5 ml-2" />
                              </>
                            )}
                          </button>
                        </div>
                      </div>
                    </div>
                  </div>
                )}

                {/* Step 3: Download */}
                {step === 3 && (
                  <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
                    <div className="text-center mb-8">
                      <div className="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
                        <CheckCircle className="w-10 h-10 text-green-600" />
                      </div>
                      <h2 className="text-2xl font-semibold text-gray-900 mb-3">Step 3: Dossier Ready!</h2>
                      <p className="text-gray-600 text-lg">Your custom property dossier has been generated successfully</p>
                    </div>
                    
                    <div className="max-w-2xl mx-auto">
                      <div className="bg-gray-50 rounded-xl p-6 mb-8">
                        <h3 className="text-lg font-semibold text-gray-900 mb-4">Report Summary</h3>
                        <div className="space-y-3">
                          <div className="flex justify-between">
                            <span className="text-gray-600">Property:</span>
                            <span className="font-medium text-gray-900">{selectedProperty?.id}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Address:</span>
                            <span className="font-medium text-gray-900 text-right">{selectedProperty?.address}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Sections Included:</span>
                            <span className="font-medium text-gray-900">{selectedSections.length} sections</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Estimated Size:</span>
                            <span className="font-medium text-gray-900">{estimatePDFSize(selectedSections)}</span>
                          </div>
                          <div className="flex justify-between">
                            <span className="text-gray-600">Generated:</span>
                            <span className="font-medium text-gray-900">{new Date().toLocaleString()}</span>
                          </div>
                        </div>
                      </div>
                      
                      <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8">
                        <h4 className="font-semibold text-blue-900 mb-4">Included Sections & Details:</h4>
                        <div className="space-y-4">
                          {selectedSections.map((section) => (
                            <div key={section.id} className="bg-white rounded-lg p-4 border border-blue-100">
                              <div className="flex items-center text-blue-800 mb-2">
                                <CheckCircle className="w-4 h-4 mr-2" />
                                <span className="font-semibold">{section.title}</span>
                              </div>
                              <p className="text-blue-700 text-sm mb-2">{section.description}</p>
                              <div className="ml-6">
                                <p className="text-xs text-blue-600 font-medium mb-1">Includes:</p>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-1">
                                  {section.details.map((detail, idx) => (
                                    <div key={idx} className="flex items-center text-xs text-blue-600">
                                      <div className="w-1 h-1 bg-blue-400 rounded-full mr-2"></div>
                                      <span>{detail}</span>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          ))}
                        </div>
                      </div>

                      <div className="flex justify-center space-x-4">
                        <button
                          onClick={handleDownloadPDF}
                          className="px-8 py-4 bg-gray-900 text-white rounded-lg hover:bg-gray-800 flex items-center text-lg font-semibold transition-colors shadow-lg border-2 border-gray-900"
                        >
                          <Download className="w-5 h-5 mr-3" />
                          Download PDF Report
                        </button>
                        <button
                          onClick={() => {
                            setStep(1);
                            setSelectedProperty(null);
                            setSearchTerm('');
                            setSections(dossierSections);
                          }}
                          className="px-6 py-4 border-2 border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
                        >
                          Generate Another Dossier
                        </button>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            )}

            {activeTab === 'faq' && (
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-8">Frequently Asked Questions</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                  <div className="space-y-6">
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">How accurate is the information?</h3>
                      <p className="text-gray-600 text-sm">All information is sourced from official government records and verified through multiple databases for maximum accuracy.</p>
                    </div>
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">How long does it take to generate?</h3>
                      <p className="text-gray-600 text-sm">Most dossiers are generated within 2-5 minutes. Complex properties with extensive history may take up to 10 minutes.</p>
                    </div>
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">Can I share the report with banks?</h3>
                      <p className="text-gray-600 text-sm">Yes, our reports are formatted to meet banking standards and can be directly submitted for loan applications.</p>
                    </div>
                  </div>
                  <div className="space-y-6">
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">What if property information is not found?</h3>
                      <p className="text-gray-600 text-sm">We'll clearly indicate missing information and provide guidance on how to obtain it from relevant authorities.</p>
                    </div>
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">Is my data secure?</h3>
                      <p className="text-gray-600 text-sm">Yes, all searches and reports are encrypted and stored securely. We follow strict data protection protocols.</p>
                    </div>
                    <div className="bg-gray-50 rounded-lg p-6">
                      <h3 className="font-semibold text-gray-900 mb-2">Can I get updates if information changes?</h3>
                      <p className="text-gray-600 text-sm">Yes, you can subscribe to property alerts to get notified of any changes in property status or ownership.</p>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}