import React from 'react';
import { useNavigate } from 'react-router-dom';
import { Shield, FileCheck, Scale, Building2, AlertTriangle, CheckCircle2, Clock, Download, ArrowRight, Award, TrendingUp } from 'lucide-react';

const LegalHealthLanding: React.FC = () => {
  const navigate = useNavigate();

  const verificationSteps = [
    {
      icon: <FileCheck className="w-8 h-8" />,
      title: "Ownership Verification",
      description: "Verify seller identity, title deed authenticity, and ownership chain",
      duration: "1-2 days"
    },
    {
      icon: <FileCheck className="w-8 h-8" />,
      title: "Document Validation",
      description: "Check sale deed, tax receipts, Khata certificate, encumbrance certificate",
      duration: "2-3 days"
    },
    {
      icon: <Scale className="w-8 h-8" />,
      title: "Litigation Check",
      description: "Search court cases, RERA complaints, property disputes",
      duration: "3-5 days"
    },
    {
      icon: <AlertTriangle className="w-8 h-8" />,
      title: "Financial Check",
      description: "Verify existing loans, property tax dues, maintenance dues",
      duration: "5-6 days"
    },
    {
      icon: <Building2 className="w-8 h-8" />,
      title: "Zoning Compliance",
      description: "Check land use legality, BBMP/BDA approvals, master plan compliance",
      duration: "6-7 days"
    },
    {
      icon: <CheckCircle2 className="w-8 h-8" />,
      title: "Project Approvals",
      description: "Verify RERA registration, building plan, occupancy certificate",
      duration: "7-8 days"
    }
  ];

  const benefits = [
    { icon: <Shield className="w-6 h-6" />, text: "Avoid buying properties with legal issues" },
    { icon: <CheckCircle2 className="w-6 h-6" />, text: "Peace of mind before investing lakhs" },
    { icon: <Award className="w-6 h-6" />, text: "Professional verification report" },
    { icon: <TrendingUp className="w-6 h-6" />, text: "Can show to bank for loan approval" },
    { icon: <FileCheck className="w-6 h-6" />, text: "Comprehensive risk assessment" },
    { icon: <Scale className="w-6 h-6" />, text: "Expert legal team verification" }
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50">
      {/* Hero Section */}
      <div className="container mx-auto px-4 py-12">
        <div className="text-center mb-16">
          <div className="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-br from-blue-600 to-purple-600 rounded-full mb-4 shadow-lg">
            <Shield className="w-10 h-10 text-white" />
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Legal Health & Title Status Report
          </h1>
          <p className="text-lg text-gray-700 mb-8 max-w-3xl mx-auto">
            Get a comprehensive legal verification report before buying property. 
            Like a health checkup, but for properties!
          </p>
          <div className="flex flex-wrap items-center justify-center gap-4 mb-8">
            <div className="flex items-center gap-2 bg-white px-4 py-2 rounded-full shadow-md">
              <CheckCircle2 className="w-5 h-5 text-green-600" />
              <span className="font-semibold text-gray-900">6-Step Verification</span>
            </div>
            <div className="flex items-center gap-2 bg-white px-4 py-2 rounded-full shadow-md">
              <Clock className="w-5 h-5 text-blue-600" />
              <span className="font-semibold text-gray-900">7-10 Days Delivery</span>
            </div>
            <div className="flex items-center gap-2 bg-white px-4 py-2 rounded-full shadow-md">
              <Download className="w-5 h-5 text-purple-600" />
              <span className="font-semibold text-gray-900">PDF Certificate</span>
            </div>
          </div>
          <div className="flex flex-wrap items-center justify-center gap-4">
            <button
              onClick={() => navigate('/buyer/legal-health/create-report')}
              className="px-8 py-3 bg-gradient-to-r from-blue-600 to-purple-600 text-white rounded-lg font-semibold text-lg hover:shadow-xl transform hover:scale-105 transition-all flex items-center gap-2"
            >
              Order Report - ₹9,999
              <ArrowRight className="w-5 h-5" />
            </button>
            <button
              onClick={() => navigate('/buyer/legal-health/my-reports')}
              className="px-8 py-3 bg-white text-blue-600 border-2 border-blue-600 rounded-lg font-semibold text-lg hover:bg-blue-50 transition-all shadow-md"
            >
              My Reports
            </button>
          </div>
        </div>

        {/* Problem Statement */}
        <div className="bg-gradient-to-r from-red-50 to-orange-50 border-l-4 border-red-500 p-6 rounded-xl mb-16 max-w-5xl mx-auto shadow-md">
          <h3 className="text-2xl font-bold text-red-900 mb-4 flex items-center gap-2">
            <AlertTriangle className="w-6 h-6" />
            Real-World Problem We Solve:
          </h3>
          <p className="text-lg text-red-800 mb-4">
            You want to buy a flat/villa for ₹50 lakhs. But you're worried:
          </p>
          <div className="grid md:grid-cols-2 gap-3">
            <div className="flex items-start gap-2 bg-white p-3 rounded-lg">
              <AlertTriangle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
              <span className="text-gray-900">Is the seller the real owner?</span>
            </div>
            <div className="flex items-start gap-2 bg-white p-3 rounded-lg">
              <AlertTriangle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
              <span className="text-gray-900">Are there any court cases on this property?</span>
            </div>
            <div className="flex items-start gap-2 bg-white p-3 rounded-lg">
              <AlertTriangle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
              <span className="text-gray-900">Does the builder have proper approvals?</span>
            </div>
            <div className="flex items-start gap-2 bg-white p-3 rounded-lg">
              <AlertTriangle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
              <span className="text-gray-900">Are there any hidden loans/debts on the property?</span>
            </div>
          </div>
          <div className="mt-4 bg-green-100 border-l-4 border-green-600 p-3 rounded-lg">
            <p className="text-green-900 font-semibold flex items-center gap-2">
              <CheckCircle2 className="w-5 h-5" />
              This feature helps buyers verify ALL legal aspects before investing.
            </p>
          </div>
        </div>

        {/* 6-Step Verification Process */}
        <div className="mb-16">
          <h2 className="text-3xl font-bold text-center text-gray-900 mb-12">
            Our 6-Step Verification Process
          </h2>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {verificationSteps.map((step, index) => (
              <div key={index} className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-all border-t-4 border-blue-600">
                <div className="flex items-start gap-3 mb-3">
                  <div className="flex-shrink-0 w-12 h-12 bg-gradient-to-br from-blue-600 to-purple-600 rounded-lg flex items-center justify-center text-white">
                    {step.icon}
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-xs font-bold text-blue-600 bg-blue-50 px-2 py-1 rounded-full">Step {index + 1}</span>
                      <span className="text-xs text-gray-500">{step.duration}</span>
                    </div>
                    <h3 className="font-bold text-gray-900 mb-2">{step.title}</h3>
                    <p className="text-sm text-gray-600">{step.description}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Benefits */}
        <div className="bg-white rounded-2xl p-8 mb-16 shadow-lg border border-gray-100">
          <h2 className="text-3xl font-bold text-center mb-8 text-gray-900">Why Choose Our Legal Health Report?</h2>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {benefits.map((benefit, index) => (
              <div key={index} className="flex items-start gap-3 p-5 bg-gradient-to-br from-blue-50 to-purple-50 rounded-xl hover:shadow-md transition-all">
                <div className="flex-shrink-0 w-12 h-12 bg-blue-600 rounded-lg flex items-center justify-center text-white">
                  {benefit.icon}
                </div>
                <span className="text-gray-900 font-medium text-base">{benefit.text}</span>
              </div>
            ))}
          </div>
        </div>

        {/* How It Works */}
        <div className="mb-16">
          <h2 className="text-3xl font-bold text-center text-gray-900 mb-12">How It Works</h2>
          <div className="max-w-4xl mx-auto space-y-4">
            {[
              { step: 1, title: "Order Report", desc: "Select property and pay ₹9,999", color: "blue" },
              { step: 2, title: "Upload Documents", desc: "Submit property papers (sale deed, tax receipts, etc.)", color: "purple" },
              { step: 3, title: "Verification Process", desc: "Our legal team performs 6-step verification (7-10 days)", color: "indigo" },
              { step: 4, title: "Get Report", desc: "Receive detailed report with risk score and recommendations", color: "green" },
              { step: 5, title: "Make Decision", desc: "Proceed with purchase or negotiate based on findings", color: "orange" }
            ].map((item) => (
              <div key={item.step} className="flex items-start gap-4 bg-white rounded-lg p-5 shadow-md hover:shadow-lg transition-all">
                <div className={`flex-shrink-0 w-12 h-12 bg-gradient-to-br from-${item.color}-600 to-${item.color}-700 text-white rounded-lg flex items-center justify-center font-bold text-lg`}>
                  {item.step}
                </div>
                <div className="flex-1">
                  <h3 className="font-bold text-gray-900 text-lg mb-1">{item.title}</h3>
                  <p className="text-gray-600">{item.desc}</p>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* CTA Section */}
        <div className="bg-gradient-to-r from-gray-900 via-blue-900 to-purple-900 rounded-2xl p-10 text-center shadow-xl">
          <h2 className="text-3xl font-bold mb-4 text-gray-900">Ready to Verify Your Property?</h2>
          <p className="text-lg text-gray-900 mb-8 max-w-2xl mx-auto">
            Get peace of mind before investing lakhs in property
          </p>
          <button
            onClick={() => navigate('/buyer/legal-health/create-report')}
            className="px-10 py-4 bg-white text-gray-900 rounded-lg font-bold text-lg hover:bg-gray-100 transition-all inline-flex items-center gap-2 shadow-lg transform hover:scale-105"
          >
            Order Report - ₹9,999
            <ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </div>
    </div>
  );
};

export default LegalHealthLanding;
