import React, { useState, useEffect } from 'react';
import { 
  Calendar, Filter, Download, Search, Users, Clock, CheckCircle, AlertCircle, 
  BarChart3, PieChart, FileText, Globe, Phone, Mail, TrendingUp, MapPin,
  Eye, Edit, Trash2, Plus, RefreshCw
} from 'lucide-react';
import { 
  NRIRequest, 
  DashboardStats, 
  ChartData,
  getStatusBadgeClass,
  getPriorityBadgeClass,
  exportToCSV,
  formatDate,
  COUNTRIES,
  CATEGORIES,
  STATUSES,
  PRIORITIES,
  DASHBOARD_COLORS
} from '../../utils/nriDashboardUtils';

// Enhanced mock data
const mockData = {
  stats: {
    totalRequests: 1247,
    resolvedRequests: 892,
    pendingRequests: 355,
    avgResolutionTime: '4.2 days'
  },
  requests: [
    {
      id: 'NRI-2024-001',
      name: 'Rajesh Kumar',
      email: 'rajesh.kumar@email.com',
      phone: '+1-555-0123',
      country: 'USA',
      category: 'Property',
      status: 'Resolved' as const,
      priority: 'High' as const,
      officer: 'Priya Sharma',
      createdDate: '2024-01-15',
      resolutionTime: '3 days',
      description: 'Property investment guidance for Mumbai apartment purchase'
    },
    {
      id: 'NRI-2024-002',
      name: 'Anita Patel',
      email: 'anita.patel@email.com',
      phone: '+971-50-123-4567',
      country: 'UAE',
      category: 'Legal',
      status: 'In Progress' as const,
      priority: 'Medium' as const,
      officer: 'Amit Singh',
      createdDate: '2024-01-18',
      resolutionTime: '2 days',
      description: 'FEMA compliance guidance for property purchase'
    },
    {
      id: 'NRI-2024-003',
      name: 'Suresh Reddy',
      email: 'suresh.reddy@email.com',
      phone: '+44-20-1234-5678',
      country: 'UK',
      category: 'Documentation',
      status: 'New' as const,
      priority: 'Low' as const,
      officer: 'Neha Gupta',
      createdDate: '2024-01-20',
      resolutionTime: '-',
      description: 'POA documentation assistance required'
    },
    {
      id: 'NRI-2024-004',
      name: 'Vikram Singh',
      email: 'vikram.singh@email.com',
      phone: '+61-2-9876-5432',
      country: 'Australia',
      category: 'Property',
      status: 'In Progress' as const,
      priority: 'High' as const,
      officer: 'Ravi Kumar',
      createdDate: '2024-01-22',
      resolutionTime: '1 day',
      description: 'Property valuation and market analysis request'
    },
    {
      id: 'NRI-2024-005',
      name: 'Meera Joshi',
      email: 'meera.joshi@email.com',
      phone: '+1-647-555-0199',
      country: 'Canada',
      category: 'Banking',
      status: 'Resolved' as const,
      priority: 'Medium' as const,
      officer: 'Kiran Patel',
      createdDate: '2024-01-10',
      resolutionTime: '5 days',
      description: 'NRE account opening guidance and documentation'
    }
  ]
};

export function NRIAssistanceDashboard() {
  const [dateRange, setDateRange] = useState({ from: '2024-01-01', to: '2024-01-31' });
  const [filters, setFilters] = useState({
    status: 'All',
    category: 'All',
    country: 'All',
    priority: 'All'
  });
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [isLoading, setIsLoading] = useState(false);

  const handleViewRequest = (requestId) => {
    alert(`Viewing details for request: ${requestId}`);
  };

  const handleEditRequest = (requestId) => {
    alert(`Editing request: ${requestId}`);
  };

  const handleDeleteRequest = (requestId) => {
    if (confirm(`Are you sure you want to delete request ${requestId}?`)) {
      alert(`Request ${requestId} deleted successfully`);
    }
  };

  const handleCreateNewRequest = () => {
    alert('Opening new request form...');
  };

  const handleGenerateReport = async () => {
    setIsLoading(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 2000));
      alert('Report generated successfully! Check your downloads folder.');
    } catch (error) {
      alert('Error generating report. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleExportData = () => {
    exportToCSV(mockData.requests, `nri_assistance_report_${new Date().toISOString().split('T')[0]}.csv`);
  };

  const filteredRequests = mockData.requests.filter(request => {
    const matchesSearch = request.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         request.id.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         request.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = filters.status === 'All' || request.status === filters.status;
    const matchesCategory = filters.category === 'All' || request.category === filters.category;
    const matchesCountry = filters.country === 'All' || request.country === filters.country;
    const matchesPriority = filters.priority === 'All' || request.priority === filters.priority;
    
    return matchesSearch && matchesStatus && matchesCategory && matchesCountry && matchesPriority;
  });

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-7xl mx-auto px-4 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-2">NRI Assistance Desk Report</h1>
          <p className="text-gray-600">Monitor and analyze NRI assistance requests</p>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-8">
          <div className="flex flex-wrap items-center gap-4">
            <div className="flex items-center space-x-2">
              <Calendar className="w-5 h-5 text-gray-400" />
              <input
                type="date"
                value={dateRange.from}
                onChange={(e) => setDateRange({...dateRange, from: e.target.value})}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm"
              />
              <span className="text-gray-500">to</span>
              <input
                type="date"
                value={dateRange.to}
                onChange={(e) => setDateRange({...dateRange, to: e.target.value})}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm"
              />
            </div>
            <select
              value={filters.status}
              onChange={(e) => setFilters({...filters, status: e.target.value})}
              className="border border-gray-300 rounded-lg px-3 py-2 text-sm"
            >
              {STATUSES.map(status => (
                <option key={status} value={status}>{status} Status</option>
              ))}
            </select>
            <select
              value={filters.category}
              onChange={(e) => setFilters({...filters, category: e.target.value})}
              className="border border-gray-300 rounded-lg px-3 py-2 text-sm"
            >
              {CATEGORIES.map(category => (
                <option key={category} value={category}>{category} Category</option>
              ))}
            </select>
            <button 
              onClick={handleGenerateReport}
              disabled={isLoading}
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50 flex items-center"
            >
              {isLoading ? <RefreshCw className="w-4 h-4 mr-2 animate-spin" /> : null}
              Generate Report
            </button>
            <button 
              onClick={handleExportData}
              className="border border-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-50 flex items-center"
            >
              <Download className="w-4 h-4 mr-2" />
              Export
            </button>
          </div>
        </div>

        {/* Statistics Cards */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Total Requests</p>
                <p className="text-2xl font-bold text-gray-900 mt-1">{mockData.stats.totalRequests.toLocaleString()}</p>
              </div>
              <div className="bg-blue-100 p-3 rounded-lg">
                <FileText className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Resolved</p>
                <p className="text-2xl font-bold text-green-600 mt-1">{mockData.stats.resolvedRequests.toLocaleString()}</p>
              </div>
              <div className="bg-green-100 p-3 rounded-lg">
                <CheckCircle className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Pending</p>
                <p className="text-2xl font-bold text-orange-600 mt-1">{mockData.stats.pendingRequests.toLocaleString()}</p>
              </div>
              <div className="bg-orange-100 p-3 rounded-lg">
                <AlertCircle className="w-6 h-6 text-orange-600" />
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Avg Resolution</p>
                <p className="text-2xl font-bold text-purple-600 mt-1">{mockData.stats.avgResolutionTime}</p>
              </div>
              <div className="bg-purple-100 p-3 rounded-lg">
                <Clock className="w-6 h-6 text-purple-600" />
              </div>
            </div>
          </div>
        </div>

        {/* Requests Table */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <h3 className="text-lg font-semibold text-gray-900">NRI Assistance Requests</h3>
              <div className="flex items-center gap-4">
                <div className="relative">
                  <Search className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
                  <input
                    type="text"
                    placeholder="Search requests..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg text-sm w-64"
                  />
                </div>
                <button 
                  onClick={handleCreateNewRequest}
                  className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center text-sm"
                >
                  <Plus className="w-4 h-4 mr-2" />
                  New Request
                </button>
              </div>
            </div>
          </div>
          
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Request ID</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">NRI Details</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Category</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Officer</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredRequests.map((request, index) => (
                  <tr key={index} className="hover:bg-gray-50">
                    <td className="px-6 py-4">
                      <div className="text-sm font-medium text-blue-600">{request.id}</div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm font-medium text-gray-900">{request.name}</div>
                      <div className="text-sm text-gray-500 flex items-center">
                        <MapPin className="w-3 h-3 mr-1" />
                        {request.country}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">{request.category}</div>
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getPriorityBadgeClass(request.priority)}`}>
                        {request.priority}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusBadgeClass(request.status)}`}>
                        {request.status}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-900">
                      {request.officer}
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">{formatDate(request.createdDate)}</div>
                      <div className="text-xs text-gray-500">{request.resolutionTime}</div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-2">
                        <button 
                          onClick={() => handleViewRequest(request.id)}
                          className="text-blue-600 hover:text-blue-800" 
                          title="View Details"
                        >
                          <Eye className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleEditRequest(request.id)}
                          className="text-green-600 hover:text-green-800" 
                          title="Edit Request"
                        >
                          <Edit className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleDeleteRequest(request.id)}
                          className="text-red-600 hover:text-red-800" 
                          title="Delete Request"
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          
          <div className="px-6 py-4 border-t border-gray-200">
            <div className="flex items-center justify-between">
              <p className="text-sm text-gray-700">
                Showing {filteredRequests.length} of {mockData.requests.length} results
              </p>
              <div className="flex space-x-2">
                <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">
                  Previous
                </button>
                <button className="px-3 py-1 text-sm bg-blue-600 text-white rounded">
                  1
                </button>
                <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">
                  Next
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}