import React, { useState, useEffect } from 'react';
import { 
  Calendar, Filter, Download, Search, Users, Clock, CheckCircle, AlertCircle, 
  BarChart3, PieChart, FileText, Globe, Phone, Mail, TrendingUp, MapPin,
  Eye, Edit, Trash2, Plus, RefreshCw, Settings
} from 'lucide-react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart as RechartsPieChart, Cell } from 'recharts';

// Enhanced mock data
const mockData = {
  stats: {
    totalRequests: 1247,
    resolvedRequests: 892,
    pendingRequests: 355,
    avgResolutionTime: '4.2 days',
    monthlyGrowth: '+12.5%',
    satisfactionRate: '94.2%'
  },
  chartData: {
    categoryWise: [
      { category: 'Property', count: 456, percentage: 36.6, color: '#0056D2' },
      { category: 'Legal', count: 324, percentage: 26.0, color: '#10B981' },
      { category: 'Documentation', count: 287, percentage: 23.0, color: '#F59E0B' },
      { category: 'Other', count: 180, percentage: 14.4, color: '#EF4444' }
    ],
    statusWise: [
      { name: 'Resolved', value: 892, color: '#10B981' },
      { name: 'In Progress', value: 234, color: '#F59E0B' },
      { name: 'New', value: 121, color: '#0056D2' }
    ],
    monthlyTrend: [
      { month: 'Jan', requests: 98, resolved: 85 },
      { month: 'Feb', requests: 112, resolved: 95 },
      { month: 'Mar', requests: 125, resolved: 108 },
      { month: 'Apr', requests: 134, resolved: 118 },
      { month: 'May', requests: 145, resolved: 128 },
      { month: 'Jun', requests: 156, resolved: 142 }
    ]
  },
  requests: [
    {
      id: 'NRI-2024-001',
      name: 'Rajesh Kumar',
      email: 'rajesh.kumar@email.com',
      phone: '+1-555-0123',
      country: 'USA',
      category: 'Property',
      status: 'Resolved',
      priority: 'High',
      officer: 'Priya Sharma',
      createdDate: '2024-01-15',
      resolutionTime: '3 days',
      description: 'Property investment guidance for Mumbai apartment'
    },
    {
      id: 'NRI-2024-002',
      name: 'Anita Patel',
      email: 'anita.patel@email.com',
      phone: '+971-50-123-4567',
      country: 'UAE',
      category: 'Legal',
      status: 'In Progress',
      priority: 'Medium',
      officer: 'Amit Singh',
      createdDate: '2024-01-18',
      resolutionTime: '2 days',
      description: 'FEMA compliance for property purchase'
    },
    {
      id: 'NRI-2024-003',
      name: 'Suresh Reddy',
      email: 'suresh.reddy@email.com',
      phone: '+44-20-1234-5678',
      country: 'UK',
      category: 'Documentation',
      status: 'New',
      priority: 'Low',
      officer: 'Neha Gupta',
      createdDate: '2024-01-20',
      resolutionTime: '-',
      description: 'POA documentation assistance'
    },
    {
      id: 'NRI-2024-004',
      name: 'Vikram Singh',
      email: 'vikram.singh@email.com',
      phone: '+61-2-9876-5432',
      country: 'Australia',
      category: 'Property',
      status: 'In Progress',
      priority: 'High',
      officer: 'Ravi Kumar',
      createdDate: '2024-01-22',
      resolutionTime: '1 day',
      description: 'Property valuation and market analysis'
    }
  ]
};

const countries = ['All', 'USA', 'UAE', 'UK', 'Australia', 'Canada', 'Singapore'];
const categories = ['All', 'Property', 'Legal', 'Documentation', 'Other'];
const statuses = ['All', 'New', 'In Progress', 'Resolved'];

export function NRIAssistanceDashboard() {
  const [dateRange, setDateRange] = useState({ from: '2024-01-01', to: '2024-01-31' });
  const [filters, setFilters] = useState({
    status: 'All',
    category: 'All',
    country: 'All'
  });
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [selectedRequest, setSelectedRequest] = useState(null);
  const [isLoading, setIsLoading] = useState(false);

  const getStatusBadge = (status: string) => {
    const styles = {
      'New': 'bg-blue-100 text-blue-800 border-blue-200',
      'In Progress': 'bg-orange-100 text-orange-800 border-orange-200',
      'Resolved': 'bg-green-100 text-green-800 border-green-200'
    };
    return styles[status as keyof typeof styles] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const getPriorityBadge = (priority: string) => {
    const styles = {
      'High': 'bg-red-100 text-red-800 border-red-200',
      'Medium': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'Low': 'bg-gray-100 text-gray-800 border-gray-200'
    };
    return styles[priority as keyof typeof styles] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const handleGenerateReport = () => {
    setIsLoading(true);
    setTimeout(() => {
      setIsLoading(false);
      alert('Report generated successfully!');
    }, 2000);
  };

  const handleExportData = () => {
    const csvContent = "data:text/csv;charset=utf-8," 
      + "Request ID,NRI Name,Country,Category,Status,Officer,Created Date,Resolution Time\n"
      + mockData.requests.map(r => 
          `${r.id},${r.name},${r.country},${r.category},${r.status},${r.officer},${r.createdDate},${r.resolutionTime}`
        ).join("\n");
    
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "nri_assistance_report.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Sidebar */}
      <div className="fixed left-0 top-0 h-full w-64 bg-white shadow-lg border-r border-gray-200 z-40">
        <div className="p-6 border-b border-gray-200">
          <h2 className="text-xl font-bold text-gray-900">Admin Dashboard</h2>
          <p className="text-sm text-gray-600 mt-1">NRI Assistance Portal</p>
        </div>
        <nav className="mt-6">
          <div className="px-4 space-y-2">
            <a href="#" className="flex items-center px-4 py-3 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors">
              <BarChart3 className="w-5 h-5 mr-3" />
              Dashboard
            </a>
            <a href="#" className="flex items-center px-4 py-3 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors">
              <Users className="w-5 h-5 mr-3" />
              NRI Requests
            </a>
            <a href="#" className="flex items-center px-4 py-3 text-white bg-[#0056D2] rounded-lg shadow-sm">
              <FileText className="w-5 h-5 mr-3" />
              Assistance Desk Report
            </a>
            <a href="#" className="flex items-center px-4 py-3 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors">
              <Users className="w-5 h-5 mr-3" />
              Users
            </a>
            <a href="#" className="flex items-center px-4 py-3 text-gray-600 hover:bg-gray-50 rounded-lg transition-colors">
              <Settings className="w-5 h-5 mr-3" />
              Settings
            </a>
          </div>
        </nav>
      </div>

      {/* Main Content */}
      <div className="ml-64">
        {/* Header */}
        <div className="bg-white shadow-sm border-b border-gray-200 px-8 py-6">
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
            <div className="mb-4 lg:mb-0">
              <h1 className="text-2xl font-bold text-gray-900">NRI Assistance Desk Report</h1>
              <p className="text-gray-600 mt-1">Monitor and analyze NRI assistance requests across all regions</p>
            </div>
            <div className="flex flex-wrap items-center gap-4">
              <div className="flex items-center space-x-2">
                <Calendar className="w-5 h-5 text-gray-400" />
                <input
                  type="date"
                  value={dateRange.from}
                  onChange={(e) => setDateRange({...dateRange, from: e.target.value})}
                  className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                <span className="text-gray-500">to</span>
                <input
                  type="date"
                  value={dateRange.to}
                  onChange={(e) => setDateRange({...dateRange, to: e.target.value})}
                  className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <select
                value={filters.status}
                onChange={(e) => setFilters({...filters, status: e.target.value})}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                {statuses.map(status => (
                  <option key={status} value={status}>{status} Status</option>
                ))}
              </select>
              <select
                value={filters.category}
                onChange={(e) => setFilters({...filters, category: e.target.value})}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                {categories.map(category => (
                  <option key={category} value={category}>{category} {category !== 'All' ? 'Category' : 'Categories'}</option>
                ))}
              </select>
              <select
                value={filters.country}
                onChange={(e) => setFilters({...filters, country: e.target.value})}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                {countries.map(country => (
                  <option key={country} value={country}>{country} {country !== 'All' ? 'Country' : 'Countries'}</option>
                ))}
              </select>
              <button 
                onClick={handleGenerateReport}
                disabled={isLoading}
                className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center disabled:opacity-50"
              >
                {isLoading ? <RefreshCw className="w-4 h-4 mr-2 animate-spin" /> : null}
                Generate Report
              </button>
              <button 
                onClick={handleExportData}
                className="border border-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-50 transition-colors flex items-center"
              >
                <Download className="w-4 h-4 mr-2" />
                Export
              </button>
            </div>
          </div>
        </div>

        {/* Statistics Cards */}
        <div className="p-8">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Requests</p>
                  <p className="text-3xl font-bold text-gray-900 mt-2">{mockData.stats.totalRequests.toLocaleString()}</p>
                  <p className="text-sm text-green-600 mt-1 flex items-center">
                    <TrendingUp className="w-4 h-4 mr-1" />
                    {mockData.stats.monthlyGrowth} from last month
                  </p>
                </div>
                <div className="bg-blue-100 p-3 rounded-lg">
                  <FileText className="w-6 h-6 text-blue-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Resolved Requests</p>
                  <p className="text-3xl font-bold text-green-600 mt-2">{mockData.stats.resolvedRequests.toLocaleString()}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {((mockData.stats.resolvedRequests / mockData.stats.totalRequests) * 100).toFixed(1)}% resolution rate
                  </p>
                </div>
                <div className="bg-green-100 p-3 rounded-lg">
                  <CheckCircle className="w-6 h-6 text-green-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Pending Requests</p>
                  <p className="text-3xl font-bold text-orange-600 mt-2">{mockData.stats.pendingRequests.toLocaleString()}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {((mockData.stats.pendingRequests / mockData.stats.totalRequests) * 100).toFixed(1)}% pending
                  </p>
                </div>
                <div className="bg-orange-100 p-3 rounded-lg">
                  <AlertCircle className="w-6 h-6 text-orange-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Avg Resolution Time</p>
                  <p className="text-3xl font-bold text-purple-600 mt-2">{mockData.stats.avgResolutionTime}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {mockData.stats.satisfactionRate} satisfaction rate
                  </p>
                </div>
                <div className="bg-purple-100 p-3 rounded-lg">
                  <Clock className="w-6 h-6 text-purple-600" />
                </div>
              </div>
            </div>
          </div>

          {/* Charts Section */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            {/* Category-wise Bar Chart */}
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Category-wise Request Distribution</h3>
              <ResponsiveContainer width="100%" height={300}>
                <BarChart data={mockData.chartData.categoryWise}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="category" />
                  <YAxis />
                  <Tooltip />
                  <Bar dataKey="count" fill="#0056D2" radius={[4, 4, 0, 0]} />
                </BarChart>
              </ResponsiveContainer>
            </div>

            {/* Status-wise Pie Chart */}
            <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Status-wise Request Distribution</h3>
              <ResponsiveContainer width="100%" height={300}>
                <RechartsPieChart>
                  <Pie
                    data={mockData.chartData.statusWise}
                    cx="50%"
                    cy="50%"
                    outerRadius={80}
                    dataKey="value"
                    label={({name, percent}) => `${name} ${(percent * 100).toFixed(0)}%`}
                  >
                    {mockData.chartData.statusWise.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={entry.color} />
                    ))}
                  </Pie>
                  <Tooltip />
                </RechartsPieChart>
              </ResponsiveContainer>
            </div>
          </div>

          {/* Requests Table */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 sm:mb-0">Detailed NRI Assistance Requests</h3>
                <div className="flex items-center space-x-4">
                  <div className="relative">
                    <Search className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
                    <input
                      type="text"
                      placeholder="Search requests..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>
                  <button className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center text-sm">
                    <Plus className="w-4 h-4 mr-2" />
                    New Request
                  </button>
                </div>
              </div>
            </div>
            
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Request Details</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Contact Info</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Assigned Officer</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timeline</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {mockData.requests.map((request, index) => (
                    <tr key={index} className="hover:bg-gray-50 transition-colors">
                      <td className="px-6 py-4">
                        <div>
                          <div className="text-sm font-medium text-blue-600">{request.id}</div>
                          <div className="text-sm text-gray-900 font-medium">{request.name}</div>
                          <div className="text-xs text-gray-500 mt-1">{request.description}</div>
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900 flex items-center">
                          <MapPin className="w-4 h-4 mr-1 text-gray-400" />
                          {request.country}
                        </div>
                        <div className="text-xs text-gray-500 flex items-center mt-1">
                          <Mail className="w-3 h-3 mr-1" />
                          {request.email}
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">{request.category}</div>
                        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border mt-1 ${getPriorityBadge(request.priority)}`}>
                          {request.priority}
                        </span>
                      </td>
                      <td className="px-6 py-4">
                        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border ${getStatusBadge(request.status)}`}>
                          {request.status}
                        </span>
                      </td>
                      <td className="px-6 py-4 text-sm text-gray-900">
                        {request.officer}
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">{request.createdDate}</div>
                        <div className="text-xs text-gray-500">{request.resolutionTime}</div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="flex items-center space-x-2">
                          <button className="text-blue-600 hover:text-blue-800 transition-colors">
                            <Eye className="w-4 h-4" />
                          </button>
                          <button className="text-green-600 hover:text-green-800 transition-colors">
                            <Edit className="w-4 h-4" />
                          </button>
                          <button className="text-red-600 hover:text-red-800 transition-colors">
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            
            <div className="px-6 py-4 border-t border-gray-200">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                <p className="text-sm text-gray-700 mb-4 sm:mb-0">
                  Showing <span className="font-medium">1</span> to <span className="font-medium">4</span> of{' '}
                  <span className="font-medium">97</span> results
                </p>
                <div className="flex space-x-2">
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    Previous
                  </button>
                  <button className="px-3 py-1 text-sm bg-[#0056D2] text-white rounded hover:bg-blue-700 transition-colors">
                    1
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    2
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    3
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    Next
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}