import React, { useState, useEffect } from 'react';
import { 
  Calendar, Filter, Download, Search, Users, Clock, CheckCircle, AlertCircle, 
  BarChart3, PieChart, FileText, Globe, Phone, Mail, TrendingUp, MapPin,
  Eye, Edit, Trash2, Plus, RefreshCw, ArrowRight, Info, Target, UserCheck,
  Workflow, ClipboardList, Timer, Award, Shield, BookOpen, HelpCircle
} from 'lucide-react';
import { 
  NRIRequest, 
  DashboardStats, 
  ChartData,
  getStatusBadgeClass,
  getPriorityBadgeClass,
  exportToCSV,
  formatDate,
  COUNTRIES,
  CATEGORIES,
  STATUSES,
  PRIORITIES,
  DASHBOARD_COLORS
} from '../../utils/nriDashboardUtils';

// Enhanced mock data with more realistic information
const mockData = {
  stats: {
    totalRequests: 1247,
    resolvedRequests: 892,
    pendingRequests: 355,
    avgResolutionTime: '4.2 days',
    monthlyGrowth: '+12.5%',
    satisfactionRate: '94.2%'
  },
  chartData: {
    categoryWise: [
      { category: 'Property', count: 456, percentage: 36.6, color: DASHBOARD_COLORS.primary },
      { category: 'Legal', count: 324, percentage: 26.0, color: DASHBOARD_COLORS.success },
      { category: 'Documentation', count: 287, percentage: 23.0, color: DASHBOARD_COLORS.warning },
      { category: 'Banking', count: 120, percentage: 9.6, color: DASHBOARD_COLORS.info },
      { category: 'Other', count: 60, percentage: 4.8, color: DASHBOARD_COLORS.danger }
    ],
    statusWise: [
      { name: 'Resolved', value: 892, color: DASHBOARD_COLORS.success },
      { name: 'In Progress', value: 234, color: DASHBOARD_COLORS.warning },
      { name: 'New', value: 121, color: DASHBOARD_COLORS.primary }
    ],
    monthlyTrend: [
      { month: 'Jan', requests: 98, resolved: 85, pending: 13 },
      { month: 'Feb', requests: 112, resolved: 95, pending: 17 },
      { month: 'Mar', requests: 125, resolved: 108, pending: 17 },
      { month: 'Apr', requests: 134, resolved: 118, pending: 16 },
      { month: 'May', requests: 145, resolved: 128, pending: 17 },
      { month: 'Jun', requests: 156, resolved: 142, pending: 14 }
    ]
  },
  requests: [
    {
      id: 'NRI-2024-001',
      name: 'Rajesh Kumar',
      email: 'rajesh.kumar@email.com',
      phone: '+1-555-0123',
      country: 'USA',
      category: 'Property',
      status: 'Resolved' as const,
      priority: 'High' as const,
      officer: 'Priya Sharma',
      createdDate: '2024-01-15',
      resolutionTime: '3 days',
      description: 'Property investment guidance for Mumbai apartment purchase'
    },
    {
      id: 'NRI-2024-002',
      name: 'Anita Patel',
      email: 'anita.patel@email.com',
      phone: '+971-50-123-4567',
      country: 'UAE',
      category: 'Legal',
      status: 'In Progress' as const,
      priority: 'Medium' as const,
      officer: 'Amit Singh',
      createdDate: '2024-01-18',
      resolutionTime: '2 days',
      description: 'FEMA compliance guidance for property purchase'
    },
    {
      id: 'NRI-2024-003',
      name: 'Suresh Reddy',
      email: 'suresh.reddy@email.com',
      phone: '+44-20-1234-5678',
      country: 'UK',
      category: 'Documentation',
      status: 'New' as const,
      priority: 'Low' as const,
      officer: 'Neha Gupta',
      createdDate: '2024-01-20',
      resolutionTime: '-',
      description: 'POA documentation assistance required'
    },
    {
      id: 'NRI-2024-004',
      name: 'Vikram Singh',
      email: 'vikram.singh@email.com',
      phone: '+61-2-9876-5432',
      country: 'Australia',
      category: 'Property',
      status: 'In Progress' as const,
      priority: 'High' as const,
      officer: 'Ravi Kumar',
      createdDate: '2024-01-22',
      resolutionTime: '1 day',
      description: 'Property valuation and market analysis request'
    },
    {
      id: 'NRI-2024-005',
      name: 'Meera Joshi',
      email: 'meera.joshi@email.com',
      phone: '+1-647-555-0199',
      country: 'Canada',
      category: 'Banking',
      status: 'Resolved' as const,
      priority: 'Medium' as const,
      officer: 'Kiran Patel',
      createdDate: '2024-01-10',
      resolutionTime: '5 days',
      description: 'NRE account opening guidance and documentation'
    }
  ]
};

export function NRIAssistanceDashboard() {
  const [dateRange, setDateRange] = useState({ from: '2024-01-01', to: '2024-01-31' });
  const [filters, setFilters] = useState({
    status: 'All',
    category: 'All',
    country: 'All',
    priority: 'All'
  });
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [isLoading, setIsLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('overview');

  const requestCategories = [
    { name: 'Property Ownership Issues', count: 456, description: 'Property disputes, ownership verification, title issues' },
    { name: 'Legal Clarifications', count: 324, description: 'FEMA compliance, legal procedures, regulatory guidance' },
    { name: 'Document Verification', count: 287, description: 'POA authentication, document attestation, notarization' },
    { name: 'Tax & Payment Issues', count: 120, description: 'TDS queries, tax compliance, payment processing' },
    { name: 'General Guidance', count: 60, description: 'Investment advice, procedural guidance, complaints' }
  ];

  const stakeholders = [
    { role: 'NRI', responsibility: 'Raises assistance request with detailed issue description', icon: Globe },
    { role: 'Desk Officer', responsibility: 'Processes, investigates, and resolves assigned requests', icon: UserCheck },
    { role: 'Manager/Admin', responsibility: 'Monitors performance, evaluates service quality, allocates resources', icon: Award }
  ];

  const workflowSteps = [
    { step: 1, title: 'Request Initiation', description: 'NRI submits request with issue details', status: 'New', icon: ClipboardList },
    { step: 2, title: 'Request Classification', description: 'Categorized and assigned priority, routed to officer', status: 'Assigned', icon: Target },
    { step: 3, title: 'Request Handling', description: 'Officer reviews, investigates, may request additional info', status: 'In Progress', icon: Workflow },
    { step: 4, title: 'Resolution', description: 'Issue addressed, resolution documented with supporting evidence', status: 'Resolved', icon: CheckCircle },
    { step: 5, title: 'Closure', description: 'NRI notified, request formally closed with timestamps', status: 'Closed', icon: Shield }
  ];

  const handleViewRequest = (requestId) => {
    alert(`Viewing details for request: ${requestId}`);
  };

  const handleEditRequest = (requestId) => {
    alert(`Editing request: ${requestId}`);
  };

  const handleDeleteRequest = (requestId) => {
    if (confirm(`Are you sure you want to delete request ${requestId}?`)) {
      alert(`Request ${requestId} deleted successfully`);
    }
  };

  const handleCreateNewRequest = () => {
    alert('Opening new request form...');
  };

  const handleGenerateReport = async () => {
    setIsLoading(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 2000));
      alert('Report generated successfully! Check your downloads folder.');
    } catch (error) {
      alert('Error generating report. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleExportData = () => {
    exportToCSV(mockData.requests, `nri_assistance_report_${new Date().toISOString().split('T')[0]}.csv`);
  };

  const filteredRequests = mockData.requests.filter(request => {
    const matchesSearch = request.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         request.id.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         request.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = filters.status === 'All' || request.status === filters.status;
    const matchesCategory = filters.category === 'All' || request.category === filters.category;
    const matchesCountry = filters.country === 'All' || request.country === filters.country;
    const matchesPriority = filters.priority === 'All' || request.priority === filters.priority;
    
    return matchesSearch && matchesStatus && matchesCategory && matchesCountry && matchesPriority;
  });

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-4 lg:px-8">
        {/* Header */}
        <div className="bg-white shadow-sm border border-gray-200 rounded-lg px-4 lg:px-8 py-6 mb-6">
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
            <div className="mb-4 lg:mb-0">
              <h1 className="text-2xl font-bold text-gray-900">NRI Assistance Desk</h1>
              <p className="text-gray-600 mt-2">Comprehensive support system for Non-Resident Indians - Single point of contact for property, legal, documentation, and governance matters</p>
            </div>
            <div className="flex items-center space-x-4">
              <div className="bg-blue-100 p-3 rounded-lg">
                <Globe className="w-8 h-8 text-blue-600" />
              </div>
            </div>
          </div>
        </div>

        {/* Navigation Tabs */}
        <div className="bg-white shadow-sm border border-gray-200 rounded-lg mb-6">
          <div className="flex flex-wrap border-b border-gray-200">
            {[
              { id: 'overview', label: 'Overview & Purpose', icon: Info },
              { id: 'workflow', label: 'Functional Flow', icon: Workflow },
              { id: 'dashboard', label: 'Dashboard & Reports', icon: BarChart3 }
            ].map((tab) => {
              const Icon = tab.icon;
              return (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex items-center px-6 py-4 text-sm font-medium border-b-2 transition-colors ${
                    activeTab === tab.id
                      ? 'border-blue-500 text-blue-600 bg-blue-50'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  <Icon className="w-4 h-4 mr-2" />
                  {tab.label}
                </button>
              );
            })}
          </div>
        </div>

        {/* Tab Content */}
        {activeTab === 'overview' && (
          <div className="space-y-6">
            {/* What is NRI Assistance Desk */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <HelpCircle className="w-6 h-6 text-blue-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">What is the NRI Assistance Desk?</h2>
              </div>
              <div className="bg-blue-50 border-l-4 border-blue-400 p-4 mb-4">
                <p className="text-gray-700 leading-relaxed">
                  The NRI Assistance Desk is a service support mechanism created to help Non-Resident Indians (NRIs) resolve issues related to property, legal, documentation, and governance matters within India. It acts as a single point of contact between NRIs and the organization.
                </p>
              </div>
            </div>

            {/* Why this feature is needed */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <Target className="w-6 h-6 text-green-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Why this feature is needed</h2>
              </div>
              <div className="grid md:grid-cols-2 gap-6">
                <div className="bg-orange-50 border border-orange-200 rounded-lg p-4">
                  <h3 className="font-semibold text-orange-800 mb-3 flex items-center">
                    <Globe className="w-5 h-5 mr-2" />
                    NRI Challenges
                  </h3>
                  <ul className="space-y-2 text-gray-700">
                    <li>• Are physically away from India</li>
                    <li>• Face difficulty in follow-ups</li>
                    <li>• Need trusted support channels</li>
                    <li>• Time zone differences</li>
                  </ul>
                </div>
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                  <h3 className="font-semibold text-blue-800 mb-3 flex items-center">
                    <Award className="w-5 h-5 mr-2" />
                    Organization Benefits
                  </h3>
                  <ul className="space-y-2 text-gray-700">
                    <li>• Structured handling of NRI requests</li>
                    <li>• Transparency & accountability</li>
                    <li>• Performance tracking</li>
                    <li>• Resource optimization</li>
                  </ul>
                </div>
              </div>
              <div className="mt-4 p-4 bg-green-50 border border-green-200 rounded-lg">
                <p className="text-green-800 font-medium">
                  👉 The Assistance Desk ensures organized intake, tracking, resolution, and reporting of NRI requests.
                </p>
              </div>
            </div>

            {/* Types of issues handled */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <ClipboardList className="w-6 h-6 text-purple-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Types of Issues Handled</h2>
              </div>
              <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-4">
                {requestCategories.map((category, index) => (
                  <div key={index} className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow">
                    <div className="flex items-center justify-between mb-2">
                      <h3 className="font-semibold text-gray-900 text-sm">{category.name}</h3>
                      <span className="bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">{category.count}</span>
                    </div>
                    <p className="text-gray-600 text-sm">{category.description}</p>
                  </div>
                ))}
              </div>
            </div>

            {/* Stakeholders */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <Users className="w-6 h-6 text-indigo-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Stakeholders & Responsibilities</h2>
              </div>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-4 py-3 text-left text-sm font-medium text-gray-500">Role</th>
                      <th className="px-4 py-3 text-left text-sm font-medium text-gray-500">Responsibility</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-200">
                    {stakeholders.map((stakeholder, index) => {
                      const Icon = stakeholder.icon;
                      return (
                        <tr key={index} className="hover:bg-gray-50">
                          <td className="px-4 py-4">
                            <div className="flex items-center">
                              <Icon className="w-5 h-5 text-blue-600 mr-2" />
                              <span className="font-medium text-gray-900">{stakeholder.role}</span>
                            </div>
                          </td>
                          <td className="px-4 py-4 text-gray-700">{stakeholder.responsibility}</td>
                        </tr>
                      );
                    })}
                  </tbody>
                </table>
              </div>
            </div>
          </div>
        )}

        {/* Workflow Tab */}
        {activeTab === 'workflow' && (
          <div className="space-y-6">
            {/* End-to-End Functional Flow */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-6">
                <Workflow className="w-6 h-6 text-blue-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">End-to-End Functional Flow</h2>
              </div>
              <div className="space-y-6">
                {workflowSteps.map((step, index) => {
                  const Icon = step.icon;
                  return (
                    <div key={index} className="flex items-start space-x-4">
                      <div className="flex-shrink-0">
                        <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                          <span className="text-blue-600 font-semibold">{step.step}</span>
                        </div>
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center space-x-3 mb-2">
                          <Icon className="w-5 h-5 text-gray-600" />
                          <h3 className="font-semibold text-gray-900">{step.title}</h3>
                          <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                            step.status === 'New' ? 'bg-blue-100 text-blue-800' :
                            step.status === 'Assigned' ? 'bg-yellow-100 text-yellow-800' :
                            step.status === 'In Progress' ? 'bg-orange-100 text-orange-800' :
                            step.status === 'Resolved' ? 'bg-green-100 text-green-800' :
                            'bg-gray-100 text-gray-800'
                          }`}>
                            {step.status}
                          </span>
                        </div>
                        <p className="text-gray-600">{step.description}</p>
                        {index < workflowSteps.length - 1 && (
                          <div className="mt-4 ml-5">
                            <ArrowRight className="w-4 h-4 text-gray-400" />
                          </div>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Core Components */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <Target className="w-6 h-6 text-purple-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Core Components of the Feature</h2>
              </div>
              <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-4">
                {[
                  { name: 'Request Intake', description: 'Structured form for NRIs to submit assistance requests', icon: ClipboardList, color: 'blue' },
                  { name: 'Request Processing', description: 'Classification, prioritization, and officer assignment', icon: Workflow, color: 'green' },
                  { name: 'Status Tracking', description: 'Real-time status updates and progress monitoring', icon: Timer, color: 'orange' },
                  { name: 'Resolution & Closure', description: 'Issue resolution with documentation and closure', icon: CheckCircle, color: 'purple' },
                  { name: 'Reporting & Analysis', description: 'Comprehensive reports and performance analytics', icon: BarChart3, color: 'indigo' },
                  { name: 'Governance & Audit', description: 'Compliance monitoring and audit trail maintenance', icon: Shield, color: 'red' }
                ].map((component, index) => {
                  const Icon = component.icon;
                  return (
                    <div key={index} className={`border border-${component.color}-200 bg-${component.color}-50 rounded-lg p-4`}>
                      <div className="flex items-center mb-2">
                        <Icon className={`w-5 h-5 text-${component.color}-600 mr-2`} />
                        <h3 className={`font-semibold text-${component.color}-900`}>{component.name}</h3>
                      </div>
                      <p className={`text-${component.color}-700 text-sm`}>{component.description}</p>
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Information Captured */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <FileText className="w-6 h-6 text-green-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Information Captured for Reporting</h2>
              </div>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h3 className="font-semibold text-gray-900 mb-3">Request Data</h3>
                  <ul className="space-y-2 text-gray-700">
                    <li>• Request category and subcategory</li>
                    <li>• Priority level and urgency</li>
                    <li>• Country of NRI origin</li>
                    <li>• Contact information and preferences</li>
                  </ul>
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-3">Process Data</h3>
                  <ul className="space-y-2 text-gray-700">
                    <li>• Status transitions and timestamps</li>
                    <li>• Resolution duration and SLA compliance</li>
                    <li>• Officer handling and workload</li>
                    <li>• Time period and seasonal trends</li>
                  </ul>
                </div>
              </div>
            </div>
          </div>
        )}
        {/* Dashboard Tab */}
        {activeTab === 'dashboard' && (
          <div className="space-y-6">
            {/* Report Purpose */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <BarChart3 className="w-6 h-6 text-blue-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">What is the NRI Assistance Desk Report?</h2>
              </div>
              <div className="bg-blue-50 border-l-4 border-blue-400 p-4 mb-4">
                <p className="text-gray-700 leading-relaxed">
                  The NRI Assistance Desk Report is a consolidated summary of all assistance desk activities over a period of time. It provides comprehensive insights into request volume, resolution efficiency, and service quality metrics.
                </p>
              </div>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                  <h3 className="font-semibold text-green-800 mb-2">Key Questions Answered</h3>
                  <ul className="space-y-1 text-green-700 text-sm">
                    <li>• How many requests were received?</li>
                    <li>• What types of issues were raised?</li>
                    <li>• How many were resolved?</li>
                    <li>• How long did resolutions take?</li>
                  </ul>
                </div>
                <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
                  <h3 className="font-semibold text-purple-800 mb-2">Report Usage</h3>
                  <ul className="space-y-1 text-purple-700 text-sm">
                    <li>• Management reviews service quality</li>
                    <li>• Identify bottlenecks and issues</li>
                    <li>• Allocate resources effectively</li>
                    <li>• Improve policies and procedures</li>
                  </ul>
                </div>
              </div>
            </div>

            {/* Insights Provided */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <PieChart className="w-6 h-6 text-indigo-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Insights Provided by the Report</h2>
              </div>
              <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4">
                {[
                  {
                    title: 'Operational Insights',
                    icon: BarChart3,
                    color: 'blue',
                    metrics: ['Total requests vs resolved', 'Pending workload', 'SLA compliance rate', 'Service availability']
                  },
                  {
                    title: 'Demand Insights',
                    icon: Globe,
                    color: 'green',
                    metrics: ['Requests by country', 'Requests by category', 'Seasonal trends', 'Peak demand periods']
                  },
                  {
                    title: 'Efficiency Insights',
                    icon: Timer,
                    color: 'orange',
                    metrics: ['Average resolution time', 'Delayed cases', 'Process bottlenecks', 'Resource utilization']
                  },
                  {
                    title: 'Performance Insights',
                    icon: Award,
                    color: 'purple',
                    metrics: ['Requests per officer', 'Resolution rate', 'Quality scores', 'Customer satisfaction']
                  }
                ].map((insight, index) => {
                  const Icon = insight.icon;
                  return (
                    <div key={index} className="border border-gray-200 bg-gray-50 rounded-lg p-4">
                      <div className="flex items-center mb-3">
                        <Icon className="w-5 h-5 text-blue-600 mr-2" />
                        <h3 className="font-semibold text-gray-900 text-sm">{insight.title}</h3>
                      </div>
                      <ul className="space-y-1 text-gray-700 text-xs">
                        {insight.metrics.map((metric, idx) => (
                          <li key={idx}>• {metric}</li>
                        ))}
                      </ul>
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Governance & Accountability */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <Shield className="w-6 h-6 text-red-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Governance & Accountability</h2>
              </div>
              <div className="grid md:grid-cols-2 gap-6">
                <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                  <h3 className="font-semibold text-red-800 mb-3">Compliance & Audit</h3>
                  <ul className="space-y-2 text-red-700">
                    <li>• Transparency in operations</li>
                    <li>• Audit readiness and trail</li>
                    <li>• SLA monitoring and compliance</li>
                    <li>• Policy adherence tracking</li>
                  </ul>
                </div>
                <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-4">
                  <h3 className="font-semibold text-indigo-800 mb-3">Quality Assurance</h3>
                  <ul className="space-y-2 text-indigo-700">
                    <li>• Service quality metrics</li>
                    <li>• Customer satisfaction tracking</li>
                    <li>• Process improvement identification</li>
                    <li>• Best practice documentation</li>
                  </ul>
                </div>
              </div>
            </div>

            {/* Simple Explanation */}
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="flex items-center mb-4">
                <BookOpen className="w-6 h-6 text-green-600 mr-3" />
                <h2 className="text-xl font-semibold text-gray-900">Simple Explanation</h2>
              </div>
              <div className="bg-green-50 border-l-4 border-green-400 p-4">
                <p className="text-green-800 font-medium text-lg">
                  "The NRI Assistance Desk feature manages and tracks service requests raised by NRIs, while the Assistance Desk Report summarizes request volume, resolution status, and service efficiency to support monitoring and decision-making."
                </p>
              </div>
            </div>

            {/* Dashboard Controls */}
            <div className="bg-white shadow-sm border border-gray-200 rounded-lg px-4 lg:px-8 py-4">
              <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                <div className="mb-3 lg:mb-0">
                  <h3 className="text-lg font-semibold text-gray-900">Live Dashboard & Analytics</h3>
                  <p className="text-sm text-gray-600 mt-1">Monitor and analyze NRI assistance requests in real-time</p>
                </div>
                <div className="flex flex-wrap items-center gap-2 lg:gap-4">
                  <div className="flex items-center space-x-2">
                    <Calendar className="w-5 h-5 text-gray-400" />
                    <input
                      type="date"
                      value={dateRange.from}
                      onChange={(e) => setDateRange({...dateRange, from: e.target.value})}
                      className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                    <span className="text-gray-500 hidden sm:inline">to</span>
                    <input
                      type="date"
                      value={dateRange.to}
                      onChange={(e) => setDateRange({...dateRange, to: e.target.value})}
                      className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    />
                  </div>
                  <select
                    value={filters.status}
                    onChange={(e) => setFilters({...filters, status: e.target.value})}
                    className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  >
                    {STATUSES.map(status => (
                      <option key={status} value={status}>{status} Status</option>
                    ))}
                  </select>
                  <button 
                    onClick={handleGenerateReport}
                    disabled={isLoading}
                    className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center disabled:opacity-50 text-sm"
                  >
                    {isLoading ? <RefreshCw className="w-4 h-4 mr-2 animate-spin" /> : null}
                    Generate
                  </button>
                  <button 
                    onClick={handleExportData}
                    className="border border-gray-300 text-gray-700 px-4 py-2 rounded-lg hover:bg-gray-50 transition-colors flex items-center text-sm"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Export
                  </button>
                </div>
              </div>
            </div>

        <div className="p-4 lg:p-8">
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 lg:gap-6 mb-6 lg:mb-8">
            <div className="bg-white p-4 lg:p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Total Requests</p>
                  <p className="text-2xl lg:text-3xl font-bold text-gray-900 mt-2">{mockData.stats.totalRequests.toLocaleString()}</p>
                  <p className="text-sm text-green-600 mt-1 flex items-center">
                    <TrendingUp className="w-4 h-4 mr-1" />
                    {mockData.stats.monthlyGrowth} from last month
                  </p>
                </div>
                <div className="bg-blue-100 p-3 rounded-lg">
                  <FileText className="w-6 h-6 text-blue-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-4 lg:p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Resolved Requests</p>
                  <p className="text-2xl lg:text-3xl font-bold text-green-600 mt-2">{mockData.stats.resolvedRequests.toLocaleString()}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {((mockData.stats.resolvedRequests / mockData.stats.totalRequests) * 100).toFixed(1)}% resolution rate
                  </p>
                </div>
                <div className="bg-green-100 p-3 rounded-lg">
                  <CheckCircle className="w-6 h-6 text-green-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-4 lg:p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Pending Requests</p>
                  <p className="text-2xl lg:text-3xl font-bold text-orange-600 mt-2">{mockData.stats.pendingRequests.toLocaleString()}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {((mockData.stats.pendingRequests / mockData.stats.totalRequests) * 100).toFixed(1)}% pending
                  </p>
                </div>
                <div className="bg-orange-100 p-3 rounded-lg">
                  <AlertCircle className="w-6 h-6 text-orange-600" />
                </div>
              </div>
            </div>

            <div className="bg-white p-4 lg:p-6 rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600">Avg Resolution Time</p>
                  <p className="text-2xl lg:text-3xl font-bold text-purple-600 mt-2">{mockData.stats.avgResolutionTime}</p>
                  <p className="text-sm text-gray-600 mt-1">
                    {mockData.stats.satisfactionRate} satisfaction rate
                  </p>
                </div>
                <div className="bg-purple-100 p-3 rounded-lg">
                  <Clock className="w-6 h-6 text-purple-600" />
                </div>
              </div>
            </div>
          </div>

          {/* Requests Table */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-4 lg:p-6 border-b border-gray-200">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 sm:mb-0">Detailed NRI Assistance Requests</h3>
                <div className="flex flex-col sm:flex-row items-stretch sm:items-center space-y-2 sm:space-y-0 sm:space-x-4">
                  <div className="relative">
                    <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2 pointer-events-none" />
                    <input
                      type="text"
                      placeholder="Search requests..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent w-full sm:w-64 bg-white"
                    />
                  </div>
                  <button 
                    onClick={handleCreateNewRequest}
                    className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center justify-center text-sm"
                  >
                    <Plus className="w-4 h-4 mr-2" />
                    New Request
                  </button>
                </div>
              </div>
            </div>
            
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Request Details</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Contact Info</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Assigned Officer</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timeline</th>
                    <th className="px-4 lg:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {filteredRequests.map((request, index) => (
                    <tr key={index} className="hover:bg-gray-50 transition-colors">
                      <td className="px-4 lg:px-6 py-4">
                        <div>
                          <div className="text-sm font-medium text-blue-600">{request.id}</div>
                          <div className="text-sm text-gray-900 font-medium">{request.name}</div>
                          <div className="text-xs text-gray-500 mt-1 max-w-xs truncate">{request.description}</div>
                        </div>
                      </td>
                      <td className="px-4 lg:px-6 py-4">
                        <div className="text-sm text-gray-900 flex items-center">
                          <MapPin className="w-4 h-4 mr-1 text-gray-400" />
                          {request.country}
                        </div>
                        <div className="text-xs text-gray-500 flex items-center mt-1">
                          <Mail className="w-3 h-3 mr-1" />
                          <span className="truncate max-w-32">{request.email}</span>
                        </div>
                      </td>
                      <td className="px-4 lg:px-6 py-4">
                        <div className="text-sm text-gray-900">{request.category}</div>
                        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border mt-1 ${getPriorityBadgeClass(request.priority)}`}>
                          {request.priority}
                        </span>
                      </td>
                      <td className="px-4 lg:px-6 py-4">
                        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border ${getStatusBadgeClass(request.status)}`}>
                          {request.status}
                        </span>
                      </td>
                      <td className="px-4 lg:px-6 py-4 text-sm text-gray-900">
                        {request.officer}
                      </td>
                      <td className="px-4 lg:px-6 py-4">
                        <div className="text-sm text-gray-900">{formatDate(request.createdDate)}</div>
                        <div className="text-xs text-gray-500">{request.resolutionTime}</div>
                      </td>
                      <td className="px-4 lg:px-6 py-4">
                        <div className="flex items-center space-x-2">
                          <button 
                            onClick={() => handleViewRequest(request.id)}
                            className="text-blue-600 hover:text-blue-800 transition-colors" 
                            title="View Details"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => handleEditRequest(request.id)}
                            className="text-green-600 hover:text-green-800 transition-colors" 
                            title="Edit Request"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => handleDeleteRequest(request.id)}
                            className="text-red-600 hover:text-red-800 transition-colors" 
                            title="Delete Request"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            
            <div className="px-4 lg:px-6 py-4 border-t border-gray-200">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                <p className="text-sm text-gray-700 mb-4 sm:mb-0">
                  Showing <span className="font-medium">1</span> to <span className="font-medium">{filteredRequests.length}</span> of{' '}
                  <span className="font-medium">{mockData.requests.length}</span> results
                </p>
                <div className="flex space-x-2">
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    Previous
                  </button>
                  <button className="px-3 py-1 text-sm bg-[#0056D2] text-white rounded hover:bg-blue-700 transition-colors">
                    1
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    2
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    3
                  </button>
                  <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50 transition-colors">
                    Next
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
        )}
      </div>
    </div>
  );
}